/* pathjumpop.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2012-2019 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "pathjumpop.hh"
#include "listermode.h"
#include "pathjumpui.hh"
#include "nmspecialsourceext.hh"
#include "worker.h"
#include "worker_locale.h"
#include "fileentry.hh"
#include "argclass.hh"
#include "nwc_path.hh"
#include "virtualdirmode.hh"
#include "datei.h"
#include <aguix/awindow.h>
#include <aguix/text.h>
#include <aguix/button.h>
#include <aguix/cyclebutton.h>

const char *PathJumpOp::name = "PathJumpOp";

PathJumpOp::PathJumpOp() : FunctionProto()
{
    m_category = FunctionProto::CAT_CURSOR;
    hasConfigure = true;
}

PathJumpOp::~PathJumpOp()
{
}

PathJumpOp *PathJumpOp::duplicate() const
{
    PathJumpOp *ta = new PathJumpOp();
    ta->m_initial_tab = m_initial_tab;
    return ta;
}

bool PathJumpOp::isName(const char *str)
{
    if ( strcmp( str, name ) == 0 )
        return true;
    else
        return false;
}

const char *PathJumpOp::getName()
{
    return name;
}

int PathJumpOp::run( WPUContext *wpu, ActionMessage *msg )
{
    Lister *l1;
    ListerMode *lm1;
    
    l1 = msg->getWorker()->getActiveLister();
    if ( l1 == NULL )
        return 1;

    msg->getWorker()->setWaitCursor();

    PathJumpUI ui( *Worker::getAGUIX(),
                   msg->getWorker()->getPathStore(),
                   *msg->getWorker() );

    lm1 = l1->getActiveMode();
    if ( lm1 ) {
        std::list< NM_specialsourceExt > sellist;
        lm1->getSelFiles( sellist, ListerMode::LM_GETFILES_ONLYACTIVE );
            
        std::string dirname, basename;

        dirname = lm1->getCurrentDirectory();

        if ( sellist.empty() == false ) {
            if ( (*sellist.begin()).entry() != NULL ) {
                basename = (*sellist.begin()).entry()->name;
                dirname = NWC::Path::dirname( sellist.begin()->entry()->fullname );
            }
        }
            
        ui.setCurrentDirname( dirname );
        ui.setCurrentBasename( basename );
    }

    int res = ui.mainLoop( m_initial_tab == SHOW_BY_FILTER ? true : false );

    if ( res == 1 ) {
        std::string sel_path = ui.getSelectedPath();

        std::list< RefCount< ArgClass > > args;

        if ( lm1 ) {
            args.push_back( new StringArg( sel_path ) );
            lm1->runCommand( "enter_dir", args );

            // if ( highlight_entry.empty() == false ) {
            //     nm1->activateEntry( highlight_entry );
            // }
        }
    } else if ( res == 2 ) {
        std::unique_ptr< NWC::Dir > resdir = ui.getResultsAsDir();

        l1->switch2Mode( 0 );

        VirtualDirMode *vdm = dynamic_cast< VirtualDirMode* >( l1->getActiveMode() );
        if ( vdm != NULL ) {
            vdm->newTab();

            vdm->showDir( resdir );
        }
    }

    msg->getWorker()->unsetWaitCursor();
    return 0;
}

const char *PathJumpOp::getDescription()
{
    return catalog.getLocale( 1301 );
}

void PathJumpOp::setInitialTab( initial_tab_t v )
{
    m_initial_tab = v;
}

bool
PathJumpOp::save( Datei *fh )
{
    if ( fh == NULL ) return false;
    switch ( m_initial_tab ) {
        case SHOW_BY_FILTER:
            fh->configPutPair( "initialtab", "showbyfilter" );
            break;
        default:
            // don't save default value to be backward compatible
            break;
    }
    return true;
}

int
PathJumpOp::configure()
{
    AGUIX *aguix = Worker::getAGUIX();
    AWindow *win;
    CycleButton *rcyb;
    AGMessage *msg;
    int endmode = -1;

    auto title = AGUIXUtils::formatStringToString( catalog.getLocale( 293 ),
                                                   getDescription() );
    win = new AWindow( aguix, 10, 10, 10, 10, title.c_str(), AWindow::AWINDOW_DIALOG );
    win->create();
    
    AContainer *ac1 = win->setContainer( new AContainer( win, 1, 2 ), true );
    ac1->setMinSpace( 5 );
    ac1->setMaxSpace( 5 );
    
    AContainer *ac1_1 = ac1->add( new AContainer( win, 2, 1 ), 0, 0 );
    ac1_1->setMinSpace( 5 );
    ac1_1->setMaxSpace( 5 );
    ac1_1->setBorderWidth( 0 );

    ac1_1->add( new Text( aguix, 0, 0, catalog.getLocale( 1224 ) ), 0, 0, AContainer::CO_FIX );
    rcyb = (CycleButton*)ac1_1->add( new CycleButton( aguix, 0, 0, 100, 0 ), 1, 0, AContainer::CO_INCWNR );
    rcyb->addOption( catalog.getLocale( 1225 ) );
    rcyb->addOption( catalog.getLocale( 1226 ) );
    rcyb->resize( rcyb->getMaxSize(), rcyb->getHeight() );
    ac1_1->readLimits();
    switch ( m_initial_tab ) {
        case SHOW_BY_FILTER:
            rcyb->setOption( 1 );
            break;
        default:
            rcyb->setOption( 0 );
            break;
    }
    
    AContainer *ac1_2 = ac1->add( new AContainer( win, 2, 1 ), 0, 1 );
    ac1_2->setMinSpace( 5 );
    ac1_2->setMaxSpace( -1 );
    ac1_2->setBorderWidth( 0 );
    Button *okb =(Button*)ac1_2->add( new Button( aguix,
                                                  0,
                                                  0,
                                                  catalog.getLocale( 11 ),
                                                  0 ), 0, 0, AContainer::CO_FIX );
    Button *cb = (Button*)ac1_2->add( new Button( aguix,
                                                  0,
                                                  0,
                                                  catalog.getLocale( 8 ),
                                                  0 ), 1, 0, AContainer::CO_FIX );
    
    win->setDoTabCycling( true );
    win->contMaximize( true );
    win->show();

    for ( ; endmode == -1; ) {
        msg = aguix->WaitMessage( win );
        if ( msg != NULL ) {
            switch ( msg->type ) {
                case AG_CLOSEWINDOW:
                    if ( msg->closewindow.window == win->getWindow() ) endmode = 1;
                    break;
                case AG_BUTTONCLICKED:
                    if ( msg->button.button == okb ) endmode = 0;
                    else if ( msg->button.button == cb ) endmode = 1;
                    break;
            }
            aguix->ReplyMessage( msg );
        }
    }
    
    if ( endmode == 0 ) {
        // ok
        switch ( rcyb->getSelectedOption() ) {
            case 1:
                m_initial_tab = SHOW_BY_FILTER;
                break;
            default:
                m_initial_tab = SHOW_BY_TIME;
                break;
        }
    }
    
    delete win;
    
    return endmode;
}
