-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================
--
-- Rules of the Dictionary:
--
-- 1. Each subprogram callable from outside of Dictionary should have 2 versions:
--    - the version without underscore used for types conversion ONLY from/to Dictionary.Symbol.
--    - the version with underscore or terminated by "_Local" containing the actual implementation on the relevant type
--      (NOT Dictionary.Symbol)
--    Example:
--    function Foo_Bar (A_B : RawDict.A_B_Info_ref;
--                      Arg : T) return RawDict.C_D
--    --# global in Dict;
--    is
--       return [THE_ACTUAL_IMPLEMENTATION];
--    end Foo_Bar;
--
--    function FooBar (AB  : Symbol;
--                     Arg : T) return Symbol is
--    begin
--       return RawDict.Get_C_D_Symbol (Foo_Bar (A_B => RawDict.Get_A_B_Info_Ref (AB),
--                                               Arg => Arg));
--    end FooBar;
--
-- 2. For each type A_B in RawDict, they are 2 functions for types conversion:
--    - RawDict.Get_A_B_Info_Ref converts a Dictionary.Symbol into a RawDict.A_B_Info_Ref. This function may raise a
--      SystemErrors.Fatal_Error if the Symbol is not of the expected type.
--    - RawDict.Get_A_B_Symbol converts a RawDict.A_B_Info_Ref into a Dictionary.Symbol. This function succeeds all the time.
--
-- 3. If a types conversion RawDict.Get_A_B_Info_Ref is safe (protected by a case/if statement on the
--    RawDict.GetSymbolDiscriminant (Symbol), the named argument is used otherwise the positional argument is used.
--    Example:
--    case RawDict.GetSymbolDiscriminant (Symbol) is
--       when A_B_Symbol =>
--          R := Foo_Bar (A_B => RawDict.Get_A_B_Info_Ref (Item => Symbol),
--                        Arg => Arg);
--       when others =>
--          ...
--    end case;
--
-- 4. The subprogram callable from outside (the one without underscore or "_Local") should never be called from inside Dictionary.
--    When a subprogram from Dictionary has to call another subprogram from Dictionary, the one with underscore or "_Local" has to
--    be used. This version is safer (no risk of SystemErrors.Fatal_Error linked with types conversion) and faster (prevent 2
--    types conversion) because it is strongly typed.
--
-- 5. If a subprogram callable from outside may get different types of Dictionary.Symbol, a case statement will be used with the
--    following pattern:
--    function FooBar (Sym : Symbol) return T is
--       R : T;
--    begin
--       case RawDict.GetSymbolDiscriminant (Symb) is
--          when A_B_Symbol =>
--             R := F (A_B => RawDict.Get_A_B_Info_Ref (Item => Sym));
--          when C_D_Symbol =>
--             R := G (C_D => RawDict.Get_C_D_Info_Ref (Item => Sym));
--          when others =>
--             R := Null_T;
--             SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
--                                       Msg     => "in FooBar");
--       end case;
--       return R;
--    end FooBar;
--
--    The "when others =>" must raise a SystemErrors.Fatal_Error! This enforces to know what type of Dictionary.Symbol that is
--    in use.
--
-- 6. If a subprogram is ONLY called from outside of Dictionary, the subprogram is called purely external.
--    Each line containing types conversion on the arguments/return variable of the subprogram will be terminated by
--    "-- GAA External". This allows "grep" filtering only on relevant types conversion inside Dictionary.
--    Let assume that the function FooBar is never called from inside Dictionary, the function will be:
--
--    function FooBar (Sym : Symbol) return Symbol is
--       R : RawDict.C_D;
--    begin
--       case RawDict.GetSymbolDiscriminant (Symb) is
--          when A_B_Symbol =>
--             R := F (A_B => RawDict.Get_A_B_Info_Ref (Item => Sym)); -- GAA External
--          when C_D_Symbol =>
--             R := G (C_D => RawDict.Get_C_D_Info_Ref (Item => Sym)); -- GAA External
--          when others =>
--             R := Null_T;
--             SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
--                                       Msg     => "in FooBar");
--       end case;
--       return RawDict.Get_C_D_Symbol (R); -- GAA External
--    end FooBar;
--
-- 7. The case statement is preferred over the if/elsif/else statement.
--    Example:
--       case RawDict.GetSymbolDiscriminant (Symb) is
--          when A_B_Symbol =>
--             R := F (A_B => RawDict.Get_A_B_Info_Ref (Item => Sym));
--          when C_D_Symbol =>
--             R := G (C_D => RawDict.Get_C_D_Info_Ref (Item => Sym));
--          when others =>
--             R := Null_T;
--             SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
--                                       Msg     => "in FooBar");
--       end case;
--
--    is preferred over
--
--       if RawDict.GetSymbolDiscriminant (Symb) = A_B_Symbol then
--          R := F (A_B => RawDict.Get_A_B_Info_Ref (Item => Sym));
--       elsif RawDict.GetSymbolDiscriminant (Symb) = C_D_Symbol then
--          R := G (C_D => RawDict.Get_C_D_Info_Ref (Item => Sym));
--       else
--          R := Null_T;
--          SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
--                                    Msg     => "in FooBar");
--       end if;
--
--------------------------------------------------------------------------------

with CommandLineData;
with FileSystem;
with Maths;
with Statistics;
with SystemErrors;
with Ada.Containers.Vectors;

use type CommandLineData.Language_Profiles;
use type Maths.ErrorCode;

package body Dictionary is

   -----------------------------------------------------------------------------
   --                                   TYPES                                 --
   -----------------------------------------------------------------------------

   type SymbolDiscriminant is (
                               Null_Symbol,
                               Declaration_Symbol,
                               Enumeration_Literal_Symbol,
                               Array_Index_Symbol,
                               Record_Component_Symbol,
                               Subcomponent_Symbol,
                               Type_Symbol,
                               Constant_Symbol,
                               Variable_Symbol,
                               Own_Variable_Symbol,
                               Global_Variable_Symbol,
                               Quantified_Variable_Symbol,
                               Implicit_Return_Variable_Symbol,
                               Implicit_In_Stream_Symbol, -- notional volatile source for reads of protected vars
                               Rule_Policy_Symbol,
                               Constituent_Symbol,
                               Context_Clause_Symbol,
                               Use_Type_Clause_Symbol,
                               Parameter_Constraint_Symbol, -- index information for unconstrained formal parameters
                               Subprogram_Parameter_Symbol,
                               Subprogram_Symbol,
                               Operator_Symbol,
                               Dependency_Symbol,
                               Package_Symbol,
                               Generic_Parameter_Symbol,
                               Generic_Unit_Symbol,
                               Generic_Association_Symbol,
                               OwnTaskSymbol,
                               ImplicitProofFunctionSymbol,
                               SuspendsListItemSymbol,
                               InterruptStreamMappingSymbol,
                               VirtualElementSymbol,
                               LoopSymbol,
                               LoopParameterSymbol,
                               LoopEntryVariableSymbol,
                               ProtectedInfoSymbol,
                               TaskInfoSymbol,
                               KnownDiscriminantSymbol,
                               SubtypeSymbol, -- extra info on Task/Protected subtypes
                               DiscriminantConstraintSymbol);  -- constrains for above

   type Type_Discriminant is (
                              Unknown_Type_Item,
                              Enumeration_Type_Item,
                              Integer_Type_Item,
                              Modular_Type_Item,
                              Floating_Point_Type_Item,
                              Fixed_Point_Type_Item,
                              Array_Type_Item,
                              Record_Type_Item,
                              Abstract_Proof_Type_Item,
                              Protected_Type_Item,
                              Task_Type_Item,
                              Access_Type_Item,
                              Generic_Type_Item);

   type Generic_Type_Discriminant is (
                                      Invalid_Generic_Type,
                                      Generic_Private_Type,
                                      Generic_Discrete_Type,
                                      Generic_Integer_Type,
                                      Generic_Modular_Type,
                                      Generic_Floating_Point_Type,
                                      Generic_Fixed_Point_Type,
                                      Generic_Array_Type);

   type TriState is (Never, Sometimes, Always);

   -- RefType needs to be big enough to be suitable for Unchecked_Conversion
   -- from/to an access type, so...
   type Ref_Type is range -2 ** (ExaminerConstants.Address_Size - 1) .. 2 ** (ExaminerConstants.Address_Size - 1) - 1;

   -- If conducting proof, it may be useful to assert the base type of RefType.
   -- This base type will be platform dependent. On 32 bit systems the correct
   -- assertion may look like:
   -- --# assert RefType'Base is Integer;
   -- While on 64 bit systems the correct assertion may look like:
   -- --# assert RefType'Base is Long_Long_Integer;

   for Ref_Type'Size use ExaminerConstants.Address_Size;

   --# inherit ContextManager,
   --#         Dictionary,
   --#         LexTokenManager,
   --#         SystemErrors;
   package Dynamic_Symbol_Table is

      subtype Valid_Symbol is Dictionary.Symbol range 1 .. Dictionary.Symbol'Last;

      type T is private;

      procedure Initialize (The_Table : out T);
      --# derives The_Table from ;

      function Get_Current_Usage (The_Table : in T) return Natural;

      procedure Add_Symbol (The_Table    : in out T;
                            Discriminant : in     Dictionary.SymbolDiscriminant;
                            Ref          : in     Dictionary.Ref_Type;
                            Comp_Unit    : in     ContextManager.UnitDescriptors;
                            Loc          : in     LexTokenManager.Token_Position;
                            Item         :    out Dictionary.Symbol);
      --# derives Item      from The_Table &
      --#         The_Table from *,
      --#                        Comp_Unit,
      --#                        Discriminant,
      --#                        Loc,
      --#                        Ref;

      function Get_Symbol_Discriminant (The_Table : in T;
                                        Item      : in Dictionary.Symbol)
                                       return Dictionary.SymbolDiscriminant;

      function Get_Symbol_Ref (The_Table : in T;
                               Item      : in Dictionary.Symbol)
                              return Dictionary.Ref_Type;

      function Get_Symbol_Compilation_Unit (The_Table : in T;
                                            Item      : in Dictionary.Symbol)
                                           return ContextManager.UnitDescriptors;

      function Get_Symbol_Location (The_Table : in T;
                                    Item      : in Dictionary.Symbol)
                                   return LexTokenManager.Token_Position;

      procedure Set_Symbol_Location (The_Table : in out T;
                                     Item      : in     Dictionary.Symbol;
                                     Location  : in     LexTokenManager.Token_Position);
      --# derives The_Table from *,
      --#                        Item,
      --#                        Location;

      -- This type should mode in the private part of the package when "hide" is removed.
      type Symbol_Info is record
         Discriminant : Dictionary.SymbolDiscriminant;
         Comp_Unit    : ContextManager.UnitDescriptors;
         Loc          : LexTokenManager.Token_Position;
         Ref          : Dictionary.Ref_Type;
      end record;

   private
      --# hide Dynamic_Symbol_Table;

      package ST_Vec is new Ada.Containers.Vectors (Index_Type   => Valid_Symbol,
                                                    Element_Type => Symbol_Info);

      type T is record
         Vec : ST_Vec.Vector;
      end record;
   end Dynamic_Symbol_Table;

   --------------------------------------------------------------------------------
   --                              RAW DICTIONARY                                --
   --------------------------------------------------------------------------------

   --# inherit ContextManager,
   --#         Dictionary,
   --#         Dynamic_Symbol_Table,
   --#         ExaminerConstants,
   --#         LexTokenManager,
   --#         SP_Symbols;
   package RawDict is

      type Declaration_Info_Ref is private;

      Null_Declaration_Info_Ref : constant Declaration_Info_Ref;

      --------------------------------------------------------------------------------

      type Enumeration_Literal_Info_Ref is private;

      Null_Enumeration_Literal_Info_Ref : constant Enumeration_Literal_Info_Ref;

      --------------------------------------------------------------------------------

      type Array_Index_Info_Ref is private;

      Null_Array_Index_Info_Ref : constant Array_Index_Info_Ref;

      --------------------------------------------------------------------------------

      type Record_Component_Info_Ref is private;

      Null_Record_Component_Info_Ref : constant Record_Component_Info_Ref;

      --------------------------------------------------------------------------------

      type Subcomponent_Info_Ref is private;

      Null_Subcomponent_Info_Ref : constant Subcomponent_Info_Ref;

      --------------------------------------------------------------------------------

      type Type_Info_Ref is private;

      Null_Type_Info_Ref : constant Type_Info_Ref;

      --------------------------------------------------------------------------------

      type Constant_Info_Ref is private;

      Null_Constant_Info_Ref : constant Constant_Info_Ref;

      --------------------------------------------------------------------------------

      type Variable_Info_Ref is private;

      Null_Variable_Info_Ref : constant Variable_Info_Ref;

      --------------------------------------------------------------------------------

      type Own_Variable_Info_Ref is private;

      Null_Own_Variable_Info_Ref : constant Own_Variable_Info_Ref;

      --------------------------------------------------------------------------------

      type Global_Variable_Info_Ref is private;

      Null_Global_Variable_Info_Ref : constant Global_Variable_Info_Ref;

      --------------------------------------------------------------------------------

      type Quantified_Variable_Info_Ref is private;

      Null_Quantified_Variable_Info_Ref : constant Quantified_Variable_Info_Ref;

      --------------------------------------------------------------------------------

      type Implicit_Return_Variable_Info_Ref is private;

      Null_Implicit_Return_Variable_Info_Ref : constant Implicit_Return_Variable_Info_Ref;

      --------------------------------------------------------------------------------

      type Implicit_In_Stream_Info_Ref is private;

      Null_Implicit_In_Stream_Info_Ref : constant Implicit_In_Stream_Info_Ref;

      --------------------------------------------------------------------------------

      type Rule_Policy_Info_Ref is private;

      Null_Rule_Policy_Info_Ref : constant Rule_Policy_Info_Ref;

      --------------------------------------------------------------------------------

      type Constituent_Info_Ref is private;

      Null_Constituent_Info_Ref : constant Constituent_Info_Ref;

      --------------------------------------------------------------------------------

      type Context_Clause_Info_Ref is private;

      Null_Context_Clause_Info_Ref : constant Context_Clause_Info_Ref;

      --------------------------------------------------------------------------------

      type Use_Type_Clause_Info_Ref is private;

      Null_Use_Type_Clause_Info_Ref : constant Use_Type_Clause_Info_Ref;

      --------------------------------------------------------------------------------

      type Parameter_Constraint_Info_Ref is private;

      Null_Parameter_Constraint_Info_Ref : constant Parameter_Constraint_Info_Ref;

      --------------------------------------------------------------------------------

      type Subprogram_Parameter_Info_Ref is private;

      Null_Subprogram_Parameter_Info_Ref : constant Subprogram_Parameter_Info_Ref;

      --------------------------------------------------------------------------------

      type Subprogram_Info_Ref is private;

      Null_Subprogram_Info_Ref : constant Subprogram_Info_Ref;

      --------------------------------------------------------------------------------

      type Operator_Info_Ref is private;

      Null_Operator_Info_Ref : constant Operator_Info_Ref;

      --------------------------------------------------------------------------------

      type Dependency_Info_Ref is private;

      Null_Dependency_Info_Ref : constant Dependency_Info_Ref;

      --------------------------------------------------------------------------------

      type Package_Info_Ref is private;

      Null_Package_Info_Ref : constant Package_Info_Ref;

      --------------------------------------------------------------------------------

      type Generic_Parameter_Info_Ref is private;

      Null_Generic_Parameter_Info_Ref : constant Generic_Parameter_Info_Ref;

      --------------------------------------------------------------------------------

      type Generic_Unit_Info_Ref is private;

      Null_Generic_Unit_Info_Ref : constant Generic_Unit_Info_Ref;

      --------------------------------------------------------------------------------

      type Generic_Association_Info_Ref is private;

      Null_Generic_Association_Info_Ref : constant Generic_Association_Info_Ref;

      --------------------------------------------------------------------------------

      function GetSymbolDiscriminant (Item : Dictionary.Symbol) return Dictionary.SymbolDiscriminant;
      --# global in Dictionary.Dict;

      function Get_Symbol_Compilation_Unit (Item : Dictionary.Symbol) return ContextManager.UnitDescriptors;
      --# global in Dictionary.Dict;

      function Get_Symbol_Location (Item : Dictionary.Symbol) return LexTokenManager.Token_Position;
      --# global in Dictionary.Dict;

      procedure Set_Symbol_Location (Item     : in Dictionary.Symbol;
                                     Location : in LexTokenManager.Token_Position);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Item,
      --#                              Location;

      --------------------------------------------------------------------------------
      --  Declaration_Info
      --------------------------------------------------------------------------------

      function Get_Declaration_Info_Ref (Item : Dictionary.Symbol) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Declaration_Info_Ref);

      procedure Create_Declaration
        (Context         : in     Dictionary.Contexts;
         Scope           : in     Dictionary.Scopes;
         Comp_Unit       : in     ContextManager.UnitDescriptors;
         Loc             : in     LexTokenManager.Token_Position;
         The_Declaration :    out Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Declaration from Comp_Unit,
      --#                              Context,
      --#                              Dictionary.Dict,
      --#                              Loc,
      --#                              Scope;
      --# post The_Declaration /= Null_Declaration_Info_Ref;

      procedure Set_Declaration_Context (The_Declaration : in Declaration_Info_Ref;
                                         Context         : in Dictionary.Contexts);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Context,
      --#                              The_Declaration;

      procedure Set_Declaration_Item (The_Declaration : in Declaration_Info_Ref;
                                      Item            : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Item,
      --#                              The_Declaration;

      procedure Set_Next_Declaration (The_Declaration, Next : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              The_Declaration;

      function Get_Declaration_Symbol (The_Declaration : Declaration_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Declaration = Null_Declaration_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Declaration_Symbol));

      function Get_Declaration_Context (The_Declaration : Declaration_Info_Ref) return Dictionary.Contexts;
      --# global in Dictionary.Dict;

      function Get_Declaration_Scope (The_Declaration : Declaration_Info_Ref) return Dictionary.Scopes;
      --# global in Dictionary.Dict;

      function Get_Declaration_Item (The_Declaration : Declaration_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Next_Declaration (The_Declaration : Declaration_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Enumeration_Literal_Info
      --------------------------------------------------------------------------------

      function Get_Enumeration_Literal_Info_Ref (Item : Dictionary.Symbol) return Enumeration_Literal_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Enumeration_Literal_Info_Ref);

      procedure Create_Enumeration_Literal
        (Name                    : in     LexTokenManager.Lex_String;
         Position                : in     LexTokenManager.Lex_String;
         Enumeration_Type        : in     Type_Info_Ref;
         Comp_Unit               : in     ContextManager.UnitDescriptors;
         Loc                     : in     LexTokenManager.Token_Position;
         The_Enumeration_Literal :    out Enumeration_Literal_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Enumeration_Literal from Comp_Unit,
      --#                                      Dictionary.Dict,
      --#                                      Enumeration_Type,
      --#                                      Loc,
      --#                                      Name,
      --#                                      Position;
      --# post The_Enumeration_Literal /= Null_Enumeration_Literal_Info_Ref;

      procedure Set_Next_Enumeration_Literal (The_Enumeration_Literal, Next : in Enumeration_Literal_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              The_Enumeration_Literal;

      function Get_Enumeration_Literal_Symbol (The_Enumeration_Literal : Enumeration_Literal_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Enumeration_Literal = Null_Enumeration_Literal_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Enumeration_Literal_Symbol));

      function Get_Enumeration_Literal_Name
        (The_Enumeration_Literal : Enumeration_Literal_Info_Ref)
        return                    LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Enumeration_Literal_Position
        (The_Enumeration_Literal : Enumeration_Literal_Info_Ref)
        return                    LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Enumeration_Literal_Type (The_Enumeration_Literal : Enumeration_Literal_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Next_Enumeration_Literal
        (The_Enumeration_Literal : Enumeration_Literal_Info_Ref)
        return                    Enumeration_Literal_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Array_Index_Info
      --------------------------------------------------------------------------------

      function Get_Array_Index_Info_Ref (Item : Dictionary.Symbol) return Array_Index_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Array_Index_Info_Ref);

      procedure Create_Array_Index
        (Index_Type      : in     Type_Info_Ref;
         Comp_Unit       : in     ContextManager.UnitDescriptors;
         Loc             : in     LexTokenManager.Token_Position;
         The_Array_Index :    out Array_Index_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Array_Index from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Index_Type,
      --#                              Loc;
      --# post The_Array_Index /= Null_Array_Index_Info_Ref;

      procedure Set_Next_Array_Index (The_Array_Index, Next : in Array_Index_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              The_Array_Index;

      function Get_Array_Index_Symbol (The_Array_Index : Array_Index_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Array_Index = Null_Array_Index_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Array_Index_Symbol));

      function Get_Array_Index_Type (The_Array_Index : Array_Index_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Next_Array_Index (The_Array_Index : Array_Index_Info_Ref) return Array_Index_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Record_Component_Info
      --------------------------------------------------------------------------------

      function Get_Record_Component_Info_Ref (Item : Dictionary.Symbol) return Record_Component_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Record_Component_Info_Ref);

      procedure Create_Record_Component
        (Name                 : in     LexTokenManager.Lex_String;
         Record_Type          : in     Type_Info_Ref;
         Component_Type       : in     Type_Info_Ref;
         Inherited_Field      : in     Boolean;
         Comp_Unit            : in     ContextManager.UnitDescriptors;
         Loc                  : in     LexTokenManager.Token_Position;
         The_Record_Component :    out Record_Component_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Record_Component from Component_Type,
      --#                                   Comp_Unit,
      --#                                   Dictionary.Dict,
      --#                                   Inherited_Field,
      --#                                   Loc,
      --#                                   Name,
      --#                                   Record_Type;
      --# post The_Record_Component /= Null_Record_Component_Info_Ref;

      procedure Set_Next_Record_Component (The_Record_Component, Next : in Record_Component_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              The_Record_Component;

      function Get_Record_Component_Symbol (The_Record_Component : Record_Component_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Record_Component = Null_Record_Component_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Record_Component_Symbol));

      function Get_Record_Component_Name (The_Record_Component : Record_Component_Info_Ref) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Record_Component_Record_Type (The_Record_Component : Record_Component_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Record_Component_Type (The_Record_Component : Record_Component_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Record_Component_Inherited_Field (The_Record_Component : Record_Component_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Next_Record_Component (The_Record_Component : Record_Component_Info_Ref) return Record_Component_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Subcomponent_Info
      --------------------------------------------------------------------------------

      function Get_Subcomponent_Info_Ref (Item : Dictionary.Symbol) return Subcomponent_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Subcomponent_Info_Ref);

      procedure Create_Subcomponent
        (Object           : in     Dictionary.Symbol;
         Record_Component : in     Record_Component_Info_Ref;
         Marked_Valid     : in     Boolean;
         Comp_Unit        : in     ContextManager.UnitDescriptors;
         Loc              : in     LexTokenManager.Token_Position;
         The_Subcomponent :    out Subcomponent_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Subcomponent from Comp_Unit,
      --#                               Dictionary.Dict,
      --#                               Loc,
      --#                               Marked_Valid,
      --#                               Object,
      --#                               Record_Component;
      --# post The_Subcomponent /= Null_Subcomponent_Info_Ref;

      procedure Set_Subcomponent_Subcomponents (The_Subcomponent : in Subcomponent_Info_Ref;
                                                Sibling          : in Subcomponent_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Sibling,
      --#                              The_Subcomponent;

      procedure Set_Subcomponent_Marked_Valid (The_Subcomponent : in Subcomponent_Info_Ref;
                                               Marked_Valid     : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Marked_Valid,
      --#                              The_Subcomponent;

      procedure Set_Next_Subcomponent (The_Subcomponent, Next : in Subcomponent_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              The_Subcomponent;

      function Get_Subcomponent_Symbol (The_Subcomponent : Subcomponent_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Subcomponent = Null_Subcomponent_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Subcomponent_Symbol));

      function Get_Subcomponent_Object (The_Subcomponent : Subcomponent_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Subcomponent_Record_Component (The_Subcomponent : Subcomponent_Info_Ref) return Record_Component_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subcomponent_Subcomponents (The_Subcomponent : Subcomponent_Info_Ref) return Subcomponent_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subcomponent_Marked_Valid (The_Subcomponent : Subcomponent_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Next_Subcomponent (The_Subcomponent : Subcomponent_Info_Ref) return Subcomponent_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Type_Info
      --------------------------------------------------------------------------------

      function Get_Type_Info_Ref (Item : Dictionary.Symbol) return Type_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Type_Info_Ref);

      procedure Create_Type
        (Name            : in     LexTokenManager.Lex_String;
         The_Declaration : in     Declaration_Info_Ref;
         Is_Private      : in     Boolean;
         Is_Announcement : in     Boolean;
         Comp_Unit       : in     ContextManager.UnitDescriptors;
         Loc             : in     LexTokenManager.Token_Position;
         Type_Mark       :    out Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         Type_Mark       from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Is_Announcement,
      --#                              Is_Private,
      --#                              Loc,
      --#                              Name,
      --#                              The_Declaration;
      --# post Type_Mark /= Null_Type_Info_Ref;

      procedure Set_Type_Parent (Type_Mark : in Type_Info_Ref;
                                 Parent    : in Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Parent,
      --#                              Type_Mark;

      procedure Set_Type_Declaration (Type_Mark       : in Type_Info_Ref;
                                      The_Declaration : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              Type_Mark;

      procedure Set_Type_Is_Full_Range_Subtype (Type_Mark : in Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Type_Mark;

      procedure Set_Type_Discriminant (Type_Mark    : in Type_Info_Ref;
                                       Discriminant : in Dictionary.Type_Discriminant);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Discriminant,
      --#                              Type_Mark;

      procedure Set_Type_Private (Type_Mark  : in Type_Info_Ref;
                                  Is_Private : in Dictionary.TriState);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Is_Private,
      --#                              Type_Mark;

      procedure Set_Type_Limited (Type_Mark  : in Type_Info_Ref;
                                  Is_Limited : in Dictionary.TriState);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Is_Limited,
      --#                              Type_Mark;

      procedure Set_Type_Is_Tagged (Type_Mark : in Type_Info_Ref;
                                    Is_Tagged : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Is_Tagged,
      --#                              Type_Mark;

      procedure Set_Type_Is_Own_Var_Type (Type_Mark : in Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Type_Mark;

      procedure Set_Type_Extends (Type_Mark : in Type_Info_Ref;
                                  Root_Type : in Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Root_Type,
      --#                              Type_Mark;

      procedure Set_Type_Accesses (Type_Mark  : in Type_Info_Ref;
                                   The_Access : in Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Access,
      --#                              Type_Mark;

      procedure Set_Type_Limited_Private (Type_Mark : in Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Type_Mark;

      procedure Set_Type_Derived (Type_Mark  : in Type_Info_Ref;
                                  Is_Derived : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Is_Derived,
      --#                              Type_Mark;

      procedure Set_Type_Equality_Defined (Type_Mark        : in Type_Info_Ref;
                                           Equality_Defined : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Equality_Defined,
      --#                              Type_Mark;

      procedure Set_Type_Contains_Float (Type_Mark      : in Type_Info_Ref;
                                         Contains_Float : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Contains_Float,
      --#                              Type_Mark;

      procedure Set_Type_Constrained (Type_Mark   : in Type_Info_Ref;
                                      Constrained : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Constrained,
      --#                              Type_Mark;

      procedure Set_Type_Static (Type_Mark : in Type_Info_Ref;
                                 Static    : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Static,
      --#                              Type_Mark;

      procedure Set_Type_Wellformed (Type_Mark  : in Type_Info_Ref;
                                     Wellformed : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Type_Mark,
      --#                              Wellformed;

      procedure Set_Type_Lower (Type_Mark : in Type_Info_Ref;
                                Lower     : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Lower,
      --#                              Type_Mark;

      procedure Set_Type_Upper (Type_Mark : in Type_Info_Ref;
                                Upper     : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Type_Mark,
      --#                              Upper;

      procedure Set_Type_Modulus (Type_Mark : in Type_Info_Ref;
                                  Modulus   : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Modulus,
      --#                              Type_Mark;

      procedure Set_Type_Error_Bound (Type_Mark   : in Type_Info_Ref;
                                      Error_Bound : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Error_Bound,
      --#                              Type_Mark;

      procedure Set_Type_Base_Type (Type_Mark : in Type_Info_Ref;
                                    Base_Type : in Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Base_Type,
      --#                              Type_Mark;

      procedure Set_Type_First_Enumeration_Literal
        (Type_Mark           : in Type_Info_Ref;
         Enumeration_Literal : in Enumeration_Literal_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Enumeration_Literal,
      --#                              Type_Mark;

      procedure Set_Type_Last_Enumeration_Literal
        (Type_Mark           : in Type_Info_Ref;
         Enumeration_Literal : in Enumeration_Literal_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Enumeration_Literal,
      --#                              Type_Mark;

      procedure Set_Type_First_Array_Index (Type_Mark   : in Type_Info_Ref;
                                            Array_Index : in Array_Index_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Array_Index,
      --#                              Type_Mark;

      procedure Set_Type_Last_Array_Index (Type_Mark   : in Type_Info_Ref;
                                           Array_Index : in Array_Index_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Array_Index,
      --#                              Type_Mark;

      procedure Set_Type_Array_Component (Type_Mark      : in Type_Info_Ref;
                                          Component_Type : in Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Component_Type,
      --#                              Type_Mark;

      procedure Set_Type_First_Record_Component (Type_Mark        : in Type_Info_Ref;
                                                 Record_Component : in Record_Component_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Record_Component,
      --#                              Type_Mark;

      procedure Set_Type_Last_Record_Component (Type_Mark        : in Type_Info_Ref;
                                                Record_Component : in Record_Component_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Record_Component,
      --#                              Type_Mark;

      procedure Set_Type_Ancillary_Fields (Type_Mark       : in Type_Info_Ref;
                                           The_Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              Type_Mark;

      procedure Set_Type_Size_Attribute (Type_Mark : in Type_Info_Ref;
                                         Size_Val  : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Size_Val,
      --#                              Type_Mark;

      procedure Set_Type_Atomic (Type_Mark : in Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Type_Mark;

      procedure Set_Type_Virtual_Element_List (Type_Mark : in Type_Info_Ref;
                                               The_List  : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_List,
      --#                              Type_Mark;

      procedure Set_Type_Private_Type_Declaration (Type_Mark       : in Type_Info_Ref;
                                                   The_Declaration : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              Type_Mark;

      procedure Set_Type_Kind_Of_Generic
        (Type_Mark       : in Type_Info_Ref;
         Kind_Of_Generic : in Dictionary.Generic_Type_Discriminant);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Kind_Of_Generic,
      --#                              Type_Mark;

      function Get_Type_Symbol (Type_Mark : Type_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> Type_Mark = Null_Type_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Type_Symbol));

      function Get_Type_Name (Type_Mark : Type_Info_Ref) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Type_Parent (Type_Mark : Type_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Type_Declaration (Type_Mark : Type_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Type_Is_Full_Range_Subtype (Type_Mark : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_First_Constrained_Subtype (Type_Mark : Type_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Type_Discriminant (Type_Mark : Type_Info_Ref) return Dictionary.Type_Discriminant;
      --# global in Dictionary.Dict;

      function Get_Type_Private (Type_Mark : Type_Info_Ref) return Dictionary.TriState;
      --# global in Dictionary.Dict;

      function Get_Type_Limited (Type_Mark : Type_Info_Ref) return Dictionary.TriState;
      --# global in Dictionary.Dict;

      function Get_Type_Limited_Private (Type_Mark : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Type_Derived (Type_Mark : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Type_Is_Tagged (Type_Mark : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Type_Is_Own_Var_Type (Type_Mark : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Type_Extends (Type_Mark : Type_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Type_Accesses (Type_Mark : Type_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Type_Equality_Defined (Type_Mark : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Type_Contains_Float (Type_Mark : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Type_Constrained (Type_Mark : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Type_Static (Type_Mark : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Type_Wellformed (Type_Mark : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Type_Base_Type (Type_Mark : Type_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Type_Lower (Type_Mark : Type_Info_Ref) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Type_Upper (Type_Mark : Type_Info_Ref) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Type_Modulus (Type_Mark : Type_Info_Ref) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Type_Error_Bound (Type_Mark : Type_Info_Ref) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Type_First_Enumeration_Literal (Type_Mark : Type_Info_Ref) return Enumeration_Literal_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Type_Last_Enumeration_Literal (Type_Mark : Type_Info_Ref) return Enumeration_Literal_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Type_First_Array_Index (Type_Mark : Type_Info_Ref) return Array_Index_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Type_Last_Array_Index (Type_Mark : Type_Info_Ref) return Array_Index_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Type_Array_Component (Type_Mark : Type_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Type_First_Record_Component (Type_Mark : Type_Info_Ref) return Record_Component_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Type_Last_Record_Component (Type_Mark : Type_Info_Ref) return Record_Component_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Type_Announcement (Type_Mark : Type_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Type_Private_Type_Declaration (Type_Mark : Type_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Type_Ancillary_Fields (Type_Mark : Type_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Type_Size_Attribute (Type_Mark : Type_Info_Ref) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Type_Atomic (Type_Mark : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Type_Virtual_Element_List (Type_Mark : Type_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Type_Kind_Of_Generic (Type_Mark : Type_Info_Ref) return Dictionary.Generic_Type_Discriminant;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Protected_Type_Info
      --------------------------------------------------------------------------------

      procedure Set_Protected_Type_Own_Variable (The_Protected_Type : in Type_Info_Ref;
                                                 Own_Variable       : in Own_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Own_Variable,
      --#                              The_Protected_Type;

      procedure Set_Protected_Type_Elements_Hidden (The_Protected_Type : in Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Protected_Type;

      procedure Set_Protected_Type_First_Visible_Declaration
        (The_Protected_Type : in Type_Info_Ref;
         The_Declaration    : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Protected_Type;

      procedure Set_Protected_Type_Last_Visible_Declaration
        (The_Protected_Type : in Type_Info_Ref;
         The_Declaration    : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Protected_Type;

      procedure Set_Protected_Type_First_Private_Declaration
        (The_Protected_Type : in Type_Info_Ref;
         The_Declaration    : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Protected_Type;

      procedure Set_Protected_Type_Last_Private_Declaration
        (The_Protected_Type : in Type_Info_Ref;
         The_Declaration    : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Protected_Type;

      procedure Set_Protected_Type_First_Local_Declaration
        (The_Protected_Type : in Type_Info_Ref;
         The_Declaration    : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Protected_Type;

      procedure Set_Protected_Type_Last_Local_Declaration
        (The_Protected_Type : in Type_Info_Ref;
         The_Declaration    : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Protected_Type;

      procedure Set_Protected_Type_The_Entry (The_Protected_Type : in Type_Info_Ref;
                                              The_Entry          : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Entry,
      --#                              The_Protected_Type;

      procedure Set_Protected_Type_First_Discriminant
        (The_Protected_Type : in Type_Info_Ref;
         Discriminant       : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Discriminant,
      --#                              The_Protected_Type;

      procedure Set_Protected_Type_Last_Discriminant
        (The_Protected_Type : in Type_Info_Ref;
         Discriminant       : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Discriminant,
      --#                              The_Protected_Type;

      procedure Set_Protected_Type_Has_Pragma
        (The_Protected_Type : in Type_Info_Ref;
         The_Pragma         : in Dictionary.RavenscarPragmas);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Pragma,
      --#                              The_Protected_Type;

      procedure Set_Protected_Type_Pragma_Value
        (The_Protected_Type : in Type_Info_Ref;
         The_Pragma         : in Dictionary.RavenscarPragmasWithValue;
         The_Value          : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Pragma,
      --#                              The_Protected_Type,
      --#                              The_Value;

      procedure Set_Protected_Type_Body (The_Protected_Type : in Type_Info_Ref;
                                         The_Body           : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Body,
      --#                              The_Protected_Type;

      procedure Set_Protected_Type_Has_Proper_Body (The_Protected_Type : in Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Protected_Type;

      procedure Set_Protected_Type_With_Clauses
        (The_Protected_Type : in Type_Info_Ref;
         The_Context_Clause : in Context_Clause_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Context_Clause,
      --#                              The_Protected_Type;

      procedure Set_Protected_Type_Use_Type_Clauses
        (The_Protected_Type  : in Type_Info_Ref;
         The_Use_Type_Clause : in Use_Type_Clause_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Protected_Type,
      --#                              The_Use_Type_Clause;

      function Get_Protected_Type_Own_Variable (The_Protected_Type : Type_Info_Ref) return Own_Variable_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Protected_Type_Elements_Hidden (The_Protected_Type : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Protected_Type_Has_Entry (The_Protected_Type : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Protected_Type_First_Visible_Declaration (The_Protected_Type : Type_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Protected_Type_Last_Visible_Declaration (The_Protected_Type : Type_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Protected_Type_First_Private_Declaration (The_Protected_Type : Type_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Protected_Type_Last_Private_Declaration (The_Protected_Type : Type_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Protected_Type_First_Local_Declaration (The_Protected_Type : Type_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Protected_Type_Last_Local_Declaration (The_Protected_Type : Type_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Protected_Type_First_Discriminant (The_Protected_Type : Type_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Protected_Type_Last_Discriminant (The_Protected_Type : Type_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Protected_Type_Has_Pragma
        (The_Protected_Type : Type_Info_Ref;
         The_Pragma         : Dictionary.RavenscarPragmas)
        return               Boolean;
      --# global in Dictionary.Dict;

      function Get_Protected_Type_Pragma_Value
        (The_Protected_Type : Type_Info_Ref;
         The_Pragma         : Dictionary.RavenscarPragmasWithValue)
        return               LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Protected_Type_Body (The_Protected_Type : Type_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Protected_Type_Has_Proper_Body (The_Protected_Type : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Protected_Type_With_Clauses (The_Protected_Type : Type_Info_Ref) return Context_Clause_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Protected_Type_Use_Type_Clauses (The_Protected_Type : Type_Info_Ref) return Use_Type_Clause_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Task_Type_Info
      --------------------------------------------------------------------------------

      procedure Set_Task_Type_Signature_Not_Wellformed
        (The_Task_Type : in Type_Info_Ref;
         Abstraction   : in Dictionary.Abstractions);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              The_Task_Type;

      procedure Set_Task_Type_Has_Second_Annotation (The_Task_Type : in Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Task_Type;

      procedure Set_Task_Type_Has_Derives_Annotation (The_Task_Type : in Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Task_Type;

      procedure Set_Task_Type_First_Local_Declaration
        (The_Task_Type   : in Type_Info_Ref;
         The_Declaration : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Task_Type;

      procedure Set_Task_Type_Last_Local_Declaration
        (The_Task_Type   : in Type_Info_Ref;
         The_Declaration : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Task_Type;

      procedure Set_Task_Type_With_Clauses (The_Task_Type      : in Type_Info_Ref;
                                            The_Context_Clause : in Context_Clause_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Context_Clause,
      --#                              The_Task_Type;

      procedure Set_Task_Type_Use_Type_Clauses
        (The_Task_Type       : in Type_Info_Ref;
         The_Use_Type_Clause : in Use_Type_Clause_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Task_Type,
      --#                              The_Use_Type_Clause;

      procedure Set_Task_Type_First_Discriminant (The_Task_Type : in Type_Info_Ref;
                                                  Discriminant  : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Discriminant,
      --#                              The_Task_Type;

      procedure Set_Task_Type_Last_Discriminant (The_Task_Type : in Type_Info_Ref;
                                                 Discriminant  : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Discriminant,
      --#                              The_Task_Type;

      procedure Set_Task_Type_First_Global_Variable
        (The_Task_Type       : in Type_Info_Ref;
         Abstraction         : in Dictionary.Abstractions;
         The_Global_Variable : in Global_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              The_Global_Variable,
      --#                              The_Task_Type;

      procedure Set_Task_Type_Last_Global_Variable
        (The_Task_Type       : in Type_Info_Ref;
         Abstraction         : in Dictionary.Abstractions;
         The_Global_Variable : in Global_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              The_Global_Variable,
      --#                              The_Task_Type;

      procedure Set_Task_Type_Has_Pragma (The_Task_Type : in Type_Info_Ref;
                                          The_Pragma    : in Dictionary.RavenscarPragmas);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Pragma,
      --#                              The_Task_Type;

      procedure Set_Task_Type_Pragma_Value
        (The_Task_Type : in Type_Info_Ref;
         The_Pragma    : in Dictionary.RavenscarPragmasWithValue;
         The_Value     : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Pragma,
      --#                              The_Task_Type,
      --#                              The_Value;

      procedure Set_Task_Type_First_Loop (The_Task_Type : in Type_Info_Ref;
                                          The_Loop      : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Loop,
      --#                              The_Task_Type;

      procedure Set_Task_Type_Last_Loop (The_Task_Type : in Type_Info_Ref;
                                         The_Loop      : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Loop,
      --#                              The_Task_Type;

      procedure Set_Task_Type_Suspends_List (The_Task_Type     : in Type_Info_Ref;
                                             The_Suspends_List : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Suspends_List,
      --#                              The_Task_Type;

      procedure Set_Task_Type_Body (The_Task_Type : in Type_Info_Ref;
                                    The_Body      : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Body,
      --#                              The_Task_Type;

      procedure Set_Task_Type_Has_Proper_Body (The_Task_Type : in Type_Info_Ref;
                                               Is_Hidden     : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Is_Hidden,
      --#                              The_Task_Type;

      procedure Set_Task_Type_Uses_Unprotected_Variables (The_Task_Type : in Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Task_Type;

      procedure Set_Task_Type_Uses_Unchecked_Conversion (The_Task_Type : in Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Task_Type;

      procedure Set_Task_Type_Assigns_From_External (The_Task_Type : in Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Task_Type;

      function Get_Task_Type_Signature_Is_Wellformed
        (The_Task_Type : Type_Info_Ref;
         Abstraction   : Dictionary.Abstractions)
        return          Boolean;
      --# global in Dictionary.Dict;

      function Get_Task_Type_Has_Second_Annotation (The_Task_Type : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Task_Type_Has_Derives_Annotation (The_Task_Type : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Task_Type_First_Local_Declaration (The_Task_Type : Type_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Task_Type_Last_Local_Declaration (The_Task_Type : Type_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Task_Type_With_Clauses (The_Task_Type : Type_Info_Ref) return Context_Clause_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Task_Type_Use_Type_Clauses (The_Task_Type : Type_Info_Ref) return Use_Type_Clause_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Task_Type_First_Discriminant (The_Task_Type : Type_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Task_Type_Last_Discriminant (The_Task_Type : Type_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Task_Type_First_Global_Variable
        (The_Task_Type : Type_Info_Ref;
         Abstraction   : Dictionary.Abstractions)
        return          Global_Variable_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Task_Type_Last_Global_Variable
        (The_Task_Type : Type_Info_Ref;
         Abstraction   : Dictionary.Abstractions)
        return          Global_Variable_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Task_Type_Has_Pragma (The_Task_Type : Type_Info_Ref;
                                         The_Pragma    : Dictionary.RavenscarPragmas) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Task_Type_Pragma_Value
        (The_Task_Type : Type_Info_Ref;
         The_Pragma    : Dictionary.RavenscarPragmasWithValue)
        return          LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Task_Type_First_Loop (The_Task_Type : Type_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Task_Type_Last_Loop (The_Task_Type : Type_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Task_Type_Suspends_List (The_Task_Type : Type_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Task_Type_Body (The_Task_Type : Type_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Task_Type_Has_Proper_Body (The_Task_Type : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Task_Type_Uses_Unprotected_Variables (The_Task_Type : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Task_Type_Uses_Unchecked_Conversion (The_Task_Type : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Task_Type_Assigns_From_External (The_Task_Type : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Task_Type_Body_Is_Hidden (The_Task_Type : Type_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Constant_Info
      --------------------------------------------------------------------------------

      function Get_Constant_Info_Ref (Item : Dictionary.Symbol) return Constant_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Constant_Info_Ref);

      procedure Create_Constant
        (Name            : in     LexTokenManager.Lex_String;
         Type_Mark       : in     Type_Info_Ref;
         Static          : in     Boolean;
         The_Declaration : in     Declaration_Info_Ref;
         Is_Deferred     : in     Boolean;
         Comp_Unit       : in     ContextManager.UnitDescriptors;
         Loc             : in     LexTokenManager.Token_Position;
         The_Constant    :    out Constant_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Constant    from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Is_Deferred,
      --#                              Loc,
      --#                              Name,
      --#                              Static,
      --#                              The_Declaration,
      --#                              Type_Mark;
      --# post The_Constant /= Null_Constant_Info_Ref;

      procedure Set_Constant_Value (The_Constant : in Constant_Info_Ref;
                                    Value        : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Constant,
      --#                              Value;

      procedure Set_Constant_Exp_Node
        (The_Constant      : in Constant_Info_Ref;
         Exp_Is_Wellformed : in Boolean;
         Exp_Node          : in ExaminerConstants.RefType);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Exp_Is_Wellformed,
      --#                              Exp_Node,
      --#                              The_Constant;

      procedure Set_Constant_Static (The_Constant : in Constant_Info_Ref;
                                     Static       : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Static,
      --#                              The_Constant;

      procedure Set_Constant_Declaration (The_Constant    : in Constant_Info_Ref;
                                          The_Declaration : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Constant,
      --#                              The_Declaration;

      procedure Set_Constant_Deferred_Declaration
        (The_Constant    : in Constant_Info_Ref;
         The_Declaration : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Constant,
      --#                              The_Declaration;

      procedure Set_Constant_Associated_Generic_Parameter
        (The_Constant          : in Constant_Info_Ref;
         The_Generic_Parameter : in Generic_Parameter_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Constant,
      --#                              The_Generic_Parameter;

      procedure Set_Constant_First_Rule_Policy (The_Constant    : in Constant_Info_Ref;
                                                The_Rule_Policy : in Rule_Policy_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Constant,
      --#                              The_Rule_Policy;

      procedure Set_Constant_Last_Rule_Policy (The_Constant    : in Constant_Info_Ref;
                                               The_Rule_Policy : in Rule_Policy_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Constant,
      --#                              The_Rule_Policy;

      function Get_Constant_Symbol (The_Constant : Constant_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Constant = Null_Constant_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Constant_Symbol));

      function Get_Constant_Name (The_Constant : Constant_Info_Ref) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Constant_Type (The_Constant : Constant_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Constant_Value (The_Constant : Constant_Info_Ref) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Constant_First_Rule_Policy (The_Constant : Constant_Info_Ref) return Rule_Policy_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Constant_Last_Rule_Policy (The_Constant : Constant_Info_Ref) return Rule_Policy_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Constant_Exp_Node (The_Constant : Constant_Info_Ref) return ExaminerConstants.RefType;
      --# global in Dictionary.Dict;

      function Get_Constant_Exp_Is_Wellformed (The_Constant : Constant_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Constant_Static (The_Constant : Constant_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Constant_Declaration (The_Constant : Constant_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Constant_Deferred_Declaration (The_Constant : Constant_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Constant_Associated_Generic_Parameter (The_Constant : Constant_Info_Ref) return Generic_Parameter_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Variable_Info
      --------------------------------------------------------------------------------

      function Get_Variable_Info_Ref (Item : Dictionary.Symbol) return Variable_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Variable_Info_Ref);

      procedure Create_Variable
        (Name         : in     LexTokenManager.Lex_String;
         Comp_Unit    : in     ContextManager.UnitDescriptors;
         Loc          : in     LexTokenManager.Token_Position;
         The_Variable :    out Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Variable    from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc,
      --#                              Name;
      --# post The_Variable /= Null_Variable_Info_Ref;

      procedure Set_Variable_Type (The_Variable : in Variable_Info_Ref;
                                   Type_Mark    : in Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Variable,
      --#                              Type_Mark;

      procedure Set_Variable_Abstract_Type (The_Variable       : in Variable_Info_Ref;
                                            Abstract_Type_Mark : in Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstract_Type_Mark,
      --#                              The_Variable;

      procedure Set_Variable_Initialized (The_Variable : in Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Variable;

      procedure Set_Variable_Has_Address_Clause (The_Variable : in Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Variable;

      procedure Set_Variable_Has_Pragma_Import (The_Variable : in Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Variable;

      procedure Set_Variable_Is_Aliased (The_Variable : in Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Variable;

      procedure Set_Variable_Marked_Valid (The_Variable : in Variable_Info_Ref;
                                           Val          : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Variable,
      --#                              Val;

      procedure Set_Variable_Declaration (The_Variable    : in Variable_Info_Ref;
                                          The_Declaration : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Variable;

      procedure Set_Variable_Exp_Node (The_Variable : in Variable_Info_Ref;
                                       Exp_Node     : in ExaminerConstants.RefType);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Exp_Node,
      --#                              The_Variable;

      procedure Set_Variable_Own_Task (The_Variable : in Variable_Info_Ref;
                                       Own_Task     : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Own_Task,
      --#                              The_Variable;

      procedure Set_Variable_Virtual_Element (The_Variable    : in Variable_Info_Ref;
                                              Virtual_Element : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Variable,
      --#                              Virtual_Element;

      procedure Set_Variable_Global_References
        (The_Variable : in Variable_Info_Ref;
         Abstraction  : in Dictionary.Abstractions;
         Reference    : in Global_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              Reference,
      --#                              The_Variable;

      procedure Set_Variable_Own_Variable (The_Variable : in Variable_Info_Ref;
                                           Own_Variable : in Own_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Own_Variable,
      --#                              The_Variable;

      procedure Set_Variable_Constituent (The_Variable    : in Variable_Info_Ref;
                                          The_Constituent : in Constituent_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Constituent,
      --#                              The_Variable;

      procedure Set_Variable_Subcomponents (The_Variable  : in Variable_Info_Ref;
                                            Subcomponents : in Subcomponent_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Subcomponents,
      --#                              The_Variable;

      function Get_Variable_Symbol (The_Variable : Variable_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Variable = Null_Variable_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Variable_Symbol));

      function Get_Variable_Name (The_Variable : Variable_Info_Ref) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Variable_Type (The_Variable : Variable_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Variable_Abstract_Type (The_Variable : Variable_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Variable_Initialized (The_Variable : Variable_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Variable_Has_Address_Clause (The_Variable : Variable_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Variable_Has_Pragma_Import (The_Variable : Variable_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Variable_Is_Aliased (The_Variable : Variable_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Variable_Marked_Valid (The_Variable : Variable_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Variable_Declaration (The_Variable : Variable_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Variable_Exp_Node (The_Variable : Variable_Info_Ref) return ExaminerConstants.RefType;
      --# global in Dictionary.Dict;

      function Get_Variable_Global_References
        (The_Variable : Variable_Info_Ref;
         Abstraction  : Dictionary.Abstractions)
        return         Global_Variable_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Variable_Own_Variable (The_Variable : Variable_Info_Ref) return Own_Variable_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Variable_Own_Task (The_Variable : Variable_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Variable_Constituent (The_Variable : Variable_Info_Ref) return Constituent_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Variable_Subcomponents (The_Variable : Variable_Info_Ref) return Subcomponent_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Variable_Virtual_Element (The_Variable : Variable_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Global_Variable_Info
      --------------------------------------------------------------------------------

      type Kind_Of_Global_Variable_T is (Subprogram_Parameter_Item, Subprogram_Variable_Item, Task_Type_Variable_Item);

      function Get_Global_Variable_Info_Ref (Item : Dictionary.Symbol) return Global_Variable_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Global_Variable_Info_Ref);

      procedure Create_Global_Variable
        (Mode                 : in     Dictionary.Modes;
         Prefix_Needed        : in     Boolean;
         The_Subprogram       : in     Subprogram_Info_Ref;
         The_Task_Type        : in     Type_Info_Ref;
         Subprogram_Parameter : in     Subprogram_Parameter_Info_Ref;
         Variable             : in     Variable_Info_Ref;
         Next_Subprogram      : in     Global_Variable_Info_Ref;
         Comp_Unit            : in     ContextManager.UnitDescriptors;
         Loc                  : in     LexTokenManager.Token_Position;
         The_Global_Variable  :    out Global_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Global_Variable from Comp_Unit,
      --#                                  Dictionary.Dict,
      --#                                  Loc,
      --#                                  Mode,
      --#                                  Next_Subprogram,
      --#                                  Prefix_Needed,
      --#                                  Subprogram_Parameter,
      --#                                  The_Subprogram,
      --#                                  The_Task_Type,
      --#                                  Variable;
      --# post The_Global_Variable /= Null_Global_Variable_Info_Ref;

      procedure Set_Global_Variable_Exported (The_Global_Variable : in Global_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Global_Variable;

      procedure Set_Global_Variable_Imported (The_Global_Variable : in Global_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Global_Variable;

      procedure Set_Global_Variable_Dependencies
        (The_Global_Variable : in Global_Variable_Info_Ref;
         Abstraction         : in Dictionary.Abstractions;
         Dependency          : in Dependency_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              Dependency,
      --#                              The_Global_Variable;

      procedure Set_Next_Global_Variable (The_Global_Variable, Next : in Global_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              The_Global_Variable;

      function Get_Global_Variable_Symbol (The_Global_Variable : Global_Variable_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Global_Variable = Null_Global_Variable_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Global_Variable_Symbol));

      function Get_Global_Variable_Mode (The_Global_Variable : Global_Variable_Info_Ref) return Dictionary.Modes;
      --# global in Dictionary.Dict;

      function Get_Global_Variable_Exported (The_Global_Variable : Global_Variable_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Global_Variable_Imported (The_Global_Variable : Global_Variable_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Global_Variable_Prefix_Needed (The_Global_Variable : Global_Variable_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Global_Variable_Dependencies
        (The_Global_Variable : Global_Variable_Info_Ref;
         Abstraction         : Dictionary.Abstractions)
        return                Dependency_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Global_Variable_Next_Subprogram
        (The_Global_Variable : Global_Variable_Info_Ref)
        return                Global_Variable_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Global_Variable_Subprogram (The_Global_Variable : Global_Variable_Info_Ref) return Subprogram_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Global_Variable_Task_Type (The_Global_Variable : Global_Variable_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Kind_Of_Global_Variable (The_Global_Variable : Global_Variable_Info_Ref) return Kind_Of_Global_Variable_T;
      --# global in Dictionary.Dict;

      function Get_Global_Variable_Variable (The_Global_Variable : Global_Variable_Info_Ref) return Variable_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Global_Variable_Parameter
        (The_Global_Variable : Global_Variable_Info_Ref)
        return                Subprogram_Parameter_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Next_Global_Variable (The_Global_Variable : Global_Variable_Info_Ref) return Global_Variable_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Own_Variable_Info
      --------------------------------------------------------------------------------

      function Get_Own_Variable_Info_Ref (Item : Dictionary.Symbol) return Own_Variable_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Own_Variable_Info_Ref);

      procedure Create_Own_Variable
        (Variable         : in     Variable_Info_Ref;
         Owner            : in     Dictionary.Symbol;
         Comp_Unit        : in     ContextManager.UnitDescriptors;
         Loc              : in     LexTokenManager.Token_Position;
         The_Own_Variable :    out Own_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Own_Variable from Comp_Unit,
      --#                               Dictionary.Dict,
      --#                               Loc,
      --#                               Owner,
      --#                               Variable;
      --# post The_Own_Variable /= Null_Own_Variable_Info_Ref;

      procedure Set_Own_Variable_Announced (The_Own_Variable : in Own_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Own_Variable;

      procedure Set_Own_Variable_Typed (The_Own_Variable : in Own_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Own_Variable;

      procedure Set_Own_Variable_Initialized (The_Own_Variable : in Own_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Own_Variable;

      procedure Set_Own_Variable_Constituents
        (The_Own_Variable : in Own_Variable_Info_Ref;
         The_Constituent  : in Constituent_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Constituent,
      --#                              The_Own_Variable;

      procedure Set_Own_Variable_Mode (The_Own_Variable : in Own_Variable_Info_Ref;
                                       Mode             : in Dictionary.Modes);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Mode,
      --#                              The_Own_Variable;

      procedure Set_Own_Variable_Protected (The_Own_Variable : in Own_Variable_Info_Ref;
                                            Is_Protected     : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Is_Protected,
      --#                              The_Own_Variable;

      procedure Set_Next_Own_Variable (The_Own_Variable, Next : in Own_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              The_Own_Variable;

      procedure Set_Own_Variable_Interrupt_Stream_Mappings
        (The_Own_Variable              : in Own_Variable_Info_Ref;
         The_Interrupt_Stream_Mappings : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Interrupt_Stream_Mappings,
      --#                              The_Own_Variable;

      procedure Set_Own_Variable_Implicit_In_Stream
        (The_Own_Variable       : in Own_Variable_Info_Ref;
         The_Implicit_In_Stream : in Implicit_In_Stream_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Implicit_In_Stream,
      --#                              The_Own_Variable;

      procedure Set_Own_Variable_Is_Interrupt_Stream
        (The_Own_Variable    : in Own_Variable_Info_Ref;
         Is_Interrupt_Stream : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Is_Interrupt_Stream,
      --#                              The_Own_Variable;

      procedure Set_Own_Variable_Unprotected_Reference
        (The_Own_Variable : in Own_Variable_Info_Ref;
         By_Thread        : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              By_Thread,
      --#                              The_Own_Variable;

      procedure Set_Own_Variable_Suspends_Reference
        (The_Own_Variable : in Own_Variable_Info_Ref;
         By_Thread        : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              By_Thread,
      --#                              The_Own_Variable;

      procedure Set_Own_Variable_Is_Suspendable (The_Own_Variable : in Own_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Own_Variable;

      procedure Set_Own_Variable_Has_Interrupt_Property (The_Own_Variable : in Own_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Own_Variable;

      procedure Set_Own_Variable_Priority_Property
        (The_Own_Variable : in Own_Variable_Info_Ref;
         The_Value        : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Own_Variable,
      --#                              The_Value;

      procedure Set_Own_Variable_Integrity_Property
        (The_Own_Variable : in Own_Variable_Info_Ref;
         The_Value        : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Own_Variable,
      --#                              The_Value;

      function Get_Own_Variable_Symbol (The_Own_Variable : Own_Variable_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Own_Variable = Null_Own_Variable_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Own_Variable_Symbol));

      function Get_Own_Variable_Variable (The_Own_Variable : Own_Variable_Info_Ref) return Variable_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Own_Variable_Owner (The_Own_Variable : Own_Variable_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Own_Variable_Announced (The_Own_Variable : Own_Variable_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Own_Variable_Typed (The_Own_Variable : Own_Variable_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Own_Variable_Initialized (The_Own_Variable : Own_Variable_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Own_Variable_Constituents (The_Own_Variable : Own_Variable_Info_Ref) return Constituent_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Own_Variable_Mode (The_Own_Variable : Own_Variable_Info_Ref) return Dictionary.Modes;
      --# global in Dictionary.Dict;

      function Get_Own_Variable_Protected (The_Own_Variable : Own_Variable_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Own_Variable_Implicit_In_Stream (The_Own_Variable : Own_Variable_Info_Ref) return Implicit_In_Stream_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Next_Own_Variable (The_Own_Variable : Own_Variable_Info_Ref) return Own_Variable_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Own_Variable_Interrupt_Stream_Mappings (The_Own_Variable : Own_Variable_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Own_Variable_Is_Interrupt_Stream (The_Own_Variable : Own_Variable_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Own_Variable_Is_Suspendable (The_Own_Variable : Own_Variable_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Own_Variable_Has_Interrupt_Property (The_Own_Variable : Own_Variable_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Own_Variable_Priority_Property (The_Own_Variable : Own_Variable_Info_Ref) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Own_Variable_Integrity_Property (The_Own_Variable : Own_Variable_Info_Ref) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Own_Variable_Unprotected_Reference (The_Own_Variable : Own_Variable_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Own_Variable_Suspends_Reference (The_Own_Variable : Own_Variable_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Own_Variable_Has_Valid_Priority_Property (The_Own_Variable : Own_Variable_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Quantified_Variable_Info
      --------------------------------------------------------------------------------

      function Get_Quantified_Variable_Info_Ref (Item : Dictionary.Symbol) return Quantified_Variable_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Quantified_Variable_Info_Ref);

      procedure Create_Quantified_Variable
        (Name                     : in     LexTokenManager.Lex_String;
         Type_Mark                : in     Type_Info_Ref;
         The_Parameter_Constraint : in     Parameter_Constraint_Info_Ref;
         Region                   : in     Dictionary.Symbol;
         Comp_Unit                : in     ContextManager.UnitDescriptors;
         Loc                      : in     LexTokenManager.Token_Position;
         The_Quantified_Variable  :    out Quantified_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Quantified_Variable from Comp_Unit,
      --#                                      Dictionary.Dict,
      --#                                      Loc,
      --#                                      Name,
      --#                                      Region,
      --#                                      The_Parameter_Constraint,
      --#                                      Type_Mark;
      --# post The_Quantified_Variable /= Null_Quantified_Variable_Info_Ref;

      function Get_Quantified_Variable_Symbol (The_Quantified_Variable : Quantified_Variable_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Quantified_Variable = Null_Quantified_Variable_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Quantified_Variable_Symbol));

      function Get_Quantified_Variable_Name
        (The_Quantified_Variable : Quantified_Variable_Info_Ref)
        return                    LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Quantified_Variable_Type (The_Quantified_Variable : Quantified_Variable_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Quantified_Variable_Constraint
        (The_Quantified_Variable : Quantified_Variable_Info_Ref)
        return                    Parameter_Constraint_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Quantified_Variable_Region (The_Quantified_Variable : Quantified_Variable_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Implicit_Return_Variable_Info
      --------------------------------------------------------------------------------

      function Get_Implicit_Return_Variable_Info_Ref (Item : Dictionary.Symbol) return Implicit_Return_Variable_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Implicit_Return_Variable_Info_Ref);

      procedure Create_Implicit_Return_Variable
        (Name                         : in     LexTokenManager.Lex_String;
         The_Function                 : in     Subprogram_Info_Ref;
         Comp_Unit                    : in     ContextManager.UnitDescriptors;
         Loc                          : in     LexTokenManager.Token_Position;
         The_Implicit_Return_Variable :    out Implicit_Return_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Implicit_Return_Variable from Comp_Unit,
      --#                                           Dictionary.Dict,
      --#                                           Loc,
      --#                                           Name,
      --#                                           The_Function;
      --# post The_Implicit_Return_Variable /= Null_Implicit_Return_Variable_Info_Ref;

      function Get_Implicit_Return_Variable_Symbol
        (The_Implicit_Return_Variable : Implicit_Return_Variable_Info_Ref)
        return                         Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Implicit_Return_Variable = Null_Implicit_Return_Variable_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Implicit_Return_Variable_Symbol));

      function Get_Implicit_Return_Variable_Name
        (The_Implicit_Return_Variable : Implicit_Return_Variable_Info_Ref)
        return                         LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Implicit_Return_Variable_Function
        (The_Implicit_Return_Variable : Implicit_Return_Variable_Info_Ref)
        return                         Subprogram_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Implicit_In_Stream_Info
      --------------------------------------------------------------------------------

      function Get_Implicit_In_Stream_Info_Ref (Item : Dictionary.Symbol) return Implicit_In_Stream_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Implicit_In_Stream_Info_Ref);

      procedure Create_Implicit_In_Stream
        (The_Own_Variable       : in     Own_Variable_Info_Ref;
         Comp_Unit              : in     ContextManager.UnitDescriptors;
         Loc                    : in     LexTokenManager.Token_Position;
         The_Implicit_In_Stream :    out Implicit_In_Stream_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Implicit_In_Stream from Comp_Unit,
      --#                                     Dictionary.Dict,
      --#                                     Loc,
      --#                                     The_Own_Variable;
      --# post The_Implicit_In_Stream /= Null_Implicit_In_Stream_Info_Ref;

      function Get_Implicit_In_Stream_Symbol (The_Implicit_In_Stream : Implicit_In_Stream_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Implicit_In_Stream = Null_Implicit_In_Stream_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Implicit_In_Stream_Symbol));

      function Get_Implicit_In_Stream_Own_Variable
        (The_Implicit_In_Stream : Implicit_In_Stream_Info_Ref)
        return                   Own_Variable_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Rule_Policy_Info
      --------------------------------------------------------------------------------

      procedure Create_Rule_Policy
        (Scope           : in     Dictionary.Scopes;
         Value           : in     Dictionary.Rule_Policies;
         Comp_Unit       : in     ContextManager.UnitDescriptors;
         Loc             : in     LexTokenManager.Token_Position;
         The_Rule_Policy :    out Rule_Policy_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Rule_Policy from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc,
      --#                              Scope,
      --#                              Value;
      --# post The_Rule_Policy /= Null_Rule_Policy_Info_Ref;

      procedure Set_Next_Rule_Policy (The_Rule_Policy, Next : in Rule_Policy_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              The_Rule_Policy;

      function Get_Rule_Policy_Symbol (The_Rule_Policy : Rule_Policy_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Rule_Policy = Null_Rule_Policy_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Rule_Policy_Symbol));

      function Get_Rule_Policy_Scope (The_Rule_Policy : Rule_Policy_Info_Ref) return Dictionary.Scopes;
      --# global in Dictionary.Dict;

      function Get_Rule_Policy_Value (The_Rule_Policy : Rule_Policy_Info_Ref) return Dictionary.Rule_Policies;
      --# global in Dictionary.Dict;

      function Get_Next_Rule_Policy (The_Rule_Policy : Rule_Policy_Info_Ref) return Rule_Policy_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Constituent_Info
      --------------------------------------------------------------------------------

      function Get_Constituent_Info_Ref (Item : Dictionary.Symbol) return Constituent_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Constituent_Info_Ref);

      procedure Create_Constituent
        (The_Own_Variable : in     Own_Variable_Info_Ref;
         The_Variable     : in     Variable_Info_Ref;
         Mode             : in     Dictionary.Modes;
         Comp_Unit        : in     ContextManager.UnitDescriptors;
         Loc              : in     LexTokenManager.Token_Position;
         The_Constituent  :    out Constituent_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Constituent from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc,
      --#                              Mode,
      --#                              The_Own_Variable,
      --#                              The_Variable;
      --# post The_Constituent /= Null_Constituent_Info_Ref;

      procedure Set_Next_Constituent (The_Constituent, Next : in Constituent_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              The_Constituent;

      function Get_Constituent_Symbol (The_Constituent : Constituent_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Constituent = Null_Constituent_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Constituent_Symbol));

      function Get_Constituent_Own_Variable (The_Constituent : Constituent_Info_Ref) return Own_Variable_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Constituent_Variable (The_Constituent : Constituent_Info_Ref) return Variable_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Constituent_Mode (The_Constituent : Constituent_Info_Ref) return Dictionary.Modes;
      --# global in Dictionary.Dict;

      function Get_Next_Constituent (The_Constituent : Constituent_Info_Ref) return Constituent_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Context_Clause_Info
      --------------------------------------------------------------------------------

      function Get_Context_Clause_Info_Ref (Item : Dictionary.Symbol) return Context_Clause_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Context_Clause_Info_Ref);

      procedure Create_Context_Clause
        (The_Package        : in     Package_Info_Ref;
         The_Subprogram     : in     Subprogram_Info_Ref;
         Explicit           : in     Boolean;
         Comp_Unit          : in     ContextManager.UnitDescriptors;
         Loc                : in     LexTokenManager.Token_Position;
         The_Context_Clause :    out Context_Clause_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Context_Clause from Comp_Unit,
      --#                                 Dictionary.Dict,
      --#                                 Explicit,
      --#                                 Loc,
      --#                                 The_Package,
      --#                                 The_Subprogram;
      --# post The_Context_Clause /= Null_Context_Clause_Info_Ref;

      procedure Set_Context_Clause_Explicit (The_Context_Clause : in Context_Clause_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Context_Clause;

      procedure Set_Next_Context_Clause (The_Context_Clause, Next : in Context_Clause_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              The_Context_Clause;

      function Get_Context_Clause_Symbol (The_Context_Clause : Context_Clause_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Context_Clause = Null_Context_Clause_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Context_Clause_Symbol));

      function Get_Context_Clause_Is_Subprogram (The_Context_Clause : Context_Clause_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Context_Clause_Package (The_Context_Clause : Context_Clause_Info_Ref) return Package_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Context_Clause_Subprogram (The_Context_Clause : Context_Clause_Info_Ref) return Subprogram_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Context_Clause_Explicit (The_Context_Clause : Context_Clause_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Next_Context_Clause (The_Context_Clause : Context_Clause_Info_Ref) return Context_Clause_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Use_Type_Clause_Info
      --------------------------------------------------------------------------------

      procedure Create_Use_Type_Clause
        (Type_Mark           : in     Type_Info_Ref;
         Comp_Unit           : in     ContextManager.UnitDescriptors;
         Loc                 : in     LexTokenManager.Token_Position;
         The_Use_Type_Clause :    out Use_Type_Clause_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Use_Type_Clause from Comp_Unit,
      --#                                  Dictionary.Dict,
      --#                                  Loc,
      --#                                  Type_Mark;
      --# post The_Use_Type_Clause /= Null_Use_Type_Clause_Info_Ref;

      procedure Set_Next_Use_Type_Clause (The_Use_Type_Clause, Next : in Use_Type_Clause_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              The_Use_Type_Clause;

      function Get_Use_Type_Clause_Type (The_Use_Type_Clause : Use_Type_Clause_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Next_Use_Type_Clause (The_Use_Type_Clause : Use_Type_Clause_Info_Ref) return Use_Type_Clause_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Parameter_Constraint_Info_Ref
      --------------------------------------------------------------------------------

      function Get_Parameter_Constraint_Info_Ref (Item : Dictionary.Symbol) return Parameter_Constraint_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Parameter_Constraint_Info_Ref);

      procedure Create_Parameter_Constraint
        (The_Subprogram_Parameter : in     Subprogram_Parameter_Info_Ref;
         Dimension                : in     Positive;
         Comp_Unit                : in     ContextManager.UnitDescriptors;
         Loc                      : in     LexTokenManager.Token_Position;
         The_Parameter_Constraint :    out Parameter_Constraint_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Parameter_Constraint from Comp_Unit,
      --#                                       Dictionary.Dict,
      --#                                       Dimension,
      --#                                       Loc,
      --#                                       The_Subprogram_Parameter;
      --# post The_Parameter_Constraint /= Null_Parameter_Constraint_Info_Ref;

      procedure Set_Next_Parameter_Constraint (The_Parameter_Constraint, Next : in Parameter_Constraint_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              The_Parameter_Constraint;

      function Get_Parameter_Constraint_Symbol
        (The_Parameter_Constraint : Parameter_Constraint_Info_Ref)
        return                     Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Parameter_Constraint = Null_Parameter_Constraint_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Parameter_Constraint_Symbol));

      function Get_Parameter_Constraint_Subprogram_Parameter
        (The_Parameter_Constraint : Parameter_Constraint_Info_Ref)
        return                     Subprogram_Parameter_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Parameter_Constraint_Dimension (The_Parameter_Constraint : Parameter_Constraint_Info_Ref) return Positive;
      --# global in Dictionary.Dict;

      function Get_Next_Parameter_Constraint
        (The_Parameter_Constraint : Parameter_Constraint_Info_Ref)
        return                     Parameter_Constraint_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Subprogram_Parameter_Info
      --------------------------------------------------------------------------------

      function Get_Subprogram_Parameter_Info_Ref (Item : Dictionary.Symbol) return Subprogram_Parameter_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Subprogram_Parameter_Info_Ref);

      procedure Create_Subprogram_Parameter
        (Name                     : in     LexTokenManager.Lex_String;
         The_Subprogram           : in     Subprogram_Info_Ref;
         Type_Mark                : in     Type_Info_Ref;
         Mode                     : in     Dictionary.Modes;
         Comp_Unit                : in     ContextManager.UnitDescriptors;
         Loc                      : in     LexTokenManager.Token_Position;
         The_Subprogram_Parameter :    out Subprogram_Parameter_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Subprogram_Parameter from Comp_Unit,
      --#                                       Dictionary.Dict,
      --#                                       Loc,
      --#                                       Mode,
      --#                                       Name,
      --#                                       The_Subprogram,
      --#                                       Type_Mark;
      --# post The_Subprogram_Parameter /= Null_Subprogram_Parameter_Info_Ref;

      procedure Set_Subprogram_Parameter_Exported
        (The_Subprogram_Parameter : in Subprogram_Parameter_Info_Ref;
         Abstraction              : in Dictionary.Abstractions);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              The_Subprogram_Parameter;

      procedure Set_Subprogram_Parameter_Imported
        (The_Subprogram_Parameter : in Subprogram_Parameter_Info_Ref;
         Abstraction              : in Dictionary.Abstractions);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              The_Subprogram_Parameter;

      procedure Set_Subprogram_Parameter_Dependencies
        (The_Subprogram_Parameter : in Subprogram_Parameter_Info_Ref;
         Abstraction              : in Dictionary.Abstractions;
         Dependency               : in Dependency_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              Dependency,
      --#                              The_Subprogram_Parameter;

      procedure Set_Subprogram_Parameter_Global_References
        (The_Subprogram_Parameter : in Subprogram_Parameter_Info_Ref;
         Abstraction              : in Dictionary.Abstractions;
         Reference                : in Global_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              Reference,
      --#                              The_Subprogram_Parameter;

      procedure Set_Subprogram_Parameter_Subcomponents
        (The_Subprogram_Parameter : in Subprogram_Parameter_Info_Ref;
         Subcomponents            : in Subcomponent_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Subcomponents,
      --#                              The_Subprogram_Parameter;

      procedure Set_Subprogram_Parameter_Index_Constraints
        (The_Subprogram_Parameter : in Subprogram_Parameter_Info_Ref;
         The_Index_Constraints    : in Parameter_Constraint_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Index_Constraints,
      --#                              The_Subprogram_Parameter;

      procedure Set_Next_Subprogram_Parameter (The_Subprogram_Parameter, Next : in Subprogram_Parameter_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              The_Subprogram_Parameter;

      function Get_Subprogram_Parameter_Symbol
        (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref)
        return                     Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Subprogram_Parameter = Null_Subprogram_Parameter_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Subprogram_Parameter_Symbol));

      function Get_Subprogram_Parameter_Name
        (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref)
        return                     LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Parameter_Subprogram
        (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref)
        return                     Subprogram_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Parameter_Type (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Parameter_Mode (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref) return Dictionary.Modes;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Parameter_Exported
        (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref;
         Abstraction              : Dictionary.Abstractions)
        return                     Boolean;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Parameter_Imported
        (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref;
         Abstraction              : Dictionary.Abstractions)
        return                     Boolean;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Parameter_Dependencies
        (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref;
         Abstraction              : Dictionary.Abstractions)
        return                     Dependency_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Parameter_Global_References
        (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref;
         Abstraction              : Dictionary.Abstractions)
        return                     Global_Variable_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Parameter_Subcomponents
        (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref)
        return                     Subcomponent_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Parameter_Index_Constraints
        (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref)
        return                     Parameter_Constraint_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Next_Subprogram_Parameter
        (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref)
        return                     Subprogram_Parameter_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Subprogram_Info
      --------------------------------------------------------------------------------

      function Get_Subprogram_Info_Ref (Item : Dictionary.Symbol) return Subprogram_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Subprogram_Info_Ref);

      procedure Create_Subprogram
        (Name            : in     LexTokenManager.Lex_String;
         The_Declaration : in     Declaration_Info_Ref;
         Comp_Unit       : in     ContextManager.UnitDescriptors;
         Loc             : in     LexTokenManager.Token_Position;
         The_Subprogram  :    out Subprogram_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Subprogram  from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc,
      --#                              Name,
      --#                              The_Declaration;
      --# post The_Subprogram /= Null_Subprogram_Info_Ref;

      procedure Set_Subprogram_Implicit_Proof_Function
        (The_Subprogram     : in Subprogram_Info_Ref;
         Abstraction        : in Dictionary.Abstractions;
         The_Proof_Function : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              The_Proof_Function,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Implicit_Return_Variable
        (The_Subprogram               : in Subprogram_Info_Ref;
         Abstraction                  : in Dictionary.Abstractions;
         The_Implicit_Return_Variable : in Implicit_Return_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              The_Implicit_Return_Variable,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Signature_Not_Wellformed
        (The_Subprogram : in Subprogram_Info_Ref;
         Abstraction    : in Dictionary.Abstractions);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Precondition
        (The_Subprogram : in Subprogram_Info_Ref;
         Abstraction    : in Dictionary.Abstractions;
         Precondition   : in ExaminerConstants.RefType);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              Precondition,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Postcondition
        (The_Subprogram : in Subprogram_Info_Ref;
         Abstraction    : in Dictionary.Abstractions;
         Postcondition  : in ExaminerConstants.RefType);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              Postcondition,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Body (The_Subprogram : in Subprogram_Info_Ref;
                                     The_Body       : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Body,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Has_Proper_Body (The_Subprogram : in Subprogram_Info_Ref;
                                                Is_Hidden      : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Is_Hidden,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Has_Second_Annotation (The_Subprogram : in Subprogram_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Has_Second_Constraint (The_Subprogram : in Subprogram_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Has_Derives_Annotation (The_Subprogram : in Subprogram_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Has_Delay_Property (The_Subprogram : in Subprogram_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Mark_Accounts_For_Delay (The_Subprogram : in Subprogram_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Generic_Unit
        (The_Subprogram   : in Subprogram_Info_Ref;
         The_Generic_Unit : in Generic_Unit_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Generic_Unit,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Instantiation_Of (The_Subprogram : in Subprogram_Info_Ref;
                                                 The_Generic    : in Subprogram_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Generic,
      --#                              The_Subprogram;

      procedure Set_Subprogram_First_Generic_Association
        (The_Subprogram          : in Subprogram_Info_Ref;
         The_Generic_Association : in Generic_Association_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Generic_Association,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Last_Generic_Association
        (The_Subprogram          : in Subprogram_Info_Ref;
         The_Generic_Association : in Generic_Association_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Generic_Association,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Suspends_List (The_Subprogram    : in Subprogram_Info_Ref;
                                              The_Suspends_List : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Subprogram,
      --#                              The_Suspends_List;

      procedure Set_Subprogram_Is_Entry (The_Subprogram : in Subprogram_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Entry_Barrier (The_Subprogram : in Subprogram_Info_Ref;
                                              The_Barrier    : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Barrier,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Is_Interrupt_Handler (The_Subprogram : in Subprogram_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Uses_Unprotected_Variables (The_Subprogram : in Subprogram_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Uses_Unchecked_Conversion (The_Subprogram : in Subprogram_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Assigns_From_External (The_Subprogram : in Subprogram_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Return_Type (The_Subprogram : in Subprogram_Info_Ref;
                                            Type_Mark      : in Type_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Subprogram,
      --#                              Type_Mark;

      procedure Set_Subprogram_With_Clauses
        (The_Subprogram     : in Subprogram_Info_Ref;
         The_Context_Clause : in Context_Clause_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Context_Clause,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Use_Type_Clauses
        (The_Subprogram      : in Subprogram_Info_Ref;
         The_Use_Type_Clause : in Use_Type_Clause_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Subprogram,
      --#                              The_Use_Type_Clause;

      procedure Set_Subprogram_Inherit_Clauses
        (The_Subprogram     : in Subprogram_Info_Ref;
         The_Inherit_Clause : in Context_Clause_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Inherit_Clause,
      --#                              The_Subprogram;

      procedure Set_Subprogram_First_Parameter
        (The_Subprogram           : in Subprogram_Info_Ref;
         The_Subprogram_Parameter : in Subprogram_Parameter_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Subprogram,
      --#                              The_Subprogram_Parameter;

      procedure Set_Subprogram_Last_Parameter
        (The_Subprogram           : in Subprogram_Info_Ref;
         The_Subprogram_Parameter : in Subprogram_Parameter_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Subprogram,
      --#                              The_Subprogram_Parameter;

      procedure Set_Subprogram_First_Global_Variable
        (The_Subprogram      : in Subprogram_Info_Ref;
         Abstraction         : in Dictionary.Abstractions;
         The_Global_Variable : in Global_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              The_Global_Variable,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Last_Global_Variable
        (The_Subprogram      : in Subprogram_Info_Ref;
         Abstraction         : in Dictionary.Abstractions;
         The_Global_Variable : in Global_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              The_Global_Variable,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Renaming_Declarations
        (The_Subprogram  : in Subprogram_Info_Ref;
         The_Declaration : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Subprogram;

      procedure Set_Subprogram_First_Declaration
        (The_Subprogram  : in Subprogram_Info_Ref;
         The_Declaration : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Last_Declaration
        (The_Subprogram  : in Subprogram_Info_Ref;
         The_Declaration : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Subprogram;

      procedure Set_Subprogram_First_Loop (The_Subprogram : in Subprogram_Info_Ref;
                                           The_Loop       : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Loop,
      --#                              The_Subprogram;

      procedure Set_Subprogram_Last_Loop (The_Subprogram : in Subprogram_Info_Ref;
                                          The_Loop       : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Loop,
      --#                              The_Subprogram;

      function Get_Subprogram_Symbol (The_Subprogram : Subprogram_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Subprogram = Null_Subprogram_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Subprogram_Symbol));

      function Get_Subprogram_Name (The_Subprogram : Subprogram_Info_Ref) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Implicit_Proof_Function
        (The_Subprogram : Subprogram_Info_Ref;
         Abstraction    : Dictionary.Abstractions)
        return           Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Implicit_Return_Variable
        (The_Subprogram : Subprogram_Info_Ref;
         Abstraction    : Dictionary.Abstractions)
        return           Implicit_Return_Variable_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Signature_Is_Wellformed
        (The_Subprogram : Subprogram_Info_Ref;
         Abstraction    : Dictionary.Abstractions)
        return           Boolean;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Precondition
        (The_Subprogram : Subprogram_Info_Ref;
         Abstraction    : Dictionary.Abstractions)
        return           ExaminerConstants.RefType;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Postcondition
        (The_Subprogram : Subprogram_Info_Ref;
         Abstraction    : Dictionary.Abstractions)
        return           ExaminerConstants.RefType;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Body (The_Subprogram : Subprogram_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Has_Proper_Body (The_Subprogram : Subprogram_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Body_Is_Hidden (The_Subprogram : Subprogram_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Has_Second_Annotation (The_Subprogram : Subprogram_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Has_Second_Constraint (The_Subprogram : Subprogram_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Has_Derives_Annotation (The_Subprogram : Subprogram_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Has_Delay_Property (The_Subprogram : Subprogram_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Delay_Property_Is_Accounted_For (The_Subprogram : Subprogram_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Generic_Unit (The_Subprogram : Subprogram_Info_Ref) return Generic_Unit_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Instantiation_Of (The_Subprogram : Subprogram_Info_Ref) return Subprogram_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_First_Generic_Association (The_Subprogram : Subprogram_Info_Ref) return Generic_Association_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Last_Generic_Association (The_Subprogram : Subprogram_Info_Ref) return Generic_Association_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Suspends_List (The_Subprogram : Subprogram_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Is_Entry (The_Subprogram : Subprogram_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Entry_Barrier (The_Subprogram : Subprogram_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Is_Interrupt_Handler (The_Subprogram : Subprogram_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Uses_Unprotected_Variables (The_Subprogram : Subprogram_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Uses_Unchecked_Conversion (The_Subprogram : Subprogram_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Assigns_From_External (The_Subprogram : Subprogram_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Return_Type (The_Subprogram : Subprogram_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_With_Clauses (The_Subprogram : Subprogram_Info_Ref) return Context_Clause_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Use_Type_Clauses (The_Subprogram : Subprogram_Info_Ref) return Use_Type_Clause_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Inherit_Clauses (The_Subprogram : Subprogram_Info_Ref) return Context_Clause_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_First_Parameter (The_Subprogram : Subprogram_Info_Ref) return Subprogram_Parameter_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Last_Parameter (The_Subprogram : Subprogram_Info_Ref) return Subprogram_Parameter_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_First_Global_Variable
        (The_Subprogram : Subprogram_Info_Ref;
         Abstraction    : Dictionary.Abstractions)
        return           Global_Variable_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Last_Global_Variable
        (The_Subprogram : Subprogram_Info_Ref;
         Abstraction    : Dictionary.Abstractions)
        return           Global_Variable_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Renaming_Declarations (The_Subprogram : Subprogram_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_First_Declaration (The_Subprogram : Subprogram_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Last_Declaration (The_Subprogram : Subprogram_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Subprogram_First_Loop (The_Subprogram : Subprogram_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Last_Loop (The_Subprogram : Subprogram_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Subprogram_Specification (The_Subprogram : Subprogram_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Operator_Info
      --------------------------------------------------------------------------------

      function Get_Operator_Info_Ref (Item : Dictionary.Symbol) return Operator_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Operator_Info_Ref);

      procedure Create_Unary_Operator
        (Name         : in     SP_Symbols.SP_Symbol;
         Operand      : in     Type_Info_Ref;
         Comp_Unit    : in     ContextManager.UnitDescriptors;
         Loc          : in     LexTokenManager.Token_Position;
         The_Operator :    out Operator_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Operator    from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc,
      --#                              Name,
      --#                              Operand;
      --# post The_Operator /= Null_Operator_Info_Ref;

      procedure Create_Binary_Operator
        (Name         : in     SP_Symbols.SP_Symbol;
         Left, Right  : in     Type_Info_Ref;
         Comp_Unit    : in     ContextManager.UnitDescriptors;
         Loc          : in     LexTokenManager.Token_Position;
         The_Operator :    out Operator_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Operator    from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Left,
      --#                              Loc,
      --#                              Name,
      --#                              Right;
      --# post The_Operator /= Null_Operator_Info_Ref;

      function Get_Operator_Symbol (The_Operator : Operator_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Operator = Null_Operator_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Operator_Symbol));

      function Get_Operator_Name (The_Operator : Operator_Info_Ref) return SP_Symbols.SP_Symbol;
      --# global in Dictionary.Dict;

      function Get_Operator_Is_Binary (The_Operator : Operator_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Operator_Operand (The_Operator : Operator_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Operator_Left_Operand (The_Operator : Operator_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Operator_Right_Operand (The_Operator : Operator_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Dependency_Info
      --------------------------------------------------------------------------------

      type Kind_Of_Dependency_T is (Dependency_Parameter_Item, Dependency_Variable_Item);

      function Get_Dependency_Info_Ref (Item : Dictionary.Symbol) return Dependency_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Dependency_Info_Ref);

      procedure Create_Dependency
        (The_Import_Parameter : in     Subprogram_Parameter_Info_Ref;
         The_Import_Variable  : in     Variable_Info_Ref;
         Comp_Unit            : in     ContextManager.UnitDescriptors;
         Loc                  : in     LexTokenManager.Token_Position;
         The_Dependency       :    out Dependency_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Dependency  from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc,
      --#                              The_Import_Parameter,
      --#                              The_Import_Variable;
      --# post The_Dependency /= Null_Dependency_Info_Ref;

      procedure Set_Next_Dependency (The_Dependency, Next : in Dependency_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              The_Dependency;

      function Get_Dependency_Symbol (The_Dependency : Dependency_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Dependency = Null_Dependency_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Dependency_Symbol));

      function Get_Kind_Of_Dependency (The_Dependency : Dependency_Info_Ref) return Kind_Of_Dependency_T;
      --# global in Dictionary.Dict;

      function Get_Dependency_Import_Parameter (The_Dependency : Dependency_Info_Ref) return Subprogram_Parameter_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Dependency_Import_Variable (The_Dependency : Dependency_Info_Ref) return Variable_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Next_Dependency (The_Dependency : Dependency_Info_Ref) return Dependency_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Package_Info
      --------------------------------------------------------------------------------

      function Get_Package_Info_Ref (Item : Dictionary.Symbol) return Package_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Package_Info_Ref);

      procedure Create_Package
        (Name            : in     LexTokenManager.Lex_String;
         The_Declaration : in     Declaration_Info_Ref;
         Comp_Unit       : in     ContextManager.UnitDescriptors;
         Loc             : in     LexTokenManager.Token_Position;
         The_Package     :    out Package_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Package     from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc,
      --#                              Name,
      --#                              The_Declaration;
      --# post The_Package /= Null_Package_Info_Ref;

      procedure Set_Package_Body (The_Package : in Package_Info_Ref;
                                  The_Body    : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Body,
      --#                              The_Package;

      procedure Set_Package_Has_Proper_Body (The_Package : in Package_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Package;

      procedure Set_Package_Inherit_Clauses (The_Package        : in Package_Info_Ref;
                                             The_Context_Clause : in Context_Clause_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Context_Clause,
      --#                              The_Package;

      procedure Set_Package_Own_Variables (The_Package   : in Package_Info_Ref;
                                           Own_Variables : in Own_Variable_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Own_Variables,
      --#                              The_Package;

      procedure Set_Package_Task_List (The_Package : in Package_Info_Ref;
                                       Task_List   : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Task_List,
      --#                              The_Package;

      procedure Set_Package_First_Loop (The_Package : in Package_Info_Ref;
                                        The_Loop    : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Loop,
      --#                              The_Package;

      procedure Set_Package_Last_Loop (The_Package : in Package_Info_Ref;
                                       The_Loop    : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Loop,
      --#                              The_Package;

      procedure Set_Package_Visible_With_Clauses
        (The_Package        : in Package_Info_Ref;
         The_Context_Clause : in Context_Clause_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Context_Clause,
      --#                              The_Package;

      procedure Set_Package_Visible_Use_Type_Clauses
        (The_Package         : in Package_Info_Ref;
         The_Use_Type_Clause : in Use_Type_Clause_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Package,
      --#                              The_Use_Type_Clause;

      procedure Set_Package_Visible_Renaming_Declarations
        (The_Package     : in Package_Info_Ref;
         The_Declaration : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Package;

      procedure Set_Package_First_Visible_Declaration
        (The_Package     : in Package_Info_Ref;
         The_Declaration : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Package;

      procedure Set_Package_Last_Visible_Declaration
        (The_Package     : in Package_Info_Ref;
         The_Declaration : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Package;

      procedure Set_Package_Local_With_Clauses
        (The_Package        : in Package_Info_Ref;
         The_Context_Clause : in Context_Clause_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Context_Clause,
      --#                              The_Package;

      procedure Set_Package_Local_Use_Type_Clauses
        (The_Package         : in Package_Info_Ref;
         The_Use_Type_Clause : in Use_Type_Clause_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Package,
      --#                              The_Use_Type_Clause;

      procedure Set_Package_Local_Renaming_Declarations
        (The_Package     : in Package_Info_Ref;
         The_Declaration : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Package;

      procedure Set_Package_First_Local_Declaration
        (The_Package     : in Package_Info_Ref;
         The_Declaration : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Package;

      procedure Set_Package_Last_Local_Declaration
        (The_Package     : in Package_Info_Ref;
         The_Declaration : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Package;

      procedure Set_Package_First_Private_Declaration
        (The_Package     : in Package_Info_Ref;
         The_Declaration : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Package;

      procedure Set_Package_Last_Private_Declaration
        (The_Package     : in Package_Info_Ref;
         The_Declaration : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Package;

      procedure Set_Package_Elaborate_Body_Found (The_Package : in Package_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Package;

      procedure Set_Package_Parent (The_Package : in Package_Info_Ref;
                                    The_Parent  : in Package_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Package,
      --#                              The_Parent;

      procedure Set_Package_Next_Sibling (The_Package : in Package_Info_Ref;
                                          The_Sibling : in Package_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Package,
      --#                              The_Sibling;

      procedure Set_Package_Extends (The_Package : in Package_Info_Ref;
                                     The_Extends : in Package_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Extends,
      --#                              The_Package;

      procedure Set_Package_Declares_Tagged_Type (The_Package : in Package_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Package;

      procedure Set_Package_First_Private_Child (The_Package : in Package_Info_Ref;
                                                 The_Child   : in Package_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Child,
      --#                              The_Package;

      procedure Set_Package_Last_Private_Child (The_Package : in Package_Info_Ref;
                                                The_Child   : in Package_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Child,
      --#                              The_Package;

      procedure Set_Package_First_Public_Child (The_Package : in Package_Info_Ref;
                                                The_Child   : in Package_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Child,
      --#                              The_Package;

      procedure Set_Package_Last_Public_Child (The_Package : in Package_Info_Ref;
                                               The_Child   : in Package_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Child,
      --#                              The_Package;

      procedure Set_Package_Is_Private (The_Package : in Package_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Package;

      procedure Set_Package_Generic_Unit (The_Package      : in Package_Info_Ref;
                                          The_Generic_Unit : in Generic_Unit_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Generic_Unit,
      --#                              The_Package;

      function Get_Package_Symbol (The_Package : Package_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Package = Null_Package_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Package_Symbol));

      function Get_Package_Name (The_Package : Package_Info_Ref) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function Get_Package_Body (The_Package : Package_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_Has_Proper_Body (The_Package : Package_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Package_Inherit_Clauses (The_Package : Package_Info_Ref) return Context_Clause_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_Own_Variables (The_Package : Package_Info_Ref) return Own_Variable_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_Task_List (The_Package : Package_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Package_First_Loop (The_Package : Package_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Package_Last_Loop (The_Package : Package_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function Get_Package_Specification (The_Package : Package_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_Visible_With_Clauses (The_Package : Package_Info_Ref) return Context_Clause_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_Visible_Use_Type_Clauses (The_Package : Package_Info_Ref) return Use_Type_Clause_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_Visible_Renaming_Declarations (The_Package : Package_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_First_Visible_Declaration (The_Package : Package_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_Last_Visible_Declaration (The_Package : Package_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_Local_With_Clauses (The_Package : Package_Info_Ref) return Context_Clause_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_Local_Use_Type_Clauses (The_Package : Package_Info_Ref) return Use_Type_Clause_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_Local_Renaming_Declarations (The_Package : Package_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_First_Local_Declaration (The_Package : Package_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_Last_Local_Declaration (The_Package : Package_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_First_Private_Declaration (The_Package : Package_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_Last_Private_Declaration (The_Package : Package_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_Elaborate_Body_Found (The_Package : Package_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Package_Parent (The_Package : Package_Info_Ref) return Package_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_Next_Sibling (The_Package : Package_Info_Ref) return Package_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_Extends (The_Package : Package_Info_Ref) return Package_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_Declares_Tagged_Type (The_Package : Package_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Package_First_Private_Child (The_Package : Package_Info_Ref) return Package_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_Last_Private_Child (The_Package : Package_Info_Ref) return Package_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_First_Public_Child (The_Package : Package_Info_Ref) return Package_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_Last_Public_Child (The_Package : Package_Info_Ref) return Package_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Package_Is_Private (The_Package : Package_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Package_Generic_Unit (The_Package : Package_Info_Ref) return Generic_Unit_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Generic_Parameter_Info
      --------------------------------------------------------------------------------

      function Get_Generic_Parameter_Info_Ref (Item : Dictionary.Symbol) return Generic_Parameter_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Generic_Parameter_Info_Ref);

      procedure Create_Generic_Parameter
        (Owning_Generic        : in     Generic_Unit_Info_Ref;
         Type_Mark             : in     Type_Info_Ref;
         Object                : in     Constant_Info_Ref;
         Comp_Unit             : in     ContextManager.UnitDescriptors;
         Loc                   : in     LexTokenManager.Token_Position;
         The_Generic_Parameter :    out Generic_Parameter_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Generic_Parameter from Comp_Unit,
      --#                                    Dictionary.Dict,
      --#                                    Loc,
      --#                                    Object,
      --#                                    Owning_Generic,
      --#                                    Type_Mark;
      --# post The_Generic_Parameter /= Null_Generic_Parameter_Info_Ref;

      procedure Set_Next_Generic_Parameter (The_Generic_Parameter, Next : in Generic_Parameter_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              The_Generic_Parameter;

      function Get_Generic_Parameter_Symbol (The_Generic_Parameter : Generic_Parameter_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Generic_Parameter = Null_Generic_Parameter_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Generic_Parameter_Symbol));

      function Get_Generic_Parameter_Owning_Generic
        (The_Generic_Parameter : Generic_Parameter_Info_Ref)
        return                  Generic_Unit_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Generic_Parameter_Kind
        (The_Generic_Parameter : Generic_Parameter_Info_Ref)
        return                  Dictionary.Generic_Parameter_Kind;
      --# global in Dictionary.Dict;

      function Get_Generic_Parameter_Type (The_Generic_Parameter : Generic_Parameter_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Generic_Parameter_Object (The_Generic_Parameter : Generic_Parameter_Info_Ref) return Constant_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Next_Generic_Parameter (The_Generic_Parameter : Generic_Parameter_Info_Ref) return Generic_Parameter_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Generic_Unit_Info
      --------------------------------------------------------------------------------

      function Get_Generic_Unit_Info_Ref (Item : Dictionary.Symbol) return Generic_Unit_Info_Ref;
      --# global in Dictionary.Dict;
      --# return S => (Item = Dictionary.NullSymbol <-> S = Null_Generic_Unit_Info_Ref);

      procedure Create_Generic_Unit
        (Kind             : in     Dictionary.Generic_Kind;
         Scope            : in     Dictionary.Scopes;
         Comp_Unit        : in     ContextManager.UnitDescriptors;
         Loc              : in     LexTokenManager.Token_Position;
         The_Generic_Unit :    out Generic_Unit_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Generic_Unit from Comp_Unit,
      --#                               Dictionary.Dict,
      --#                               Kind,
      --#                               Loc,
      --#                               Scope;
      --# post The_Generic_Unit /= Null_Generic_Unit_Info_Ref;

      procedure Set_Generic_Unit_First_Declaration
        (The_Generic_Unit : in Generic_Unit_Info_Ref;
         The_Declaration  : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Generic_Unit;

      procedure Set_Generic_Unit_Last_Declaration
        (The_Generic_Unit : in Generic_Unit_Info_Ref;
         The_Declaration  : in Declaration_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Declaration,
      --#                              The_Generic_Unit;

      procedure Set_Generic_Unit_First_Generic_Parameter
        (The_Generic_Unit      : in Generic_Unit_Info_Ref;
         The_Generic_Parameter : in Generic_Parameter_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Generic_Parameter,
      --#                              The_Generic_Unit;

      procedure Set_Generic_Unit_Last_Generic_Parameter
        (The_Generic_Unit      : in Generic_Unit_Info_Ref;
         The_Generic_Parameter : in Generic_Parameter_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Generic_Parameter,
      --#                              The_Generic_Unit;

      procedure Set_Generic_Unit_Owning_Subprogram
        (The_Generic_Unit : in Generic_Unit_Info_Ref;
         The_Subprogram   : in Subprogram_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Generic_Unit,
      --#                              The_Subprogram;

      procedure Set_Generic_Unit_Owning_Package
        (The_Generic_Unit : in Generic_Unit_Info_Ref;
         The_Package      : in Package_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              The_Generic_Unit,
      --#                              The_Package;

      function Get_Generic_Unit_Symbol (The_Generic_Unit : Generic_Unit_Info_Ref) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      --# return S => ((S = Dictionary.NullSymbol <-> The_Generic_Unit = Null_Generic_Unit_Info_Ref) and
      --#                (S = Dictionary.NullSymbol or GetSymbolDiscriminant (S, Dictionary.Dict) = Dictionary.Generic_Unit_Symbol));

      function Get_Generic_Unit_Scope (The_Generic_Unit : Generic_Unit_Info_Ref) return Dictionary.Scopes;
      --# global in Dictionary.Dict;

      function Get_Generic_Unit_First_Declaration (The_Generic_Unit : Generic_Unit_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Generic_Unit_Last_Declaration (The_Generic_Unit : Generic_Unit_Info_Ref) return Declaration_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Generic_Unit_First_Generic_Parameter
        (The_Generic_Unit : Generic_Unit_Info_Ref)
        return             Generic_Parameter_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Generic_Unit_Last_Generic_Parameter
        (The_Generic_Unit : Generic_Unit_Info_Ref)
        return             Generic_Parameter_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Generic_Unit_Kind (The_Generic_Unit : Generic_Unit_Info_Ref) return Dictionary.Generic_Kind;
      --# global in Dictionary.Dict;

      function Get_Generic_Unit_Owning_Subprogram (The_Generic_Unit : Generic_Unit_Info_Ref) return Subprogram_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Generic_Unit_Owning_Package (The_Generic_Unit : Generic_Unit_Info_Ref) return Package_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --  Generic_Association_Info
      --------------------------------------------------------------------------------

      procedure Create_Generic_Type_Association
        (Formal_Type             : in     Type_Info_Ref;
         Actual_Type             : in     Type_Info_Ref;
         Comp_Unit               : in     ContextManager.UnitDescriptors;
         Loc                     : in     LexTokenManager.Token_Position;
         The_Generic_Association :    out Generic_Association_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Generic_Association from Actual_Type,
      --#                                      Comp_Unit,
      --#                                      Dictionary.Dict,
      --#                                      Formal_Type,
      --#                                      Loc;
      --# post The_Generic_Association /= Null_Generic_Association_Info_Ref;

      procedure Create_Generic_Object_Association
        (Formal_Object           : in     Constant_Info_Ref;
         Actual_Object           : in     Constant_Info_Ref;
         Comp_Unit               : in     ContextManager.UnitDescriptors;
         Loc                     : in     LexTokenManager.Token_Position;
         The_Generic_Association :    out Generic_Association_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         The_Generic_Association from Actual_Object,
      --#                                      Comp_Unit,
      --#                                      Dictionary.Dict,
      --#                                      Formal_Object,
      --#                                      Loc;
      --# post The_Generic_Association /= Null_Generic_Association_Info_Ref;

      procedure Set_Next_Generic_Association (The_Generic_Association, Next : in Generic_Association_Info_Ref);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              The_Generic_Association;

      function Get_Generic_Association_Is_Object (The_Generic_Association : Generic_Association_Info_Ref) return Boolean;
      --# global in Dictionary.Dict;

      function Get_Generic_Association_Formal_Type (The_Generic_Association : Generic_Association_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Generic_Association_Formal_Object (The_Generic_Association : Generic_Association_Info_Ref) return Constant_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Generic_Association_Actual_Type (The_Generic_Association : Generic_Association_Info_Ref) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Generic_Association_Actual_Object (The_Generic_Association : Generic_Association_Info_Ref) return Constant_Info_Ref;
      --# global in Dictionary.Dict;

      function Get_Next_Generic_Association (The_Generic_Association : Generic_Association_Info_Ref) return Generic_Association_Info_Ref;
      --# global in Dictionary.Dict;

      --------------------------------------------------------------------------------
      --------------------------------------------------------------------------------
      --------------------------------------------------------------------------------
      --------------------------------------------------------------------------------

      procedure CreateOwnTask
        (Variable  : in     Variable_Info_Ref;
         Owner     : in     Package_Info_Ref;
         Comp_Unit : in     ContextManager.UnitDescriptors;
         Loc       : in     LexTokenManager.Token_Position;
         OwnTask   :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              Owner,
      --#                              Variable &
      --#         OwnTask         from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;

      procedure SetNextOwnTask (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;

      function GetNextOwnTask (Current : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function GetOwnTaskVariable (OwnTask : Dictionary.Symbol) return Variable_Info_Ref;
      --# global in Dictionary.Dict;

      function GetOwnTaskOwner (OwnTask : Dictionary.Symbol) return Package_Info_Ref;
      --# global in Dictionary.Dict;

      procedure CreateImplicitProofFunction
        (Ada_Function  : in     Subprogram_Info_Ref;
         Comp_Unit     : in     ContextManager.UnitDescriptors;
         Loc           : in     LexTokenManager.Token_Position;
         ProofFunction :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Ada_Function,
      --#                              Comp_Unit,
      --#                              Loc &
      --#         ProofFunction   from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;

      function GetImplicitProofFunctionAdaFunction (ProofFunction : Dictionary.Symbol) return Subprogram_Info_Ref;
      --# global in Dictionary.Dict;

      -- discriminants -----------------------------------------------------------------------

      procedure CreateKnownDiscriminant
        (Name              : in     LexTokenManager.Lex_String;
         Protected_Type    : in     Type_Info_Ref;
         Type_Mark         : in     Type_Info_Ref;
         Comp_Unit         : in     ContextManager.UnitDescriptors;
         Loc               : in     LexTokenManager.Token_Position;
         KnownDiscriminant :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         KnownDiscriminant from Comp_Unit,
      --#                                Dictionary.Dict,
      --#                                Loc,
      --#                                Name,
      --#                                Protected_Type,
      --#                                Type_Mark;

      procedure SetNextDiscriminant (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;

      procedure SetDiscriminantSetsPriority (Discriminant : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Discriminant;

      function GetNextDiscriminant (Discriminant : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function GetDiscriminantName (Discriminant : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function GetDiscriminantTypeMark (Discriminant : Dictionary.Symbol) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function GetDiscriminantProtectedType (Discriminant : Dictionary.Symbol) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function GetDiscriminantSetsPriority (Discriminant : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;

      -- subtype here means extra infor for task and protected subtypes only
      procedure CreateSubtype
        (Comp_Unit  : in     ContextManager.UnitDescriptors;
         Loc        : in     LexTokenManager.Token_Position;
         TheSubtype :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         TheSubtype      from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;

      procedure SetSubtypeInfoPriority (TheSubtype : in Dictionary.Symbol;
                                        Priority   : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Priority,
      --#                              TheSubtype;

      procedure SetSubtypeInfoFirstConstraint (TheSubtype    : in Dictionary.Symbol;
                                               TheConstraint : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheConstraint,
      --#                              TheSubtype;

      procedure SetSubtypeInfoLastConstraint (TheSubtype    : in Dictionary.Symbol;
                                              TheConstraint : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheConstraint,
      --#                              TheSubtype;

      function GetSubtypeInfoPriority (TheSubtype : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function GetSubtypeInfoFirstConstraint (TheSubtype : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function GetSubtypeInfoLastConstraint (TheSubtype : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      procedure CreateDiscriminantConstraint
        (Comp_Unit                 : in     ContextManager.UnitDescriptors;
         Loc                       : in     LexTokenManager.Token_Position;
         TheDiscriminantConstraint :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         TheDiscriminantConstraint from Comp_Unit,
      --#                                        Dictionary.Dict,
      --#                                        Loc;

      procedure SetDiscriminantConstraintStaticValue
        (TheConstraint : in Dictionary.Symbol;
         TheValue      : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheConstraint,
      --#                              TheValue;

      procedure SetDiscriminantConstraintAccessedObject (TheConstraint : in Dictionary.Symbol;
                                                         TheObject     : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheConstraint,
      --#                              TheObject;

      procedure SetNextDiscriminantConstraint (TheConstraint : in Dictionary.Symbol;
                                               Next          : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              TheConstraint;

      function GetDiscriminantConstraintStaticValue (TheConstraint : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function GetDiscriminantConstraintAccessedObject (TheConstraint : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function GetNextDiscriminantConstraint (TheConstraint : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      procedure CreateInterruptStreamMapping
        (TheHandler         : in     LexTokenManager.Lex_String;
         TheInterruptStream : in     LexTokenManager.Lex_String;
         Comp_Unit          : in     ContextManager.UnitDescriptors;
         Loc                : in     LexTokenManager.Token_Position;
         TheMapping         :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              TheHandler,
      --#                              TheInterruptStream &
      --#         TheMapping      from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;

      procedure SetNextInterruptStreamMapping (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;

      function GetNextInterruptStreamMapping (TheMapping : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function GetInterruptStreamMappingHandler (TheMapping : Dictionary.Symbol) return LexTokenManager.Lex_String;

      function GetInterruptStreamMappingStream (TheMapping : Dictionary.Symbol) return LexTokenManager.Lex_String;

      procedure CreateProtectedInfo
        (Comp_Unit : in     ContextManager.UnitDescriptors;
         Loc       : in     LexTokenManager.Token_Position;
         InfoSym   :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         InfoSym         from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;

      procedure CreateSuspendsListItem
        (ThePOorSO        : in     Dictionary.Symbol;
         Comp_Unit        : in     ContextManager.UnitDescriptors;
         Loc              : in     LexTokenManager.Token_Position;
         SuspendsListItem :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict  from *,
      --#                               Comp_Unit,
      --#                               Loc,
      --#                               ThePOorSO &
      --#         SuspendsListItem from Comp_Unit,
      --#                               Dictionary.Dict,
      --#                               Loc;

      procedure SetNextSuspendsListItem (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;

      function GetSuspendsListItem (SuspendsListItem : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      procedure SetSuspendsListItemIsAccountedFor (SuspendsListItem : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              SuspendsListItem;

      function GetSuspendsListItemIsAccountedFor (SuspendsListItem : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;

      function GetNextSuspendsListItem (SuspendsListItem : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      procedure CreateVirtualElement
        (The_Variable      : in     Variable_Info_Ref;
         TheOwner          : in     Dictionary.Symbol;
         Comp_Unit         : in     ContextManager.UnitDescriptors;
         Loc               : in     LexTokenManager.Token_Position;
         TheVirtualElement :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict   from *,
      --#                                Comp_Unit,
      --#                                Loc,
      --#                                TheOwner,
      --#                                The_Variable &
      --#         TheVirtualElement from Comp_Unit,
      --#                                Dictionary.Dict,
      --#                                Loc;

      function GetVirtualElementVariable (VirtualElement : Dictionary.Symbol) return Variable_Info_Ref;
      --# global in Dictionary.Dict;

      function GetVirtualElementOwner (VirtualElement : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      procedure SetVirtualElementSeenByOwner (VirtualElement : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              VirtualElement;

      function GetVirtualElementSeenByOwner (VirtualElement : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;

      procedure SetNextVirtualElement (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;

      function GetNextVirtualElement (VirtualElement : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      procedure CreateLoop
        (Region    : in     Dictionary.Symbol;
         Comp_Unit : in     ContextManager.UnitDescriptors;
         Loc       : in     LexTokenManager.Token_Position;
         TheLoop   :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              Region &
      --#         TheLoop         from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;

      procedure SetLoopName (Name    : in LexTokenManager.Lex_String;
                             TheLoop : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Name,
      --#                              TheLoop;

      procedure SetLoopParameter (ForLoop, LoopParameter : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ForLoop,
      --#                              LoopParameter;

      procedure SetLoopOnEntryVars (ForLoop, OnEntryVars : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ForLoop,
      --#                              OnEntryVars;

      procedure SetLoopExitExpn (ForLoop : in Dictionary.Symbol;
                                 Expn    : in Natural);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Expn,
      --#                              ForLoop;

      procedure SetLoopEntryExpn (ForLoop : in Dictionary.Symbol;
                                  Expn    : in Natural);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Expn,
      --#                              ForLoop;

      procedure SetLoopHasExits (TheLoop : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheLoop;

      procedure SetNextLoop (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;

      function GetLoopName (TheLoop : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function GetLoopRegion (TheLoop : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function GetLoopOnEntryVars (TheLoop : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function GetLoopExitExpn (TheLoop : Dictionary.Symbol) return Natural;
      --# global in Dictionary.Dict;

      function GetLoopEntryExpn (TheLoop : Dictionary.Symbol) return Natural;
      --# global in Dictionary.Dict;

      function GetLoopParameter (TheLoop : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function GetLoopHasExits (TheLoop : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;

      function GetNextLoop (TheLoop : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      procedure CreateTaskInfo
        (Comp_Unit : in     ContextManager.UnitDescriptors;
         Loc       : in     LexTokenManager.Token_Position;
         InfoSym   :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         InfoSym         from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;

      procedure CreateLoopParameter
        (Name           : in     LexTokenManager.Lex_String;
         Type_Mark      : in     Type_Info_Ref;
         TheLoop        : in     Dictionary.Symbol;
         HasStaticRange : in     Boolean;
         IsReverse      : in     Boolean;
         Comp_Unit      : in     ContextManager.UnitDescriptors;
         Loc            : in     LexTokenManager.Token_Position;
         LoopParameter  :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         LoopParameter   from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              HasStaticRange,
      --#                              IsReverse,
      --#                              Loc,
      --#                              Name,
      --#                              TheLoop,
      --#                              Type_Mark;

      function GetLoopParameterName (LoopParameter : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function GetLoopParameterType (LoopParameter : Dictionary.Symbol) return Type_Info_Ref;
      --# global in Dictionary.Dict;

      function GetLoopParameterHasStaticRange (LoopParameter : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;

      function GetLoopParameterIsReverse (LoopParameter : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;

      function GetLoopParameterLoop (LoopParameter : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      procedure CreateLoopEntryVariable
        (OriginalVar       : in     Dictionary.Symbol;
         TheLoop           : in     Dictionary.Symbol;
         Comp_Unit         : in     ContextManager.UnitDescriptors;
         Loc               : in     LexTokenManager.Token_Position;
         LoopEntryVariable :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict   from *,
      --#                                Comp_Unit,
      --#                                Loc,
      --#                                OriginalVar,
      --#                                TheLoop &
      --#         LoopEntryVariable from Comp_Unit,
      --#                                Dictionary.Dict,
      --#                                Loc;

      procedure SetLoopEntryVariableNext (LoopEntryVariable : in Dictionary.Symbol;
                                          Next              : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              LoopEntryVariable,
      --#                              Next;

      function GetLoopEntryVariableOriginalVar (LoopEntryVariable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function GetLoopEntryVariableTheLoop (LoopEntryVariable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function GetLoopEntryVariableNext (LoopEntryVariable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

   private
      --# hide RawDict;

      type Declaration_Info;

      type Declaration_Info_Ref is access all Declaration_Info;

      Null_Declaration_Info_Ref : constant Declaration_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Enumeration_Literal_Info;

      type Enumeration_Literal_Info_Ref is access all Enumeration_Literal_Info;

      Null_Enumeration_Literal_Info_Ref : constant Enumeration_Literal_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Array_Index_Info;

      type Array_Index_Info_Ref is access all Array_Index_Info;

      Null_Array_Index_Info_Ref : constant Array_Index_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Record_Component_Info;

      type Record_Component_Info_Ref is access all Record_Component_Info;

      Null_Record_Component_Info_Ref : constant Record_Component_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Subcomponent_Info;

      type Subcomponent_Info_Ref is access all Subcomponent_Info;

      Null_Subcomponent_Info_Ref : constant Subcomponent_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Type_Info;

      type Type_Info_Ref is access all Type_Info;

      Null_Type_Info_Ref : constant Type_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Constant_Info;

      type Constant_Info_Ref is access all Constant_Info;

      Null_Constant_Info_Ref : constant Constant_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Variable_Info;

      type Variable_Info_Ref is access all Variable_Info;

      Null_Variable_Info_Ref : constant Variable_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Global_Variable_Info;

      type Global_Variable_Info_Ref is access all Global_Variable_Info;

      Null_Global_Variable_Info_Ref : constant Global_Variable_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Own_Variable_Info;

      type Own_Variable_Info_Ref is access all Own_Variable_Info;

      Null_Own_Variable_Info_Ref : constant Own_Variable_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Quantified_Variable_Info;

      type Quantified_Variable_Info_Ref is access all Quantified_Variable_Info;

      Null_Quantified_Variable_Info_Ref : constant Quantified_Variable_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Implicit_Return_Variable_Info;

      type Implicit_Return_Variable_Info_Ref is access all Implicit_Return_Variable_Info;

      Null_Implicit_Return_Variable_Info_Ref : constant Implicit_Return_Variable_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Implicit_In_Stream_Info;

      type Implicit_In_Stream_Info_Ref is access all Implicit_In_Stream_Info;

      Null_Implicit_In_Stream_Info_Ref : constant Implicit_In_Stream_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Rule_Policy_Info;

      type Rule_Policy_Info_Ref is access all Rule_Policy_Info;

      Null_Rule_Policy_Info_Ref : constant Rule_Policy_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Constituent_Info;

      type Constituent_Info_Ref is access all Constituent_Info;

      Null_Constituent_Info_Ref : constant Constituent_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Context_Clause_Info;

      type Context_Clause_Info_Ref is access all Context_Clause_Info;

      Null_Context_Clause_Info_Ref : constant Context_Clause_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Use_Type_Clause_Info;

      type Use_Type_Clause_Info_Ref is access all Use_Type_Clause_Info;

      Null_Use_Type_Clause_Info_Ref : constant Use_Type_Clause_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Parameter_Constraint_Info;

      type Parameter_Constraint_Info_Ref is access all Parameter_Constraint_Info;

      Null_Parameter_Constraint_Info_Ref : constant Parameter_Constraint_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Subprogram_Parameter_Info;

      type Subprogram_Parameter_Info_Ref is access all Subprogram_Parameter_Info;

      Null_Subprogram_Parameter_Info_Ref : constant Subprogram_Parameter_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Subprogram_Info;

      type Subprogram_Info_Ref is access all Subprogram_Info;

      Null_Subprogram_Info_Ref : constant Subprogram_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Operator_Info;

      type Operator_Info_Ref is access all Operator_Info;

      Null_Operator_Info_Ref : constant Operator_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Dependency_Info;

      type Dependency_Info_Ref is access all Dependency_Info;

      Null_Dependency_Info_Ref : constant Dependency_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Package_Info;

      type Package_Info_Ref is access all Package_Info;

      Null_Package_Info_Ref : constant Package_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Generic_Parameter_Info;

      type Generic_Parameter_Info_Ref is access all Generic_Parameter_Info;

      Null_Generic_Parameter_Info_Ref : constant Generic_Parameter_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Generic_Unit_Info;

      type Generic_Unit_Info_Ref is access all Generic_Unit_Info;

      Null_Generic_Unit_Info_Ref : constant Generic_Unit_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Generic_Association_Info;

      type Generic_Association_Info_Ref is access all Generic_Association_Info;

      Null_Generic_Association_Info_Ref : constant Generic_Association_Info_Ref := null;

      --------------------------------------------------------------------------------

      type Symbols is array (Dictionary.Abstractions) of Dictionary.Symbol;

      Null_Symbols : constant Symbols := Symbols'(Dictionary.Abstractions => Dictionary.NullSymbol);

      --------------------------------------------------------------------------------

      type Booleans is array (Dictionary.Abstractions) of Boolean;

      type Constraints is array (Dictionary.Abstractions) of ExaminerConstants.RefType;

      type Global_Variables_T is array (Dictionary.Abstractions) of Global_Variable_Info_Ref;

      type Implicit_Return_Variables_T is array (Dictionary.Abstractions) of Implicit_Return_Variable_Info_Ref;

      type Dependencies_T is array (Dictionary.Abstractions) of Dependency_Info_Ref;

      --------------------------------------------------------------------------------

      type Declaration_Info is record
         Self    : Dictionary.Symbol;
         Context : Dictionary.Contexts;
         Scope   : Dictionary.Scopes;
         Item    : Dictionary.Symbol;
         Next    : Declaration_Info_Ref;
      end record;

      type Enumeration_Literal_Info is record
         Self     : Dictionary.Symbol;
         Name     : LexTokenManager.Lex_String;
         Position : LexTokenManager.Lex_String;
         The_Type : Type_Info_Ref;
         Next     : Enumeration_Literal_Info_Ref;
      end record;

      type Array_Index_Info is record
         Self       : Dictionary.Symbol;
         Index_Type : Type_Info_Ref;
         Next       : Array_Index_Info_Ref;
      end record;

      type Record_Component_Info is record
         Self            : Dictionary.Symbol;
         Name            : LexTokenManager.Lex_String;
         Record_Type     : Type_Info_Ref;
         Component_Type  : Type_Info_Ref;
         Inherited_Field : Boolean; -- tag
         Next            : Record_Component_Info_Ref;
      end record;

      type Subcomponent_Info is record
         Self             : Dictionary.Symbol;
         Object           : Dictionary.Symbol;
         Record_Component : Record_Component_Info_Ref;
         Subcomponents    : Subcomponent_Info_Ref;
         Marked_Valid     : Boolean;
         Next             : Subcomponent_Info_Ref;
      end record;

      type Type_Info (Is_Announcement_Or_Private : Boolean) is record
         Self                  : Dictionary.Symbol;
         Name                  : LexTokenManager.Lex_String;
         Parent                : Type_Info_Ref;
         Declaration           : Declaration_Info_Ref;
         Is_Full_Range_Subtype : Boolean;
         Discriminant          : Dictionary.Type_Discriminant;
         Is_Private            : Dictionary.TriState;
         Is_Limited            : Dictionary.TriState;
         Is_Limited_Private    : Boolean;
         Is_Derived            : Boolean;
         Is_Tagged             : Boolean;
         Is_Own_Var_Type       : Boolean; -- type can be used for abstract view of own variable
         Is_Atomic             : Boolean;
         Equality_Defined      : Boolean;
         Contains_Float        : Boolean;
         Constrained           : Boolean;
         Static                : Boolean;
         Wellformed            : Boolean;
         Base_Type             : Type_Info_Ref;
         Extends               : Type_Info_Ref; -- root type for tagged type extension
         Accesses              : Type_Info_Ref;
         Lower                 : LexTokenManager.Lex_String;
         Upper                 : LexTokenManager.Lex_String;
         Modulus               : LexTokenManager.Lex_String;
         Error_Bound           : LexTokenManager.Lex_String;
         --  linked list of indexes for an array type, or record
         --  components for a record type
         Head                     : Dictionary.Symbol;
         Tail                     : Dictionary.Symbol;
         Component_Type           : Type_Info_Ref;
         The_Virtual_Element_List : Dictionary.Symbol;
         Ancillary_Fields         : Dictionary.Symbol; -- points to ProtectedInfo, TaskInfo or SubtypeInfo record
         Size_Attribute           : LexTokenManager.Lex_String;
         case Is_Announcement_Or_Private is
            when True =>
               Is_Announcement          : Boolean;
               Announcement_Declaration : Declaration_Info_Ref;
               Private_Declaration      : Declaration_Info_Ref;
            when False =>
               Kind_Of_Generic : Dictionary.Generic_Type_Discriminant;
         end case;
      end record;

      type Constant_Info (Is_Deferred : Boolean) is record
         Self              : Dictionary.Symbol;
         Name              : LexTokenManager.Lex_String;
         Static            : Boolean;
         Type_Mark         : Type_Info_Ref;
         Value             : LexTokenManager.Lex_String;
         Exp_Is_Wellformed : Boolean;
         Exp_Node          : ExaminerConstants.RefType;
         -- points back to GenericParameterInfo if const is generic object
         Associated_Generic_Parameter : Generic_Parameter_Info_Ref;
         First_Rule_Policy            : Rule_Policy_Info_Ref;
         Last_Rule_Policy             : Rule_Policy_Info_Ref;
         Declaration                  : Declaration_Info_Ref;
         case Is_Deferred is
            when True =>
               Deferred_Declaration : Declaration_Info_Ref;
            when False =>
               null;
         end case;
      end record;

      type Variable_Info is record
         Self               : Dictionary.Symbol;
         Name               : LexTokenManager.Lex_String;
         Type_Mark          : Type_Info_Ref;
         Abstract_Type_Mark : Type_Info_Ref;
         Initialized        : Boolean;
         Has_Address_Clause : Boolean;
         Has_Pragma_Import  : Boolean;
         Is_Aliased         : Boolean;
         Marked_Valid       : Boolean;
         Exp_Node           : ExaminerConstants.RefType;
         Declaration        : Declaration_Info_Ref;
         Global_References  : Global_Variables_T;
         Own_Variable       : Own_Variable_Info_Ref;
         Own_Task           : Dictionary.Symbol;
         Virtual_Element    : Dictionary.Symbol;
         Constituent        : Constituent_Info_Ref;
         Subcomponents      : Subcomponent_Info_Ref;
      end record;

      type Global_Variable_Info (Kind_Of_Global_Variable : Kind_Of_Global_Variable_T) is record
         Self            : Dictionary.Symbol;
         Mode            : Modes;
         Exported        : Boolean;
         Imported        : Boolean;
         Prefix_Needed   : Boolean;
         Dependencies    : Dependencies_T;
         Next_Subprogram : Global_Variable_Info_Ref;
         Next_Variable   : Global_Variable_Info_Ref;
         case Kind_Of_Global_Variable is
            when Subprogram_Parameter_Item =>
               Subprogram_With_Parameter : Subprogram_Info_Ref;
               Subprogram_Parameter      : Subprogram_Parameter_Info_Ref;
            when Subprogram_Variable_Item =>
               Subprogram_With_Variable : Subprogram_Info_Ref;
               Variable                 : Variable_Info_Ref;
            when Task_Type_Variable_Item =>
               Task_Type          : Type_Info_Ref;
               Task_Type_Variable : Variable_Info_Ref;
         end case;
      end record;

      type Own_Variable_Info is record
         Self                      : Dictionary.Symbol;
         Variable                  : Variable_Info_Ref;
         Owner                     : Dictionary.Symbol;
         Announced                 : Boolean;
         Typed                     : Boolean;
         Initialized               : Boolean;
         Mode                      : Dictionary.Modes;
         Is_Protected              : Boolean;
         Is_Interrupt_Stream       : Boolean;
         Interrupt_Stream_Mappings : Dictionary.Symbol;
         Unprotected_Reference     : Dictionary.Symbol;
         Suspends_Reference        : Dictionary.Symbol;
         Implicit_In_Stream        : Implicit_In_Stream_Info_Ref; -- only set for un-moded, protected own variables
         Priority                  : LexTokenManager.Lex_String; -- StorageRep of number
         Integrity                 : LexTokenManager.Lex_String; -- StorageRep of number
         Suspendable               : Boolean;
         Interruptable             : Boolean;
         Constituents              : Constituent_Info_Ref;
         Next                      : Own_Variable_Info_Ref;
      end record;

      type Quantified_Variable_Info is record
         Self                 : Dictionary.Symbol;
         Name                 : LexTokenManager.Lex_String;
         Type_Mark            : Type_Info_Ref;
         Parameter_Constraint : Parameter_Constraint_Info_Ref;
         Region               : Dictionary.Symbol;
      end record;

      type Implicit_Return_Variable_Info is record
         Self         : Dictionary.Symbol;
         Name         : LexTokenManager.Lex_String;
         The_Function : Subprogram_Info_Ref;
      end record;

      type Implicit_In_Stream_Info is record
         Self         : Dictionary.Symbol;
         Own_Variable : Own_Variable_Info_Ref; -- points back to the own variable with which this stream is associated
      end record;

      type Rule_Policy_Info is record
         Self  : Dictionary.Symbol;
         Scope : Dictionary.Scopes;
         Value : Dictionary.Rule_Policies;
         Next  : Rule_Policy_Info_Ref;
      end record;

      type Constituent_Info is record
         Self         : Dictionary.Symbol;
         Own_Variable : Own_Variable_Info_Ref;
         Variable     : Variable_Info_Ref;
         Mode         : Modes;
         Next         : Constituent_Info_Ref;
      end record;

      type Context_Clause_Info (Is_Subprogram : Boolean) is record
         Self     : Dictionary.Symbol;
         Explicit : Boolean; -- To allow for implicit withing of parents when child with'd
         Next     : Context_Clause_Info_Ref;
         case Is_Subprogram is
            when False =>
               The_Package : Package_Info_Ref;
            when True =>
               The_Subprogram : Subprogram_Info_Ref;
         end case;
      end record;

      type Use_Type_Clause_Info is record
         The_Type : Type_Info_Ref;
         Next     : Use_Type_Clause_Info_Ref;
      end record;

      -- If a subprogram parameter is unconstrained we need symbols to represent the anonymous constraints that
      -- are provided by the calling environment.  These are Parameter_Constraint symbols and they are in a linked list
      -- because the unconstrained formal parameter may have more than one dimension.

      type Parameter_Constraint_Info is record
         Self                 : Dictionary.Symbol;
         Subprogram_Parameter : Subprogram_Parameter_Info_Ref; -- links back to SubprogramParameterInfo
         Dimension            : Positive;
         Next                 : Parameter_Constraint_Info_Ref; -- links to another ParameterConstraintInfo for next dimension
      end record;

      type Subprogram_Parameter_Info is record
         Self              : Dictionary.Symbol;
         Name              : LexTokenManager.Lex_String;
         Subprogram        : Subprogram_Info_Ref;
         Type_Mark         : Type_Info_Ref;
         Mode              : Modes;
         Exported          : Booleans;
         Imported          : Booleans;
         Dependencies      : Dependencies_T;
         Global_References : Global_Variables_T;
         Subcomponents     : Subcomponent_Info_Ref;
         Index_Constraints : Parameter_Constraint_Info_Ref; -- links to Parameter_Constraint_Info record
         Next              : Subprogram_Parameter_Info_Ref;
      end record;

      type Subprogram_Info is record
         Self                      : Dictionary.Symbol;
         Name                      : LexTokenManager.Lex_String;
         Implicit_Proof_Functions  : Symbols;
         Implicit_Return_Variables : Implicit_Return_Variables_T;
         Signature_Is_Wellformed   : Booleans;
         Precondition              : Constraints;
         Postcondition             : Constraints;
         Subprogram_Body           : Declaration_Info_Ref;
         Has_Proper_Body           : Boolean;
         Body_Is_Hidden            : Boolean;
         Has_Second_Annotation     : Boolean;
         Has_Second_Constraint     : Boolean;
         -- true for procedures or entries with no explicit derives annotation
         Has_Derives_Annotation          : Boolean;
         Has_Delay_Property              : Boolean;
         Delay_Property_Is_Accounted_For : Boolean;
         -- set for generic subprogram
         Generic_Unit : Generic_Unit_Info_Ref;
         -- points to generic subprogram symbol if this subprogram is an instantiation
         Instantiation_Of : Subprogram_Info_Ref;

         -- linked list of generic formal/actual pairings, only used if Instantiation_Of is not null
         First_Generic_Association  : Generic_Association_Info_Ref;
         Last_Generic_Association   : Generic_Association_Info_Ref;
         Suspends_List              : Dictionary.Symbol;
         Is_Entry                   : Boolean;
         Entry_Barrier              : Dictionary.Symbol;
         Is_Interrupt_Handler       : Boolean;
         Uses_Unprotected_Variables : Boolean;
         Uses_Unchecked_Conversion  : Boolean;
         Assigns_From_External      : Boolean;
         Type_Mark                  : Type_Info_Ref;
         With_Clauses               : Context_Clause_Info_Ref;
         Use_Type_Clauses           : Use_Type_Clause_Info_Ref;
         Inherit_Clauses            : Context_Clause_Info_Ref;
         First_Subprogram_Parameter : Subprogram_Parameter_Info_Ref;
         Last_Subprogram_Parameter  : Subprogram_Parameter_Info_Ref;
         First_Global_Variable      : Global_Variables_T;
         Last_Global_Variable       : Global_Variables_T;
         Renaming_Declarations      : Declaration_Info_Ref;
         First_Declaration          : Declaration_Info_Ref;
         Last_Declaration           : Declaration_Info_Ref;
         First_Loop                 : Dictionary.Symbol;
         Last_Loop                  : Dictionary.Symbol;
         Specification              : Declaration_Info_Ref;
      end record;

      type Operator_Info (Is_Binary : Boolean) is record
         Self : Dictionary.Symbol;
         Name : SP_Symbols.SP_Symbol;
         case Is_Binary is
            when False =>
               Operand : Type_Info_Ref;
            when True =>
               Left_Operand  : Type_Info_Ref;
               Right_Operand : Type_Info_Ref;
         end case;
      end record;

      type Dependency_Info (Kind_Of_Dependency : Kind_Of_Dependency_T) is record
         Self : Dictionary.Symbol;
         Next : Dependency_Info_Ref;
         case Kind_Of_Dependency is
            when Dependency_Parameter_Item =>
               Import_Parameter : Subprogram_Parameter_Info_Ref;
            when Dependency_Variable_Item =>
               Import_Variable : Variable_Info_Ref;
         end case;
      end record;

      type Part_Info is record
         With_Clauses          : Context_Clause_Info_Ref;
         Use_Type_Clauses      : Use_Type_Clause_Info_Ref;
         Renaming_Declarations : Declaration_Info_Ref;
         First_Declaration     : Declaration_Info_Ref;
         Last_Declaration      : Declaration_Info_Ref;
      end record;

      type Child_Info is record
         First_Child : Package_Info_Ref;
         Last_Child  : Package_Info_Ref;
      end record;

      type Package_Info is record
         Self                    : Dictionary.Symbol;
         Name                    : LexTokenManager.Lex_String;
         Package_Body            : Declaration_Info_Ref;
         Package_Has_Proper_Body : Boolean;
         Inherit_Clauses         : Context_Clause_Info_Ref;
         Own_Variables           : Own_Variable_Info_Ref;
         Task_List               : Dictionary.Symbol;
         First_Loop              : Dictionary.Symbol;
         Last_Loop               : Dictionary.Symbol;
         Specification           : Declaration_Info_Ref;
         Visible_Part            : Part_Info;
         Local_Part              : Part_Info;
         Private_Part            : Part_Info;
         Elaborate_Body_Found    : Boolean;
         Parent                  : Package_Info_Ref;
         Next_Sibling            : Package_Info_Ref;
         Extends                 : Package_Info_Ref; -- type extension/tagged types
         Declares_Tagged_Type    : Boolean;
         Private_Children        : Child_Info;
         Public_Children         : Child_Info;
         Is_Private              : Boolean;
         Generic_Unit            : Generic_Unit_Info_Ref; -- set for generic package
      end record;

      type Generic_Parameter_Info (Kind : Dictionary.Generic_Parameter_Kind) is record
         Self           : Dictionary.Symbol;
         Owning_Generic : Generic_Unit_Info_Ref;
         Next           : Generic_Parameter_Info_Ref;
         case Kind is
            when Dictionary.Generic_Type_Parameter =>
               Type_Mark : Type_Info_Ref;
            when Dictionary.Generic_Object_Parameter =>
               Object : Constant_Info_Ref;
         end case;
      end record;

      type Generic_Unit_Info (Kind : Dictionary.Generic_Kind) is record
         Self              : Dictionary.Symbol;
         Scope             : Scopes;
         First_Declaration : Declaration_Info_Ref;
         Last_Declaration  : Declaration_Info_Ref;
         -- linked list of generic formals
         First_Generic_Parameter : Generic_Parameter_Info_Ref;
         Last_Generic_Parameter  : Generic_Parameter_Info_Ref;
         case Kind is
            when Dictionary.Generic_Of_Subprogram =>
               Owning_Subprogram : Subprogram_Info_Ref;
            when Dictionary.Generic_Of_Package =>
               Owning_Package : Package_Info_Ref;
         end case;
      end record;

      type Generic_Association_Info (Is_Object : Boolean) is record
         Next : Generic_Association_Info_Ref;
         case Is_Object is
            when False =>
               Formal_Type : Type_Info_Ref;
               Actual_Type : Type_Info_Ref;
            when True =>
               Formal_Object : Constant_Info_Ref;
               Actual_Object : Constant_Info_Ref;
         end case;
      end record;

   end RawDict;

   function "=" (Left, Right : RawDict.Kind_Of_Global_Variable_T) return Boolean renames RawDict."=";

   function "=" (Left, Right : RawDict.Declaration_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Enumeration_Literal_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Array_Index_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Record_Component_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Subcomponent_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Type_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Constant_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Variable_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Own_Variable_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Global_Variable_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Quantified_Variable_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Implicit_Return_Variable_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Implicit_In_Stream_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Rule_Policy_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Constituent_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Context_Clause_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Use_Type_Clause_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Parameter_Constraint_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Subprogram_Parameter_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Subprogram_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Operator_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Dependency_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Package_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Generic_Parameter_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Generic_Unit_Info_Ref) return Boolean renames RawDict."=";
   function "=" (Left, Right : RawDict.Generic_Association_Info_Ref) return Boolean renames RawDict."=";

   type DeclarationDiscriminant is (TypeAnnouncement, TypeDeclaration, PrivateTypeDeclaration);

   type ProofStatementDiscriminant is (AssertStatement, CheckStatement);

   type ReferenceDiscriminant is (ReadRef, WriteRef, CallRef, OtherRef);

   type Universal_Types is (Universal_Integer_Type, Universal_Fixed_Type, Universal_Real_Type);

   type Universal_Type_List is array (Universal_Types) of RawDict.Type_Info_Ref;

   type Predefined_Types is (
                             Predefined_Boolean_Type,
                             Predefined_Integer_Type,
                             Predefined_Long_Integer_Type,
                             Predefined_Float_Type,
                             Predefined_Long_Float_Type,
                             Predefined_Character_Type,
                             Predefined_String_Type,
                             Predefined_Natural_Subtype,
                             Predefined_Positive_Subtype,
                             Predefined_Duration_Type,
                             Predefined_Time_Type,
                             Predefined_Time_Span_Type,
                             Predefined_Suspension_Object_Type);

   type Predefined_Type_List is array (Predefined_Types) of RawDict.Type_Info_Ref;

   type Predefined_Literals is (Predefined_False, Predefined_True);

   type Predefined_Literal_List is array (Predefined_Literals) of RawDict.Enumeration_Literal_Info_Ref;

   type Type_Set is record
      Unknown_Type_Mark       : RawDict.Type_Info_Ref;
      The_Universal_Types     : Universal_Type_List;
      The_Predefined_Types    : Predefined_Type_List;
      The_Predefined_Literals : Predefined_Literal_List;
   end record;

   type Main_Program_Set is record
      Subprogram     : RawDict.Subprogram_Info_Ref;
      Priority_Given : Boolean;
      The_Priority   : LexTokenManager.Lex_String;
   end record;

   ------------------------------------------------------------------
   -- Symbols for the predefined packages.  In SPARK83 mode,
   -- these are set to NullSymbol where such packages do not
   -- exist.
   --
   -- Note - package System is NOT included here, since some
   -- older SPARK83 projects supply it explicitly as a shadow
   -- specification which gets read in via the normal index file
   -- mechanism.  More recent projects specify System in the config
   -- file.  In that case, System is added in configfile.adb
   ------------------------------------------------------------------
   type Package_Set is record
      Package_Standard                 : RawDict.Package_Info_Ref; -- All
      Package_ASCII                    : RawDict.Package_Info_Ref; -- All (but considered obsolete in SPARK95)
      Package_Ada                      : RawDict.Package_Info_Ref; -- SPARK95 and RavenSPARK only
      Package_Ada_Characters           : RawDict.Package_Info_Ref; -- SPARK95 nad RavenSPARK only
      Package_Ada_Characters_Latin1    : RawDict.Package_Info_Ref; -- SPARK95 nad RavenSPARK only
      Package_Real_Time                : RawDict.Package_Info_Ref; -- RavenSPARK only
      Package_Synchronous_Task_Control : RawDict.Package_Info_Ref; -- RavenSPARK only
      Package_Interrupts               : RawDict.Package_Info_Ref; -- RavenSPARK only
   end record;

   type Subprogram_Set is record
      STC_Suspend_Until_True : RawDict.Subprogram_Info_Ref; -- STC: Synchronous Task Control
      Ada_Real_Time_Clock    : RawDict.Subprogram_Info_Ref;
      Unchecked_Conversion   : RawDict.Subprogram_Info_Ref;
   end record;

   type Dictionaries is record
      File_Name     : E_Strings.T;
      TemporaryFile : SPARK_IO.File_Type;
      Symbols       : Dynamic_Symbol_Table.T;
      Types         : Type_Set;
      Main          : Main_Program_Set;
      Packages      : Package_Set;
      Subprograms   : Subprogram_Set;
      Null_Variable : RawDict.Variable_Info_Ref; -- an own variable of package standard used as "data sink"
      The_Partition : RawDict.Subprogram_Info_Ref; -- a subprogram record used to store the partition-wide flow relation
   end record;

   --------------------------------------------------------------------------------
   --                                 VARIABLES                                  --
   --------------------------------------------------------------------------------

   Dict : Dictionaries;

   ------------------------------------------------------------------------
   --# inherit CommandLineData,
   --#         CommandLineHandler,
   --#         Dictionary,
   --#         E_Strings,
   --#         FileSystem,
   --#         LexTokenManager,
   --#         Maths,
   --#         RawDict,
   --#         ScreenEcho,
   --#         SPARK_IO,
   --#         SystemErrors,
   --#         XMLReport;
   package TargetData is

      procedure Read_Target_Data_File;
      --# global in     CommandLineData.Content;
      --#        in out Dictionary.Dict;
      --#        in out LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dictionary.Dict,
      --#         LexTokenManager.State,
      --#         SPARK_IO.File_Sys     from *,
      --#                                    CommandLineData.Content,
      --#                                    LexTokenManager.State,
      --#                                    SPARK_IO.File_Sys;

      procedure Output_Target_Data_File (To_File : in SPARK_IO.File_Type);
      --# global in     CommandLineData.Content;
      --#        in out LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives LexTokenManager.State,
      --#         SPARK_IO.File_Sys,
      --#         XMLReport.State       from CommandLineData.Content,
      --#                                    LexTokenManager.State,
      --#                                    SPARK_IO.File_Sys,
      --#                                    To_File,
      --#                                    XMLReport.State;

   end TargetData;

   ---------------------------------------------------------------------------

   --# inherit CommandLineData,
   --#         Dictionary,
   --#         RawDict,
   --#         SP_Symbols,
   --#         SystemErrors;
   package Operator_Is_Visible is

      function Unary_Operator_Is_Visible
        (Name      : SP_Symbols.SP_Symbol;
         Type_Mark : RawDict.Type_Info_Ref;
         Scope     : Dictionary.Scopes)
        return    Boolean;
      --# global in CommandLineData.Content;
      --#        in Dictionary.Dict;

      function Binary_Operator_Is_Visible
        (Name           : SP_Symbols.SP_Symbol;
         The_Left_Type  : RawDict.Type_Info_Ref;
         The_Right_Type : RawDict.Type_Info_Ref;
         Scope          : Dictionary.Scopes)
        return  Boolean;
      --# global in CommandLineData.Content;
      --#        in Dictionary.Dict;

   end Operator_Is_Visible;

   --------------------------------------------------------------------------------

   package body Dynamic_Symbol_Table is separate;

   package body RawDict is separate;

   --------------------------------------------------------------------------------
   --                                SYMBOLS and ORDERING                        --
   --------------------------------------------------------------------------------

   function Declared_Before (Left, Right : Symbol) return Boolean is
   begin
      return Left < Right;
   end Declared_Before;

   --------------------------------------------------------------------------------
   --                                SUBPROGRAMS                                 --
   --------------------------------------------------------------------------------

   function Is_Null_Symbol (TheSymbol : Symbol) return Boolean is
   begin
      return TheSymbol = NullSymbol;
   end Is_Null_Symbol;


   --------------------------------------------------------------------------------

   function IsDeclaration (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Declaration_Symbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Declaration_Symbol;
   end IsDeclaration;

   --------------------------------------------------------------------------------

   function IsEnumerationLiteral (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Enumeration_Literal_Symbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Enumeration_Literal_Symbol;
   end IsEnumerationLiteral;

   --------------------------------------------------------------------------------

   function IsArrayIndex (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Array_Index_Symbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Array_Index_Symbol;
   end IsArrayIndex;

   --------------------------------------------------------------------------------

   function IsSubcomponent (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Subcomponent_Symbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Subcomponent_Symbol;
   end IsSubcomponent;

   --------------------------------------------------------------------------------

   function IsTypeMark (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Type_Symbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol;
   end IsTypeMark;

   --------------------------------------------------------------------------------

   function IsConstant (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Constant_Symbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Constant_Symbol;
   end IsConstant;

   --------------------------------------------------------------------------------

   function IsVariable (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Variable_Symbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Variable_Symbol;
   end IsVariable;

   --------------------------------------------------------------------------------

   function IsGlobalVariable (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Global_Variable_Symbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Global_Variable_Symbol;
   end IsGlobalVariable;

   --------------------------------------------------------------------------------

   function IsQuantifiedVariable (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Quantified_Variable_Symbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Quantified_Variable_Symbol;
   end IsQuantifiedVariable;

   --------------------------------------------------------------------------------

   function IsImplicitReturnVariable (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Implicit_Return_Variable_Symbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Implicit_Return_Variable_Symbol;
   end IsImplicitReturnVariable;

   --------------------------------------------------------------------------------

   function IsImplicitInStream (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Implicit_In_Stream_Symbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Implicit_In_Stream_Symbol;
   end IsImplicitInStream;

   --------------------------------------------------------------------------------

   function IsRulePolicy (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Rule_Policy_Symbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Rule_Policy_Symbol;
   end IsRulePolicy;

   --------------------------------------------------------------------------------

   function IsParameterConstraint (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Parameter_Constraint_Symbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Parameter_Constraint_Symbol;
   end IsParameterConstraint;

   --------------------------------------------------------------------------------

   function IsSubprogramParameter (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Subprogram_Parameter_Symbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Subprogram_Parameter_Symbol;
   end IsSubprogramParameter;

   --------------------------------------------------------------------------------

   function IsSubprogram (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Subprogram_Symbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Subprogram_Symbol;
   end IsSubprogram;

   --------------------------------------------------------------------------------

   function IsOperator (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Operator_Symbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Operator_Symbol;
   end IsOperator;

   --------------------------------------------------------------------------------

   function IsDependency (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Dependency_Symbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Dependency_Symbol;
   end IsDependency;

   --------------------------------------------------------------------------------

   function IsPackage (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Package_Symbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Package_Symbol;
   end IsPackage;

   --------------------------------------------------------------------------------

   function IsGenericParameterSymbol (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Generic_Parameter_Symbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Generic_Parameter_Symbol;
   end IsGenericParameterSymbol;

   --------------------------------------------------------------------------------

   function IsKnownDiscriminant (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = KnownDiscriminantSymbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = KnownDiscriminantSymbol;
   end IsKnownDiscriminant;

   --------------------------------------------------------------------------------

   function IsLoop (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = LoopSymbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = LoopSymbol;
   end IsLoop;

   --------------------------------------------------------------------------------

   function IsImplicitProofFunction (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = ImplicitProofFunctionSymbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = ImplicitProofFunctionSymbol;
   end IsImplicitProofFunction;

   --------------------------------------------------------------------------------

   function IsLoopParameter (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = LoopParameterSymbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = LoopParameterSymbol;
   end IsLoopParameter;

   --------------------------------------------------------------------------------
   --  Are_Equal
   --------------------------------------------------------------------------------

   function Enumeration_Literals_Are_Equal (Left_Symbol, Right_Symbol : Symbol) return Boolean is
   begin
      SystemErrors.RT_Assert
        (C       => (Left_Symbol = NullSymbol
                       or else RawDict.GetSymbolDiscriminant (Left_Symbol) = Enumeration_Literal_Symbol)
           and then (Right_Symbol = NullSymbol
                       or else RawDict.GetSymbolDiscriminant (Right_Symbol) = Enumeration_Literal_Symbol),
         Sys_Err => SystemErrors.Invalid_Symbol_Table,
         Msg     => "in Dictionary.Enumeration_Literals_Are_Equal");
      --# accept Flow, 35, Dict, "Importation of the initial value is ineffective OK" &
      --#        Flow, 50, Dict, "Value is not derived from the imported value OK";
      return Left_Symbol = Right_Symbol;
   end Enumeration_Literals_Are_Equal;

   --------------------------------------------------------------------------------

   function Record_Components_Are_Equal (Left_Symbol, Right_Symbol : Symbol) return Boolean is
   begin
      SystemErrors.RT_Assert
        (C       => (Left_Symbol = NullSymbol
                       or else RawDict.GetSymbolDiscriminant (Left_Symbol) = Record_Component_Symbol)
           and then (Right_Symbol = NullSymbol
                       or else RawDict.GetSymbolDiscriminant (Right_Symbol) = Record_Component_Symbol),
         Sys_Err => SystemErrors.Invalid_Symbol_Table,
         Msg     => "in Dictionary.Record_Components_Are_Equal");
      --# accept Flow, 35, Dict, "Importation of the initial value is ineffective OK" &
      --#        Flow, 50, Dict, "Value is not derived from the imported value OK";
      return Left_Symbol = Right_Symbol;
   end Record_Components_Are_Equal;

   --------------------------------------------------------------------------------

   function Types_Are_Equal (Left_Symbol, Right_Symbol : Symbol;
                             Full_Range_Subtype        : Boolean) return Boolean is
      Result : Boolean;
   begin
      if Full_Range_Subtype then
         Result := RawDict.Get_First_Constrained_Subtype (Type_Mark => RawDict.Get_Type_Info_Ref (Left_Symbol)) = -- GAA External
           RawDict.Get_First_Constrained_Subtype (Type_Mark => RawDict.Get_Type_Info_Ref (Right_Symbol)); -- GAA External
      else
         SystemErrors.RT_Assert
           (C       => (Left_Symbol = NullSymbol
                          or else RawDict.GetSymbolDiscriminant (Left_Symbol) = Type_Symbol)
              and then (Right_Symbol = NullSymbol
                          or else RawDict.GetSymbolDiscriminant (Right_Symbol) = Type_Symbol),
            Sys_Err => SystemErrors.Invalid_Symbol_Table,
            Msg     => "in Dictionary.Types_Are_Equal");
         Result := Left_Symbol = Right_Symbol;
      end if;
      return Result;
   end Types_Are_Equal;

   --------------------------------------------------------------------------------

   function Variables_Are_Equal (Left_Symbol, Right_Symbol : Symbol) return Boolean is
   begin
      SystemErrors.RT_Assert
        (C       => (Left_Symbol = NullSymbol
                       or else RawDict.GetSymbolDiscriminant (Left_Symbol) = Variable_Symbol)
           and then (Right_Symbol = NullSymbol
                       or else RawDict.GetSymbolDiscriminant (Right_Symbol) = Variable_Symbol),
         Sys_Err => SystemErrors.Invalid_Symbol_Table,
         Msg     => "in Dictionary.Variables_Are_Equal");
      --# accept Flow, 35, Dict, "Importation of the initial value is ineffective OK" &
      --#        Flow, 50, Dict, "Value is not derived from the imported value OK";
      return Left_Symbol = Right_Symbol;
   end Variables_Are_Equal;

   --------------------------------------------------------------------------------

   function Implicit_Return_Variables_Are_Equal (Left_Symbol, Right_Symbol : Symbol) return Boolean is
   begin
      SystemErrors.RT_Assert
        (C       => (Left_Symbol = NullSymbol
                       or else RawDict.GetSymbolDiscriminant (Left_Symbol) = Implicit_Return_Variable_Symbol)
           and then (Right_Symbol = NullSymbol
                       or else RawDict.GetSymbolDiscriminant (Right_Symbol) = Implicit_Return_Variable_Symbol),
         Sys_Err => SystemErrors.Invalid_Symbol_Table,
         Msg     => "in Dictionary.Implicit_Return_Variables_Are_Equal");
      --# accept Flow, 35, Dict, "Importation of the initial value is ineffective OK" &
      --#        Flow, 50, Dict, "Value is not derived from the imported value OK";
      return Left_Symbol = Right_Symbol;
   end Implicit_Return_Variables_Are_Equal;

   --------------------------------------------------------------------------------

   function Subprograms_Are_Equal (Left_Symbol, Right_Symbol : Symbol) return Boolean is
   begin
      SystemErrors.RT_Assert
        (C       => (Left_Symbol = NullSymbol
                       or else RawDict.GetSymbolDiscriminant (Left_Symbol) = Subprogram_Symbol)
           and then (Right_Symbol = NullSymbol
                       or else RawDict.GetSymbolDiscriminant (Right_Symbol) = Subprogram_Symbol),
         Sys_Err => SystemErrors.Invalid_Symbol_Table,
         Msg     => "in Dictionary.Subprograms_Are_Equal");
      --# accept Flow, 35, Dict, "Importation of the initial value is ineffective OK" &
      --#        Flow, 50, Dict, "Value is not derived from the imported value OK";
      return Left_Symbol = Right_Symbol;
   end Subprograms_Are_Equal;

   --------------------------------------------------------------------------------

   function Packages_Are_Equal (Left_Symbol, Right_Symbol : Symbol) return Boolean is
   begin
      SystemErrors.RT_Assert
        (C       => (Left_Symbol = NullSymbol
                       or else RawDict.GetSymbolDiscriminant (Left_Symbol) = Package_Symbol)
           and then (Right_Symbol = NullSymbol
                       or else RawDict.GetSymbolDiscriminant (Right_Symbol) = Package_Symbol),
         Sys_Err => SystemErrors.Invalid_Symbol_Table,
         Msg     => "in Dictionary.Packages_Are_Equal");
      --# accept Flow, 35, Dict, "Importation of the initial value is ineffective OK" &
      --#        Flow, 50, Dict, "Value is not derived from the imported value OK";
      return Left_Symbol = Right_Symbol;
   end Packages_Are_Equal;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   procedure Set_Current_File_Name (File_Name : in E_Strings.T) is
   begin
      if CommandLineData.Content.Plain_Output then
         Dict.File_Name := FileSystem.Base_Name (Path   => File_Name,
                                                 Suffix => "");
      else
         Dict.File_Name := File_Name;
      end if;
   end Set_Current_File_Name;

   --------------------------------------------------------------------------------

   function GetRegion (Scope : Scopes) return Symbol is
   begin
      return Scope.The_Unit;
   end GetRegion;

   --------------------------------------------------------------------------------
   --  Get_Unknown_Type_Mark
   --------------------------------------------------------------------------------

   function Get_Unknown_Type_Mark return RawDict.Type_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Types.Unknown_Type_Mark;
   end Get_Unknown_Type_Mark;

   --------------------------------------------------------------------------------

   function GetUnknownTypeMark return Symbol is
   begin
      return RawDict.Get_Type_Symbol (Get_Unknown_Type_Mark); -- GAA External
   end GetUnknownTypeMark;

   --------------------------------------------------------------------------------

   function IsUnknownTypeMark (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.Get_Type_Info_Ref (TheSymbol) = Get_Unknown_Type_Mark; -- GAA External
   end IsUnknownTypeMark;

   ------------------------------------------------------------------------------
   --  Is_Subtype
   ------------------------------------------------------------------------------

   function Is_Subtype (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Type_Parent (Type_Mark => Type_Mark) /= RawDict.Null_Type_Info_Ref;
   end Is_Subtype;

   --------------------------------------------------------------------------------

   function IsSubtype (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Subtype (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol)); -- GAA External
   end IsSubtype;

   --------------------------------------------------------------------------------
   --  Is_Type
   --------------------------------------------------------------------------------

   function Is_Type (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global Dict;
   is
   begin
      return Type_Mark /= Get_Unknown_Type_Mark and then not Is_Subtype (Type_Mark => Type_Mark);
   end Is_Type;

   --------------------------------------------------------------------------------

   function IsType (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol)); -- GAA External
   end IsType;

   --------------------------------------------------------------------------------
   --  Is_Task_Type
   --------------------------------------------------------------------------------

   function Is_Task_Type (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return Is_Type (Type_Mark => Type_Mark)
        and then RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark) = Task_Type_Item;
   end Is_Task_Type;

   --------------------------------------------------------------------------------

   function IsTaskType (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol)); -- GAA External
   end IsTaskType;

   --------------------------------------------------------------------------------
   --  Is_Protected_Type
   --------------------------------------------------------------------------------

   function Is_Protected_Type (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return Is_Type (Type_Mark => Type_Mark)
        and then RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark) = Protected_Type_Item;
   end Is_Protected_Type;

   --------------------------------------------------------------------------------

   function IsProtectedType (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Protected_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol)); -- GAA External
   end IsProtectedType;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function IsProtectedTypeMark (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then RawDict.Get_Type_Discriminant (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol)) = Protected_Type_Item; -- GAA External
   end IsProtectedTypeMark;

   --------------------------------------------------------------------------------

   function Set_Visibility (The_Visibility : Visibility;
                            The_Unit       : Symbol) return Scopes is
   begin
      return Scopes'(The_Visibility => The_Visibility,
                     The_Unit       => The_Unit);
   end Set_Visibility;

   --------------------------------------------------------------------------------

   function Get_Visibility (Scope : Scopes) return Visibility is
   begin
      return Scope.The_Visibility;
   end Get_Visibility;

   --------------------------------------------------------------------------------

   procedure Add_Declaration
     (Comp_Unit       : in     ContextManager.UnitDescriptors;
      Loc             : in     Location;
      Scope           : in     Scopes;
      Context         : in     Contexts;
      The_Declaration :    out RawDict.Declaration_Info_Ref)
   --# global in out Dict;
   --# derives Dict,
   --#         The_Declaration from Comp_Unit,
   --#                              Context,
   --#                              Dict,
   --#                              Loc,
   --#                              Scope;
   --# post The_Declaration /= RawDict.Null_Declaration_Info_Ref;
      is separate;

   --------------------------------------------------------------------------------
   --  Type_Is_Private
   --------------------------------------------------------------------------------

   function Type_Is_Private (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Type_Private_Type_Declaration
        (Type_Mark => RawDict.Get_First_Constrained_Subtype (Type_Mark => Type_Mark)) /= RawDict.Null_Declaration_Info_Ref;
   end Type_Is_Private;

   --------------------------------------------------------------------------------

   function TypeIsPrivate (TheType : Symbol) return Boolean is
   begin
      return Type_Is_Private (Type_Mark => RawDict.Get_Type_Info_Ref (TheType)); -- GAA External
   end TypeIsPrivate;

   --------------------------------------------------------------------------------
   --  Type_Is_Tagged
   --------------------------------------------------------------------------------

   function Type_Is_Tagged (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Type_Is_Tagged (Type_Mark => Type_Mark)
        or else RawDict.Get_Type_Extends (Type_Mark => Type_Mark) /= RawDict.Null_Type_Info_Ref;
      -- derived type of tagged type also tagged
   end Type_Is_Tagged;

   --------------------------------------------------------------------------------

   function TypeIsTagged (TheType : Symbol) return Boolean is
   begin
      return Type_Is_Tagged (Type_Mark => RawDict.Get_Type_Info_Ref (TheType)); -- GAA External
   end TypeIsTagged;

   --------------------------------------------------------------------------------
   --  Type_Is_Extended_Tagged
   --------------------------------------------------------------------------------

   function Type_Is_Extended_Tagged (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Type_Extends (Type_Mark => Type_Mark) /= RawDict.Null_Type_Info_Ref;
   end Type_Is_Extended_Tagged;

   --------------------------------------------------------------------------------

   function TypeIsExtendedTagged (TheType : Symbol) return Boolean is
   begin
      return Type_Is_Extended_Tagged (Type_Mark => RawDict.Get_Type_Info_Ref (TheType)); -- GAA External
   end TypeIsExtendedTagged;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function TypeIsTask (TheType : Symbol) return Boolean is
   begin
      return RawDict.Get_Type_Discriminant (Type_Mark => RawDict.Get_Type_Info_Ref (TheType)) = Task_Type_Item; -- GAA External
   end TypeIsTask;

   --------------------------------------------------------------------------------

   function TypeIsAccess (TheType : Symbol) return Boolean is
   begin
      return RawDict.Get_Type_Discriminant (Type_Mark => RawDict.Get_Type_Info_Ref (TheType)) = Access_Type_Item; -- GAA External
   end TypeIsAccess;

   --------------------------------------------------------------------------------
   --  Is_An_Extension_Of
   --------------------------------------------------------------------------------

   function Is_An_Extension_Of (Root_Type, Extended_Type : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
      Result         : Boolean := False;
      Extended_Local : RawDict.Type_Info_Ref;
   begin
      if Type_Is_Tagged (Type_Mark => Root_Type)
        and then -- to trap null symbol
        Type_Is_Extended_Tagged (Type_Mark => Extended_Type) then
         Extended_Local := Extended_Type;
         loop
            Result := Root_Type = Extended_Local;
            exit when Result;

            Extended_Local := RawDict.Get_Type_Extends (Type_Mark => Extended_Local);
            exit when Extended_Local = RawDict.Null_Type_Info_Ref;
         end loop;
      end if;
      return Result;
   end Is_An_Extension_Of;

   --------------------------------------------------------------------------------

   function IsAnExtensionOf (RootType, ExtendedType : Symbol) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (ExtendedType) = Type_Symbol
        and then Is_An_Extension_Of
        (Root_Type     => RawDict.Get_Type_Info_Ref (RootType), -- GAA External
         Extended_Type => RawDict.Get_Type_Info_Ref (Item => ExtendedType)); -- GAA External
   end IsAnExtensionOf;

   --------------------------------------------------------------------------------
   --  TypeIsAnnounced
   --------------------------------------------------------------------------------

   function TypeIsAnnounced (TheType : Symbol) return Boolean is

      function Type_Is_Announced (Type_Mark : RawDict.Type_Info_Ref) return Boolean
      --# global in Dict;
     is
      begin
         return RawDict.Get_Type_Announcement (Type_Mark => Type_Mark) /= RawDict.Null_Declaration_Info_Ref;
      end Type_Is_Announced;

   begin -- TypeIsAnnounced
      return Type_Is_Announced (Type_Mark => RawDict.Get_Type_Info_Ref (TheType)); -- GAA External
   end TypeIsAnnounced;

   --------------------------------------------------------------------------------
   --  Constant_Is_Deferred
   --------------------------------------------------------------------------------

   function Constant_Is_Deferred (The_Constant : RawDict.Constant_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Constant_Deferred_Declaration (The_Constant => The_Constant) /= RawDict.Null_Declaration_Info_Ref;
   end Constant_Is_Deferred;

   --------------------------------------------------------------------------------

   function IsDeferredConstant (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Constant_Symbol
        and then Constant_Is_Deferred (The_Constant => RawDict.Get_Constant_Info_Ref (Item => TheSymbol)); -- GAA External
   end IsDeferredConstant;

   --------------------------------------------------------------------------------
   --  Get_Declaration
   --------------------------------------------------------------------------------

   function Get_Type_Declaration (Type_Mark : RawDict.Type_Info_Ref) return RawDict.Declaration_Info_Ref
   --# global in Dict;
   is
      The_Declaration : RawDict.Declaration_Info_Ref;
   begin
      -- for a protected or task type we want to return the Ada declaration if available
      if (RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark) = Protected_Type_Item
            or else RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark) = Task_Type_Item)
        and then RawDict.Get_Type_Declaration (Type_Mark => Type_Mark) /= RawDict.Null_Declaration_Info_Ref then
         The_Declaration := RawDict.Get_Type_Declaration (Type_Mark => Type_Mark);
         -- if not one of these types, or if there is not Ada declaration,
         -- return type announcement if available
      elsif RawDict.Get_Type_Announcement (Type_Mark => Type_Mark) /= RawDict.Null_Declaration_Info_Ref
        and then RawDict.Get_Type_Declaration (Type_Mark => Type_Mark) = RawDict.Null_Declaration_Info_Ref then
         -- we have a type announcement but no Ada declaration
         The_Declaration := RawDict.Get_Type_Announcement (Type_Mark => Type_Mark);
         -- or proceed to check for private/visible Ada declarations
      elsif RawDict.Get_Type_Private_Type_Declaration (Type_Mark => Type_Mark) /= RawDict.Null_Declaration_Info_Ref then
         The_Declaration := RawDict.Get_Type_Private_Type_Declaration (Type_Mark => Type_Mark);
      else -- this case now also handles type that is announced _and_ declared
         The_Declaration := RawDict.Get_Type_Declaration (Type_Mark => Type_Mark);
      end if;
      return The_Declaration;
   end Get_Type_Declaration;

   --------------------------------------------------------------------------------

   function Get_Constant_Declaration (The_Constant : RawDict.Constant_Info_Ref) return RawDict.Declaration_Info_Ref
   --# global in Dict;
   is
      The_Declaration : RawDict.Declaration_Info_Ref;
   begin
      if Constant_Is_Deferred (The_Constant => The_Constant) then
         The_Declaration := RawDict.Get_Constant_Deferred_Declaration (The_Constant => The_Constant);
      else
         The_Declaration := RawDict.Get_Constant_Declaration (The_Constant => The_Constant);
      end if;
      return The_Declaration;
   end Get_Constant_Declaration;

   --------------------------------------------------------------------------------

   function SubprogramSignatureIsWellformed (Abstraction : Abstractions;
                                             Subprogram  : Symbol) return Boolean is
      Result : Boolean;
   begin
      case RawDict.GetSymbolDiscriminant (Subprogram) is
         when Subprogram_Symbol =>
            Result :=
              RawDict.Get_Subprogram_Signature_Is_Wellformed
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Subprogram), -- GAA External
               Abstraction    => Abstraction);
         when Type_Symbol =>
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Subprogram)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.SubprogramSignatureIsWellformed");
            Result :=
              RawDict.Get_Task_Type_Signature_Is_Wellformed
              (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => Subprogram), -- GAA External
               Abstraction   => Abstraction);
         when others => -- non-exec code
            Result := False;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.SubprogramSignatureIsWellformed");
      end case;
      return Result;
   end SubprogramSignatureIsWellformed;

   --------------------------------------------------------------------------------

   function GetAdaFunction (ProofFunction : Symbol) return Symbol is
   begin
      return RawDict.Get_Subprogram_Symbol (RawDict.GetImplicitProofFunctionAdaFunction (ProofFunction)); -- GAA External
   end GetAdaFunction;

   --------------------------------------------------------------------------------

   function GetImplicitProofFunction (Abstraction : Abstractions;
                                      TheFunction : Symbol) return Symbol is
   begin
      return RawDict.Get_Subprogram_Implicit_Proof_Function
        (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (TheFunction), -- GAA External
         Abstraction    => Abstraction);
   end GetImplicitProofFunction;

   --------------------------------------------------------------------------------
   --  Has_Implicit_Return_Variable
   --------------------------------------------------------------------------------

   function HasImplicitReturnVariable (Abstraction : Abstractions;
                                       TheFunction : Symbol) return Boolean is

      function Has_Implicit_Return_Variable (The_Function : RawDict.Subprogram_Info_Ref;
                                             Abstraction  : Abstractions) return Boolean
      --# global in Dict;
      is
      begin
         return RawDict.Get_Subprogram_Implicit_Return_Variable
           (The_Subprogram => The_Function,
            Abstraction    => Abstraction) /= RawDict.Null_Implicit_Return_Variable_Info_Ref;
      end Has_Implicit_Return_Variable;

   begin -- HasImplicitReturnVariable
      return Has_Implicit_Return_Variable (The_Function => RawDict.Get_Subprogram_Info_Ref (TheFunction), -- GAA External
                                           Abstraction    => Abstraction);
   end HasImplicitReturnVariable;

   --------------------------------------------------------------------------------
   --  Get_Implicit_Return_Variable
   --------------------------------------------------------------------------------

   function GetImplicitReturnVariable (Abstraction : Abstractions;
                                       TheFunction : Symbol) return Symbol is

      function Get_Implicit_Return_Variable (The_Function : RawDict.Subprogram_Info_Ref;
                                             Abstraction  : Abstractions) return RawDict.Implicit_Return_Variable_Info_Ref
      --# global in Dict;
      is
         Result : RawDict.Implicit_Return_Variable_Info_Ref;
      begin
         Result := RawDict.Get_Subprogram_Implicit_Return_Variable
           (The_Subprogram => The_Function,
            Abstraction    => Abstraction);
         if Result = RawDict.Null_Implicit_Return_Variable_Info_Ref
           and then RawDict.Get_Subprogram_Instantiation_Of (The_Subprogram => The_Function) /= RawDict.Null_Subprogram_Info_Ref
           and then RawDict.Get_Subprogram_Postcondition (The_Subprogram => The_Function,
                                                          Abstraction    => Abstraction) = 0 then
            Result := RawDict.Get_Subprogram_Implicit_Return_Variable
              (The_Subprogram => RawDict.Get_Subprogram_Instantiation_Of (The_Subprogram => The_Function),
               Abstraction    => Abstraction);
         end if;
         return Result;
      end Get_Implicit_Return_Variable;

   begin -- GetImplicitReturnVariable
      return RawDict.Get_Implicit_Return_Variable_Symbol -- GAA External
        (Get_Implicit_Return_Variable
           (The_Function => RawDict.Get_Subprogram_Info_Ref (TheFunction), -- GAA External
            Abstraction  => Abstraction));
   end GetImplicitReturnVariable;

   --------------------------------------------------------------------------------
   --  Get_Type
   --------------------------------------------------------------------------------

   function Get_Subcomponent_Type (The_Subcomponent : RawDict.Subcomponent_Info_Ref) return RawDict.Type_Info_Ref
   --# global in Dict;
   is
   begin
      return RawDict.Get_Record_Component_Type
        (The_Record_Component => RawDict.Get_Subcomponent_Record_Component (The_Subcomponent => The_Subcomponent));
   end Get_Subcomponent_Type;

   --------------------------------------------------------------------------------

   function Get_Parameter_Constraint_Type
     (The_Parameter_Constraint : RawDict.Parameter_Constraint_Info_Ref)
     return                     RawDict.Type_Info_Ref
   --# global in Dict;
   is
      The_Dimension   : Positive;
      The_Array_Index : RawDict.Array_Index_Info_Ref;
   begin
      -- Parameter constraint symbols are only used to communicate information about
      -- unconstrained parameters bewteen the WFFs and te VCG.  As such they don't really
      -- have a type other than for FDL declaration purposes where they behave as if they
      -- were subtypes.  For this purpose it is enough to return the type of matching index of the
      -- uncontrained type of teh obhject associated with the constraint symbol.
      -- So if parameter X is an unconstrained array of type T then the type of X__index__subtype__2 is
      -- the type of the 2nd dimension of T.

      -- The rather cumbersome use of RawDict calls here is caused by ordering problems between Get_Type and the
      -- higher-level Dictionary iterator calls that could otherwise be used.

      The_Dimension := RawDict.Get_Parameter_Constraint_Dimension (The_Parameter_Constraint => The_Parameter_Constraint);

      -- Follow linked list of indexes
      The_Array_Index :=
        RawDict.Get_Type_First_Array_Index
        (Type_Mark => RawDict.Get_Subprogram_Parameter_Type
           (The_Subprogram_Parameter => RawDict.Get_Parameter_Constraint_Subprogram_Parameter
              (The_Parameter_Constraint => The_Parameter_Constraint)));
      while The_Dimension > 1 loop
         The_Array_Index := RawDict.Get_Next_Array_Index (The_Array_Index => The_Array_Index);
         The_Dimension   := The_Dimension - 1;
      end loop;
      return RawDict.Get_Array_Index_Type (The_Array_Index => The_Array_Index);
   end Get_Parameter_Constraint_Type;

   function Get_Type (The_Symbol : Symbol) return RawDict.Type_Info_Ref
   --# global in Dict;
   is
      Result       : RawDict.Type_Info_Ref;
      Local_Symbol : Symbol;

      --------------------------------------------------------------------------------

      function Get_Implicit_Return_Variable_Type
        (The_Implicit_Return_Variable : RawDict.Implicit_Return_Variable_Info_Ref)
        return                         RawDict.Type_Info_Ref
      --# global in Dict;
      is
      begin
         return RawDict.Get_Subprogram_Return_Type
           (The_Subprogram => RawDict.Get_Implicit_Return_Variable_Function
              (The_Implicit_Return_Variable => The_Implicit_Return_Variable));
      end Get_Implicit_Return_Variable_Type;

      --------------------------------------------------------------------------------

      function Get_Generic_Parameter_Type
        (The_Generic_Parameter : RawDict.Generic_Parameter_Info_Ref)
        return                  RawDict.Type_Info_Ref
      --# global in Dict;
      is
         Result : RawDict.Type_Info_Ref;
      begin
         case RawDict.Get_Generic_Parameter_Kind (The_Generic_Parameter => The_Generic_Parameter) is
            when Generic_Type_Parameter =>
               -- if its a type parameter then this is enough, we have the type mark
               Result := RawDict.Get_Generic_Parameter_Type (The_Generic_Parameter => The_Generic_Parameter);
            when Generic_Object_Parameter =>
               -- its an object so above gave us a constant and next line gets its type
               Result :=
                 RawDict.Get_Constant_Type
                 (The_Constant => RawDict.Get_Generic_Parameter_Object (The_Generic_Parameter => The_Generic_Parameter));
         end case;
         return Result;
      end Get_Generic_Parameter_Type;

   begin -- Get_Type

      -- If the symbol is a special variable used to store values on entry to a for loop
      -- then we "dereference" it back to the original variable before seeking the type
      if RawDict.GetSymbolDiscriminant (The_Symbol) = LoopEntryVariableSymbol then
         Local_Symbol := RawDict.GetLoopEntryVariableOriginalVar (The_Symbol);
      else
         Local_Symbol := The_Symbol;
      end if;

      case RawDict.GetSymbolDiscriminant (Local_Symbol) is
         when Type_Symbol =>
            Result := RawDict.Get_Type_Parent (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Local_Symbol));
         when Enumeration_Literal_Symbol =>
            Result :=
              RawDict.Get_Enumeration_Literal_Type
              (The_Enumeration_Literal => RawDict.Get_Enumeration_Literal_Info_Ref (Item => Local_Symbol));
         when Record_Component_Symbol =>
            Result :=
              RawDict.Get_Record_Component_Type
              (The_Record_Component => RawDict.Get_Record_Component_Info_Ref (Item => Local_Symbol));
         when Subcomponent_Symbol =>
            Result := Get_Subcomponent_Type (The_Subcomponent => RawDict.Get_Subcomponent_Info_Ref (Item => Local_Symbol));
         when Variable_Symbol =>
            Result := RawDict.Get_Variable_Type (The_Variable => RawDict.Get_Variable_Info_Ref (Item => Local_Symbol));
         when Quantified_Variable_Symbol =>
            Result :=
              RawDict.Get_Quantified_Variable_Type
              (The_Quantified_Variable => RawDict.Get_Quantified_Variable_Info_Ref (Item => Local_Symbol));
         when Implicit_Return_Variable_Symbol =>
            Result :=
              Get_Implicit_Return_Variable_Type
              (The_Implicit_Return_Variable => RawDict.Get_Implicit_Return_Variable_Info_Ref (Item => Local_Symbol));
         when Constant_Symbol =>
            Result := RawDict.Get_Constant_Type (The_Constant => RawDict.Get_Constant_Info_Ref (Item => Local_Symbol));
         when Subprogram_Symbol =>
            Result := RawDict.Get_Subprogram_Return_Type (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Local_Symbol));
         when ImplicitProofFunctionSymbol =>
            Result :=
              RawDict.Get_Subprogram_Return_Type (The_Subprogram => RawDict.GetImplicitProofFunctionAdaFunction (Local_Symbol));
         when Subprogram_Parameter_Symbol =>
            Result :=
              RawDict.Get_Subprogram_Parameter_Type
              (The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => Local_Symbol));
         when Parameter_Constraint_Symbol =>
            Result :=
              Get_Parameter_Constraint_Type
              (The_Parameter_Constraint => RawDict.Get_Parameter_Constraint_Info_Ref (Item => Local_Symbol));
         when Generic_Parameter_Symbol =>
            Result :=
              Get_Generic_Parameter_Type (The_Generic_Parameter => RawDict.Get_Generic_Parameter_Info_Ref (Item => Local_Symbol));
         when LoopParameterSymbol =>
            Result := RawDict.GetLoopParameterType (Local_Symbol);
         when KnownDiscriminantSymbol =>
            Result := RawDict.GetDiscriminantTypeMark (Local_Symbol);
         when others => -- non-exec code
            Result := RawDict.Null_Type_Info_Ref;
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.Get_Type");
      end case;
      return Result;
   end Get_Type;

   ------------------------------------------------------------------------------

   function GetType (TheSymbol : Symbol) return Symbol is
   begin
      return RawDict.Get_Type_Symbol (Get_Type (The_Symbol => TheSymbol)); -- GAA External
   end GetType;

   --------------------------------------------------------------------------------
   --  Get_Root_Type
   --------------------------------------------------------------------------------

   function Get_Root_Type (Type_Mark : RawDict.Type_Info_Ref) return RawDict.Type_Info_Ref
   --# global in Dict;
   is
      Current : RawDict.Type_Info_Ref;
   begin
      Current := Type_Mark;
      -- loop through possible chain of subtype to get actual type
      while Current /= Get_Unknown_Type_Mark and then Is_Subtype (Type_Mark => Current) loop
         Current := RawDict.Get_Type_Parent (Type_Mark => Current);
      end loop;
      return Current;
   end Get_Root_Type;

   --------------------------------------------------------------------------------

   function GetRootType (TypeMark : Symbol) return Symbol is
      Current : RawDict.Type_Info_Ref;
   begin
      case RawDict.GetSymbolDiscriminant (TypeMark) is
         when Parameter_Constraint_Symbol =>
            -- synthetic index type of unconst array params, first get type of the constraint
            Current := Get_Parameter_Constraint_Type
              (The_Parameter_Constraint => RawDict.Get_Parameter_Constraint_Info_Ref (Item => TypeMark)); -- GAA External
         when Type_Symbol =>
            Current := RawDict.Get_Type_Info_Ref (Item => TypeMark); -- GAA External
         when others =>
            Current := RawDict.Null_Type_Info_Ref;
      end case;
      return RawDict.Get_Type_Symbol (Get_Root_Type (Type_Mark => Current)); -- GAA External
   end GetRootType;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function Is_Variable (TheSymbol : Symbol) return Boolean is
      Result : Boolean;
   begin
      case RawDict.GetSymbolDiscriminant (TheSymbol) is
         when Variable_Symbol                 |
           LoopEntryVariableSymbol         |
           Quantified_Variable_Symbol      |
           Implicit_Return_Variable_Symbol |
           Subprogram_Parameter_Symbol     |
           LoopParameterSymbol             =>
            Result := True;
         when others =>
            Result := False;
      end case;
      return Result;
   end Is_Variable;

   --------------------------------------------------------------------------------
   --  Type_Is_Atomic
   --------------------------------------------------------------------------------

   function Type_Is_Atomic (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return -- If this is a subtype then check the base type
        (Is_Subtype (Type_Mark => Type_Mark)
           and then RawDict.Get_Type_Atomic (Type_Mark => Get_Root_Type (Type_Mark => Type_Mark)))
        or else RawDict.Get_Type_Atomic (Type_Mark => Type_Mark); -- Otherwise just check the type itself
   end Type_Is_Atomic;

   ------------------------------------------------------------------------------

   function IsAtomic (TheSymbol : Symbol) return Boolean is
   begin
      return (Is_Variable (TheSymbol) and then Type_Is_Atomic (Get_Type (The_Symbol => TheSymbol)))
        or else (RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
                   and then Type_Is_Atomic (RawDict.Get_Type_Info_Ref (Item => TheSymbol)));
   end IsAtomic;

   ------------------------------------------------------------------------------
   --  Is_Predefined_Suspension_Object_Type
   ------------------------------------------------------------------------------

   function Is_Predefined_Suspension_Object_Type (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in CommandLineData.Content;
   --#        in Dict;
   is
   begin
      return CommandLineData.Ravenscar_Selected
        and then Type_Mark = Dict.Types.The_Predefined_Types (Predefined_Suspension_Object_Type);
   end Is_Predefined_Suspension_Object_Type;

   --------------------------------------------------------------------------------

   function IsPredefinedSuspensionObjectType (TheSymbol : Symbol) return Boolean is
   begin
      return Is_Predefined_Suspension_Object_Type (Type_Mark => RawDict.Get_Type_Info_Ref (TheSymbol)); -- GAA External
   end IsPredefinedSuspensionObjectType;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function IsPragmaAtomic (TheSymbol : Symbol) return Boolean is
   begin
      return IsAtomic (TheSymbol)
        and then not (RawDict.Get_Type_Discriminant (Type_Mark => Get_Type (The_Symbol => TheSymbol)) = Protected_Type_Item)
        and then not Is_Predefined_Suspension_Object_Type (Type_Mark => Get_Type (The_Symbol => TheSymbol));
   end IsPragmaAtomic;

   ------------------------------------------------------------------------------

   function Get_Unconstrained_Array_Index (TheSymbol : Symbol) return Symbol is
   begin
      return RawDict.Get_Parameter_Constraint_Symbol -- GAA External
        (RawDict.Get_Quantified_Variable_Constraint
           (The_Quantified_Variable => RawDict.Get_Quantified_Variable_Info_Ref (TheSymbol))); -- GAA External
   end Get_Unconstrained_Array_Index;

   ------------------------------------------------------------------------------

   function GetAccess (TheProtectedType : Symbol) return Symbol is
   begin
      return RawDict.Get_Type_Symbol -- GAA External
        (RawDict.Get_Type_Accesses (Type_Mark => RawDict.Get_Type_Info_Ref (TheProtectedType))); -- GAA External
   end GetAccess;

   ------------------------------------------------------------------------------
   --  Get_Context
   ------------------------------------------------------------------------------

   function Get_Type_Context (Type_Mark : RawDict.Type_Info_Ref) return Contexts
   --# global in Dict;
   is
   begin
      return RawDict.Get_Declaration_Context (The_Declaration => Get_Type_Declaration (Type_Mark => Type_Mark));
   end Get_Type_Context;

   ------------------------------------------------------------------------------

   function Get_Constant_Context (The_Constant : RawDict.Constant_Info_Ref) return Contexts
   --# global in Dict;
   is
   begin
      return RawDict.Get_Declaration_Context (The_Declaration => Get_Constant_Declaration (The_Constant => The_Constant));
   end Get_Constant_Context;

   ------------------------------------------------------------------------------

   function Get_Subprogram_Context (The_Subprogram : RawDict.Subprogram_Info_Ref) return Contexts
   --# global in Dict;
   is
   begin
      return RawDict.Get_Declaration_Context
        (The_Declaration => RawDict.Get_Subprogram_Specification (The_Subprogram => The_Subprogram));
   end Get_Subprogram_Context;

   ------------------------------------------------------------------------------

   function GetContext (TheSymbol : Symbol) return Contexts is
      Context : Contexts;

      ------------------------------------------------------------------------------

      function Get_Package_Context (The_Package : RawDict.Package_Info_Ref) return Contexts
      --# global in Dict;
      is
      begin
         return RawDict.Get_Declaration_Context (The_Declaration => RawDict.Get_Package_Specification (The_Package => The_Package));
      end Get_Package_Context;

      ------------------------------------------------------------------------------

      function Get_Variable_Context (The_Variable : RawDict.Variable_Info_Ref) return Contexts
      --# global in Dict;
      is
         Context : Contexts;
      begin
         if RawDict.Get_Variable_Declaration (The_Variable => The_Variable) = RawDict.Null_Declaration_Info_Ref then
            Context := ProofContext;
         else
            Context :=
              RawDict.Get_Declaration_Context
              (The_Declaration => RawDict.Get_Variable_Declaration (The_Variable => The_Variable));
         end if;
         return Context;
      end Get_Variable_Context;

      ------------------------------------------------------------------------------

      function Get_Enumeration_Literal_Context (The_Enumeration_Literal : RawDict.Enumeration_Literal_Info_Ref) return Contexts
      --# global in Dict;
      is
      begin
         return RawDict.Get_Declaration_Context
           (The_Declaration => Get_Type_Declaration
              (Type_Mark => RawDict.Get_Enumeration_Literal_Type (The_Enumeration_Literal => The_Enumeration_Literal)));
      end Get_Enumeration_Literal_Context;

      ------------------------------------------------------------------------------

      function Get_Record_Component_Context (The_Record_Component : RawDict.Record_Component_Info_Ref) return Contexts
      --# global in Dict;
      is
      begin
         return RawDict.Get_Declaration_Context
           (The_Declaration => Get_Type_Declaration
              (Type_Mark => RawDict.Get_Record_Component_Type (The_Record_Component => The_Record_Component)));
      end Get_Record_Component_Context;

      ------------------------------------------------------------------------------

      function Get_Subprogram_Parameter_Context
        (The_Subprogram_Parameter : RawDict.Subprogram_Parameter_Info_Ref)
        return                     Contexts
      --# global in Dict;
      is
      begin
         return RawDict.Get_Declaration_Context
           (The_Declaration => RawDict.Get_Subprogram_Specification
              (The_Subprogram => RawDict.Get_Subprogram_Parameter_Subprogram
                 (The_Subprogram_Parameter => The_Subprogram_Parameter)));
      end Get_Subprogram_Parameter_Context;

   begin -- GetContext
      case RawDict.GetSymbolDiscriminant (TheSymbol) is
         when Type_Symbol =>
            Context := Get_Type_Context (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol)); -- GAA External
         when Constant_Symbol =>
            Context := Get_Constant_Context (The_Constant => RawDict.Get_Constant_Info_Ref (Item => TheSymbol)); -- GAA External
         when Package_Symbol =>
            Context := Get_Package_Context (The_Package => RawDict.Get_Package_Info_Ref (Item => TheSymbol)); -- GAA External
         when Subprogram_Symbol =>
            Context := Get_Subprogram_Context
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => TheSymbol)); -- GAA External
         when Variable_Symbol =>
            Context := Get_Variable_Context (The_Variable => RawDict.Get_Variable_Info_Ref (Item => TheSymbol)); -- GAA External
         when Quantified_Variable_Symbol | ImplicitProofFunctionSymbol | Implicit_Return_Variable_Symbol =>
            Context := ProofContext;
         when Enumeration_Literal_Symbol =>
            Context :=
              Get_Enumeration_Literal_Context
              (The_Enumeration_Literal => RawDict.Get_Enumeration_Literal_Info_Ref (Item => TheSymbol)); -- GAA External
         when Record_Component_Symbol =>
            Context := Get_Record_Component_Context
              (The_Record_Component => RawDict.Get_Record_Component_Info_Ref (Item => TheSymbol)); -- GAA External
         when LoopSymbol | LoopParameterSymbol =>
            Context := ProgramContext;
         when Subprogram_Parameter_Symbol =>
            Context :=
              Get_Subprogram_Parameter_Context
              (The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => TheSymbol)); -- GAA External
         when others =>
            Context := ProgramContext;
      end case;
      return Context;
   end GetContext;

   --------------------------------------------------------------------------------
   --  Get_Root_Package
   --------------------------------------------------------------------------------

   function Get_Root_Package (The_Package : RawDict.Package_Info_Ref) return RawDict.Package_Info_Ref
   --# global in Dict;
   is
      Current_Package, Next_Package : RawDict.Package_Info_Ref;
   begin
      Next_Package := The_Package;
      loop
         Current_Package := Next_Package;
         Next_Package    := RawDict.Get_Package_Parent (The_Package => Current_Package);
         exit when Next_Package = RawDict.Null_Package_Info_Ref;
      end loop;
      return Current_Package;
   end Get_Root_Package;

   --------------------------------------------------------------------------------

   function GetRootPackage (ThePackage : Symbol) return Symbol is
   begin
      return RawDict.Get_Package_Symbol (Get_Root_Package (The_Package => RawDict.Get_Package_Info_Ref (ThePackage))); -- GAA External
   end GetRootPackage;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function IsPrivatePackage (ThePackage : Symbol) return Boolean is
   begin
      return RawDict.Get_Package_Is_Private (The_Package => RawDict.Get_Package_Info_Ref (ThePackage)); -- GAA External
   end IsPrivatePackage;

   --------------------------------------------------------------------------------

   function GetPackageParent (ThePackage : Symbol) return Symbol is
   begin
      return RawDict.Get_Package_Symbol (RawDict.Get_Package_Parent (The_Package => RawDict.Get_Package_Info_Ref (ThePackage))); -- GAA External
   end GetPackageParent;

   --------------------------------------------------------------------------------

   function PackageDeclaresTaggedType (ThePackage : Symbol) return Boolean is
   begin
      return RawDict.Get_Package_Declares_Tagged_Type (The_Package => RawDict.Get_Package_Info_Ref (ThePackage)); -- GAA External
   end PackageDeclaresTaggedType;

   --------------------------------------------------------------------------------
   --  Package_Extends_Another_Package
   --------------------------------------------------------------------------------

   function PackageExtendsAnotherPackage (ThePackage : Symbol) return Boolean is

      function Package_Extends_Another_Package (The_Package : RawDict.Package_Info_Ref) return Boolean
      --# global in Dict;
      is
      begin
         return RawDict.Get_Package_Extends (The_Package => The_Package) /= RawDict.Null_Package_Info_Ref;
      end Package_Extends_Another_Package;

   begin -- PackageExtendsAnotherPackage
      return Package_Extends_Another_Package (The_Package => RawDict.Get_Package_Info_Ref (ThePackage)); -- GAA External
   end PackageExtendsAnotherPackage;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function GetPackageThatIsExtended (ThePackage : Symbol) return Symbol is
   begin
      return RawDict.Get_Package_Symbol (RawDict.Get_Package_Extends (The_Package => RawDict.Get_Package_Info_Ref (ThePackage))); -- GAA External
   end GetPackageThatIsExtended;

   --------------------------------------------------------------------------------
   --  Is_Proper_Descendent
   --------------------------------------------------------------------------------

   function Is_Proper_Descendent (Inner_Package, Outer_Package : RawDict.Package_Info_Ref) return Boolean
   --# global in Dict;
   is
      Current_Package : RawDict.Package_Info_Ref;
   begin
      Current_Package := Inner_Package;
      loop
         Current_Package := RawDict.Get_Package_Parent (The_Package => Current_Package);
         exit when Current_Package = Outer_Package or else Current_Package = RawDict.Null_Package_Info_Ref;
      end loop;
      return Current_Package = Outer_Package;
   end Is_Proper_Descendent;

   --------------------------------------------------------------------------------

   function IsProperDescendent (Inner, Outer : Symbol) return Boolean is
   begin
      return Is_Proper_Descendent (Inner_Package => RawDict.Get_Package_Info_Ref (Inner), -- GAA External
                                   Outer_Package => RawDict.Get_Package_Info_Ref (Outer)); -- GAA External
   end IsProperDescendent;

   --------------------------------------------------------------------------------
   --  Is_Private_Descendent
   --------------------------------------------------------------------------------

   function IsPrivateDescendent (Inner, Outer : Symbol) return Boolean is

      function Is_Private_Descendent (Inner_Package, Outer_Package : RawDict.Package_Info_Ref) return Boolean
      --# global in Dict;
      is
         Current_Package : RawDict.Package_Info_Ref;
         Is_Private      : Boolean := False;
      begin
         Current_Package := Inner_Package;
         loop
            Is_Private      := Is_Private or else RawDict.Get_Package_Is_Private (The_Package => Current_Package);
            Current_Package := RawDict.Get_Package_Parent (The_Package => Current_Package);
            exit when Current_Package = Outer_Package or else Current_Package = RawDict.Null_Package_Info_Ref;
         end loop;
         return Current_Package = Outer_Package and then Is_Private;
      end Is_Private_Descendent;

   begin -- IsPrivateDescendent
      return Is_Private_Descendent (Inner_Package => RawDict.Get_Package_Info_Ref (Inner), -- GAA External
                                    Outer_Package => RawDict.Get_Package_Info_Ref (Outer)); -- GAA External
   end IsPrivateDescendent;

   --------------------------------------------------------------------------------
   --  Is_Descendent_Of_Private_Sibling
   --------------------------------------------------------------------------------

   function IsDescendentOfPrivateSibling (Candidate, ThePackage : Symbol) return Boolean is

      function Is_Descendent_Of_Private_Sibling (Candidate, The_Package : RawDict.Package_Info_Ref) return Boolean
      --# global in Dict;
      is
         Parent_Package, Current_Package, Next_Package : RawDict.Package_Info_Ref;
      begin
         Parent_Package  := RawDict.Get_Package_Parent (The_Package => The_Package);
         Current_Package := Candidate;
         loop
            Next_Package := RawDict.Get_Package_Parent (The_Package => Current_Package);
            exit when Next_Package = Parent_Package or else Next_Package = RawDict.Null_Package_Info_Ref;
            Current_Package := Next_Package;
         end loop;
         return Next_Package = Parent_Package and then RawDict.Get_Package_Is_Private (The_Package => Current_Package);
      end Is_Descendent_Of_Private_Sibling;

   begin -- IsDescendentOfPrivateSibling
      return Is_Descendent_Of_Private_Sibling (Candidate   => RawDict.Get_Package_Info_Ref (Candidate), -- GAA External
                                               The_Package => RawDict.Get_Package_Info_Ref (ThePackage)); -- GAA External
   end IsDescendentOfPrivateSibling;

   --------------------------------------------------------------------------------
   --  Is_Declared
   --------------------------------------------------------------------------------

   function Variable_Is_Declared (The_Variable : RawDict.Variable_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Variable_Declaration (The_Variable => The_Variable) /= RawDict.Null_Declaration_Info_Ref;
   end Variable_Is_Declared;

   --------------------------------------------------------------------------------

   function Is_Declared (Item : Symbol) return Boolean is
      Result : Boolean;

      --------------------------------------------------------------------------------

      function Type_Is_Declared (Type_Mark : RawDict.Type_Info_Ref) return Boolean
      --# global in Dict;
      is
      begin
         return RawDict.Get_Type_Declaration (Type_Mark => Type_Mark) /= RawDict.Null_Declaration_Info_Ref;
      end Type_Is_Declared;

      --------------------------------------------------------------------------------

      function Constant_Is_Declared (The_Constant : RawDict.Constant_Info_Ref) return Boolean
      --# global in Dict;
      is
      begin
         return RawDict.Get_Constant_Declaration (The_Constant => The_Constant) /= RawDict.Null_Declaration_Info_Ref;
      end Constant_Is_Declared;

   begin -- Is_Declared
      case RawDict.GetSymbolDiscriminant (Item) is
         when Type_Symbol =>
            Result := Type_Is_Declared (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Item)); -- GAA External
         when Variable_Symbol =>
            Result := Variable_Is_Declared (The_Variable => RawDict.Get_Variable_Info_Ref (Item => Item)); -- GAA External
         when Constant_Symbol =>
            Result := Constant_Is_Declared (The_Constant => RawDict.Get_Constant_Info_Ref (Item => Item)); -- GAA External
         when Quantified_Variable_Symbol      |
           Subprogram_Symbol               |
           Package_Symbol                  |
           LoopParameterSymbol             |
           Enumeration_Literal_Symbol      |
           Record_Component_Symbol         |
           ImplicitProofFunctionSymbol     |
           Subcomponent_Symbol             |
           Implicit_Return_Variable_Symbol |
           Subprogram_Parameter_Symbol     =>
            Result := True;
         when others =>
            Result := False;
      end case;
      return Result;
   end Is_Declared;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function Get_Value (The_Constant : Symbol) return LexTokenManager.Lex_String is
      Result : LexTokenManager.Lex_String := LexTokenManager.Null_String;
   begin
      if RawDict.GetSymbolDiscriminant (The_Constant) = Constant_Symbol then
         Result := RawDict.Get_Constant_Value
           (The_Constant => RawDict.Get_Constant_Info_Ref (Item => The_Constant)); -- GAA External
      end if;
      return Result;
   end Get_Value;

   --------------------------------------------------------------------------------

   function GetConstantExpNode (TheConstant : Symbol) return ExaminerConstants.RefType is
   begin
      return RawDict.Get_Constant_Exp_Node (The_Constant => RawDict.Get_Constant_Info_Ref (TheConstant)); -- GAA External
   end GetConstantExpNode;

   --------------------------------------------------------------------------------

   function ConstantExpIsWellformed (TheConstant : Symbol) return Boolean is
   begin
      return RawDict.Get_Constant_Exp_Is_Wellformed (The_Constant => RawDict.Get_Constant_Info_Ref (TheConstant)); -- GAA External
   end ConstantExpIsWellformed;

   --------------------------------------------------------------------------------
   --  Is_Own_Variable
   --------------------------------------------------------------------------------

   function Is_Own_Variable (The_Variable : RawDict.Variable_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Variable_Own_Variable (The_Variable => The_Variable) /= RawDict.Null_Own_Variable_Info_Ref;
   end Is_Own_Variable;

   --------------------------------------------------------------------------------

   function IsOwnVariable (Variable : Symbol) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (Variable) = Variable_Symbol
        and then Is_Own_Variable (The_Variable => RawDict.Get_Variable_Info_Ref (Item => Variable)); -- GAA External
   end IsOwnVariable;

   --------------------------------------------------------------------------------
   --  Is_Own_Task
   --------------------------------------------------------------------------------

   function Is_Own_Task (The_Variable : RawDict.Variable_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Variable_Own_Task (The_Variable => The_Variable) /= NullSymbol;
   end Is_Own_Task;

   --------------------------------------------------------------------------------

   function IsOwnTask (Variable : Symbol) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (Variable) = Variable_Symbol
        and then Is_Own_Task (The_Variable => RawDict.Get_Variable_Info_Ref (Item => Variable)); -- GAA External
   end IsOwnTask;

   --------------------------------------------------------------------------------
   --  Get_Owner
   --------------------------------------------------------------------------------

   function Get_Owner (The_Variable : RawDict.Variable_Info_Ref) return Symbol
   --# global in Dict;
   is
      Result : Symbol;
   begin
      if Is_Own_Variable (The_Variable => The_Variable) then
         Result := RawDict.Get_Own_Variable_Owner
           (The_Own_Variable => RawDict.Get_Variable_Own_Variable (The_Variable => The_Variable));
      elsif Is_Own_Task (The_Variable => The_Variable) then
         Result := RawDict.Get_Package_Symbol
           (RawDict.GetOwnTaskOwner (RawDict.Get_Variable_Own_Task (The_Variable => The_Variable)));
      else
         Result := RawDict.Get_Own_Variable_Owner
           (The_Own_Variable => RawDict.Get_Constituent_Own_Variable
              (The_Constituent => RawDict.Get_Variable_Constituent (The_Variable => The_Variable)));
      end if;
      return Result;
   end Get_Owner;

   --------------------------------------------------------------------------------

   function GetOwner (Variable : Symbol) return Symbol is
   begin
      return Get_Owner (The_Variable => RawDict.Get_Variable_Info_Ref (Variable)); -- GAA External
   end GetOwner;

   --------------------------------------------------------------------------------
   --  Type_Is_Enumeration
   --------------------------------------------------------------------------------

   function Type_Is_Enumeration (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark) = Enumeration_Type_Item
        or else (RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark) = Generic_Type_Item
                   and then RawDict.Get_Type_Kind_Of_Generic (Type_Mark => Type_Mark) = Generic_Discrete_Type);
   end Type_Is_Enumeration;

   --------------------------------------------------------------------------------

   function TypeIsEnumeration (TypeMark : Symbol) return Boolean is
   begin
      return Type_Is_Enumeration (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)); -- GAA External
   end TypeIsEnumeration;

   --------------------------------------------------------------------------------
   --  Type_Is_Integer
   --------------------------------------------------------------------------------

   function Type_Is_Integer (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark) = Integer_Type_Item
        or else (RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark) = Generic_Type_Item
                   and then RawDict.Get_Type_Kind_Of_Generic (Type_Mark => Type_Mark) = Generic_Integer_Type);
   end Type_Is_Integer;

   --------------------------------------------------------------------------------

   function TypeIsInteger (TypeMark : Symbol) return Boolean is
   begin
      return Type_Is_Integer (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)); -- GAA External
   end TypeIsInteger;

   --------------------------------------------------------------------------------
   --  Type_Is_Modular
   --------------------------------------------------------------------------------

   function Type_Is_Modular (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark) = Modular_Type_Item
        or else (RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark) = Generic_Type_Item
                   and then RawDict.Get_Type_Kind_Of_Generic (Type_Mark => Type_Mark) = Generic_Modular_Type);
   end Type_Is_Modular;

   --------------------------------------------------------------------------------

   function TypeIsModular (TypeMark : Symbol) return Boolean is
   begin
      return Type_Is_Modular (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)); -- GAA External
   end TypeIsModular;

   --------------------------------------------------------------------------------
   --  Type_Is_Floating_Point
   --------------------------------------------------------------------------------

   function Type_Is_Floating_Point (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark) = Floating_Point_Type_Item
        or else (RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark) = Generic_Type_Item
                   and then RawDict.Get_Type_Kind_Of_Generic (Type_Mark => Type_Mark) = Generic_Floating_Point_Type);
   end Type_Is_Floating_Point;

   --------------------------------------------------------------------------------

   function TypeIsFloatingPoint (TypeMark : Symbol) return Boolean is
   begin
      return Type_Is_Floating_Point (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)); -- GAA External
   end TypeIsFloatingPoint;

   --------------------------------------------------------------------------------
   --  Type_Is_Fixed_Point
   --------------------------------------------------------------------------------

   function Type_Is_Fixed_Point (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark) = Fixed_Point_Type_Item
        or else (RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark) = Generic_Type_Item
                   and then RawDict.Get_Type_Kind_Of_Generic (Type_Mark => Type_Mark) = Generic_Fixed_Point_Type);
   end Type_Is_Fixed_Point;

   --------------------------------------------------------------------------------

   function TypeIsFixedPoint (TypeMark : Symbol) return Boolean is
   begin
      return Type_Is_Fixed_Point (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)); -- GAA External
   end TypeIsFixedPoint;

   --------------------------------------------------------------------------------
   --  Type_Is_Array
   --------------------------------------------------------------------------------

   function Type_Is_Array (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark) = Array_Type_Item
        or else (RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark) = Generic_Type_Item
                   and then RawDict.Get_Type_Kind_Of_Generic (Type_Mark => Type_Mark) = Generic_Array_Type);
   end Type_Is_Array;

   --------------------------------------------------------------------------------

   function TypeIsArray (TypeMark : Symbol) return Boolean is
   begin
      return Type_Is_Array (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)); -- GAA External
   end TypeIsArray;

   --------------------------------------------------------------------------------
   --  Get_Predefined_Boolean_Type
   --------------------------------------------------------------------------------

   function Get_Predefined_Boolean_Type return  RawDict.Type_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Types.The_Predefined_Types (Predefined_Boolean_Type);
   end Get_Predefined_Boolean_Type;

   --------------------------------------------------------------------------------

   function GetPredefinedBooleanType return Symbol is
   begin
      return RawDict.Get_Type_Symbol (Get_Predefined_Boolean_Type); -- GAA External
   end GetPredefinedBooleanType;

   --------------------------------------------------------------------------------

   function IsPredefinedBooleanType (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.Get_Type_Info_Ref (TheSymbol) = Get_Predefined_Boolean_Type; -- GAA External
   end IsPredefinedBooleanType;

   --------------------------------------------------------------------------------
   --  Get_False
   --------------------------------------------------------------------------------

   function Get_False return RawDict.Enumeration_Literal_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Types.The_Predefined_Literals (Predefined_False);
   end Get_False;

   --------------------------------------------------------------------------------

   function GetFalse return Symbol is
   begin
      return RawDict.Get_Enumeration_Literal_Symbol (Get_False); -- GAA External
   end GetFalse;

   --------------------------------------------------------------------------------
   --  Get_True
   --------------------------------------------------------------------------------

   function Get_True return RawDict.Enumeration_Literal_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Types.The_Predefined_Literals (Predefined_True);
   end Get_True;

   --------------------------------------------------------------------------------

   function GetTrue return Symbol is
   begin
      return RawDict.Get_Enumeration_Literal_Symbol (Get_True); -- GAA External
   end GetTrue;

   --------------------------------------------------------------------------------
   --  Get_Universal_Real_Type
   --------------------------------------------------------------------------------

   function Get_Universal_Real_Type return RawDict.Type_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Types.The_Universal_Types (Universal_Real_Type);
   end Get_Universal_Real_Type;

   --------------------------------------------------------------------------------

   function GetUniversalRealType return Symbol is
   begin
      return RawDict.Get_Type_Symbol (Get_Universal_Real_Type); -- GAA External
   end GetUniversalRealType;

   --------------------------------------------------------------------------------

   function IsUniversalRealType (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.Get_Type_Info_Ref (TheSymbol) = Get_Universal_Real_Type; -- GAA External
   end IsUniversalRealType;

   --------------------------------------------------------------------------------
   --  Get_Universal_Fixed_Type
   --------------------------------------------------------------------------------

   function Get_Universal_Fixed_Type return RawDict.Type_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Types.The_Universal_Types (Universal_Fixed_Type);
   end Get_Universal_Fixed_Type;

   --------------------------------------------------------------------------------

   function GetUniversalFixedType return Symbol is
   begin
      return RawDict.Get_Type_Symbol (Get_Universal_Fixed_Type); -- GAA External
   end GetUniversalFixedType;

   --------------------------------------------------------------------------------

   function IsUniversalFixedType (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.Get_Type_Info_Ref (TheSymbol) = Get_Universal_Fixed_Type; -- GAA External
   end IsUniversalFixedType;

   --------------------------------------------------------------------------------
   --  Get_Universal_Integer_Type
   --------------------------------------------------------------------------------

   function Get_Universal_Integer_Type return RawDict.Type_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Types.The_Universal_Types (Universal_Integer_Type);
   end Get_Universal_Integer_Type;

   --------------------------------------------------------------------------------

   function GetUniversalIntegerType return Symbol is
   begin
      return RawDict.Get_Type_Symbol (Get_Universal_Integer_Type); -- GAA External
   end GetUniversalIntegerType;

   --------------------------------------------------------------------------------

   function IsUniversalIntegerType (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.Get_Type_Info_Ref (TheSymbol) = Get_Universal_Integer_Type; -- GAA External
   end IsUniversalIntegerType;

   --------------------------------------------------------------------------------
   --  Get_Predefined_Integer_Type
   --------------------------------------------------------------------------------

   function Get_Predefined_Integer_Type return  RawDict.Type_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Types.The_Predefined_Types (Predefined_Integer_Type);
   end Get_Predefined_Integer_Type;

   --------------------------------------------------------------------------------

   function GetPredefinedIntegerType return Symbol is
   begin
      return RawDict.Get_Type_Symbol (Get_Predefined_Integer_Type); -- GAA External
   end GetPredefinedIntegerType;

   --------------------------------------------------------------------------------

   function IsPredefinedIntegerType (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.Get_Type_Info_Ref (TheSymbol) = Get_Predefined_Integer_Type; -- GAA External
   end IsPredefinedIntegerType;

   --------------------------------------------------------------------------------
   --  Get_Predefined_Long_Integer_Type
   --------------------------------------------------------------------------------

   function Get_Predefined_Long_Integer_Type return  RawDict.Type_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Types.The_Predefined_Types (Predefined_Long_Integer_Type);
   end Get_Predefined_Long_Integer_Type;

   --------------------------------------------------------------------------------

   function GetPredefinedLongIntegerType return Symbol is
   begin
      return RawDict.Get_Type_Symbol (Get_Predefined_Long_Integer_Type); -- GAA External
   end GetPredefinedLongIntegerType;

   --------------------------------------------------------------------------------
   --  Get_Predefined_Float_Type
   --------------------------------------------------------------------------------

   function Get_Predefined_Float_Type return  RawDict.Type_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Types.The_Predefined_Types (Predefined_Float_Type);
   end Get_Predefined_Float_Type;

   --------------------------------------------------------------------------------

   function IsPredefinedFloatType (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.Get_Type_Info_Ref (TheSymbol) = Get_Predefined_Float_Type; -- GAA External
   end IsPredefinedFloatType;

   --------------------------------------------------------------------------------
   --  Get_Predefined_Time_Type
   --------------------------------------------------------------------------------

   function Get_Predefined_Time_Type return  RawDict.Type_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Types.The_Predefined_Types (Predefined_Time_Type);
   end Get_Predefined_Time_Type;

   --------------------------------------------------------------------------------

   function GetPredefinedTimeType return Symbol is
   begin
      return RawDict.Get_Type_Symbol (Get_Predefined_Time_Type); -- GAA External
   end GetPredefinedTimeType;

   --------------------------------------------------------------------------------
   --  Get_Predefined_Time_Span_Type
   --------------------------------------------------------------------------------

   function Get_Predefined_Time_Span_Type return  RawDict.Type_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Types.The_Predefined_Types (Predefined_Time_Span_Type);
   end Get_Predefined_Time_Span_Type;

   --------------------------------------------------------------------------------

   function IsPredefinedTimeSpanType (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.Get_Type_Info_Ref (TheSymbol) = Get_Predefined_Time_Span_Type; -- GAA External
   end IsPredefinedTimeSpanType;

   --------------------------------------------------------------------------------
   --  Get_Null_Variable
   --------------------------------------------------------------------------------

   function Get_Null_Variable return RawDict.Variable_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Null_Variable;
   end Get_Null_Variable;

   --------------------------------------------------------------------------------

   function GetNullVariable return Symbol is
   begin
      return RawDict.Get_Variable_Symbol (Get_Null_Variable); -- GAA External
   end GetNullVariable;

   --------------------------------------------------------------------------------

   function Is_Null_Variable (TheVariable : Symbol) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheVariable) = Variable_Symbol
        and then RawDict.Get_Variable_Info_Ref (TheVariable) = Get_Null_Variable; -- GAA External
   end Is_Null_Variable;

   --------------------------------------------------------------------------------
   --  Get_The_Partition
   --------------------------------------------------------------------------------

   function Get_The_Partition return  RawDict.Subprogram_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.The_Partition;
   end Get_The_Partition;

   --------------------------------------------------------------------------------

   function GetThePartition return Symbol is
   begin
      return RawDict.Get_Subprogram_Symbol (Get_The_Partition); -- GAA External
   end GetThePartition;

   --------------------------------------------------------------------------------
   --  Is_Predefined_Time_Type
   --------------------------------------------------------------------------------

   function Is_Predefined_Time_Type (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return Type_Mark = Get_Predefined_Time_Type or else Type_Mark = Get_Predefined_Time_Span_Type;
   end Is_Predefined_Time_Type;

   --------------------------------------------------------------------------------

   function IsPredefinedTimeType (TheSymbol : Symbol) return Boolean is
   begin
      return Is_Predefined_Time_Type (Type_Mark => RawDict.Get_Type_Info_Ref (TheSymbol)); -- GAA External
   end IsPredefinedTimeType;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function CurrentSymbol (CurrentIterator : Iterator) return Symbol is
      Result : Symbol;
   begin
      case CurrentIterator.Discriminant is
         when NullSymIterator .. EmbeddedPackageIterator =>
            Result := CurrentIterator.Current;
         when OwnVariableIterator .. AbstractOwnVariableIterator =>
            Result := RawDict.Get_Variable_Symbol
              (RawDict.Get_Own_Variable_Variable
                 (The_Own_Variable => RawDict.Get_Own_Variable_Info_Ref (CurrentIterator.Current)));
         when ConstituentIterator =>
            Result := RawDict.Get_Variable_Symbol
              (RawDict.Get_Constituent_Variable
                 (The_Constituent => RawDict.Get_Constituent_Info_Ref (CurrentIterator.Current)));
      end case;
      return Result;
   end CurrentSymbol;

   --------------------------------------------------------------------------------

   function GetEnclosingObject (Object : Symbol) return Symbol is
   begin
      return RawDict.Get_Subcomponent_Object (The_Subcomponent => RawDict.Get_Subcomponent_Info_Ref (Object)); -- GAA External
   end GetEnclosingObject;

   --------------------------------------------------------------------------------

   function GetMostEnclosingObject (Object : Symbol) return Symbol is
      Current : Symbol;
   begin
      Current := Object;
      while RawDict.GetSymbolDiscriminant (Current) = Subcomponent_Symbol loop
         Current := RawDict.Get_Subcomponent_Object (The_Subcomponent => RawDict.Get_Subcomponent_Info_Ref (Item => Current));
      end loop;
      return Current;
   end GetMostEnclosingObject;

   --------------------------------------------------------------------------------

   function Is_Constant (TheSymbol : Symbol) return Boolean is
      Result : Boolean;
   begin
      case RawDict.GetSymbolDiscriminant (TheSymbol) is
         when Constant_Symbol =>
            Result := True;
         when Generic_Parameter_Symbol =>
            -- we can regard a generic formal object as a constant since we will only allow
            -- it to be instantiated with a constant
            Result := RawDict.Get_Generic_Parameter_Kind
              (The_Generic_Parameter => RawDict.Get_Generic_Parameter_Info_Ref (Item => TheSymbol)) = -- GAA External
              Generic_Object_Parameter;
         when others =>
            Result := False;
      end case;
      return Result;
   end Is_Constant;

   --------------------------------------------------------------------------------
   --  First_Global_Variable
   --------------------------------------------------------------------------------

   function First_Subprogram_Global_Variable
     (The_Subprogram : RawDict.Subprogram_Info_Ref;
      Abstraction    : Abstractions)
     return           Iterator
   --# global in Dict;
   is
      function First_Global_Variable (The_Global_Variable : RawDict.Global_Variable_Info_Ref) return Iterator
      --# global in Dict;
      is
         Global_Variables : Iterator := NullIterator;
      begin
         if The_Global_Variable /= RawDict.Null_Global_Variable_Info_Ref then
            case RawDict.Get_Kind_Of_Global_Variable (The_Global_Variable => The_Global_Variable) is
               when RawDict.Subprogram_Variable_Item =>
                  Global_Variables := Iterator'
                    (GlobalVariableIterator,
                     IsAbstract,
                     RawDict.Get_Variable_Symbol
                       (RawDict.Get_Global_Variable_Variable (The_Global_Variable => The_Global_Variable)),
                     RawDict.Get_Global_Variable_Symbol (The_Global_Variable));
               when RawDict.Subprogram_Parameter_Item =>
                  Global_Variables := Iterator'
                    (GlobalVariableIterator,
                     IsAbstract,
                     RawDict.Get_Subprogram_Parameter_Symbol
                       (RawDict.Get_Global_Variable_Parameter (The_Global_Variable => The_Global_Variable)),
                     RawDict.Get_Global_Variable_Symbol (The_Global_Variable));
               when others => -- non-exec code
                  Global_Variables := NullIterator;
                  SystemErrors.Fatal_Error
                    (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                     Msg     => "in Dictionary.First_Global_Variable");
            end case;
         end if;
         return Global_Variables;
      end First_Global_Variable;

   begin -- First_Subprogram_Global_Variable
      return First_Global_Variable
        (The_Global_Variable => RawDict.Get_Subprogram_First_Global_Variable (The_Subprogram => The_Subprogram,
                                                                              Abstraction    => Abstraction));
   end First_Subprogram_Global_Variable;

   --------------------------------------------------------------------------------

   function First_Task_Type_Global_Variable (The_Task_Type : RawDict.Type_Info_Ref;
                                             Abstraction   : Abstractions) return Iterator
   --# global in Dict;
   is
      function First_Global_Variable (The_Global_Variable : RawDict.Global_Variable_Info_Ref) return Iterator
      --# global in Dict;
      is
         Global_Variables : Iterator := NullIterator;
      begin
         if The_Global_Variable /= RawDict.Null_Global_Variable_Info_Ref then
            Global_Variables := Iterator'
              (GlobalVariableIterator,
               IsAbstract,
               RawDict.Get_Variable_Symbol (RawDict.Get_Global_Variable_Variable (The_Global_Variable => The_Global_Variable)),
               RawDict.Get_Global_Variable_Symbol (The_Global_Variable));
         end if;
         return Global_Variables;
      end First_Global_Variable;

   begin -- First_Task_Type_Global_Variable
      return First_Global_Variable
        (The_Global_Variable => RawDict.Get_Task_Type_First_Global_Variable (The_Task_Type => The_Task_Type,
                                                                             Abstraction   => Abstraction));
   end First_Task_Type_Global_Variable;

   --------------------------------------------------------------------------------

   function FirstGlobalVariable (Abstraction : Abstractions;
                                 Subprogram  : Symbol) return Iterator is
      Global_Variables : Iterator;
   begin
      case RawDict.GetSymbolDiscriminant (Subprogram) is
         when Subprogram_Symbol =>
            Global_Variables := First_Subprogram_Global_Variable
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Subprogram), -- GAA External
               Abstraction    => Abstraction);
         when Type_Symbol =>
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Subprogram)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.FirstGlobalVariable");
            Global_Variables := First_Task_Type_Global_Variable
              (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => Subprogram), -- GAA External
               Abstraction   => Abstraction);
         when others => -- non-exec code
            Global_Variables := NullIterator;
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.FirstGlobalVariable");
      end case;
      return Global_Variables;
   end FirstGlobalVariable;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function IsVariableOrSubcomponent (TheSymbol : Symbol) return Boolean is
   begin
      return Is_Variable (TheSymbol) or else RawDict.GetSymbolDiscriminant (TheSymbol) = Subcomponent_Symbol;
   end IsVariableOrSubcomponent;

   --------------------------------------------------------------------------------
   --  Get_First_Record_Subcomponent
   --------------------------------------------------------------------------------

   function GetFirstRecordSubcomponent (TheSymbol : Symbol) return Symbol is

      function Get_First_Record_Subcomponent (TheSymbol : Symbol) return RawDict.Subcomponent_Info_Ref
      --# global in Dict;
      is
         The_Subcomponent : RawDict.Subcomponent_Info_Ref;
      begin
         case RawDict.GetSymbolDiscriminant (TheSymbol) is
            when Subcomponent_Symbol =>
               The_Subcomponent := RawDict.Get_Subcomponent_Subcomponents
                 (The_Subcomponent => RawDict.Get_Subcomponent_Info_Ref (Item => TheSymbol)); -- GAA External
            when Variable_Symbol =>
               The_Subcomponent := RawDict.Get_Variable_Subcomponents
                 (The_Variable => RawDict.Get_Variable_Info_Ref (Item => TheSymbol)); -- GAA External
            when Subprogram_Parameter_Symbol =>
               The_Subcomponent := RawDict.Get_Subprogram_Parameter_Subcomponents
                 (The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => TheSymbol)); -- GAA External
            when others =>
               The_Subcomponent := RawDict.Null_Subcomponent_Info_Ref;
         end case;
         return The_Subcomponent;
      end Get_First_Record_Subcomponent;

   begin -- GetFirstRecordSubcomponent
      return RawDict.Get_Subcomponent_Symbol (Get_First_Record_Subcomponent (TheSymbol)); -- GAA External
   end GetFirstRecordSubcomponent;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function GetNextRecordSubcomponent (TheSubcomponent : Symbol) return Symbol is
   begin
      return RawDict.Get_Subcomponent_Symbol -- GAA External
        (RawDict.Get_Next_Subcomponent
           (The_Subcomponent => RawDict.Get_Subcomponent_Info_Ref (TheSubcomponent))); -- GAA External
   end GetNextRecordSubcomponent;

   --------------------------------------------------------------------------------

   function VariableIsInitialized (Variable : Symbol) return Boolean is
   begin
      return RawDict.Get_Variable_Initialized (The_Variable => RawDict.Get_Variable_Info_Ref (Variable)); -- GAA External
   end VariableIsInitialized;

   --------------------------------------------------------------------------------

   function VariableHasAddressClause (Variable : Symbol) return Boolean is
   begin
      return RawDict.Get_Variable_Has_Address_Clause (The_Variable => RawDict.Get_Variable_Info_Ref (Variable)); -- GAA External
   end VariableHasAddressClause;

   --------------------------------------------------------------------------------

   function VariableHasPragmaImport (Variable : Symbol) return Boolean is
   begin
      return RawDict.Get_Variable_Has_Pragma_Import (The_Variable => RawDict.Get_Variable_Info_Ref (Variable)); -- GAA External
   end VariableHasPragmaImport;

   --------------------------------------------------------------------------------

   function TypeSizeAttribute (TypeMark : Symbol) return LexTokenManager.Lex_String is
   begin
      return RawDict.Get_Type_Size_Attribute (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)); -- GAA External
   end TypeSizeAttribute;

   --------------------------------------------------------------------------------

   function VariableIsAliased (Variable : Symbol) return Boolean is
   begin
      return RawDict.Get_Variable_Is_Aliased (The_Variable => RawDict.Get_Variable_Info_Ref (Variable)); -- GAA External
   end VariableIsAliased;

   --------------------------------------------------------------------------------

   function VariableIsMarkedValid (TheVariable : Symbol) return Boolean is
   begin
      return RawDict.Get_Variable_Marked_Valid (The_Variable => RawDict.Get_Variable_Info_Ref (TheVariable)); -- GAA External
   end VariableIsMarkedValid;

   --------------------------------------------------------------------------------

   function SubcomponentIsMarkedValid (TheSubcomponent : Symbol) return Boolean is
   begin
      return RawDict.Get_Subcomponent_Marked_Valid
        (The_Subcomponent => RawDict.Get_Subcomponent_Info_Ref (TheSubcomponent)); -- GAA External
   end SubcomponentIsMarkedValid;

   --------------------------------------------------------------------------------

   function VariableOrSubcomponentIsMarkedValid (TheSym : Symbol) return Boolean is
      Result : Boolean;
   begin
      case RawDict.GetSymbolDiscriminant (TheSym) is
         when Variable_Symbol =>
            Result := RawDict.Get_Variable_Marked_Valid
              (The_Variable => RawDict.Get_Variable_Info_Ref (Item => TheSym)); -- GAA External
         when Subcomponent_Symbol =>
            Result := RawDict.Get_Subcomponent_Marked_Valid
              (The_Subcomponent => RawDict.Get_Subcomponent_Info_Ref (Item => TheSym)); -- GAA External
         when LoopEntryVariableSymbol         |
           Quantified_Variable_Symbol      |
           Implicit_Return_Variable_Symbol |
           Subprogram_Parameter_Symbol     |
           LoopParameterSymbol             =>
            Result := True;
         when others => -- non-exec code
            Result := False;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.VariableOrSubcomponentIsMarkedValid");
      end case;
      return Result;
   end VariableOrSubcomponentIsMarkedValid;

   --------------------------------------------------------------------------------

   function SetsPriority (TheDiscriminant : Symbol) return Boolean is
   begin
      return RawDict.GetDiscriminantSetsPriority (TheDiscriminant);
   end SetsPriority;

   --------------------------------------------------------------------------------

   function IsFormalParameter (Subprogram, Parameter : Symbol) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (Parameter) = Subprogram_Parameter_Symbol
        and then RawDict.GetSymbolDiscriminant (Subprogram) = Subprogram_Symbol
        and then RawDict.Get_Subprogram_Parameter_Subprogram
        (The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => Parameter)) =   -- GAA External
        RawDict.Get_Subprogram_Info_Ref (Item => Subprogram); -- GAA External
   end IsFormalParameter;

   --------------------------------------------------------------------------------

   function IsGenericFormalParameter (TheGeneric, Parameter : Symbol) return Boolean is
      Parameter_Or_Constant                 : RawDict.Generic_Parameter_Info_Ref;
      Associated_Generic_Formal_Of_Constant : RawDict.Generic_Parameter_Info_Ref;
   begin
      -- if the generic formal is an object then a constant will have been declared in the visible
      -- scope of the geenric unit and linked to the list of its generic formal parameters.  In many
      -- cases, e.g. in the VCG, the symbol that we will find when looking up a generic formal object parameter
      -- will be the symbol of this implicitly-declared constant.  We want such constants to be correctly identified
      -- as generic formal parameters.  The following test checks for this case before going on to check whether
      -- the parameter supplied is the one we are looking for.
      case RawDict.GetSymbolDiscriminant (Parameter) is
         when Constant_Symbol =>
            -- if its a constant then it might be an implicitly-declared constant associated with a generic formal
            Associated_Generic_Formal_Of_Constant := RawDict.Get_Constant_Associated_Generic_Parameter
              (The_Constant => RawDict.Get_Constant_Info_Ref (Item => Parameter)); -- GAA External
            if Associated_Generic_Formal_Of_Constant /= RawDict.Null_Generic_Parameter_Info_Ref then
               -- then it really is an implicitly-declared constant associated with the generic formal
               Parameter_Or_Constant := Associated_Generic_Formal_Of_Constant;
            else
               Parameter_Or_Constant := RawDict.Null_Generic_Parameter_Info_Ref;
            end if;
         when Generic_Parameter_Symbol =>
            Parameter_Or_Constant := RawDict.Get_Generic_Parameter_Info_Ref (Item => Parameter); -- GAA External
         when others =>
            Parameter_Or_Constant := RawDict.Null_Generic_Parameter_Info_Ref;
      end case;
      -- now check whether Parameter, perhaps modified by code above, is actually a generic formal of the TheGeneric
      return Parameter_Or_Constant /= RawDict.Null_Generic_Parameter_Info_Ref
        and then RawDict.Get_Generic_Unit_Owning_Subprogram
        (The_Generic_Unit => RawDict.Get_Generic_Parameter_Owning_Generic (The_Generic_Parameter => Parameter_Or_Constant)) =
        RawDict.Get_Subprogram_Info_Ref (TheGeneric); -- GAA External
   end IsGenericFormalParameter;

   --------------------------------------------------------------------------------

   function Is_Subprogram (TheSymbol : Symbol) return Boolean
   --# return RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = Subprogram_Symbol or
   --#   RawDict.GetSymbolDiscriminant (TheSymbol, Dict) = ImplicitProofFunctionSymbol;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Subprogram_Symbol
        or else RawDict.GetSymbolDiscriminant (TheSymbol) = ImplicitProofFunctionSymbol;
   end Is_Subprogram;

   --------------------------------------------------------------------------------

   function Is_Generic_Subprogram (The_Symbol : Symbol) return Boolean is
      The_Subprogram : RawDict.Subprogram_Info_Ref;
   begin
      case RawDict.GetSymbolDiscriminant (The_Symbol) is
         when Subprogram_Symbol =>
            The_Subprogram := RawDict.Get_Subprogram_Info_Ref (Item => The_Symbol);
         when ImplicitProofFunctionSymbol =>
            The_Subprogram := RawDict.GetImplicitProofFunctionAdaFunction (The_Symbol);
         when others =>
            The_Subprogram := RawDict.Null_Subprogram_Info_Ref;
      end case;
      return RawDict.Get_Subprogram_Generic_Unit (The_Subprogram => The_Subprogram) /= RawDict.Null_Generic_Unit_Info_Ref;
   end Is_Generic_Subprogram;

   --------------------------------------------------------------------------------
   --  Get_Precondition
   --------------------------------------------------------------------------------

   function GetPrecondition (Abstraction : Abstractions;
                             Subprogram  : Symbol) return ExaminerConstants.RefType is
      Precondition : ExaminerConstants.RefType;

      --------------------------------------------------------------------------------

      function Get_Precondition (The_Subprogram : RawDict.Subprogram_Info_Ref;
                                 Abstraction    : Abstractions) return ExaminerConstants.RefType
      --# global in Dict;
      is
         Precondition : ExaminerConstants.RefType;
      begin
         Precondition := RawDict.Get_Subprogram_Precondition
           (The_Subprogram => The_Subprogram,
            Abstraction    => Abstraction);
         if Precondition = 0
           and then RawDict.Get_Subprogram_Instantiation_Of (The_Subprogram => The_Subprogram) /= RawDict.Null_Subprogram_Info_Ref then
            Precondition := RawDict.Get_Subprogram_Precondition
              (The_Subprogram => RawDict.Get_Subprogram_Instantiation_Of (The_Subprogram => The_Subprogram),
               Abstraction    => Abstraction);
         end if;
         return Precondition;
      end Get_Precondition;

   begin -- GetPrecondition
      case RawDict.GetSymbolDiscriminant (Subprogram) is
         when Subprogram_Symbol =>
            Precondition := Get_Precondition
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Subprogram), -- GAA External
               Abstraction    => Abstraction);
         when ImplicitProofFunctionSymbol =>
            Precondition := Get_Precondition
              (The_Subprogram => RawDict.GetImplicitProofFunctionAdaFunction (Subprogram),
               Abstraction    => Abstraction);
         when Type_Symbol =>
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Subprogram)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.GetPrecondition");
            Precondition := 0; -- valid but pointless request for a task's precondition
         when others => -- non-exec code
            Precondition := 0;
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.GetPrecondition");
      end case;
      return Precondition;
   end GetPrecondition;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function HasPrecondition (Abstraction : Abstractions;
                             Subprogram  : Symbol) return Boolean is
   begin
      return GetPrecondition (Abstraction, Subprogram) /= 0;
   end HasPrecondition;

   --------------------------------------------------------------------------------
   --  Get_Postcondition
   --------------------------------------------------------------------------------

   function GetPostcondition (Abstraction : Abstractions;
                              Subprogram  : Symbol) return ExaminerConstants.RefType is
      Postcondition : ExaminerConstants.RefType;

      --------------------------------------------------------------------------------

      function Get_Postcondition (The_Subprogram : RawDict.Subprogram_Info_Ref;
                                  Abstraction    : Abstractions) return ExaminerConstants.RefType
      --# global in Dict;
      is
         Postcondition : ExaminerConstants.RefType;
      begin
         Postcondition := RawDict.Get_Subprogram_Postcondition
           (The_Subprogram => The_Subprogram,
            Abstraction    => Abstraction);
         if Postcondition = 0
           and then RawDict.Get_Subprogram_Instantiation_Of (The_Subprogram => The_Subprogram) /= RawDict.Null_Subprogram_Info_Ref then
            Postcondition := RawDict.Get_Subprogram_Postcondition
              (The_Subprogram => RawDict.Get_Subprogram_Instantiation_Of (The_Subprogram => The_Subprogram),
               Abstraction    => Abstraction);
         end if;
         return Postcondition;
      end Get_Postcondition;

   begin -- GetPostcondition
      case RawDict.GetSymbolDiscriminant (Subprogram) is
         when Subprogram_Symbol =>
            Postcondition := Get_Postcondition
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Subprogram), -- GAA External
               Abstraction    => Abstraction);
         when ImplicitProofFunctionSymbol =>
            Postcondition := Get_Postcondition
              (The_Subprogram => RawDict.GetImplicitProofFunctionAdaFunction (Subprogram),
               Abstraction    => Abstraction);
         when Type_Symbol =>
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Subprogram)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.GetPostcondition");
            Postcondition := 0; -- valid but pointless request for a task's postcondition
         when others => -- non-exec code
            Postcondition := 0;
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.GetPostcondition");
      end case;
      return Postcondition;
   end GetPostcondition;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function HasPostcondition (Abstraction : Abstractions;
                              Subprogram  : Symbol) return Boolean is
   begin
      return GetPostcondition (Abstraction, Subprogram) /= 0;
   end HasPostcondition;

   --------------------------------------------------------------------------------
   --  Is_Import
   --------------------------------------------------------------------------------

   function Is_Imported_Variable
     (The_Global_Variable : RawDict.Global_Variable_Info_Ref;
      The_Variable        : RawDict.Variable_Info_Ref)
     return                Boolean
   --# global in Dict;
   is
      Tmp_Global_Variable : RawDict.Global_Variable_Info_Ref;
      Result              : Boolean;
   begin
      Tmp_Global_Variable := The_Global_Variable;
      loop
         if Tmp_Global_Variable = RawDict.Null_Global_Variable_Info_Ref then
            Result := False;
            exit;
         end if;
         if (RawDict.Get_Kind_Of_Global_Variable (The_Global_Variable => Tmp_Global_Variable) = RawDict.Subprogram_Variable_Item
               or else RawDict.Get_Kind_Of_Global_Variable (The_Global_Variable => Tmp_Global_Variable) =
               RawDict.Task_Type_Variable_Item)
           and then RawDict.Get_Global_Variable_Variable (The_Global_Variable => Tmp_Global_Variable) = The_Variable then
            Result := RawDict.Get_Global_Variable_Imported (The_Global_Variable => Tmp_Global_Variable);
            exit;
         end if;
         Tmp_Global_Variable := RawDict.Get_Next_Global_Variable (The_Global_Variable => Tmp_Global_Variable);
      end loop;
      return Result;
   end Is_Imported_Variable;

   --------------------------------------------------------------------------------

   function Is_Imported_Subprogram_Variable
     (The_Subprogram : RawDict.Subprogram_Info_Ref;
      Abstraction    : Abstractions;
      The_Variable   : RawDict.Variable_Info_Ref)
     return           Boolean
   --# global in Dict;
   is
   begin
      return Is_Imported_Variable
        (The_Global_Variable => RawDict.Get_Subprogram_First_Global_Variable (The_Subprogram => The_Subprogram,
                                                                              Abstraction    => Abstraction),
         The_Variable        => The_Variable);
   end Is_Imported_Subprogram_Variable;

   --------------------------------------------------------------------------------

   function Is_Imported_Subprogram_Subprogram_Parameter
     (The_Subprogram             : RawDict.Subprogram_Info_Ref;
      Abstraction                : Abstractions;
      The_Subprogram_Parameter   : RawDict.Subprogram_Parameter_Info_Ref;
      Is_Implicit_Proof_Function : Boolean)
     return                       Boolean
   --# global in Dict;
   is
      Result : Boolean;

      --------------------------------------------------------------------------------

      function Is_Imported_Subprogram_Parameter
        (The_Global_Variable      : RawDict.Global_Variable_Info_Ref;
         The_Subprogram_Parameter : RawDict.Subprogram_Parameter_Info_Ref)
        return                     Boolean
      --# global in Dict;
      is
         Tmp_Global_Variable : RawDict.Global_Variable_Info_Ref;
         Result              : Boolean;
      begin
         Tmp_Global_Variable := The_Global_Variable;
         loop
            if Tmp_Global_Variable = RawDict.Null_Global_Variable_Info_Ref then
               Result := False;
               exit;
            end if;
            if RawDict.Get_Kind_Of_Global_Variable (The_Global_Variable => Tmp_Global_Variable) =
              RawDict.Subprogram_Parameter_Item
              and then RawDict.Get_Global_Variable_Parameter (The_Global_Variable => Tmp_Global_Variable) =
              The_Subprogram_Parameter then
               Result := RawDict.Get_Global_Variable_Imported (The_Global_Variable => Tmp_Global_Variable);
               exit;
            end if;
            Tmp_Global_Variable := RawDict.Get_Next_Global_Variable (The_Global_Variable => Tmp_Global_Variable);
         end loop;
         return Result;
      end Is_Imported_Subprogram_Parameter;

   begin -- Is_Imported_Subprogram_Subprogram_Parameter
      if Is_Implicit_Proof_Function then
         Result := False;
      else
         Result :=
           RawDict.Get_Subprogram_Parameter_Subprogram (The_Subprogram_Parameter => The_Subprogram_Parameter) = The_Subprogram
           and then RawDict.Get_Subprogram_Parameter_Imported
           (The_Subprogram_Parameter => The_Subprogram_Parameter,
            Abstraction              => Abstraction);
      end if;

      if not Result then
         Result :=
           Is_Imported_Subprogram_Parameter
           (The_Global_Variable      => RawDict.Get_Subprogram_First_Global_Variable (The_Subprogram => The_Subprogram,
                                                                                      Abstraction    => Abstraction),
            The_Subprogram_Parameter => The_Subprogram_Parameter);
      end if;
      return Result;
   end Is_Imported_Subprogram_Subprogram_Parameter;

   --------------------------------------------------------------------------------

   function Is_Imported_Task_Type_Variable
     (The_Task_Type : RawDict.Type_Info_Ref;
      Abstraction   : Abstractions;
      The_Variable  : RawDict.Variable_Info_Ref)
     return          Boolean
   --# global in Dict;
   is
   begin
      return Is_Imported_Variable
        (The_Global_Variable => RawDict.Get_Task_Type_First_Global_Variable (The_Task_Type => The_Task_Type,
                                                                             Abstraction   => Abstraction),
         The_Variable        => The_Variable);
   end Is_Imported_Task_Type_Variable;

   --------------------------------------------------------------------------------

   function IsImport
     (Abstraction  : Abstractions;
      TheProcedure : Symbol;
      Variable     : Symbol)
     return         Boolean
   is
      Result : Boolean;
   begin
      -- allow use with task type as well as procedure
      case RawDict.GetSymbolDiscriminant (TheProcedure) is
         when Subprogram_Symbol =>
            case RawDict.GetSymbolDiscriminant (Variable) is
               when Subprogram_Parameter_Symbol =>
                  Result := Is_Imported_Subprogram_Subprogram_Parameter
                    (The_Subprogram             => RawDict.Get_Subprogram_Info_Ref (Item => TheProcedure), -- GAA External
                     Abstraction                => Abstraction,
                     The_Subprogram_Parameter   => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => Variable), -- GAA External
                     Is_Implicit_Proof_Function => False);
               when Variable_Symbol =>
                  Result := Is_Imported_Subprogram_Variable
                    (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => TheProcedure), -- GAA External
                     Abstraction    => Abstraction,
                     The_Variable   => RawDict.Get_Variable_Info_Ref (Item => Variable)); -- GAA External
               when others => -- non-exec code
                  Result := False;
                  SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                            Msg     => "in Dictionary.IsImport");
            end case;
         when ImplicitProofFunctionSymbol =>
            case RawDict.GetSymbolDiscriminant (Variable) is
               when Subprogram_Parameter_Symbol =>
                  Result := Is_Imported_Subprogram_Subprogram_Parameter
                    (The_Subprogram             => RawDict.GetImplicitProofFunctionAdaFunction (TheProcedure),
                     Abstraction                => Abstraction,
                     The_Subprogram_Parameter   => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => Variable), -- GAA External
                     Is_Implicit_Proof_Function => True);
               when Variable_Symbol =>
                  Result := Is_Imported_Subprogram_Variable
                    (The_Subprogram => RawDict.GetImplicitProofFunctionAdaFunction (TheProcedure),
                     Abstraction    => Abstraction,
                     The_Variable   => RawDict.Get_Variable_Info_Ref (Item => Variable)); -- GAA External
               when others => -- non-exec code
                  Result := False;
                  SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                            Msg     => "in Dictionary.IsImport");
            end case;
         when Type_Symbol =>
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheProcedure)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.IsImport");
            Result := Is_Imported_Task_Type_Variable
              (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => TheProcedure), -- GAA External
               Abstraction   => Abstraction,
               The_Variable  => RawDict.Get_Variable_Info_Ref (Variable)); -- GAA External
         when others => -- non-exec code
            Result := False;
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.IsImport");
      end case;
      return Result;
   end IsImport;

   --------------------------------------------------------------------------------
   --  Is_Export
   --------------------------------------------------------------------------------

   function Is_Exported_Variable
     (The_Global_Variable : RawDict.Global_Variable_Info_Ref;
      The_Variable        : RawDict.Variable_Info_Ref)
     return                Boolean
   --# global in Dict;
   is
      Tmp_Global_Variable : RawDict.Global_Variable_Info_Ref;
      Result              : Boolean;
   begin
      Tmp_Global_Variable := The_Global_Variable;
      loop
         if Tmp_Global_Variable = RawDict.Null_Global_Variable_Info_Ref then
            Result := False;
            exit;
         end if;
         if (RawDict.Get_Kind_Of_Global_Variable (The_Global_Variable => Tmp_Global_Variable) = RawDict.Subprogram_Variable_Item
               or else RawDict.Get_Kind_Of_Global_Variable (The_Global_Variable => Tmp_Global_Variable) =
               RawDict.Task_Type_Variable_Item)
           and then RawDict.Get_Global_Variable_Variable (The_Global_Variable => Tmp_Global_Variable) = The_Variable then
            Result := RawDict.Get_Global_Variable_Exported (The_Global_Variable => Tmp_Global_Variable);
            exit;
         end if;
         Tmp_Global_Variable := RawDict.Get_Next_Global_Variable (The_Global_Variable => Tmp_Global_Variable);
      end loop;
      return Result;
   end Is_Exported_Variable;

   --------------------------------------------------------------------------------

   function Is_Exported_Subprogram_Variable
     (The_Subprogram : RawDict.Subprogram_Info_Ref;
      Abstraction    : Abstractions;
      The_Variable   : RawDict.Variable_Info_Ref)
     return           Boolean
   --# global in Dict;
   is
   begin
      return Is_Exported_Variable
        (The_Global_Variable => RawDict.Get_Subprogram_First_Global_Variable (The_Subprogram => The_Subprogram,
                                                                              Abstraction    => Abstraction),
         The_Variable        => The_Variable);
   end Is_Exported_Subprogram_Variable;

   --------------------------------------------------------------------------------

   function Is_Exported_Subprogram_Subprogram_Parameter
     (The_Subprogram             : RawDict.Subprogram_Info_Ref;
      Abstraction                : Abstractions;
      The_Subprogram_Parameter   : RawDict.Subprogram_Parameter_Info_Ref;
      Is_Implicit_Proof_Function : Boolean)
     return                       Boolean
   --# global in Dict;
   is
      Result : Boolean;

      --------------------------------------------------------------------------------

      function Is_Exported_Subprogram_Parameter
        (The_Global_Variable      : RawDict.Global_Variable_Info_Ref;
         The_Subprogram_Parameter : RawDict.Subprogram_Parameter_Info_Ref)
        return                     Boolean
      --# global in Dict;
      is
         Tmp_Global_Variable : RawDict.Global_Variable_Info_Ref;
         Result              : Boolean;
      begin
         Tmp_Global_Variable := The_Global_Variable;
         loop
            if Tmp_Global_Variable = RawDict.Null_Global_Variable_Info_Ref then
               Result := False;
               exit;
            end if;
            if RawDict.Get_Kind_Of_Global_Variable (The_Global_Variable => Tmp_Global_Variable) =
              RawDict.Subprogram_Parameter_Item
              and then RawDict.Get_Global_Variable_Parameter (The_Global_Variable => Tmp_Global_Variable) =
              The_Subprogram_Parameter then
               Result := RawDict.Get_Global_Variable_Exported (The_Global_Variable => Tmp_Global_Variable);
               exit;
            end if;
            Tmp_Global_Variable := RawDict.Get_Next_Global_Variable (The_Global_Variable => Tmp_Global_Variable);
         end loop;
         return Result;
      end Is_Exported_Subprogram_Parameter;

   begin -- Is_Exported_Subprogram_Subprogram_Parameter
      if Is_Implicit_Proof_Function then
         Result := False;
      else
         Result :=
           RawDict.Get_Subprogram_Parameter_Subprogram (The_Subprogram_Parameter => The_Subprogram_Parameter) = The_Subprogram
           and then RawDict.Get_Subprogram_Parameter_Exported
           (The_Subprogram_Parameter => The_Subprogram_Parameter,
            Abstraction              => Abstraction);
      end if;

      if not Result then
         Result :=
           Is_Exported_Subprogram_Parameter
           (The_Global_Variable      => RawDict.Get_Subprogram_First_Global_Variable (The_Subprogram => The_Subprogram,
                                                                                      Abstraction    => Abstraction),
            The_Subprogram_Parameter => The_Subprogram_Parameter);
      end if;
      return Result;
   end Is_Exported_Subprogram_Subprogram_Parameter;

   --------------------------------------------------------------------------------

   function Is_Exported_Task_Type_Variable
     (The_Task_Type : RawDict.Type_Info_Ref;
      Abstraction   : Abstractions;
      The_Variable  : RawDict.Variable_Info_Ref)
     return          Boolean
   --# global in Dict;
   is
   begin
      return Is_Exported_Variable
        (The_Global_Variable => RawDict.Get_Task_Type_First_Global_Variable (The_Task_Type => The_Task_Type,
                                                                             Abstraction   => Abstraction),
         The_Variable        => The_Variable);
   end Is_Exported_Task_Type_Variable;

   --------------------------------------------------------------------------------

   function IsExport
     (Abstraction  : Abstractions;
      TheProcedure : Symbol;
      Variable     : Symbol)
     return         Boolean
   is
      Result : Boolean;
   begin
      -- allow use with task type as well as procedure
      case RawDict.GetSymbolDiscriminant (TheProcedure) is
         when Subprogram_Symbol =>
            case RawDict.GetSymbolDiscriminant (Variable) is
               when Subprogram_Parameter_Symbol =>
                  Result := Is_Exported_Subprogram_Subprogram_Parameter
                    (The_Subprogram             => RawDict.Get_Subprogram_Info_Ref (Item => TheProcedure), -- GAA External
                     Abstraction                => Abstraction,
                     The_Subprogram_Parameter   => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => Variable),
                     Is_Implicit_Proof_Function => False);
               when Variable_Symbol =>
                  Result := Is_Exported_Subprogram_Variable
                    (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => TheProcedure), -- GAA External
                     Abstraction    => Abstraction,
                     The_Variable   => RawDict.Get_Variable_Info_Ref (Item => Variable)); -- GAA External
               when Parameter_Constraint_Symbol | Quantified_Variable_Symbol | Type_Symbol =>
                  Result := False;
               when LoopParameterSymbol =>
                  Result := False;
               when others => -- non-exec code
                  Result := False;
                  SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                            Msg     => "in Dictionary.IsExport");
            end case;
         when ImplicitProofFunctionSymbol =>
            case RawDict.GetSymbolDiscriminant (Variable) is
               when Subprogram_Parameter_Symbol =>
                  Result := Is_Exported_Subprogram_Subprogram_Parameter
                    (The_Subprogram             => RawDict.GetImplicitProofFunctionAdaFunction (TheProcedure),
                     Abstraction                => Abstraction,
                     The_Subprogram_Parameter   => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => Variable), -- GAA External
                     Is_Implicit_Proof_Function => True);
               when Variable_Symbol =>
                  Result := Is_Exported_Subprogram_Variable
                    (The_Subprogram => RawDict.GetImplicitProofFunctionAdaFunction (TheProcedure),
                     Abstraction    => Abstraction,
                     The_Variable   => RawDict.Get_Variable_Info_Ref (Item => Variable)); -- GAA External
               when Parameter_Constraint_Symbol | Quantified_Variable_Symbol | Type_Symbol =>
                  Result := False;
               when others => -- non-exec code
                  Result := False;
                  SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                            Msg     => "in Dictionary.IsExport");
            end case;
         when Type_Symbol =>
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheProcedure)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.IsExport");
            Result := Is_Exported_Task_Type_Variable
              (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => TheProcedure), -- GAA External
               Abstraction   => Abstraction,
               The_Variable  => RawDict.Get_Variable_Info_Ref (Variable)); -- GAA External
         when others => -- non-exec code
            Result := False;
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.IsExport");
      end case;
      return Result;
   end IsExport;

   --------------------------------------------------------------------------------
   --  Is_Import_Export
   --------------------------------------------------------------------------------

   function Is_Subprogram_Import_Export_Variable
     (The_Subprogram : RawDict.Subprogram_Info_Ref;
      Abstraction    : Abstractions;
      The_Variable   : RawDict.Variable_Info_Ref)
     return           Boolean
   --# global in Dict;
   is
   begin
      return Is_Imported_Subprogram_Variable
        (The_Subprogram => The_Subprogram,
         Abstraction    => Abstraction,
         The_Variable   => The_Variable)
        and then Is_Exported_Subprogram_Variable
        (The_Subprogram => The_Subprogram,
         Abstraction    => Abstraction,
         The_Variable   => The_Variable);
   end Is_Subprogram_Import_Export_Variable;

   --------------------------------------------------------------------------------

   function Is_Subprogram_Import_Export_Subprogram_Parameter
     (The_Subprogram             : RawDict.Subprogram_Info_Ref;
      Abstraction                : Abstractions;
      The_Subprogram_Parameter   : RawDict.Subprogram_Parameter_Info_Ref;
      Is_Implicit_Proof_Function : Boolean)
     return                       Boolean
   --# global in Dict;
   is
   begin
      return Is_Imported_Subprogram_Subprogram_Parameter
        (The_Subprogram             => The_Subprogram,
         Abstraction                => Abstraction,
         The_Subprogram_Parameter   => The_Subprogram_Parameter,
         Is_Implicit_Proof_Function => Is_Implicit_Proof_Function)
        and then Is_Exported_Subprogram_Subprogram_Parameter
        (The_Subprogram             => The_Subprogram,
         Abstraction                => Abstraction,
         The_Subprogram_Parameter   => The_Subprogram_Parameter,
         Is_Implicit_Proof_Function => Is_Implicit_Proof_Function);
   end Is_Subprogram_Import_Export_Subprogram_Parameter;

   --------------------------------------------------------------------------------

   function Is_Task_Type_Import_Export
     (The_Task_Type : RawDict.Type_Info_Ref;
      Abstraction   : Abstractions;
      The_Variable  : RawDict.Variable_Info_Ref)
     return          Boolean
   --# global in Dict;
   is
   begin
      return Is_Imported_Task_Type_Variable
        (The_Task_Type => The_Task_Type,
         Abstraction   => Abstraction,
         The_Variable  => The_Variable)
        and then Is_Exported_Task_Type_Variable
        (The_Task_Type => The_Task_Type,
         Abstraction   => Abstraction,
         The_Variable  => The_Variable);
   end Is_Task_Type_Import_Export;

   --------------------------------------------------------------------------------

   function IsImportExport
     (Abstraction  : Abstractions;
      TheProcedure : Symbol;
      Variable     : Symbol)
     return         Boolean
   is
      Result : Boolean;
   begin
      case RawDict.GetSymbolDiscriminant (TheProcedure) is
         when Subprogram_Symbol =>
            case RawDict.GetSymbolDiscriminant (Variable) is
               when Subprogram_Parameter_Symbol =>
                  Result := Is_Subprogram_Import_Export_Subprogram_Parameter
                    (The_Subprogram             => RawDict.Get_Subprogram_Info_Ref (Item => TheProcedure), -- GAA External
                     Abstraction                => Abstraction,
                     The_Subprogram_Parameter   => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => Variable), -- GAA External
                     Is_Implicit_Proof_Function => False);
               when Variable_Symbol =>
                  Result := Is_Subprogram_Import_Export_Variable
                    (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => TheProcedure), -- GAA External
                     Abstraction    => Abstraction,
                     The_Variable   => RawDict.Get_Variable_Info_Ref (Item => Variable)); -- GAA External
               when Parameter_Constraint_Symbol | Quantified_Variable_Symbol | Type_Symbol =>
                  Result := False;
               when others => -- non-exec code
                  Result := False;
                  SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                            Msg     => "in Dictionary.IsImportExport");
            end case;
         when ImplicitProofFunctionSymbol =>
            case RawDict.GetSymbolDiscriminant (Variable) is
               when Subprogram_Parameter_Symbol =>
                  Result := Is_Subprogram_Import_Export_Subprogram_Parameter
                    (The_Subprogram             => RawDict.GetImplicitProofFunctionAdaFunction (TheProcedure),
                     Abstraction                => Abstraction,
                     The_Subprogram_Parameter   => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => Variable), -- GAA External
                     Is_Implicit_Proof_Function => True);
               when Variable_Symbol =>
                  Result := Is_Subprogram_Import_Export_Variable
                    (The_Subprogram => RawDict.GetImplicitProofFunctionAdaFunction (TheProcedure),
                     Abstraction    => Abstraction,
                     The_Variable   => RawDict.Get_Variable_Info_Ref (Item => Variable)); -- GAA External
               when Parameter_Constraint_Symbol | Quantified_Variable_Symbol | Type_Symbol =>
                  Result := False;
               when others => -- non-exec code
                  Result := False;
                  SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                            Msg     => "in Dictionary.IsImportExport");
            end case;
         when Type_Symbol =>
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheProcedure)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.IsImportExport");
            Result := Is_Task_Type_Import_Export
              (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => TheProcedure), -- GAA External
               Abstraction   => Abstraction,
               The_Variable  => RawDict.Get_Variable_Info_Ref (Variable)); -- GAA External
         when others => -- non-exec code
            Result := False;
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.IsImportExport");
      end case;
      return Result;
   end IsImportExport;

   --------------------------------------------------------------------------------
   --  First_Record_Component
   --------------------------------------------------------------------------------

   function First_Record_Component (Type_Mark : RawDict.Type_Info_Ref) return Iterator
   --# global in Dict;
   is
      The_Record_Component : RawDict.Record_Component_Info_Ref;
      Components           : Iterator := NullIterator;
   begin
      The_Record_Component := RawDict.Get_Type_First_Record_Component (Type_Mark => Type_Mark);
      if The_Record_Component /= RawDict.Null_Record_Component_Info_Ref then
         Components := Iterator'
           (RecordComponentIterator,
            IsAbstract,
            RawDict.Get_Record_Component_Symbol (The_Record_Component),
            NullSymbol);
      end if;
      return Components;
   end First_Record_Component;

   --------------------------------------------------------------------------------

   function FirstRecordComponent (TheRecordType : Symbol) return Iterator is
   begin
      return First_Record_Component (Type_Mark => RawDict.Get_Type_Info_Ref (TheRecordType)); -- GAA External
   end FirstRecordComponent;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   -- This function is to do with iterating over the fields of extended tagged records.
   -- It is declared here because it is used by FirstExtendedRecordComponent and NextSymbol
   function Back_Track_Up_Inherit_Chain (Start_Sym, Stop_Sym : RawDict.Type_Info_Ref) return RawDict.Type_Info_Ref
   --# global in Dict;
   is
      Result : RawDict.Type_Info_Ref;
   begin
      -- Starting at extended record Start_Sym follow the Inherit pointers to the
      -- record _preceding_ Stop_Sym.  If Start_Sym = Stop_Sym return NullSymbol.
      if Start_Sym = Stop_Sym then
         Result := RawDict.Null_Type_Info_Ref;
      else
         Result := Start_Sym;
         loop
            exit when Get_Type (The_Symbol => CurrentSymbol (First_Record_Component (Type_Mark => Result))) = Stop_Sym;
            Result := Get_Type (The_Symbol => CurrentSymbol (First_Record_Component (Type_Mark => Result)));
         end loop;
      end if;
      return Result;
   end Back_Track_Up_Inherit_Chain;

   --------------------------------------------------------------------------------
   --  First_Enumeration_Literal
   --------------------------------------------------------------------------------

   function First_Enumeration_Literal (Type_Mark : RawDict.Type_Info_Ref) return Iterator
   --# global in Dict;
   is
      The_Enumeration_Literal : RawDict.Enumeration_Literal_Info_Ref;
      Literals                : Iterator := NullIterator;
   begin
      The_Enumeration_Literal := RawDict.Get_Type_First_Enumeration_Literal (Type_Mark => Type_Mark);
      if The_Enumeration_Literal /= RawDict.Null_Enumeration_Literal_Info_Ref then
         Literals := Iterator'
           (EnumerationLiteralIterator,
            IsAbstract,
            RawDict.Get_Enumeration_Literal_Symbol (The_Enumeration_Literal),
            NullSymbol);
      end if;
      return Literals;
   end First_Enumeration_Literal;

   --------------------------------------------------------------------------------

   function FirstEnumerationLiteral (EnumerationType : Symbol) return Iterator is
   begin
      return First_Enumeration_Literal (Type_Mark => RawDict.Get_Type_Info_Ref (EnumerationType)); -- GAA External
   end FirstEnumerationLiteral;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function GetPositionNumber (Literal : Symbol) return LexTokenManager.Lex_String is
   begin
      return RawDict.Get_Enumeration_Literal_Position
        (The_Enumeration_Literal => RawDict.Get_Enumeration_Literal_Info_Ref (Literal)); -- GAA External
   end GetPositionNumber;

   --------------------------------------------------------------------------------
   --  Get_Record_Type
   --------------------------------------------------------------------------------

   function GetRecordType (Component : Symbol) return Symbol is

      function Get_Record_Type (Component : Symbol) return RawDict.Type_Info_Ref
      --# global in Dict;
      is
         The_Record_Component : RawDict.Record_Component_Info_Ref;
      begin
         case RawDict.GetSymbolDiscriminant (Component) is
            when Record_Component_Symbol =>
               The_Record_Component := RawDict.Get_Record_Component_Info_Ref (Item => Component); -- GAA External
            when Subcomponent_Symbol =>
               The_Record_Component := RawDict.Get_Subcomponent_Record_Component
                 (The_Subcomponent => RawDict.Get_Subcomponent_Info_Ref (Item => Component)); -- GAA External
            when others =>
               The_Record_Component := RawDict.Null_Record_Component_Info_Ref;
         end case;
         return RawDict.Get_Record_Component_Record_Type (The_Record_Component => The_Record_Component);
      end Get_Record_Type;

   begin -- GetRecordType
      return RawDict.Get_Type_Symbol (Get_Record_Type (Component)); -- GAA External
   end GetRecordType;

   --------------------------------------------------------------------------------
   --  Get_Predefined_Package_Standard
   --------------------------------------------------------------------------------

   function Get_Predefined_Package_Standard return RawDict.Package_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Packages.Package_Standard;
   end Get_Predefined_Package_Standard;

   --------------------------------------------------------------------------------

   function GetPredefinedPackageStandard return Symbol is
   begin
      return RawDict.Get_Package_Symbol (Get_Predefined_Package_Standard); -- GAA External
   end GetPredefinedPackageStandard;

   --------------------------------------------------------------------------------

   function IsPredefinedPackageStandard (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.Get_Package_Info_Ref (TheSymbol) = Get_Predefined_Package_Standard; -- GAA External
   end IsPredefinedPackageStandard;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function Get_Predefined_Package_Ada return RawDict.Package_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Packages.Package_Ada;
   end Get_Predefined_Package_Ada;

   --------------------------------------------------------------------------------

   function Predefined_Scope return Scopes
   --# global in Dict;
   is
   begin
      return Set_Visibility (The_Visibility => Visible,
                             The_Unit       => RawDict.Get_Package_Symbol (Get_Predefined_Package_Standard));
   end Predefined_Scope;

   --------------------------------------------------------------------------------

   function GetLocalScope (Scope : Scopes) return Scopes is
   begin
      return Set_Visibility (The_Visibility => Local,
                             The_Unit       => GetRegion (Scope));
   end GetLocalScope;

   --------------------------------------------------------------------------------

   function GlobalScope return Scopes is
   begin
      return Set_Visibility (The_Visibility => Local,
                             The_Unit       => RawDict.Get_Package_Symbol (Get_Predefined_Package_Standard));
   end GlobalScope;

   --------------------------------------------------------------------------------
   --  Get_Scope
   --------------------------------------------------------------------------------

   function Get_Type_Scope (Type_Mark : RawDict.Type_Info_Ref) return Scopes
   --# global in Dict;
   is
   begin
      return RawDict.Get_Declaration_Scope (The_Declaration => Get_Type_Declaration (Type_Mark => Type_Mark));
   end Get_Type_Scope;

   --------------------------------------------------------------------------------

   function Get_Constant_Scope (The_Constant : RawDict.Constant_Info_Ref) return Scopes
   --# global in Dict;
   is
   begin
      return RawDict.Get_Declaration_Scope (The_Declaration => Get_Constant_Declaration (The_Constant => The_Constant));
   end Get_Constant_Scope;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Scope (The_Subprogram : RawDict.Subprogram_Info_Ref) return Scopes
   --# global in Dict;
   is
   begin
      return RawDict.Get_Declaration_Scope
        (The_Declaration => RawDict.Get_Subprogram_Specification (The_Subprogram => The_Subprogram));
   end Get_Subprogram_Scope;

   --------------------------------------------------------------------------------

   function Get_Package_Scope (The_Package : RawDict.Package_Info_Ref) return Scopes
   --# global in Dict;
   is
   begin
      return RawDict.Get_Declaration_Scope
        (The_Declaration => RawDict.Get_Package_Specification (The_Package => The_Package));
   end Get_Package_Scope;

   --------------------------------------------------------------------------------

   function Get_Variable_Scope (The_Variable : RawDict.Variable_Info_Ref) return Scopes
   --# global in Dict;
   is
      Scope       : Scopes;
      The_Package : Symbol;
   begin
      if RawDict.Get_Variable_Declaration (The_Variable => The_Variable) = RawDict.Null_Declaration_Info_Ref then
         The_Package := Get_Owner (The_Variable => The_Variable);
         if RawDict.Get_Variable_Own_Variable (The_Variable => The_Variable) = RawDict.Null_Own_Variable_Info_Ref
           and then RawDict.Get_Variable_Own_Task (The_Variable => The_Variable) = NullSymbol then
            Scope := Set_Visibility (The_Visibility => Local,
                                     The_Unit       => The_Package);
         else
            Scope := Set_Visibility (The_Visibility => Visible,
                                     The_Unit       => The_Package);
         end if;
      else
         Scope :=
           RawDict.Get_Declaration_Scope (The_Declaration => RawDict.Get_Variable_Declaration (The_Variable => The_Variable));
      end if;
      return Scope;
   end Get_Variable_Scope;

   --------------------------------------------------------------------------------

   function GetScope (Item : Symbol) return Scopes is separate;

   --------------------------------------------------------------------------------

   function NextSymbol (Previous : Iterator) return Iterator is separate;

   --------------------------------------------------------------------------------

   function GetEnclosingScope (Scope : Scopes) return Scopes is
      Region          : Symbol;
      Enclosing_Scope : Scopes;
   begin
      Region          := GetRegion (Scope);
      Enclosing_Scope := GetScope (Region);
      if Is_Subprogram (Region) then
         if Get_Visibility (Scope => Scope) = Local
           and then (Get_Visibility (Scope => Enclosing_Scope) = Visible
                       or else Get_Visibility (Scope => Enclosing_Scope) = Privat) then
            Enclosing_Scope := GetLocalScope (Enclosing_Scope);
         elsif Is_Generic_Subprogram (Region) then
            Enclosing_Scope :=
              Set_Visibility
              (The_Visibility => Visible,
               The_Unit       => RawDict.Get_Generic_Unit_Symbol
                 (RawDict.Get_Subprogram_Generic_Unit (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Region))));
         end if;
      elsif RawDict.GetSymbolDiscriminant (Region) = Type_Symbol
        and then (Is_Protected_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Region))
                    or else Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Region)))
        and then Get_Visibility (Scope => Scope) = Local then
         Enclosing_Scope := GetLocalScope (Enclosing_Scope);
      elsif RawDict.GetSymbolDiscriminant (Region) = Package_Symbol
        and then RawDict.Get_Package_Generic_Unit (The_Package => RawDict.Get_Package_Info_Ref (Item => Region)) /=
        RawDict.Null_Generic_Unit_Info_Ref then
         Enclosing_Scope :=
           Set_Visibility
           (The_Visibility => Visible,
            The_Unit       => RawDict.Get_Generic_Unit_Symbol
              (RawDict.Get_Package_Generic_Unit (The_Package => RawDict.Get_Package_Info_Ref (Item => Region))));
      end if;
      return Enclosing_Scope;
   end GetEnclosingScope;

   --------------------------------------------------------------------------------

   function IsPredefinedScope (Scope : Scopes) return Boolean is
   begin
      return Scope = Predefined_Scope;
   end IsPredefinedScope;

   --------------------------------------------------------------------------------

   function IsGlobalScope (Scope : Scopes) return Boolean is
   begin
      return Scope = GlobalScope;
   end IsGlobalScope;

   --------------------------------------------------------------------------------

   function IsLibraryLevel (Scope : Scopes) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (GetRegion (Scope)) = Package_Symbol
        and then (RawDict.Get_Package_Info_Ref (Item => GetRegion (Scope)) = Get_Predefined_Package_Standard
                    or else (RawDict.GetSymbolDiscriminant (GetRegion (GetEnclosingScope (Scope))) = Package_Symbol
                               and then RawDict.Get_Package_Info_Ref (Item => GetRegion (GetEnclosingScope (Scope))) =
                               Get_Predefined_Package_Standard));
   end IsLibraryLevel;

   --------------------------------------------------------------------------------

   function IsLocal (Inner, Outer : Scopes) return Boolean is
      Current_Scope  : Scopes;
      Current_Region : Symbol;
      Outer_Region   : Symbol;
   begin
      Current_Scope := Inner;
      Outer_Region  := GetRegion (Outer);
      loop
         Current_Region := GetRegion (Current_Scope);
         exit when Current_Region = Outer_Region
           or else (RawDict.GetSymbolDiscriminant (Current_Region) = Package_Symbol
                      and then RawDict.Get_Package_Info_Ref (Item => Current_Region) = Get_Predefined_Package_Standard);
         Current_Scope := GetEnclosingScope (Current_Scope);
      end loop;
      return Current_Region = Outer_Region;
   end IsLocal;

   --------------------------------------------------------------------------------
   --  Get_Enclosing_Package
   --------------------------------------------------------------------------------

   function Get_Enclosing_Package (Scope : Scopes) return RawDict.Package_Info_Ref
   --# global in Dict;
   is
      Current : Scopes;
      Region  : Symbol;
   begin
      Current := Scope;
      loop
         Region := GetRegion (Current);
         exit when RawDict.GetSymbolDiscriminant (Region) = Package_Symbol;
         Current := GetEnclosingScope (Current);
      end loop;
      return RawDict.Get_Package_Info_Ref (Item => Region);
   end Get_Enclosing_Package;

   --------------------------------------------------------------------------------

   function GetEnclosingPackage (Scope : Scopes) return Symbol is
   begin
      return RawDict.Get_Package_Symbol (Get_Enclosing_Package (Scope => Scope)); -- GAA External
   end GetEnclosingPackage;

   --------------------------------------------------------------------------------
   --  Get_Library_Package
   --------------------------------------------------------------------------------

   function Get_Library_Package (Scope : Scopes) return RawDict.Package_Info_Ref
   --# global in Dict;
   is
      Current_Package, Enclosing_Package : RawDict.Package_Info_Ref;
   begin
      Current_Package := Get_Enclosing_Package (Scope => Scope);
      if Current_Package /= Get_Predefined_Package_Standard then
         loop
            exit when RawDict.Get_Package_Parent (The_Package => Current_Package) /= RawDict.Null_Package_Info_Ref;
            Enclosing_Package := Get_Enclosing_Package (Scope => Get_Package_Scope (The_Package => Current_Package));
            exit when Enclosing_Package = Get_Predefined_Package_Standard;
            Current_Package := Enclosing_Package;
         end loop;
      end if;
      return Current_Package;
   end Get_Library_Package;

   --------------------------------------------------------------------------------

   function GetLibraryPackage (Scope : Scopes) return Symbol is
   begin
      return RawDict.Get_Package_Symbol (Get_Library_Package (Scope => Scope)); -- GAA External
   end GetLibraryPackage;

   --------------------------------------------------------------------------------
   --  Get_Package_Owner
   --------------------------------------------------------------------------------

   -- The owner of a package (if any) is defined to be the
   -- the parent of its closest private ancestor, where the term ancestor is
   -- as defined 10.1.1(11) of the Ada 95 LRM [2], to include the package itself.
   function GetPackageOwner (ThePackage : Symbol) return Symbol is

      function Get_Package_Owner (The_Package : RawDict.Package_Info_Ref) return RawDict.Package_Info_Ref
      --# global in Dict;
      is
         Current_Package : RawDict.Package_Info_Ref;
      begin
         Current_Package := The_Package;
         loop
            exit when RawDict.Get_Package_Is_Private (The_Package => Current_Package);
            Current_Package := RawDict.Get_Package_Parent (The_Package => Current_Package);
            exit when Current_Package = RawDict.Null_Package_Info_Ref;
         end loop;

         if Current_Package /= RawDict.Null_Package_Info_Ref then
            Current_Package := RawDict.Get_Package_Parent (The_Package => Current_Package);
         end if;

         return Current_Package;
      end Get_Package_Owner;

   begin -- GetPackageOwner
      return RawDict.Get_Package_Symbol (Get_Package_Owner (The_Package => RawDict.Get_Package_Info_Ref (ThePackage))); -- GAA External
   end GetPackageOwner;

   --------------------------------------------------------------------------------
   --  Get_Constant_Rule_Policy_For_Current_Scope
   --------------------------------------------------------------------------------

   function Get_Constant_Rule_Policy_For_Current_Scope
     (The_Constant : RawDict.Constant_Info_Ref;
      The_Scope    : Scopes)
     return         Rule_Policies
   --# global in Dict;
   is
      The_Rule_Policy : RawDict.Rule_Policy_Info_Ref;
      Rule_Policy     : Rule_Policies := Unspecified;
   begin
      The_Rule_Policy := RawDict.Get_Constant_First_Rule_Policy (The_Constant => The_Constant);
      while The_Rule_Policy /= RawDict.Null_Rule_Policy_Info_Ref loop
         if RawDict.Get_Rule_Policy_Scope (The_Rule_Policy => The_Rule_Policy) = The_Scope then
            Rule_Policy := RawDict.Get_Rule_Policy_Value (The_Rule_Policy => The_Rule_Policy);
            exit;
         end if;
         The_Rule_Policy := RawDict.Get_Next_Rule_Policy (The_Rule_Policy => The_Rule_Policy);
      end loop;
      return Rule_Policy;
   end Get_Constant_Rule_Policy_For_Current_Scope;

   --------------------------------------------------------------------------------

   function IsConstantRulePolicyPresent (TheConstant : Symbol;
                                         TheScope    : Scopes) return Boolean is
   begin
      return Get_Constant_Rule_Policy_For_Current_Scope
        (The_Constant => RawDict.Get_Constant_Info_Ref (TheConstant), -- GAA External
         The_Scope    => TheScope) /= Unspecified;
   end IsConstantRulePolicyPresent;

   --------------------------------------------------------------------------------
   --  Get_Constant_Rule_Policy
   --------------------------------------------------------------------------------

   function GetConstantRulePolicy (TheConstant : Symbol;
                                   TheScope    : Scopes) return Rule_Policies is

      function Get_Constant_Rule_Policy (The_Constant : RawDict.Constant_Info_Ref;
                                         The_Scope    : Scopes) return Rule_Policies
      --# global in Dict;
      is
         Current_Scope : Scopes;
         Rule_Policy   : Rule_Policies;
      begin
         Current_Scope := The_Scope;

         -- check the list of policies for the current scope and any enclosing scope.
         loop
            Rule_Policy := Get_Constant_Rule_Policy_For_Current_Scope (The_Constant => The_Constant,
                                                                       The_Scope    => Current_Scope);
            exit when Rule_Policy /= Unspecified;

            exit when IsPredefinedScope (Current_Scope);

            -- if we have a local package scope check the visible scope for this package next
            if Get_Visibility (Scope => Current_Scope) = Local
              and then RawDict.GetSymbolDiscriminant (GetRegion (Current_Scope)) = Package_Symbol then
               Current_Scope := Set_Visibility (The_Visibility => Visible,
                                                The_Unit       => GetRegion (Current_Scope));
            else
               Current_Scope := GetEnclosingScope (Current_Scope);
            end if;
         end loop;

         -- if still unspecified then check the declaritive region for the constant.
         if Rule_Policy = Unspecified then
            Current_Scope := Get_Constant_Scope (The_Constant => The_Constant);
            Rule_Policy   := Get_Constant_Rule_Policy_For_Current_Scope (The_Constant => The_Constant,
                                                                         The_Scope    => Current_Scope);
         end if;
         return Rule_Policy;
      end Get_Constant_Rule_Policy;

   begin -- GetConstantRulePolicy
      return Get_Constant_Rule_Policy (The_Constant => RawDict.Get_Constant_Info_Ref (TheConstant), -- GAA External
                                       The_Scope    => TheScope);
   end GetConstantRulePolicy;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function TypeIsWellformed (TheType : Symbol) return Boolean is
   begin
      return RawDict.Get_Type_Wellformed (Type_Mark => RawDict.Get_Type_Info_Ref (TheType)); -- GAA External
   end TypeIsWellformed;

   --------------------------------------------------------------------------------

   function Descendent_Is_Private (Inner_Package, Outer_Package : RawDict.Package_Info_Ref) return Boolean
   --# global in Dict;
   is
      Current_Package : RawDict.Package_Info_Ref;
   begin
      Current_Package := Inner_Package;
      loop
         exit when RawDict.Get_Package_Is_Private (The_Package => Current_Package);
         Current_Package := RawDict.Get_Package_Parent (The_Package => Current_Package);
         exit when Current_Package = Outer_Package or else Current_Package = RawDict.Null_Package_Info_Ref;
      end loop;
      return Current_Package /= Outer_Package and then Current_Package /= RawDict.Null_Package_Info_Ref;
   end Descendent_Is_Private;

   --------------------------------------------------------------------------------

   function Is_Private_Seeing_Descendent (Scope       : Scopes;
                                          The_Package : RawDict.Package_Info_Ref) return Boolean
   --# global in Dict;
   is
      Library_Package : RawDict.Package_Info_Ref;
   begin
      Library_Package := Get_Library_Package (Scope => Scope);

      return Is_Proper_Descendent (Inner_Package => Library_Package,
                                   Outer_Package => The_Package)
        and then (Scope /= Set_Visibility (The_Visibility => Visible,
                                           The_Unit       => RawDict.Get_Package_Symbol (Library_Package))
                    or else Descendent_Is_Private (Inner_Package => Library_Package,
                                                   Outer_Package => The_Package));
   end Is_Private_Seeing_Descendent;

   --------------------------------------------------------------------------------
   --  Type_Is_Incomplete_Here
   --------------------------------------------------------------------------------

   function Type_Is_Incomplete_Here (Type_Mark : RawDict.Type_Info_Ref;
                                     Scope     : Scopes) return Boolean
   --# global in Dict;
   is
      Result : Boolean;
   begin
      if RawDict.Get_Type_Declaration (Type_Mark => Type_Mark) /= RawDict.Null_Declaration_Info_Ref then
         Result :=
           (not ((not Type_Is_Private (Type_Mark => Type_Mark)
                    -- Type_Mark is declared in the visible part of package spec
                    and then Get_Visibility (Scope => Get_Type_Scope (Type_Mark => Type_Mark)) = Visible) -- and isn't private
                 or else IsLocal
                   (Scope, Get_Type_Scope (Type_Mark => RawDict.Get_First_Constrained_Subtype (Type_Mark => Type_Mark)))
                   -- Scope is Local relative to Type_Mark (e.g. in same package body)
                   or else Is_Private_Seeing_Descendent
                   (Scope       => Scope,
                    The_Package => RawDict.Get_Package_Info_Ref
                      (GetRegion (Get_Type_Scope (Type_Mark => RawDict.Get_First_Constrained_Subtype (Type_Mark => Type_Mark))))))
              or else
              -- if we are in the visible scope of the place where the type is declared it can't be complete yet
              Scope = Set_Visibility (The_Visibility => Visible,
                                      The_Unit       => GetRegion (Get_Type_Scope (Type_Mark => Type_Mark))));
      else
         -- No Declaration - must be an announced type with no completion
         Result := True;
      end if;
      return Result;
   end Type_Is_Incomplete_Here;

   --------------------------------------------------------------------------------

   function TypeIsIncompleteHere (TypeMark : Symbol;
                                  Scope    : Scopes) return Boolean is
   begin
      return Type_Is_Incomplete_Here (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark), -- GAA External
                                      Scope     => Scope);
   end TypeIsIncompleteHere;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function Type_Is_Private_Here (Type_Mark : RawDict.Type_Info_Ref;
                                  Scope     : Scopes) return Boolean
   --# global in Dict;
   is
   begin
      return Type_Is_Private (Type_Mark => Type_Mark)
        and then -- type must be private to be private here!
        Type_Is_Incomplete_Here (Type_Mark => Type_Mark,
                                 Scope     => Scope);
   end Type_Is_Private_Here;

   -------------------------------------------------------------------------------
   --  Type_Is_Real
   -------------------------------------------------------------------------------

   function Type_Is_Real (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return Type_Is_Fixed_Point (Type_Mark => Type_Mark)
        or else Type_Is_Floating_Point (Type_Mark => Type_Mark);
   end Type_Is_Real;

   --------------------------------------------------------------------------------

   function TypeIsReal (TypeMark : Symbol) return Boolean is
   begin
      return Type_Is_Real (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)); -- GAA External
   end TypeIsReal;

   --------------------------------------------------------------------------------
   --  Is_Real_Type_Mark
   --------------------------------------------------------------------------------

   function Is_Real_Type_Mark (Type_Mark : RawDict.Type_Info_Ref;
                               Scope     : Scopes) return Boolean
   --# global in Dict;
   is
   begin
      return Type_Is_Real (Type_Mark => Type_Mark)
        and then not Type_Is_Private_Here (Type_Mark => Type_Mark,
                                           Scope     => Scope);
   end Is_Real_Type_Mark;

   --------------------------------------------------------------------------------

   function IsRealTypeMark (TheSymbol : Symbol;
                            Scope     : Scopes) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Real_Type_Mark (Type_Mark => RawDict.Get_Type_Info_Ref (TheSymbol), -- GAA External
                                    Scope     => Scope);
   end IsRealTypeMark;

   --------------------------------------------------------------------------------
   --  Get_Base_Type
   --------------------------------------------------------------------------------

   function Get_Base_Type (Type_Mark : RawDict.Type_Info_Ref) return RawDict.Type_Info_Ref
   --# global in Dict;
   is
      Result : RawDict.Type_Info_Ref;
   begin
      Result := Get_Root_Type (Type_Mark => Type_Mark); -- go to first named subtype
      if RawDict.Get_Type_Derived (Type_Mark => Result) then
         -- find base type
         Result := RawDict.Get_Type_Base_Type (Type_Mark => Result);
      end if;
      return Result;
   end Get_Base_Type;

   --------------------------------------------------------------------------------

   function GetBaseType (TypeMark : Symbol) return Symbol is
   begin
      return RawDict.Get_Type_Symbol (Get_Base_Type (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark))); -- GAA External
   end GetBaseType;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function GetRootOfExtendedType (TypeMark : Symbol) return Symbol is
   begin
      return RawDict.Get_Type_Symbol (RawDict.Get_Type_Extends (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark))); -- GAA External
   end GetRootOfExtendedType;

   --------------------------------------------------------------------------------
   --  Get_Enumeration_Literal
   --------------------------------------------------------------------------------

   function Get_Enumeration_Literal (Type_Mark : RawDict.Type_Info_Ref;
                                     Position  : LexTokenManager.Lex_String) return Symbol
   --# global in Dict;
   --#        in LexTokenManager.State;
   is
      Literals : Iterator;
   begin
      Literals := First_Enumeration_Literal (Type_Mark => Get_Root_Type (Type_Mark => Type_Mark));
      loop
         exit when LexTokenManager.Lex_String_Case_Insensitive_Compare
           (Lex_Str1 => RawDict.Get_Enumeration_Literal_Position
              (The_Enumeration_Literal => RawDict.Get_Enumeration_Literal_Info_Ref (CurrentSymbol (Literals))),
            Lex_Str2 => Position) =
           LexTokenManager.Str_Eq;
         Literals := NextSymbol (Literals);
      end loop;
      return CurrentSymbol (Literals);
   end Get_Enumeration_Literal;

   --------------------------------------------------------------------------------

   function GetEnumerationLiteral (EnumerationType : Symbol;
                                   Position        : LexTokenManager.Lex_String) return Symbol is
   begin
      return Get_Enumeration_Literal (Type_Mark => RawDict.Get_Type_Info_Ref (EnumerationType), -- GAA External
                                      Position  => Position);
   end GetEnumerationLiteral;

   --------------------------------------------------------------------------------
   --  Is_Integer_Type_Mark
   --------------------------------------------------------------------------------

   function Is_Integer_Type_Mark (Type_Mark : RawDict.Type_Info_Ref;
                                  Scope     : Scopes) return Boolean
   --# global in Dict;
   is
   begin
      return Type_Is_Integer (Type_Mark => Type_Mark)
        and then not Type_Is_Private_Here (Type_Mark => Type_Mark,
                                           Scope     => Scope);
   end Is_Integer_Type_Mark;

   --------------------------------------------------------------------------------

   function IsIntegerTypeMark (TheSymbol : Symbol;
                               Scope     : Scopes) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Integer_Type_Mark (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol), -- GAA External
                                       Scope     => Scope);
   end IsIntegerTypeMark;

   --------------------------------------------------------------------------------
   --  Is_Modular_Type_Mark
   --------------------------------------------------------------------------------

   function Is_Modular_Type_Mark (Type_Mark : RawDict.Type_Info_Ref;
                                  Scope     : Scopes) return Boolean
   --# global in Dict;
   is
   begin
      return Type_Is_Modular (Type_Mark => Type_Mark)
        and then not Type_Is_Private_Here (Type_Mark => Type_Mark,
                                           Scope     => Scope);
   end Is_Modular_Type_Mark;

   --------------------------------------------------------------------------------

   function IsModularTypeMark (TheSymbol : Symbol;
                               Scope     : Scopes) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Modular_Type_Mark (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol), -- GAA External
                                       Scope     => Scope);
   end IsModularTypeMark;

   --------------------------------------------------------------------------------
   --  Is_Floating_Point_Type_Mark
   --------------------------------------------------------------------------------

   function Is_Floating_Point_Type_Mark (Type_Mark : RawDict.Type_Info_Ref;
                                         Scope     : Scopes) return Boolean
   --# global in Dict;
   is
   begin
      return Type_Is_Floating_Point (Type_Mark => Type_Mark)
        and then not Type_Is_Private_Here (Type_Mark => Type_Mark,
                                           Scope     => Scope);
   end Is_Floating_Point_Type_Mark;

   --------------------------------------------------------------------------------

   function IsFloatingPointTypeMark (TheSymbol : Symbol;
                                     Scope     : Scopes) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Floating_Point_Type_Mark (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol), -- GAA External
                                              Scope     => Scope);
   end IsFloatingPointTypeMark;

   --------------------------------------------------------------------------------
   --  Is_Fixed_Point_Type_Mark
   --------------------------------------------------------------------------------

   function Is_Fixed_Point_Type_Mark (Type_Mark : RawDict.Type_Info_Ref;
                                      Scope     : Scopes) return Boolean
   --# global in Dict;
   is
   begin
      return Type_Is_Fixed_Point (Type_Mark => Type_Mark)
        and then not Type_Is_Private_Here (Type_Mark => Type_Mark,
                                           Scope     => Scope);
   end Is_Fixed_Point_Type_Mark;

   --------------------------------------------------------------------------------

   function IsFixedPointTypeMark (TheSymbol : Symbol;
                                  Scope     : Scopes) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Fixed_Point_Type_Mark (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol), -- GAA External
                                           Scope     => Scope);
   end IsFixedPointTypeMark;

   --------------------------------------------------------------------------------
   --  Is_Array_Type_Mark
   --------------------------------------------------------------------------------

   function Is_Array_Type_Mark (Type_Mark : RawDict.Type_Info_Ref;
                                Scope     : Scopes) return Boolean
   --# global in Dict;
   is
   begin
      return Type_Is_Array (Type_Mark => Type_Mark)
        and then not Type_Is_Private_Here (Type_Mark => Type_Mark,
                                           Scope     => Scope);
   end Is_Array_Type_Mark;

   --------------------------------------------------------------------------------

   function IsArrayTypeMark (TheSymbol : Symbol;
                             Scope     : Scopes) return Boolean is
   begin
      return Is_Array_Type_Mark (Type_Mark => RawDict.Get_Type_Info_Ref (TheSymbol), -- GAA External
                                 Scope     => Scope);
   end IsArrayTypeMark;

   --------------------------------------------------------------------------------
   --  Is_Integer_Type
   --------------------------------------------------------------------------------

   function IsIntegerType (TheSymbol : Symbol;
                           Scope     : Scopes) return Boolean is

      function Is_Integer_Type (Type_Mark : RawDict.Type_Info_Ref;
                                Scope     : Scopes) return Boolean
      --# global in Dict;
      is
      begin
         return Is_Type (Type_Mark => Type_Mark)
           and then Is_Integer_Type_Mark (Type_Mark => Type_Mark,
                                          Scope     => Scope);
      end Is_Integer_Type;

   begin -- IsIntegerType
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Integer_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol), -- GAA External
                                  Scope     => Scope);
   end IsIntegerType;

   --------------------------------------------------------------------------------
   --  Is_Modular_Type
   --------------------------------------------------------------------------------

   function IsModularType (TheSymbol : Symbol;
                           Scope     : Scopes) return Boolean is

      function Is_Modular_Type (Type_Mark : RawDict.Type_Info_Ref;
                                Scope     : Scopes) return Boolean
      --# global in Dict;
      is
      begin
         return Is_Type (Type_Mark => Type_Mark)
           and then Is_Modular_Type_Mark (Type_Mark => Type_Mark,
                                          Scope     => Scope);
      end Is_Modular_Type;

   begin -- IsModularType
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Modular_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol), -- GAA External
                                  Scope     => Scope);
   end IsModularType;

   --------------------------------------------------------------------------------
   --  Is_Floating_Point_Type
   --------------------------------------------------------------------------------

   function IsFloatingPointType (TheSymbol : Symbol;
                                 Scope     : Scopes) return Boolean is

      function Is_Floating_Point_Type (Type_Mark : RawDict.Type_Info_Ref;
                                       Scope     : Scopes) return Boolean
      --# global in Dict;
      is
      begin
         return Is_Type (Type_Mark => Type_Mark)
           and then Is_Floating_Point_Type_Mark (Type_Mark => Type_Mark,
                                                 Scope     => Scope);
      end Is_Floating_Point_Type;

   begin -- IsFloatingPointType
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Floating_Point_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol), -- GAA External
                                         Scope     => Scope);
   end IsFloatingPointType;

   --------------------------------------------------------------------------------
   --  Is_Fixed_Point_Type
   --------------------------------------------------------------------------------

   function IsFixedPointType (TheSymbol : Symbol;
                              Scope     : Scopes) return Boolean is

      function Is_Fixed_Point_Type (Type_Mark : RawDict.Type_Info_Ref;
                                    Scope     : Scopes) return Boolean
      --# global in Dict;
      is
      begin
         return Is_Type (Type_Mark => Type_Mark)
           and then Is_Fixed_Point_Type_Mark (Type_Mark => Type_Mark,
                                              Scope     => Scope);
      end Is_Fixed_Point_Type;

   begin -- IsFixedPointType
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Fixed_Point_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol), -- GAA External
                                      Scope     => Scope);
   end IsFixedPointType;

   --------------------------------------------------------------------------------
   --  Compatible_Types
   --------------------------------------------------------------------------------

   function CompatibleTypes
     (Scope : Scopes;
      Left  : Symbol;
      Right : Symbol)
     return  Boolean is

      function Compatible_Types
        (Scope          : Scopes;
         The_Left_Type  : RawDict.Type_Info_Ref;
         The_Right_Type : RawDict.Type_Info_Ref)
        return  Boolean
      --# global in Dict;
      is
         Base1, Base2 : RawDict.Type_Info_Ref;
         Result       : Boolean;
      begin
         if The_Left_Type = Get_Unknown_Type_Mark
           or else The_Right_Type = Get_Unknown_Type_Mark then
            Result := True;
         else
            Base1 := Get_Root_Type (Type_Mark => The_Left_Type);
            Base2 := Get_Root_Type (Type_Mark => The_Right_Type);
            if Base1 = Base2 then
               Result := True;
            elsif Base1 = Get_Universal_Integer_Type then
               Result := Is_Integer_Type_Mark (Type_Mark => Base2,
                                               Scope     => Scope)
                 or else Is_Modular_Type_Mark (Type_Mark => Base2,
                                               Scope     => Scope);
            elsif Base2 = Get_Universal_Integer_Type then
               Result := Is_Integer_Type_Mark (Type_Mark => Base1,
                                               Scope     => Scope)
                 or else Is_Modular_Type_Mark (Type_Mark => Base1,
                                               Scope     => Scope);
            elsif Base1 = Get_Universal_Real_Type then
               Result := Is_Real_Type_Mark (Type_Mark => Base2,
                                            Scope     => Scope);
            elsif Base2 = Get_Universal_Real_Type then
               Result := Is_Real_Type_Mark (Type_Mark => Base1,
                                            Scope     => Scope);
            else
               Result := False;
            end if;
         end if;
         return Result;
      end Compatible_Types;

   begin -- CompatibleTypes
      return Compatible_Types (Scope          => Scope,
                               The_Left_Type  => RawDict.Get_Type_Info_Ref (Left), -- GAA External
                               The_Right_Type => RawDict.Get_Type_Info_Ref (Right)); -- GAA External
   end CompatibleTypes;

   --------------------------------------------------------------------------------
   --  Get_Predefined_Character_Type
   --------------------------------------------------------------------------------

   function Get_Predefined_Character_Type return  RawDict.Type_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Types.The_Predefined_Types (Predefined_Character_Type);
   end Get_Predefined_Character_Type;

   --------------------------------------------------------------------------------

   function GetPredefinedCharacterType return Symbol is
   begin
      return RawDict.Get_Type_Symbol (Get_Predefined_Character_Type); -- GAA External
   end GetPredefinedCharacterType;

   --------------------------------------------------------------------------------

   function IsPredefinedCharacterType (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.Get_Type_Info_Ref (TheSymbol) = Get_Predefined_Character_Type; -- GAA External
   end IsPredefinedCharacterType;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function TypeIsGeneric (TheType : Symbol) return Boolean is
   begin
      return RawDict.Get_Type_Discriminant (Type_Mark => RawDict.Get_Type_Info_Ref (TheType)) = Generic_Type_Item; -- GAA External
   end TypeIsGeneric;

   --------------------------------------------------------------------------------
   --  Type_Is_Discrete
   --------------------------------------------------------------------------------

   function Type_Is_Discrete (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return Type_Is_Enumeration (Type_Mark => Type_Mark)
        or else Type_Is_Integer (Type_Mark => Type_Mark)
        or else Type_Is_Modular (Type_Mark => Type_Mark);
   end Type_Is_Discrete;

   --------------------------------------------------------------------------------

   function TypeIsDiscrete (TypeMark : Symbol) return Boolean is
   begin
      return Type_Is_Discrete (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)); -- GAA External
   end TypeIsDiscrete;

   --------------------------------------------------------------------------------
   --  Type_Is_Numeric
   --------------------------------------------------------------------------------

   function Type_Is_Numeric (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return Type_Is_Integer (Type_Mark => Type_Mark)
        or else Type_Is_Modular (Type_Mark => Type_Mark)
        or else Type_Is_Real (Type_Mark => Type_Mark);
   end Type_Is_Numeric;

   --------------------------------------------------------------------------------

   function TypeIsNumeric (TypeMark : Symbol) return Boolean is
   begin
      return Type_Is_Numeric (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)); -- GAA External
   end TypeIsNumeric;

   --------------------------------------------------------------------------------
   --  Type_Is_Scalar
   --------------------------------------------------------------------------------

   function Type_Is_Scalar (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return Type_Is_Discrete (Type_Mark => Type_Mark) or else Type_Is_Numeric (Type_Mark => Type_Mark);
   end Type_Is_Scalar;

   --------------------------------------------------------------------------------

   function TypeIsScalar (TypeMark : Symbol) return Boolean is
   begin
      return Type_Is_Scalar (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)); -- GAA External
   end TypeIsScalar;

   --------------------------------------------------------------------------------
   --  First_Array_Index
   --------------------------------------------------------------------------------

   function First_Array_Index (Type_Mark : RawDict.Type_Info_Ref) return Iterator
   --# global in Dict;
   is
      The_Array_Index : RawDict.Array_Index_Info_Ref;
      Array_Indices   : Iterator := NullIterator;
   begin
      The_Array_Index := RawDict.Get_Type_First_Array_Index (Type_Mark => Type_Mark);
      if The_Array_Index /= RawDict.Null_Array_Index_Info_Ref then
         Array_Indices := Iterator'
           (ArrayIndexIterator,
            IsAbstract,
            RawDict.Get_Type_Symbol (RawDict.Get_Array_Index_Type (The_Array_Index => The_Array_Index)),
            RawDict.Get_Array_Index_Symbol (The_Array_Index));
      end if;
      return Array_Indices;
   end First_Array_Index;

   --------------------------------------------------------------------------------

   function FirstArrayIndex (TypeMark : Symbol) return Iterator is
   begin
      return First_Array_Index (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)); -- GAA External
   end FirstArrayIndex;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function IsPrivateTypeMark (TheSymbol : Symbol;
                               Scope     : Scopes) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Type_Is_Private_Here (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol), -- GAA External
                                       Scope     => Scope);
   end IsPrivateTypeMark;

   --------------------------------------------------------------------------------
   --  Is_Private_Type
   --------------------------------------------------------------------------------

   function IsPrivateType (TheSymbol : Symbol;
                           Scope     : Scopes) return Boolean is

      function Is_Private_Type (Type_Mark : RawDict.Type_Info_Ref;
                                Scope     : Scopes) return Boolean
      --# global in Dict;
      is
      begin
         return Is_Type (Type_Mark => Type_Mark)
           and then Type_Is_Private_Here (Type_Mark => Type_Mark,
                                          Scope     => Scope);
      end Is_Private_Type;

   begin -- IsPrivateType
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Private_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol), -- GAA External
                                  Scope     => Scope);
   end IsPrivateType;

   --------------------------------------------------------------------------------
   --  Is_Limited_Private_Type
   --------------------------------------------------------------------------------

   function IsLimitedPrivateType (TheSymbol : Symbol) return Boolean is

      function Is_Limited_Private_Type (Type_Mark : RawDict.Type_Info_Ref) return Boolean
      --# global in Dict;
      is
      begin
         return Is_Type (Type_Mark => Type_Mark)
           and then RawDict.Get_Type_Limited_Private (Type_Mark => Type_Mark);
      end Is_Limited_Private_Type;

   begin -- IsLimitedPrivateType
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Limited_Private_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol)); -- GAA External
   end IsLimitedPrivateType;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function TypeIsAbstractProof (TypeMark : Symbol) return Boolean is
   begin
      return RawDict.Get_Type_Discriminant (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)) = -- GAA External
        Abstract_Proof_Type_Item;
   end TypeIsAbstractProof;

   --------------------------------------------------------------------------------

   function Is_Proof_Type (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return not Is_Subtype (Type_Mark => Type_Mark)
        and then (Get_Type_Context (Type_Mark => Type_Mark) = ProofContext
                    or else RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark) = Abstract_Proof_Type_Item);
   end Is_Proof_Type;

   --------------------------------------------------------------------------------
   --  Type_Is_Own_Abstract_Here
   --------------------------------------------------------------------------------

   -- This function determines whether the declaration of TypeMark is
   -- not visible from the current viewpoint (Scope) i.e. "abstract here".
   -- In the case of TypeMark being overloaded to mean both a concrete
   -- type and a type that has been "announced" in an own variable announcement
   -- the function determines which is the right view of TypeMark in the
   -- given scope (abstract or concrete).

   function TypeIsOwnAbstractHere (TypeMark : Symbol;
                                   Scope    : Scopes) return Boolean is

      function Type_Is_Own_Abstract_Here (Type_Mark : RawDict.Type_Info_Ref;
                                          Scope     : Scopes) return Boolean
      --# global in Dict;
      is
         DeclScope                      : Scopes;
         DeclarationVisibleHere, Result : Boolean;
      begin
         DeclScope              := Get_Type_Scope (Type_Mark => Type_Mark);
         DeclarationVisibleHere :=
           IsLocal (Scope, DeclScope)
           or else (RawDict.GetSymbolDiscriminant (GetRegion (Get_Type_Scope (Type_Mark => Type_Mark))) = Package_Symbol
                      and then Is_Private_Seeing_Descendent (Scope       => Scope,
                                                             The_Package => RawDict.Get_Package_Info_Ref (GetRegion (DeclScope))))
           or else (not Type_Is_Private (Type_Mark => Type_Mark)
                      and then Get_Visibility (Scope => Get_Type_Scope (Type_Mark => Type_Mark)) = Visible);
         Result                 := RawDict.Get_Type_Is_Own_Var_Type (Type_Mark => Type_Mark)
           and then (Is_Proof_Type (Type_Mark => Type_Mark) or else not DeclarationVisibleHere);
         return Result;
      end Type_Is_Own_Abstract_Here;

   begin -- TypeIsOwnAbstractHere
      return Type_Is_Own_Abstract_Here (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark), -- GAA External
                                        Scope     => Scope);
   end TypeIsOwnAbstractHere;

   --------------------------------------------------------------------------------
   --  Is_Unconstrained_Array_Type
   --------------------------------------------------------------------------------

   function Is_Unconstrained_Array_Type (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return Type_Is_Array (Type_Mark => Type_Mark)
        and then not RawDict.Get_Type_Constrained (Type_Mark => Type_Mark);
   end Is_Unconstrained_Array_Type;

   --------------------------------------------------------------------------------

   function IsUnconstrainedArrayType (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Unconstrained_Array_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol)); -- GAA External
   end IsUnconstrainedArrayType;

   --------------------------------------------------------------------------------
   --  Is_Unconstrained_Task_Type
   --------------------------------------------------------------------------------

   function IsUnconstrainedTaskType (TheSymbol : Symbol) return Boolean is

      function Is_Unconstrained_Task_Type (Type_Mark : RawDict.Type_Info_Ref) return Boolean
      --# global in Dict;
      is
      begin
         return Is_Task_Type (Type_Mark => Type_Mark)
           and then not RawDict.Get_Type_Constrained (Type_Mark => Type_Mark);
      end Is_Unconstrained_Task_Type;

   begin -- IsUnconstrainedTaskType
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Unconstrained_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol)); -- GAA External
   end IsUnconstrainedTaskType;

   --------------------------------------------------------------------------------
   --  Is_Unconstrained_Protected_Type
   --------------------------------------------------------------------------------

   function IsUnconstrainedProtectedType (TheSymbol : Symbol) return Boolean is

      function Is_Unconstrained_Protected_Type (Type_Mark : RawDict.Type_Info_Ref) return Boolean
      --# global in Dict;
      is
      begin
         return Is_Protected_Type (Type_Mark => Type_Mark)
           and then not RawDict.Get_Type_Constrained (Type_Mark => Type_Mark);
      end Is_Unconstrained_Protected_Type;

   begin -- IsUnconstrainedProtectedType
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Unconstrained_Protected_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol)); -- GAA External
   end IsUnconstrainedProtectedType;

   --------------------------------------------------------------------------------
   --  Is_Array_Attribute
   --------------------------------------------------------------------------------

   function IsArrayAttribute (Name     : LexTokenManager.Lex_String;
                              TypeMark : Symbol) return Boolean is

      function Is_Array_Attribute (Name      : LexTokenManager.Lex_String;
                                   Type_Mark : RawDict.Type_Info_Ref) return Boolean
      --# global in Dict;
      --#        in LexTokenManager.State;
      is
         Result : Boolean;
      begin
         if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Name,
                                                                 Lex_Str2 => LexTokenManager.First_Token) =
           LexTokenManager.Str_Eq
           or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Name,
                                                                        Lex_Str2 => LexTokenManager.Last_Token) =
           LexTokenManager.Str_Eq then
            Result := Type_Is_Array (Type_Mark => Type_Mark);
         elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Name,
                                                                    Lex_Str2 => LexTokenManager.Length_Token) =
           LexTokenManager.Str_Eq
           or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Name,
                                                                        Lex_Str2 => LexTokenManager.Range_Token) =
           LexTokenManager.Str_Eq
           or else LexTokenManager.Lex_String_Case_Insensitive_Compare
           (Lex_Str1 => Name,
            Lex_Str2 => LexTokenManager.Component_Size_Token) =
           LexTokenManager.Str_Eq then
            Result := True;
         else
            Result := False;
         end if;
         return Result;
      end Is_Array_Attribute;

   begin -- IsArrayAttribute
      return Is_Array_Attribute (Name      => Name,
                                 Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)); -- GAA External
   end IsArrayAttribute;

   --------------------------------------------------------------------------------
   --  Own_Variable_Has_Type
   --------------------------------------------------------------------------------

   -- Note that this function returns false if the type of the own variable is
   -- an abstract proof type. Note also that it may return true (and does)
   -- even if the variable is not declared (Is_Declared). The function checks
   -- for visibility of type declaration from the current scope, but does not
   -- take into account whether the declaration itself (which may not exist)
   -- is visible from the given scope or whether the viewpoint is a child
   -- package seeing private declarations. See also GetOwnVariableTypeHere and
   -- TypeIsOwnAbstractHere.
   function OwnVariableHasType (OwnVariable : Symbol;
                                Scope       : Scopes) return Boolean is

      function Own_Variable_Has_Type (Own_Variable : RawDict.Variable_Info_Ref;
                                      Scope        : Scopes) return Boolean
      --# global in Dict;
      is
         Type_Mark : RawDict.Type_Info_Ref;
         Result    : Boolean;
      begin
         Type_Mark := RawDict.Get_Variable_Type (The_Variable => Own_Variable);
         if Type_Mark = Get_Unknown_Type_Mark then
            Result := False;
         elsif RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark) = Abstract_Proof_Type_Item then
            Result := False;
         elsif RawDict.Get_Own_Variable_Typed
           (The_Own_Variable => RawDict.Get_Variable_Own_Variable
              (The_Variable => Own_Variable)) then
            Result := True;
         else
            Result := Get_Visibility (Scope => Get_Type_Scope (Type_Mark => Type_Mark)) = Visible
              or else IsLocal (Scope, Get_Type_Scope (Type_Mark => Type_Mark));
         end if;
         return Result;
      end Own_Variable_Has_Type;

   begin -- OwnVariableHasType
      return Own_Variable_Has_Type (Own_Variable => RawDict.Get_Variable_Info_Ref (OwnVariable), -- GAA External
                                    Scope        => Scope);
   end OwnVariableHasType;

   --------------------------------------------------------------------------------
   --  Own_Variable_Is_Announced
   --------------------------------------------------------------------------------

   function OwnVariableIsAnnounced (Variable : Symbol) return Boolean is

      function Own_Variable_Is_Announced (The_Variable : RawDict.Variable_Info_Ref) return Boolean
      --# global in Dict;
      is
      begin
         return RawDict.Get_Own_Variable_Announced
           (The_Own_Variable => RawDict.Get_Variable_Own_Variable (The_Variable => The_Variable));
      end Own_Variable_Is_Announced;

   begin -- OwnVariableIsAnnounced
      return Own_Variable_Is_Announced (The_Variable => RawDict.Get_Variable_Info_Ref (Variable)); -- GAA External
   end OwnVariableIsAnnounced;

   --------------------------------------------------------------------------------
   --  Own_Variable_Is_Initialized
   --------------------------------------------------------------------------------

   function Own_Variable_Is_Initialized (The_Variable : RawDict.Variable_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Own_Variable_Initialized
        (The_Own_Variable => RawDict.Get_Variable_Own_Variable (The_Variable => The_Variable));
   end Own_Variable_Is_Initialized;

   --------------------------------------------------------------------------------

   function OwnVariableIsInitialized (Variable : Symbol) return Boolean is
   begin
      return Own_Variable_Is_Initialized (The_Variable => RawDict.Get_Variable_Info_Ref (Variable)); -- GAA External
   end OwnVariableIsInitialized;

   --------------------------------------------------------------------------------
   --  Own_Variable_Has_Constituents
   --------------------------------------------------------------------------------

   function Own_Variable_Has_Constituents (The_Variable : RawDict.Variable_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Own_Variable_Constituents
        (The_Own_Variable => RawDict.Get_Variable_Own_Variable (The_Variable => The_Variable)) /=
        RawDict.Null_Constituent_Info_Ref;
   end Own_Variable_Has_Constituents;

   --------------------------------------------------------------------------------

   function OwnVariableHasConstituents (Variable : Symbol) return Boolean is
   begin
      return Own_Variable_Has_Constituents (The_Variable => RawDict.Get_Variable_Info_Ref (Variable)); -- GAA External
   end OwnVariableHasConstituents;

   --------------------------------------------------------------------------------
   --  First_Constituent
   --------------------------------------------------------------------------------

   function First_Constituent (The_Variable : RawDict.Variable_Info_Ref) return Iterator
   --# global in Dict;
   is
      The_Constituent : RawDict.Constituent_Info_Ref;
      Constituents    : Iterator := NullIterator;
   begin
      The_Constituent := RawDict.Get_Own_Variable_Constituents
        (The_Own_Variable => RawDict.Get_Variable_Own_Variable (The_Variable => The_Variable));
      if The_Constituent /= RawDict.Null_Constituent_Info_Ref then
         Constituents := Iterator'
           (ConstituentIterator,
            IsAbstract,
            RawDict.Get_Constituent_Symbol (The_Constituent),
            NullSymbol);
      end if;
      return Constituents;
   end First_Constituent;

   --------------------------------------------------------------------------------

   function FirstConstituent (Subject : Symbol) return Iterator is
   begin
      return First_Constituent (The_Variable => RawDict.Get_Variable_Info_Ref (Subject)); -- GAA External
   end FirstConstituent;

   --------------------------------------------------------------------------------
   --  Get_Main_Program
   --------------------------------------------------------------------------------

   function Get_Main_Program return  RawDict.Subprogram_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Main.Subprogram;
   end Get_Main_Program;

   --------------------------------------------------------------------------------

   function GetMainProgram return Symbol is
   begin
      return RawDict.Get_Subprogram_Symbol (Get_Main_Program); -- GAA External
   end GetMainProgram;

   --------------------------------------------------------------------------------
   --  Is_Main_Program
   --------------------------------------------------------------------------------

   -- special handling that affects the main subprogram also affects the "subprogram"
   -- that handles partition-wide stuff so this function returns true for either

   function Is_Main_Program (The_Subprogram : RawDict.Subprogram_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return The_Subprogram = Get_Main_Program or else The_Subprogram = Get_The_Partition;
   end Is_Main_Program;

   --------------------------------------------------------------------------------

   function IsMainProgram (Subprogram : Symbol) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (Subprogram) = Subprogram_Symbol
        and then Is_Main_Program (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Subprogram)); -- GAA External
   end IsMainProgram;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   -- this can be used to distinguish the main program symbol from the partition sym.
   function IsThePartition (Subprogram : Symbol) return Boolean is
   begin
      return RawDict.Get_Subprogram_Info_Ref (Subprogram) = Get_The_Partition; -- GAA External
   end IsThePartition;

   --------------------------------------------------------------------------------

   function FirstInheritsClause (Sym : Symbol) return Iterator is
      Inherit_Clause  : RawDict.Context_Clause_Info_Ref;
      Inherit_Clauses : Iterator := NullIterator;
   begin
      case RawDict.GetSymbolDiscriminant (Sym) is
         when Package_Symbol =>
            Inherit_Clause := RawDict.Get_Package_Inherit_Clauses
              (The_Package => RawDict.Get_Package_Info_Ref (Item => Sym)); -- GAA External
         when Subprogram_Symbol =>
            if Is_Main_Program (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Sym)) then -- GAA External
               Inherit_Clause := RawDict.Get_Subprogram_Inherit_Clauses
                 (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Sym)); -- GAA External
            else
               Inherit_Clause := RawDict.Null_Context_Clause_Info_Ref;
            end if;
         when others => -- non-exec code
            Inherit_Clause := RawDict.Null_Context_Clause_Info_Ref;
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.FirstInheritsClause");
      end case;

      if Inherit_Clause /= RawDict.Null_Context_Clause_Info_Ref then
         case RawDict.Get_Context_Clause_Is_Subprogram (The_Context_Clause => Inherit_Clause) is
            when False =>
               Inherit_Clauses := Iterator'
                 (InheritedPackageIterator,
                  IsAbstract,
                  RawDict.Get_Package_Symbol (RawDict.Get_Context_Clause_Package (The_Context_Clause => Inherit_Clause)),
                  RawDict.Get_Context_Clause_Symbol (Inherit_Clause));
            when True =>
               Inherit_Clauses := Iterator'
                 (InheritedPackageIterator,
                  IsAbstract,
                  RawDict.Get_Subprogram_Symbol (RawDict.Get_Context_Clause_Subprogram (The_Context_Clause => Inherit_Clause)),
                  RawDict.Get_Context_Clause_Symbol (Inherit_Clause));
         end case;
      end if;
      return Inherit_Clauses;
   end FirstInheritsClause;

   --------------------------------------------------------------------------------
   --  First_Local_Variable
   --------------------------------------------------------------------------------

   function First_Local_Variable (Subprogram : Symbol) return Iterator is
      Variables : Iterator;

      --------------------------------------------------------------------------------

      function First_Local_Variable_Local (The_Declaration : RawDict.Declaration_Info_Ref) return Iterator
      --# global in Dict;
      is
         Tmp_Declaration : RawDict.Declaration_Info_Ref;
         Item            : Symbol;
         Variables       : Iterator := NullIterator;
      begin
         Tmp_Declaration := The_Declaration;
         while Tmp_Declaration /= RawDict.Null_Declaration_Info_Ref loop
            Item := RawDict.Get_Declaration_Item (The_Declaration => Tmp_Declaration);
            if Is_Variable (Item) then
               Variables       := Iterator'
                 (LocalVariableIterator,
                  IsAbstract,
                  Item,
                  RawDict.Get_Declaration_Symbol (Tmp_Declaration));
               Tmp_Declaration := RawDict.Null_Declaration_Info_Ref;
            else
               Tmp_Declaration := RawDict.Get_Next_Declaration (The_Declaration => Tmp_Declaration);
            end if;
         end loop;
         return Variables;
      end First_Local_Variable_Local;

   begin -- First_Local_Variable
      case RawDict.GetSymbolDiscriminant (Subprogram) is
         when Subprogram_Symbol =>
            Variables := First_Local_Variable_Local
              (The_Declaration => RawDict.Get_Subprogram_First_Declaration
                 (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Subprogram))); -- GAA External
         when Type_Symbol =>
            case RawDict.Get_Type_Discriminant (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Subprogram)) is -- GAA External
               when Task_Type_Item =>
                  Variables := First_Local_Variable_Local
                    (The_Declaration => RawDict.Get_Task_Type_First_Local_Declaration
                       (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => Subprogram))); -- GAA External
               when Protected_Type_Item =>
                  Variables := NullIterator;
               when others => -- non-exec code
                  Variables := NullIterator;
                  SystemErrors.Fatal_Error
                    (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                     Msg     => "in Dictionary.First_Local_Variable");
            end case;
         when others => -- non-exec code
            Variables := NullIterator;
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.First_Local_Variable");
      end case;
      return Variables;
   end First_Local_Variable;

   --------------------------------------------------------------------------------
   --  First_Initialized_Variable
   --------------------------------------------------------------------------------

   function First_Initialized_Variable (Subprogram : Symbol) return Iterator is
      Variables : Iterator;

      --------------------------------------------------------------------------------

      function First_Initialized_Variable_Local (The_Declaration : RawDict.Declaration_Info_Ref) return Iterator
      --# global in Dict;
      is
         Tmp_Declaration : RawDict.Declaration_Info_Ref;
         Item            : Symbol;
         Variables       : Iterator := NullIterator;
      begin
         Tmp_Declaration := The_Declaration;
         while Tmp_Declaration /= RawDict.Null_Declaration_Info_Ref loop
            Item := RawDict.Get_Declaration_Item (The_Declaration => Tmp_Declaration);
            if Is_Variable (Item)
              and then RawDict.Get_Variable_Initialized (The_Variable => RawDict.Get_Variable_Info_Ref (Item)) then
               Variables       := Iterator'
                 (InitializedVariableIterator,
                  IsAbstract,
                  Item,
                  RawDict.Get_Declaration_Symbol (Tmp_Declaration));
               Tmp_Declaration := RawDict.Null_Declaration_Info_Ref;
            else
               Tmp_Declaration := RawDict.Get_Next_Declaration (The_Declaration => Tmp_Declaration);
            end if;
         end loop;
         return Variables;
      end First_Initialized_Variable_Local;

   begin -- First_Initialized_Variable
      case RawDict.GetSymbolDiscriminant (Subprogram) is
         when Subprogram_Symbol =>
            Variables := First_Initialized_Variable_Local
              (The_Declaration => RawDict.Get_Subprogram_First_Declaration
                 (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Subprogram))); -- GAA External
         when Type_Symbol =>
            case RawDict.Get_Type_Discriminant (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Subprogram)) is -- GAA External
               when Task_Type_Item =>
                  Variables := First_Initialized_Variable_Local
                    (The_Declaration => RawDict.Get_Task_Type_First_Local_Declaration
                       (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => Subprogram))); -- GAA External
               when Protected_Type_Item =>
                  Variables := NullIterator;
               when others => -- non-exec code
                  Variables := NullIterator;
                  SystemErrors.Fatal_Error
                    (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                     Msg     => "in Dictionary.First_Initialized_Variable");
            end case;
         when others => -- non-exec code
            Variables := NullIterator;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.First_Initialized_Variable");
      end case;
      return Variables;
   end First_Initialized_Variable;

   --------------------------------------------------------------------------------

   function IsNullIterator (Current : Iterator) return Boolean is
   begin
      return Current.Discriminant = NullSymIterator;
   end IsNullIterator;

   --------------------------------------------------------------------------------

   function GetNumberOfGlobalVariables (Abstraction : Abstractions;
                                        Subprogram  : Symbol) return Natural is
      GlobalVariable : Iterator;
      Count          : Natural := Natural'First;
   begin
      GlobalVariable := FirstGlobalVariable (Abstraction, Subprogram);
      while not IsNullIterator (GlobalVariable) and then Count < Natural'Last loop
         GlobalVariable := NextSymbol (GlobalVariable);
         Count          := Count + 1;
      end loop;
      return Count;
   end GetNumberOfGlobalVariables;

   --------------------------------------------------------------------------------
   --  Get_Array_Index
   --------------------------------------------------------------------------------

   function Get_Array_Index (Type_Mark : RawDict.Type_Info_Ref;
                             Dimension : Positive) return RawDict.Type_Info_Ref
   --# global in Dict;
   is
      Array_Index : Iterator;
   begin
      Array_Index := First_Array_Index (Type_Mark => Type_Mark);
      for Dim in Positive range 1 .. Dimension - 1 loop
         Array_Index := NextSymbol (Array_Index);
      end loop;
      return RawDict.Get_Type_Info_Ref (CurrentSymbol (Array_Index));
   end Get_Array_Index;

   --------------------------------------------------------------------------------

   function GetArrayIndex (TypeMark  : Symbol;
                           Dimension : Positive) return Symbol is
   begin
      return RawDict.Get_Type_Symbol -- GAA External
        (Get_Array_Index (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark), -- GAA External
                          Dimension => Dimension));
   end GetArrayIndex;

   --------------------------------------------------------------------------------
   --  Get_Simple_Name
   --------------------------------------------------------------------------------

   function Get_Subcomponent_Simple_Name (The_Subcomponent : RawDict.Subcomponent_Info_Ref) return LexTokenManager.Lex_String
   --# global Dict;
   is
   begin
      return RawDict.Get_Record_Component_Name
        (The_Record_Component => RawDict.Get_Subcomponent_Record_Component (The_Subcomponent => The_Subcomponent));
   end Get_Subcomponent_Simple_Name;

   --------------------------------------------------------------------------------

   function Get_Parameter_Constraint_Simple_Name
     (The_Parameter_Constraint : RawDict.Parameter_Constraint_Info_Ref)
     return                     LexTokenManager.Lex_String
   --# global Dict;
   is
   begin
      return RawDict.Get_Subprogram_Parameter_Name
        (The_Subprogram_Parameter => RawDict.Get_Parameter_Constraint_Subprogram_Parameter
           (The_Parameter_Constraint => The_Parameter_Constraint));
   end Get_Parameter_Constraint_Simple_Name;

   --------------------------------------------------------------------------------

   function GetSimpleName (Item : Symbol) return LexTokenManager.Lex_String is
      Name       : LexTokenManager.Lex_String;
      Item_Local : Symbol;

      --------------------------------------------------------------------------------

      function Get_Generic_Parameter_Simple_Name
        (The_Generic_Parameter : RawDict.Generic_Parameter_Info_Ref)
        return                  LexTokenManager.Lex_String
      --# global Dict;
      is
         Result : LexTokenManager.Lex_String;
      begin
         -- A generic parameter might be a type or an object.  The Kind field tells us which.  The name
         -- lives in either the type or the object pointed at.
         case RawDict.Get_Generic_Parameter_Kind (The_Generic_Parameter => The_Generic_Parameter) is
            when Generic_Type_Parameter =>
               Result :=
                 RawDict.Get_Type_Name
                 (Type_Mark => RawDict.Get_Generic_Parameter_Type (The_Generic_Parameter => The_Generic_Parameter));
            when Generic_Object_Parameter =>
               Result :=
                 RawDict.Get_Constant_Name
                 (The_Constant => RawDict.Get_Generic_Parameter_Object (The_Generic_Parameter => The_Generic_Parameter));
         end case;
         return Result;
      end Get_Generic_Parameter_Simple_Name;

      --------------------------------------------------------------------------------

      function Get_Constituent_Simple_Name (The_Constituent : RawDict.Constituent_Info_Ref) return LexTokenManager.Lex_String
      --# global Dict;
      is
      begin
         return RawDict.Get_Variable_Name (The_Variable => RawDict.Get_Constituent_Variable (The_Constituent => The_Constituent));
      end Get_Constituent_Simple_Name;

      --------------------------------------------------------------------------------

      function Get_Generic_Unit_Simple_Name (The_Generic_Unit : RawDict.Generic_Unit_Info_Ref) return LexTokenManager.Lex_String
      --# global Dict;
      is
         Name : LexTokenManager.Lex_String;
      begin
         case RawDict.Get_Generic_Unit_Kind (The_Generic_Unit => The_Generic_Unit) is
            when Generic_Of_Package =>
               Name := RawDict.Get_Package_Name
                 (The_Package => RawDict.Get_Generic_Unit_Owning_Package (The_Generic_Unit => The_Generic_Unit));
            when Generic_Of_Subprogram =>
               Name := RawDict.Get_Subprogram_Name
                 (The_Subprogram => RawDict.Get_Generic_Unit_Owning_Subprogram (The_Generic_Unit => The_Generic_Unit));
         end case;
         return Name;
      end Get_Generic_Unit_Simple_Name;

   begin -- GetSimpleName
      Item_Local := Item;
      -- A special symbol kind is used to store the value of variables on entry to a for
      -- loop.  We "dereference" these back to the original variable before attempting
      -- to retrieve the name
      if RawDict.GetSymbolDiscriminant (Item_Local) = LoopEntryVariableSymbol then
         Item_Local := RawDict.GetLoopEntryVariableOriginalVar (Item);
      end if;

      case RawDict.GetSymbolDiscriminant (Item_Local) is
         when Null_Symbol =>
            Name := LexTokenManager.Null_String;
         when Type_Symbol =>
            Name := RawDict.Get_Type_Name (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Item_Local));
         when Enumeration_Literal_Symbol =>
            Name := RawDict.Get_Enumeration_Literal_Name
              (The_Enumeration_Literal => RawDict.Get_Enumeration_Literal_Info_Ref (Item => Item_Local));
         when Record_Component_Symbol =>
            Name := RawDict.Get_Record_Component_Name
              (The_Record_Component => RawDict.Get_Record_Component_Info_Ref (Item => Item_Local));
         when Subcomponent_Symbol =>
            Name := Get_Subcomponent_Simple_Name (The_Subcomponent => RawDict.Get_Subcomponent_Info_Ref (Item => Item_Local));
         when Variable_Symbol =>
            Name := RawDict.Get_Variable_Name (The_Variable => RawDict.Get_Variable_Info_Ref (Item => Item_Local));
         when Quantified_Variable_Symbol =>
            Name := RawDict.Get_Quantified_Variable_Name
              (The_Quantified_Variable => RawDict.Get_Quantified_Variable_Info_Ref (Item => Item_Local));
         when Constant_Symbol =>
            Name := RawDict.Get_Constant_Name (The_Constant => RawDict.Get_Constant_Info_Ref (Item => Item_Local));
         when Package_Symbol =>
            Name := RawDict.Get_Package_Name (The_Package => RawDict.Get_Package_Info_Ref (Item => Item_Local));
         when Subprogram_Symbol =>
            Name := RawDict.Get_Subprogram_Name (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Item_Local));
         when ImplicitProofFunctionSymbol =>
            Name := RawDict.Get_Subprogram_Name (The_Subprogram => RawDict.GetImplicitProofFunctionAdaFunction (Item_Local));
         when Implicit_Return_Variable_Symbol =>
            Name := RawDict.Get_Implicit_Return_Variable_Name
              (The_Implicit_Return_Variable => RawDict.Get_Implicit_Return_Variable_Info_Ref (Item => Item_Local));
         when Subprogram_Parameter_Symbol =>
            Name := RawDict.Get_Subprogram_Parameter_Name
              (The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => Item_Local));
         when Parameter_Constraint_Symbol =>
            -- Parameter constraint symbols don't have name of their own.  We return the name of the parameter
            -- with which they are associated.  Dictionary subunit GenerateSimpleName creates a synthetic name
            -- for teh constraint using the parameter name as a prefix
            Name := Get_Parameter_Constraint_Simple_Name
              (The_Parameter_Constraint => RawDict.Get_Parameter_Constraint_Info_Ref (Item => Item_Local));
         when LoopSymbol =>
            Name := RawDict.GetLoopName (Item_Local);
         when KnownDiscriminantSymbol =>
            Name := RawDict.GetDiscriminantName (Item_Local);
         when LoopParameterSymbol =>
            Name := RawDict.GetLoopParameterName (Item_Local);
         when Generic_Parameter_Symbol =>
            Name := Get_Generic_Parameter_Simple_Name
              (The_Generic_Parameter => RawDict.Get_Generic_Parameter_Info_Ref (Item => Item_Local));
            --1606 Item below added by JEB as part of generic package work
         when Constituent_Symbol =>
            Name := Get_Constituent_Simple_Name (The_Constituent => RawDict.Get_Constituent_Info_Ref (Item => Item_Local));
         when Generic_Unit_Symbol =>
            Name := Get_Generic_Unit_Simple_Name (The_Generic_Unit => RawDict.Get_Generic_Unit_Info_Ref (Item => Item_Local));
         when others => -- non-exec code
            Name := LexTokenManager.Null_String;
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.GetSimpleName");
      end case;
      return Name;
   end GetSimpleName;

   --------------------------------------------------------------------------------
   --  Get_Scalar_Attribute_Type
   --------------------------------------------------------------------------------

   function Get_Scalar_Attribute_Type (Name      : LexTokenManager.Lex_String;
                                       Type_Mark : RawDict.Type_Info_Ref) return RawDict.Type_Info_Ref
   --# global in Dict;
   --#        in LexTokenManager.State;
      is separate;

   --------------------------------------------------------------------------------

   function GetScalarAttributeType (Name     : LexTokenManager.Lex_String;
                                    TypeMark : Symbol) return Symbol is
   begin
      return RawDict.Get_Type_Symbol -- GAA External
        (Get_Scalar_Attribute_Type (Name      => Name,
                                    Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark))); -- GAA External
   end GetScalarAttributeType;

   --------------------------------------------------------------------------------
   --  Get_Scalar_Attribute_Value
   --------------------------------------------------------------------------------

   function Get_Scalar_Attribute_Value
     (Base      : Boolean;
      Name      : LexTokenManager.Lex_String;
      Type_Mark : RawDict.Type_Info_Ref)
     return      LexTokenManager.Lex_String
   --# global in Dict;
   --#        in LexTokenManager.State;
      is separate;

   --------------------------------------------------------------------------------

   function GetScalarAttributeValue
     (Base     : Boolean;
      Name     : LexTokenManager.Lex_String;
      TypeMark : Symbol)
     return     LexTokenManager.Lex_String
   is
   begin
      return Get_Scalar_Attribute_Value (Base      => Base,
                                         Name      => Name,
                                         Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)); -- GAA External
   end GetScalarAttributeValue;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function ContainsFloat (TypeMark : Symbol) return Boolean is
   begin
      return RawDict.Get_Type_Contains_Float (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)); -- GAA External
   end ContainsFloat;

   --------------------------------------------------------------------------------
   --  Type_Is_Limited
   --------------------------------------------------------------------------------

   function Type_Is_Limited (Type_Mark : RawDict.Type_Info_Ref;
                             Scope     : Scopes) return Boolean
   --# global in Dict;
   is
      Result : Boolean;
   begin
      case RawDict.Get_Type_Limited (Type_Mark => Type_Mark) is
         when Never =>
            Result := False;
         when Sometimes =>
            Result := not IsLocal (Scope, Get_Type_Scope (Type_Mark => Type_Mark));
         when Always =>
            Result := True;
      end case;
      return Result;
   end Type_Is_Limited;

   --------------------------------------------------------------------------------

   function TypeIsLimited (TypeMark : Symbol;
                           Scope    : Scopes) return Boolean is
   begin
      return Type_Is_Limited (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark), -- GAA External
                              Scope     => Scope);
   end TypeIsLimited;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   procedure Add_Integer_Type_Mark
     (Type_Mark : in RawDict.Type_Info_Ref;
      Static    : in Boolean;
      Lower     : in LexTokenManager.Lex_String;
      Upper     : in LexTokenManager.Lex_String)
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Lower,
   --#                   Static,
   --#                   Type_Mark,
   --#                   Upper;
   is
   begin
      RawDict.Set_Type_Discriminant (Type_Mark    => Type_Mark,
                                     Discriminant => Integer_Type_Item);
      RawDict.Set_Type_Static (Type_Mark => Type_Mark,
                               Static    => Static);
      RawDict.Set_Type_Lower (Type_Mark => Type_Mark,
                              Lower     => Lower);
      RawDict.Set_Type_Upper (Type_Mark => Type_Mark,
                              Upper     => Upper);
   end Add_Integer_Type_Mark;

   --------------------------------------------------------------------------------

   procedure Add_Modular_Type_Mark
     (Type_Mark : in RawDict.Type_Info_Ref;
      Static    : in Boolean;
      Lower     : in LexTokenManager.Lex_String;
      Upper     : in LexTokenManager.Lex_String;
      Modulus   : in LexTokenManager.Lex_String)
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Lower,
   --#                   Modulus,
   --#                   Static,
   --#                   Type_Mark,
   --#                   Upper;
   is
   begin
      RawDict.Set_Type_Discriminant (Type_Mark    => Type_Mark,
                                     Discriminant => Modular_Type_Item);
      RawDict.Set_Type_Static (Type_Mark => Type_Mark,
                               Static    => Static);
      RawDict.Set_Type_Lower (Type_Mark => Type_Mark,
                              Lower     => Lower);
      RawDict.Set_Type_Upper (Type_Mark => Type_Mark,
                              Upper     => Upper);
      RawDict.Set_Type_Modulus (Type_Mark => Type_Mark,
                                Modulus   => Modulus);
   end Add_Modular_Type_Mark;

   --------------------------------------------------------------------------------

   procedure Add_Floating_Point_Type_Mark (Type_Mark : in RawDict.Type_Info_Ref;
                                           Static    : in Boolean)
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Static,
   --#                   Type_Mark;
   is
   begin
      RawDict.Set_Type_Discriminant (Type_Mark    => Type_Mark,
                                     Discriminant => Floating_Point_Type_Item);
      RawDict.Set_Type_Static (Type_Mark => Type_Mark,
                               Static    => Static);
      RawDict.Set_Type_Contains_Float (Type_Mark      => Type_Mark,
                                       Contains_Float => True);
   end Add_Floating_Point_Type_Mark;

   --------------------------------------------------------------------------------

   procedure Add_Fixed_Point_Type_Mark (Type_Mark : in RawDict.Type_Info_Ref;
                                        Static    : in Boolean)
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Static,
   --#                   Type_Mark;
   is
   begin
      RawDict.Set_Type_Discriminant (Type_Mark    => Type_Mark,
                                     Discriminant => Fixed_Point_Type_Item);
      RawDict.Set_Type_Static (Type_Mark => Type_Mark,
                               Static    => Static);
   end Add_Fixed_Point_Type_Mark;

   --------------------------------------------------------------------------------

   procedure Add_Array_Type_Mark
     (Type_Mark      : in RawDict.Type_Info_Ref;
      Component_Type : in RawDict.Type_Info_Ref;
      Static         : in Boolean)
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Component_Type,
   --#                   Static,
   --#                   Type_Mark;
   is
      Is_Private, Is_Limited : TriState;
   begin
      case RawDict.Get_Type_Private (Type_Mark => Component_Type) is
         when Never =>
            Is_Private := Never;
         when Sometimes =>
            if IsLocal (Get_Type_Scope (Type_Mark => Type_Mark), Get_Type_Scope (Type_Mark => Component_Type)) then
               Is_Private := Sometimes;
            else
               Is_Private := Always;
            end if;
         when Always =>
            Is_Private := Always;
      end case;

      case RawDict.Get_Type_Limited (Type_Mark => Component_Type) is
         when Never =>
            Is_Limited := Never;
         when Sometimes =>
            if IsLocal (Get_Type_Scope (Type_Mark => Type_Mark), Get_Type_Scope (Type_Mark => Component_Type)) then
               Is_Limited := Sometimes;
            else
               Is_Limited := Always;
            end if;
         when Always =>
            Is_Limited := Always;
      end case;

      RawDict.Set_Type_Discriminant (Type_Mark    => Type_Mark,
                                     Discriminant => Array_Type_Item);
      RawDict.Set_Type_Static (Type_Mark => Type_Mark,
                               Static    => Static);
      RawDict.Set_Type_Private (Type_Mark  => Type_Mark,
                                Is_Private => Is_Private);
      RawDict.Set_Type_Limited (Type_Mark  => Type_Mark,
                                Is_Limited => Is_Limited);
      RawDict.Set_Type_Equality_Defined
        (Type_Mark        => Type_Mark,
         Equality_Defined => RawDict.Get_Type_Equality_Defined (Type_Mark => Component_Type));
      RawDict.Set_Type_Contains_Float
        (Type_Mark      => Type_Mark,
         Contains_Float => RawDict.Get_Type_Contains_Float (Type_Mark => Component_Type));
      RawDict.Set_Type_Array_Component (Type_Mark      => Type_Mark,
                                        Component_Type => Component_Type);
   end Add_Array_Type_Mark;

   --------------------------------------------------------------------------------

   procedure Add_Enumeration_Type_Mark (Type_Mark : in RawDict.Type_Info_Ref;
                                        Static    : in Boolean)
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Static,
   --#                   Type_Mark;
   is
   begin
      RawDict.Set_Type_Discriminant (Type_Mark    => Type_Mark,
                                     Discriminant => Enumeration_Type_Item);
      RawDict.Set_Type_Static (Type_Mark => Type_Mark,
                               Static    => Static);
   end Add_Enumeration_Type_Mark;

   --------------------------------------------------------------------------------
   --  Add_Enumeration_Literal
   --------------------------------------------------------------------------------

   procedure Add_Enumeration_Literal
     (Name                    : in     LexTokenManager.Lex_String;
      Comp_Unit               : in     ContextManager.UnitDescriptors;
      Declaration             : in     Location;
      Position                : in     LexTokenManager.Lex_String;
      Type_Mark               : in     RawDict.Type_Info_Ref;
      The_Enumeration_Literal :    out RawDict.Enumeration_Literal_Info_Ref)
   --# global in out Dict;
   --# derives Dict,
   --#         The_Enumeration_Literal from Comp_Unit,
   --#                                      Declaration,
   --#                                      Dict,
   --#                                      Name,
   --#                                      Position,
   --#                                      Type_Mark;
   --# post The_Enumeration_Literal /= RawDict.Null_Enumeration_Literal_Info_Ref;
   is
      Current : RawDict.Enumeration_Literal_Info_Ref;
   begin
      RawDict.Create_Enumeration_Literal
        (Name                    => Name,
         Position                => Position,
         Enumeration_Type        => Type_Mark,
         Comp_Unit               => Comp_Unit,
         Loc                     => Declaration.Start_Position,
         The_Enumeration_Literal => The_Enumeration_Literal);

      Current := RawDict.Get_Type_Last_Enumeration_Literal (Type_Mark => Type_Mark);
      if Current = RawDict.Null_Enumeration_Literal_Info_Ref then
         RawDict.Set_Type_First_Enumeration_Literal (Type_Mark           => Type_Mark,
                                                     Enumeration_Literal => The_Enumeration_Literal);
         RawDict.Set_Type_Lower (Type_Mark => Type_Mark,
                                 Lower     => Position);
      else
         RawDict.Set_Next_Enumeration_Literal (The_Enumeration_Literal => Current,
                                               Next                    => The_Enumeration_Literal);
      end if;
      RawDict.Set_Type_Last_Enumeration_Literal (Type_Mark           => Type_Mark,
                                                 Enumeration_Literal => The_Enumeration_Literal);
      RawDict.Set_Type_Upper (Type_Mark => Type_Mark,
                              Upper     => Position);
   end Add_Enumeration_Literal;

   --------------------------------------------------------------------------------

   procedure AddEnumerationLiteral
     (Name                  : in     LexTokenManager.Lex_String;
      Comp_Unit             : in     ContextManager.UnitDescriptors;
      Declaration           : in     Location;
      Position              : in     LexTokenManager.Lex_String;
      The_Type              : in     Symbol;
      TheEnumerationLiteral :    out Symbol) is
      The_Enumeration_Literal : RawDict.Enumeration_Literal_Info_Ref;
   begin
      Add_Enumeration_Literal
        (Name                    => Name,
         Comp_Unit               => Comp_Unit,
         Declaration             => Declaration,
         Position                => Position,
         Type_Mark               => RawDict.Get_Type_Info_Ref (The_Type), -- GAA External
         The_Enumeration_Literal => The_Enumeration_Literal);
      TheEnumerationLiteral := RawDict.Get_Enumeration_Literal_Symbol (The_Enumeration_Literal); -- GAA External
   end AddEnumerationLiteral;

   --------------------------------------------------------------------------------
   --  Get_Predefined_Positive_Subtype
   --------------------------------------------------------------------------------

   function Get_Predefined_Positive_Subtype return  RawDict.Type_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Types.The_Predefined_Types (Predefined_Positive_Subtype);
   end Get_Predefined_Positive_Subtype;

   --------------------------------------------------------------------------------

   function GetPredefinedPositiveSubtype return Symbol is
   begin
      return RawDict.Get_Type_Symbol (Get_Predefined_Positive_Subtype); -- GAA External
   end GetPredefinedPositiveSubtype;

   --------------------------------------------------------------------------------
   --  Get_Predefined_Package_ASCII
   --------------------------------------------------------------------------------

   function Get_Predefined_Package_ASCII return RawDict.Package_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Packages.Package_ASCII;
   end Get_Predefined_Package_ASCII;

   --------------------------------------------------------------------------------

   function GetPredefinedPackageASCII return Symbol is
   begin
      return RawDict.Get_Package_Symbol (Get_Predefined_Package_ASCII); -- GAA External
   end GetPredefinedPackageASCII;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function TypeIsRecord (TypeMark : Symbol) return Boolean is
   begin
      return RawDict.Get_Type_Discriminant (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)) = Record_Type_Item; -- GAA External
   end TypeIsRecord;

   --------------------------------------------------------------------------------
   --  Get_Array_Component
   --------------------------------------------------------------------------------

   function Get_Array_Component (Type_Mark : RawDict.Type_Info_Ref) return RawDict.Type_Info_Ref
   --# global in Dict;
   is
      Result : RawDict.Type_Info_Ref;
   begin
      if Type_Mark = Get_Unknown_Type_Mark then
         Result := Type_Mark;
      else
         Result := RawDict.Get_Type_Array_Component (Type_Mark => Get_Root_Type (Type_Mark => Type_Mark));
      end if;
      return Result;
   end Get_Array_Component;

   --------------------------------------------------------------------------------

   function GetArrayComponent (TypeMark : Symbol) return Symbol is
   begin
      return RawDict.Get_Type_Symbol (Get_Array_Component (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark))); -- GAA External
   end GetArrayComponent;

   --------------------------------------------------------------------------------
   --  Get_Number_Of_Dimensions
   --------------------------------------------------------------------------------

   function Get_Number_Of_Dimensions (Type_Mark : RawDict.Type_Info_Ref) return Positive
   --# global in Dict;
   is
      Array_Index : Iterator;
      Count       : Positive := Positive'First;
   begin
      -- If Type_Mark denotes a subtype, then find its root type
      Array_Index := First_Array_Index (Type_Mark => Get_Root_Type (Type_Mark => Type_Mark));
      loop
         Array_Index := NextSymbol (Array_Index);
         exit when IsNullIterator (Array_Index) or else Count = Positive'Last;
         Count := Count + 1;
      end loop;
      return Count;
   end Get_Number_Of_Dimensions;

   --------------------------------------------------------------------------------

   function GetNumberOfDimensions (TypeMark : Symbol) return Positive is
   begin
      return Get_Number_Of_Dimensions (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)); -- GAA External
   end GetNumberOfDimensions;

   --------------------------------------------------------------------------------
   --  Type_Is_Boolean
   --------------------------------------------------------------------------------

   function Type_Is_Boolean (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return Get_Root_Type (Type_Mark => Type_Mark) = Get_Predefined_Boolean_Type;
   end Type_Is_Boolean;

   --------------------------------------------------------------------------------

   function TypeIsBoolean (TypeMark : Symbol) return Boolean is
   begin
      return Type_Is_Boolean (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)); -- GAA External
   end TypeIsBoolean;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function Type_Is_Boolean_Array (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return Type_Is_Array (Type_Mark => Type_Mark)
        and then Type_Is_Boolean (Type_Mark => Get_Array_Component (Type_Mark => Type_Mark))
        and then Get_Number_Of_Dimensions (Type_Mark => Type_Mark) = 1;
   end Type_Is_Boolean_Array;

   --------------------------------------------------------------------------------

   procedure Write_Space (File : in SPARK_IO.File_Type)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File;
   is
   begin
      SPARK_IO.Put_Char (File, ' ');
   end Write_Space;

   --------------------------------------------------------------------------------

   procedure Write_String (File : in SPARK_IO.File_Type;
                           Str  : in String)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                Str;
   is
   begin
      SPARK_IO.Put_String (File, Str, 0);
   end Write_String;

   --------------------------------------------------------------------------------

   procedure Write_Line (File : in SPARK_IO.File_Type;
                         Str  : in String)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                Str;
   is
   begin
      SPARK_IO.Put_Line (File, Str, 0);
   end Write_Line;

   --------------------------------------------------------------------------------

   function LoopHasName (TheLoop : Symbol) return Boolean is
   begin
      return LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => GetSimpleName (TheLoop),
         Lex_Str2 => LexTokenManager.Null_String) /=
        LexTokenManager.Str_Eq;
   end LoopHasName;

   --------------------------------------------------------------------------------

   function First_Loop (CompilationUnit : Symbol) return Iterator
   --# global in Dict;
   is
      The_Generic_Unit : RawDict.Generic_Unit_Info_Ref;
      TheLoop          : Symbol;
      Loops            : Iterator;
   begin
      case RawDict.GetSymbolDiscriminant (CompilationUnit) is
         when Subprogram_Symbol =>
            TheLoop :=
              RawDict.Get_Subprogram_First_Loop (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => CompilationUnit));
         when Type_Symbol =>
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => CompilationUnit)),
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.First_Loop");
            TheLoop := RawDict.Get_Task_Type_First_Loop (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => CompilationUnit));
         when Package_Symbol =>
            TheLoop := RawDict.Get_Package_First_Loop (The_Package => RawDict.Get_Package_Info_Ref (Item => CompilationUnit));
         when Generic_Unit_Symbol =>
            The_Generic_Unit := RawDict.Get_Generic_Unit_Info_Ref (Item => CompilationUnit);
            case RawDict.Get_Generic_Unit_Kind (The_Generic_Unit => The_Generic_Unit) is
               when Generic_Of_Package =>
                  TheLoop := RawDict.Get_Package_First_Loop
                    (The_Package => RawDict.Get_Generic_Unit_Owning_Package (The_Generic_Unit => The_Generic_Unit));
               when Generic_Of_Subprogram =>
                  TheLoop := RawDict.Get_Subprogram_First_Loop
                    (The_Subprogram => RawDict.Get_Generic_Unit_Owning_Subprogram (The_Generic_Unit => The_Generic_Unit));
            end case;
         when others => -- non-exec code
            TheLoop := NullSymbol;
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.First_Loop");
      end case;

      if TheLoop = NullSymbol then
         Loops := NullIterator;
      else
         Loops := Iterator'(LoopIterator, IsAbstract, TheLoop, NullSymbol);
      end if;
      return Loops;
   end First_Loop;

   --------------------------------------------------------------------------------

   function FirstLoopOnEntryVar (TheLoop : Symbol) return Iterator is
      OnEntryVars : Iterator;
      TheVar      : Symbol;
   begin
      TheVar := RawDict.GetLoopOnEntryVars (TheLoop);
      if TheVar = NullSymbol then
         OnEntryVars := NullIterator;
      else
         OnEntryVars := Iterator'(LoopOnEntryVarIterator, IsAbstract, TheVar, NullSymbol);
      end if;
      return OnEntryVars;
   end FirstLoopOnEntryVar;

   --------------------------------------------------------------------------------

   function IsCompilationUnit (TheSymbol : Symbol) return Boolean is
   begin
      return Is_Subprogram (TheSymbol)
        or else RawDict.GetSymbolDiscriminant (TheSymbol) = Package_Symbol
        or else RawDict.GetSymbolDiscriminant (TheSymbol) = Generic_Unit_Symbol;
   end IsCompilationUnit;

   --------------------------------------------------------------------------------

   procedure AddUsesUncheckedConversion (TheUnit : in Symbol) is
   begin
      case RawDict.GetSymbolDiscriminant (TheUnit) is
         when Subprogram_Symbol =>
            -- covers procedures, functions and entries
            RawDict.Set_Subprogram_Uses_Unchecked_Conversion
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => TheUnit)); -- GAA External
         when Type_Symbol =>
            -- covers task bodies
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheUnit)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in AddUsesUncheckedConversion");
            RawDict.Set_Task_Type_Uses_Unchecked_Conversion
              (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => TheUnit)); -- GAA External
         when Package_Symbol =>
            -- a package could only use U_C in elabotration code where it would be illegal
            null;
         when others => -- non-exec code
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.AddUsesUncheckedConversion");
      end case;
   end AddUsesUncheckedConversion;

   --------------------------------------------------------------------------------

   procedure AddAssignsFromExternal (TheUnit : in Symbol) is
   begin
      case RawDict.GetSymbolDiscriminant (TheUnit) is
         when Subprogram_Symbol =>
            -- covers procedures, functions and entries
            RawDict.Set_Subprogram_Assigns_From_External
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => TheUnit)); -- GAA External
         when Type_Symbol =>
            -- covers task bodies
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheUnit)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in AddAssignsFromExternal");
            RawDict.Set_Task_Type_Assigns_From_External
              (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => TheUnit)); -- GAA External
         when Package_Symbol =>
            -- a package could only read external variable in elabotration code
            -- where it would be illegal
            null;
         when others => -- non-exec code
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.AddAssignsFromExternal");
      end case;
   end AddAssignsFromExternal;

   --------------------------------------------------------------------------------

   function LastMostEnclosingLoop (CompilationUnit : Symbol) return Symbol is
      Sym       : Symbol;
      TheResult : Symbol;
   begin
      case RawDict.GetSymbolDiscriminant (CompilationUnit) is
         when Subprogram_Symbol =>
            Sym := RawDict.Get_Subprogram_Last_Loop
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => CompilationUnit)); -- GAA External
         when Type_Symbol =>
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => CompilationUnit)),
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in LastMostEnclosingLoop");
            Sym := RawDict.Get_Task_Type_Last_Loop
              (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => CompilationUnit)); -- GAA External
         when Package_Symbol =>
            Sym := RawDict.Get_Package_Last_Loop (RawDict.Get_Package_Info_Ref (Item => CompilationUnit)); -- GAA External
         when others => -- non-exec code
            Sym := NullSymbol;
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.LastMostEnclosingLoop");
      end case;
      TheResult := Sym;
      while not IsCompilationUnit (Sym) loop
         Sym := GetRegion (GetScope (Sym));
         if RawDict.GetSymbolDiscriminant (Sym) = LoopSymbol then
            TheResult := Sym;
         end if;
      end loop;
      return TheResult;
   end LastMostEnclosingLoop;

   --------------------------------------------------------------------------------

   function GetEnclosingCompilationUnit (Scope : Scopes) return Symbol is
      Current : Scopes;
      Region  : Symbol;
   begin
      Current := Scope;
      loop
         Region := GetRegion (Current);
         exit when IsCompilationUnit (Region)
           or else (RawDict.GetSymbolDiscriminant (Region) = Type_Symbol
                      -- to deal with task bodies
                      and then Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Region)));
         Current := GetEnclosingScope (Current);
      end loop;
      return Region;
   end GetEnclosingCompilationUnit;

   --------------------------------------------------------------------------------

   function GetLoopNumber (TheLoop : Symbol) return Positive is
      Loops  : Iterator;
      Number : Positive := Positive'First;
   begin
      Loops :=
        First_Loop
        (CompilationUnit => GetEnclosingCompilationUnit (Set_Visibility (The_Visibility => Local,
                                                                         The_Unit       => TheLoop)));
      while CurrentSymbol (Loops) /= TheLoop and then Number < Positive'Last loop
         Loops  := NextSymbol (Loops);
         Number := Number + 1;
      end loop;
      return Number;
   end GetLoopNumber;

   --------------------------------------------------------------------------------

   function RecordComponentIsInherited (TheComponent : Symbol) return Boolean is
   begin
      return RawDict.Get_Record_Component_Inherited_Field
        (The_Record_Component => RawDict.Get_Record_Component_Info_Ref (TheComponent)); -- GAA External
   end RecordComponentIsInherited;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Of_Protected_Implicit_In_Stream
     (The_Implicit_In_Stream : RawDict.Implicit_In_Stream_Info_Ref)
     return                   RawDict.Variable_Info_Ref
   --# global in Dict;
   is
   begin
      return RawDict.Get_Own_Variable_Variable
        (The_Own_Variable => RawDict.Get_Implicit_In_Stream_Own_Variable (The_Implicit_In_Stream => The_Implicit_In_Stream));
   end Get_Own_Variable_Of_Protected_Implicit_In_Stream;

   --------------------------------------------------------------------------------

   function GetSubprogramParameterConstraintDimension (TheConstraint : Symbol) return Positive is
   begin
      return RawDict.Get_Parameter_Constraint_Dimension
        (The_Parameter_Constraint => RawDict.Get_Parameter_Constraint_Info_Ref (TheConstraint)); -- GAA External
   end GetSubprogramParameterConstraintDimension;

   --------------------------------------------------------------------------------
   --  Generate_Simple_Name
   --------------------------------------------------------------------------------

   function Fetch_Simple_Name (Type_Mark : RawDict.Type_Info_Ref) return E_Strings.T
   --# global in Dict;
   --#        in LexTokenManager.State;
   is
      Type_Mark_Local : RawDict.Type_Info_Ref;
   begin
      Type_Mark_Local := Type_Mark;
      if Is_Type (Type_Mark => Type_Mark_Local)
        and then RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark_Local) = Access_Type_Item then
         Type_Mark_Local := RawDict.Get_Type_Accesses (Type_Mark => Type_Mark_Local);
      end if;
      return LexTokenManager.Lex_String_To_String (Lex_Str => RawDict.Get_Type_Name (Type_Mark => Type_Mark_Local));
   end Fetch_Simple_Name;

   --------------------------------------------------------------------------------

   function GenerateSimpleName (Item      : Symbol;
                                Separator : String) return E_Strings.T is separate;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function GetAnyPrefixNeeded
     (Sym       : Symbol;
      Scope     : Scopes;
      Separator : String)
     return      E_Strings.T
      is separate;

   --------------------------------------------------------------------------------
   --  Is_Valid_Generic_Type_Association
   --------------------------------------------------------------------------------

   function IsValidGenericTypeAssociation (Formal, Actual : Symbol;
                                           Scope          : Scopes) return Boolean is

      function Is_Valid_Generic_Type_Association (Formal_Type, Actual_Type : RawDict.Type_Info_Ref;
                                                  Scope                    : Scopes) return Boolean
      --# global in Dict;
      is
         type Array_Actual_Type is array (Type_Discriminant) of Boolean;
         type Array_Formal_Type is array (Generic_Type_Discriminant) of Array_Actual_Type;
         pragma Style_Checks (Off);
         The_Table : constant Array_Formal_Type := Array_Formal_Type'
           -- Actual type    Unknown, Enumeration, Integer, Modular, Float, Fixed, Array, Record, Abstract, Protected,  Task, Access, Generic    -- Generic type
           (Array_Actual_Type'(False,       False,   False,   False, False, False, False,  False,    False,     False, False,  False,   False),  -- Invalid_Generic_Type
            Array_Actual_Type'( True,        True,    True,    True,  True,  True,  True,   True,    False,      True,  True,   True,   False),  -- Generic_Private_Type
            Array_Actual_Type'(False,        True,    True,    True, False, False, False,  False,    False,     False, False,  False,   False),  -- Generic_Discrete_Type
            Array_Actual_Type'(False,       False,    True,   False, False, False, False,  False,    False,     False, False,  False,   False),  -- Generic_Integer_Type
            Array_Actual_Type'(False,       False,   False,    True, False, False, False,  False,    False,     False, False,  False,   False),  -- Generic_Modular_Type
            Array_Actual_Type'(False,       False,   False,   False,  True, False, False,  False,    False,     False, False,  False,   False),  -- Generic_Floating_Point_Type
            Array_Actual_Type'(False,       False,   False,   False, False,  True, False,  False,    False,     False, False,  False,   False),  -- Generic_Fixed_Point_Type
            Array_Actual_Type'(False,       False,   False,   False, False, False, False,  False,    False,     False, False,  False,   False)); -- Generic_Array_Type
         pragma Style_Checks (On);
      begin
         return RawDict.Get_Type_Discriminant (Type_Mark => Formal_Type) = Generic_Type_Item
           and then ((Type_Is_Limited (Type_Mark => Actual_Type,
                                       Scope     => Scope)
                        and then RawDict.Get_Type_Limited_Private (Type_Mark => Formal_Type))
                     or else (not Type_Is_Limited (Type_Mark => Actual_Type,
                                                   Scope     => Scope)
                                and then not (Type_Is_Boolean (Type_Mark => Actual_Type)
                                                and then RawDict.Get_Type_Kind_Of_Generic (Type_Mark => Formal_Type) = Generic_Discrete_Type)
                                and then not Is_Unconstrained_Array_Type (Type_Mark => Actual_Type)
                                and then The_Table (RawDict.Get_Type_Kind_Of_Generic (Type_Mark => Formal_Type))
                                (RawDict.Get_Type_Discriminant (Type_Mark => Actual_Type))));
      end Is_Valid_Generic_Type_Association;

   begin -- IsValidGenericTypeAssociation
      return Is_Valid_Generic_Type_Association (Formal_Type => RawDict.Get_Type_Info_Ref (Formal), -- GAA External
                                                Actual_Type => RawDict.Get_Type_Info_Ref (Actual), -- GAA External
                                                Scope       => Scope);
   end IsValidGenericTypeAssociation;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function Get_Symbol_Compilation_Unit (Item : Symbol) return ContextManager.UnitDescriptors is
   begin
      return RawDict.Get_Symbol_Compilation_Unit (Item);
   end Get_Symbol_Compilation_Unit;

   --------------------------------------------------------------------------------

   function Get_Symbol_Location (Item : Symbol) return LexTokenManager.Token_Position is
   begin
      return RawDict.Get_Symbol_Location (Item);
   end Get_Symbol_Location;

   --------------------------------------------------------------------------------

   procedure Get_SLI_Type (Item   : in     Symbol;
                           Result :    out SLI_Type) is
      Type_Mark : RawDict.Type_Info_Ref;

      procedure Get_Type_Discriminant (Type_Mark : in     RawDict.Type_Info_Ref;
                                       Is_A_Type : in     Boolean;
                                       Result    :    out SLI_Type)
      --# global in     CommandLineData.Content;
      --#        in     Dict;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives Result            from Dict,
      --#                                Is_A_Type,
      --#                                Type_Mark &
      --#         SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                Dict,
      --#                                Is_A_Type,
      --#                                LexTokenManager.State,
      --#                                Type_Mark;
      is
      begin
         if CommandLineData.Content.Debug.SLI then
            --  Debug
            SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                 Item => " Type symbol = ",
                                 Stop => 0);
            E_Strings.Put_String (File  => SPARK_IO.Standard_Output,
                                  E_Str => Fetch_Simple_Name (Type_Mark => Type_Mark));
            SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                 Item => " : ",
                                 Stop => 0);
         end if;
         case RawDict.Get_Type_Discriminant (Type_Mark => Type_Mark) is
            when Unknown_Type_Item =>
               if CommandLineData.Content.Debug.SLI then
                  --  Debug
                  SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                       Item => "Unknown_Type",
                                       Stop => 0);
               end if;
               Result := SLI_Unknown_Type;
            when Enumeration_Type_Item =>
               if Is_A_Type then
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Enumeration_Type",
                                          Stop => 0);
                  end if;
                  Result := SLI_Enumeration_Type;
               else
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Enumeration_Object",
                                          Stop => 0);
                  end if;
                  Result := SLI_Enumeration_Object;
               end if;
            when Integer_Type_Item =>
               if Is_A_Type then
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Signed_Integer_Type",
                                          Stop => 0);
                  end if;
                  Result := SLI_Signed_Integer_Type;
               else
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Signed_Integer_Object",
                                          Stop => 0);
                  end if;
                  Result := SLI_Signed_Integer_Object;
               end if;
            when Modular_Type_Item =>
               if Is_A_Type then
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Modular_Integer_Type",
                                          Stop => 0);
                  end if;
                  Result := SLI_Modular_Integer_Type;
               else
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Modular_Integer_Object",
                                          Stop => 0);
                  end if;
                  Result := SLI_Modular_Integer_Object;
               end if;
            when Floating_Point_Type_Item =>
               if Is_A_Type then
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Floating_Point_Type",
                                          Stop => 0);
                  end if;
                  Result := SLI_Floating_Point_Type;
               else
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Floating_Point_Object",
                                          Stop => 0);
                  end if;
                  Result := SLI_Floating_Point_Object;
               end if;
            when Fixed_Point_Type_Item =>
               if Is_A_Type then
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Fixed_Point_Type",
                                          Stop => 0);
                  end if;
                  Result := SLI_Fixed_Point_Type;
               else
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Fixed_Point_Object",
                                          Stop => 0);
                  end if;
                  Result := SLI_Fixed_Point_Object;
               end if;
            when Array_Type_Item =>
               if Is_A_Type then
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Array_Type",
                                          Stop => 0);
                  end if;
                  Result := SLI_Array_Type;
               else
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Array_Object",
                                          Stop => 0);
                  end if;
                  Result := SLI_Array_Object;
               end if;
            when Record_Type_Item =>
               if Is_A_Type then
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Record_Type",
                                          Stop => 0);
                  end if;
                  Result := SLI_Record_Type;
               else
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Record_Object",
                                          Stop => 0);
                  end if;
                  Result := SLI_Record_Object;
               end if;
            when Abstract_Proof_Type_Item =>
               if CommandLineData.Content.Debug.SLI then
                  --  Debug
                  SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                       Item => "Abstract_Type",
                                       Stop => 0);
               end if;
               Result := SLI_Abstract_Type;
            when Protected_Type_Item =>
               if Is_A_Type then
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Protected_Type",
                                          Stop => 0);
                  end if;
                  Result := SLI_Protected_Type;
               else
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Protected_Object",
                                          Stop => 0);
                  end if;
                  Result := SLI_Protected_Object;
               end if;
            when Task_Type_Item =>
               if Is_A_Type then
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Task_Type",
                                          Stop => 0);
                  end if;
                  Result := SLI_Task_Type;
               else
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Task_Object",
                                          Stop => 0);
                  end if;
                  Result := SLI_Task_Object;
               end if;
            when Generic_Type_Item =>
               if CommandLineData.Content.Debug.SLI then
                  --  Debug
                  SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                       Item => "Generic_Formal_Parameter",
                                       Stop => 0);
               end if;
               Result := SLI_Generic_Formal_Parameter;
            when others =>
               if CommandLineData.Content.Debug.SLI then
                  --  Debug
                  SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                       Item => "Unknown_Type",
                                       Stop => 0);
               end if;
               Result := SLI_Unknown_Type;
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Other_Internal_Error,
                  Msg     => "DICTIONARY.GET_TYPE_DISCRIMINANT : PROGRAM ERRROR");
         end case;
      end Get_Type_Discriminant;

   begin -- Get_SLI_Type
      if CommandLineData.Content.Debug.SLI then
         --  Debug
         SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                              Item => " [DICTIONARY.GET_SLI_TYPE : Symbol = ",
                              Stop => 0);
         E_Strings.Put_String (File  => SPARK_IO.Standard_Output,
                               E_Str => GenerateSimpleName (Item      => Item,
                                                            Separator => "."));
      end if;
      case RawDict.GetSymbolDiscriminant (Item => Item) is
         when Type_Symbol =>
            Get_Type_Discriminant (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Item), -- GAA External
                                   Is_A_Type => True,
                                   Result    => Result);
         when Enumeration_Literal_Symbol =>
            if CommandLineData.Content.Debug.SLI then
               --  Debug
               SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                    Item => " : Enumeration_Literal",
                                    Stop => 0);
            end if;
            Result := SLI_Enumeration_Literal;
         when Record_Component_Symbol =>
            Get_Type_Discriminant
              (Type_Mark => RawDict.Get_Record_Component_Type
                 (The_Record_Component => RawDict.Get_Record_Component_Info_Ref (Item => Item)), -- GAA External
               Is_A_Type => False,
               Result    => Result);
         when Subcomponent_Symbol =>
            Get_Type_Discriminant
              (Type_Mark => RawDict.Get_Record_Component_Type
                 (The_Record_Component => RawDict.Get_Subcomponent_Record_Component
                    (The_Subcomponent => RawDict.Get_Subcomponent_Info_Ref (Item => Item))), -- GAA External
               Is_A_Type => False,
               Result    => Result);
         when Variable_Symbol =>
            Get_Type_Discriminant
              (Type_Mark => RawDict.Get_Variable_Type
                 (The_Variable => RawDict.Get_Variable_Info_Ref (Item => Item)), -- GAA External
               Is_A_Type => False,
               Result    => Result);
         when Quantified_Variable_Symbol =>
            Get_Type_Discriminant
              (Type_Mark => RawDict.Get_Quantified_Variable_Type
                 (The_Quantified_Variable => RawDict.Get_Quantified_Variable_Info_Ref (Item => Item)), -- GAA External
               Is_A_Type => False,
               Result    => Result);
         when Implicit_Return_Variable_Symbol =>
            Get_Type_Discriminant
              (Type_Mark => RawDict.Get_Subprogram_Return_Type
                 (The_Subprogram => RawDict.Get_Implicit_Return_Variable_Function
                    (The_Implicit_Return_Variable => RawDict.Get_Implicit_Return_Variable_Info_Ref (Item => Item))), -- GAA External
               Is_A_Type => False,
               Result    => Result);
         when Own_Variable_Symbol =>
            if CommandLineData.Content.Debug.SLI then
               --  Debug
               SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                    Item => " : Record_Object",
                                    Stop => 0);
            end if;
            Result := SLI_Record_Object;
         when OwnTaskSymbol =>
            if CommandLineData.Content.Debug.SLI then
               --  Debug
               SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                    Item => " : Task_Object",
                                    Stop => 0);
            end if;
            Result := SLI_Task_Object;
         when Constant_Symbol =>
            Type_Mark := RawDict.Get_Constant_Type (The_Constant => RawDict.Get_Constant_Info_Ref (Item => Item)); -- GAA External
            if Type_Mark = Get_Universal_Integer_Type
              or else Type_Mark = Get_Universal_Real_Type then
               if CommandLineData.Content.Debug.SLI then
                  --  Debug
                  SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                       Item => " : Named_Number",
                                       Stop => 0);
               end if;
               Result := SLI_Named_Number;
            elsif Type_Mark = Get_Unknown_Type_Mark then
               if CommandLineData.Content.Debug.SLI then
                  --  Debug
                  SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                       Item => " : Unknown_Type",
                                       Stop => 0);
               end if;
               Result := SLI_Unknown_Type;
            else
               Get_Type_Discriminant (Type_Mark => Type_Mark,
                                      Is_A_Type => False,
                                      Result    => Result);
            end if;
         when Package_Symbol =>
            if CommandLineData.Content.Debug.SLI then
               --  Debug
               SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                    Item => " : Package_Type",
                                    Stop => 0);
            end if;
            Result := SLI_Package_Type;
         when Subprogram_Symbol =>
            if RawDict.Get_Subprogram_Return_Type
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Item)) = -- GAA External
              RawDict.Null_Type_Info_Ref then
               if RawDict.Get_Subprogram_Generic_Unit
                 (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Item)) /= -- GAA External
                 RawDict.Null_Generic_Unit_Info_Ref then
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => " : Generic_Procedure_Type",
                                          Stop => 0);
                  end if;
                  Result := SLI_Generic_Procedure_Type;
               else
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => " : Procedure_Type",
                                          Stop => 0);
                  end if;
                  Result := SLI_Procedure_Type;
               end if;
            else
               if RawDict.Get_Subprogram_Generic_Unit
                 (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Item)) /= -- GAA External
                 RawDict.Null_Generic_Unit_Info_Ref then
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => " : Generic_Function_Op",
                                          Stop => 0);
                  end if;
                  Result := SLI_Generic_Function_Op;
               else
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => " : Function_Op",
                                          Stop => 0);
                  end if;
                  Result := SLI_Function_Op;
               end if;
            end if;
         when ImplicitProofFunctionSymbol =>
            if CommandLineData.Content.Debug.SLI then
               --  Debug
               SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                    Item => " : Function_Op",
                                    Stop => 0);
            end if;
            Result := SLI_Function_Op;
         when Subprogram_Parameter_Symbol =>
            Get_Type_Discriminant
              (Type_Mark => RawDict.Get_Subprogram_Parameter_Type
                 (The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => Item)), -- GAA External
               Is_A_Type => False,
               Result    => Result);
         when Operator_Symbol =>
            if CommandLineData.Content.Debug.SLI then
               --  Debug
               SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                    Item => " : Function_Op",
                                    Stop => 0);
            end if;
            Result := SLI_Function_Op;
         when others =>
            if CommandLineData.Content.Debug.SLI then
               --  Debug
               SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                    Item => " : Unknown_Type",
                                    Stop => 0);
            end if;
            Result := SLI_Unknown_Type;
      end case;
      if CommandLineData.Content.Debug.SLI then
         SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                              Item => "] ",
                              Stop => 0);
      end if;
   end Get_SLI_Type;

   --------------------------------------------------------------------------------

   procedure Write_Simple_Name (File : in SPARK_IO.File_Type;
                                Item : in Symbol)
   --# global in     Dict;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                File,
   --#                                Item,
   --#                                LexTokenManager.State;
   is
   begin
      E_Strings.Put_String (File  => File,
                            E_Str => GenerateSimpleName (Item      => Item,
                                                         Separator => "."));
   end Write_Simple_Name;

   --------------------------------------------------------------------------------

   procedure Write_Name (File : in SPARK_IO.File_Type;
                         Item : in Symbol)
   --# global in     Dict;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                File,
   --#                                Item,
   --#                                LexTokenManager.State;
   is
      Local_Item : Symbol;

      procedure Write_Prefix (File : in SPARK_IO.File_Type;
                              Item : in Symbol)
      --# global in     Dict;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Dict,
      --#                                File,
      --#                                Item,
      --#                                LexTokenManager.State;
      is
         Prefix : E_Strings.T;
         Region : Symbol;
         Scope  : Scopes;

         function Append_Name (Prefix, Suffix : E_Strings.T) return E_Strings.T is
            Name : E_Strings.T;
         begin
            Name := Prefix;
            E_Strings.Append_String (E_Str => Name,
                                     Str   => ".");
            E_Strings.Append_Examiner_String (E_Str1 => Name,
                                              E_Str2 => Suffix);
            return Name;
         end Append_Name;

      begin -- Write_Prefix
         Prefix := E_Strings.Empty_String;
         if RawDict.GetSymbolDiscriminant (Item) /= Package_Symbol
           or else RawDict.Get_Package_Info_Ref (Item => Item) /= Get_Predefined_Package_Standard then -- don't try and get a prefix for this!
            Scope := GetScope (Item);
            loop
               Region := GetRegion (Scope);
               exit when RawDict.GetSymbolDiscriminant (Region) = Package_Symbol
                 and then RawDict.Get_Package_Info_Ref (Item => Region) = Get_Predefined_Package_Standard;
               Prefix := Append_Name (Prefix => GenerateSimpleName (Item      => Region,
                                                                    Separator => "."),
                                      Suffix => Prefix);
               Scope  := GetEnclosingScope (Scope);
            end loop;
         end if;
         E_Strings.Put_String (File  => File,
                               E_Str => Prefix);
      end Write_Prefix; -- Init of Prefix is partial but effective.

   begin -- Write_Name
      Local_Item := Item;

      if RawDict.GetSymbolDiscriminant (Local_Item) = Type_Symbol
        and then Is_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Local_Item))
        and then RawDict.Get_Type_Discriminant (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Local_Item)) = Access_Type_Item
      then
         SPARK_IO.Put_String (File, "access ", 0);
         Local_Item :=
           RawDict.Get_Type_Symbol (RawDict.Get_Type_Accesses (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Local_Item)));
      end if;

      if RawDict.GetSymbolDiscriminant (Item) /= Type_Symbol
        or else RawDict.Get_Type_Info_Ref (Item => Item) /= Get_Unknown_Type_Mark then
         Write_Prefix (File => File,
                       Item => Local_Item);
      end if;
      Write_Simple_Name (File => File,
                         Item => Local_Item);
   end Write_Name;

   --------------------------------------------------------------------------------

   procedure Write_Scope (File  : in SPARK_IO.File_Type;
                          Scope : in Scopes)
   --# global in     Dict;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                File,
   --#                                LexTokenManager.State,
   --#                                Scope;
   is
   begin
      if Get_Visibility (Scope => Scope) = Visible then
         Write_String (File, "specification");
      elsif Get_Visibility (Scope => Scope) = Privat then
         Write_String (File, "private part of");
      else
         Write_String (File, "body");
      end if;
      Write_String (File, " of ");
      Write_Name (File => File,
                  Item => GetRegion (Scope));
   end Write_Scope;

   --------------------------------------------------------------------------------

   procedure Write_Integer (File : in SPARK_IO.File_Type;
                            Int  : in Integer)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                Int;
   is
   begin
      SPARK_IO.Put_Integer (File, Int, 0, 10);
   end Write_Integer;

   --------------------------------------------------------------------------------
   --  First_Own_Variable
   --------------------------------------------------------------------------------

   function First_Own_Variable (The_Package : RawDict.Package_Info_Ref) return Iterator
   --# global in Dict;
   is
      Own_Variable  : RawDict.Own_Variable_Info_Ref;
      Own_Variables : Iterator;
   begin
      Own_Variable := RawDict.Get_Package_Own_Variables (The_Package => The_Package);
      if Own_Variable = RawDict.Null_Own_Variable_Info_Ref then
         Own_Variables := NullIterator;
      else
         Own_Variables := Iterator'(OwnVariableIterator,
                                    IsAbstract,
                                    RawDict.Get_Own_Variable_Symbol (Own_Variable),
                                    NullSymbol);
      end if;
      return Own_Variables;
   end First_Own_Variable;

   --------------------------------------------------------------------------------

   function FirstOwnVariable (ThePackage : Symbol) return Iterator is
   begin
      return First_Own_Variable (The_Package => RawDict.Get_Package_Info_Ref (ThePackage)); -- GAA External
   end FirstOwnVariable;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function First_Abstract_Own_Variable (The_Package : RawDict.Package_Info_Ref) return Iterator
   --# global in Dict;
   is
      Own_Variables, Abstract_Own_Variables : Iterator;
   begin
      Own_Variables := First_Own_Variable (The_Package => The_Package);
      loop
         if IsNullIterator (Own_Variables) then
            Abstract_Own_Variables := NullIterator;
            exit;
         end if;
         if Own_Variable_Has_Constituents (The_Variable => RawDict.Get_Variable_Info_Ref (CurrentSymbol (Own_Variables))) then
            Abstract_Own_Variables := Iterator'(AbstractOwnVariableIterator, IsAbstract, Own_Variables.Current, NullSymbol);
            exit;
         end if;
         Own_Variables := NextSymbol (Own_Variables);
      end loop;
      return Abstract_Own_Variables;
   end First_Abstract_Own_Variable;

   --------------------------------------------------------------------------------

   function GetLoopParameter (TheLoop : Symbol) return Symbol is
   begin
      return RawDict.GetLoopParameter (TheLoop);
   end GetLoopParameter;

   --------------------------------------------------------------------------------

   function GetLoopExitExpn (TheLoop : Symbol) return Natural is
   begin
      return RawDict.GetLoopExitExpn (TheLoop);
   end GetLoopExitExpn;

   --------------------------------------------------------------------------------

   function GetLoopEntryExpn (TheLoop : Symbol) return Natural is
   begin
      return RawDict.GetLoopEntryExpn (TheLoop);
   end GetLoopEntryExpn;

   --------------------------------------------------------------------------------

   function GetLoopHasExits (TheLoop : Symbol) return Boolean is
   begin
      return RawDict.GetLoopHasExits (TheLoop);
   end GetLoopHasExits;

   --------------------------------------------------------------------------------

   function Is_For_Loop (TheSymbol : Symbol) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = LoopSymbol and then RawDict.GetLoopParameter (TheSymbol) /= NullSymbol;
   end Is_For_Loop;

   --------------------------------------------------------------------------------

   function LoopParameterHasStaticRange (TheLoopParameter : Symbol) return Boolean is
   begin
      return RawDict.GetLoopParameterHasStaticRange (TheLoopParameter);
   end LoopParameterHasStaticRange;

   --------------------------------------------------------------------------------

   function LoopParameterMovesInReverse (TheLoopParameter : Symbol) return Boolean is
   begin
      return RawDict.GetLoopParameterIsReverse (TheLoopParameter);
   end LoopParameterMovesInReverse;

   --------------------------------------------------------------------------------
   --  First_Generic_Formal_Subprogram_Parameter
   --------------------------------------------------------------------------------

   function First_Generic_Formal_Subprogram_Parameter (The_Subprogram : RawDict.Subprogram_Info_Ref) return Iterator
   --# global in Dict;
   is
      The_Generic_Unit      : RawDict.Generic_Unit_Info_Ref;
      The_Generic_Parameter : RawDict.Generic_Parameter_Info_Ref;
      FormalParameters      : Iterator;
   begin
      The_Generic_Unit := RawDict.Get_Subprogram_Generic_Unit (The_Subprogram => The_Subprogram);
      SystemErrors.RT_Assert
        (C       => The_Generic_Unit /= RawDict.Null_Generic_Unit_Info_Ref,
         Sys_Err => SystemErrors.Precondition_Failure,
         Msg     => "Wrong unit kind in Dictionary.First_Generic_Formal_Subprogram_Parameter");
      The_Generic_Parameter := RawDict.Get_Generic_Unit_First_Generic_Parameter (The_Generic_Unit => The_Generic_Unit);
      if The_Generic_Parameter = RawDict.Null_Generic_Parameter_Info_Ref then
         FormalParameters := NullIterator;
      else
         FormalParameters := Iterator'
           (GenericFormalParameterIterator,
            IsAbstract,
            RawDict.Get_Generic_Parameter_Symbol (The_Generic_Parameter),
            NullSymbol);
      end if;
      return FormalParameters;
   end First_Generic_Formal_Subprogram_Parameter;

   --------------------------------------------------------------------------------

   function FirstGenericFormalParameter (TheGeneric : Symbol) return Iterator is
      FormalParameters : Iterator;
      -- initialization strictly not needed, fatal error occurs if attempt to
      -- return an undefined value occurs
   begin
      case RawDict.GetSymbolDiscriminant (TheGeneric) is
         when Subprogram_Symbol =>
            FormalParameters := First_Generic_Formal_Subprogram_Parameter
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => TheGeneric)); -- GAA External
         when ImplicitProofFunctionSymbol =>
            FormalParameters := First_Generic_Formal_Subprogram_Parameter
              (The_Subprogram => RawDict.GetImplicitProofFunctionAdaFunction (TheGeneric));
         when others => -- non-exec code
            FormalParameters := NullIterator;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.FirstGenericFormalParameter");
      end case;
      return FormalParameters;
   end FirstGenericFormalParameter;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function Is_Loop_Name (Name  : LexTokenManager.Lex_String;
                          Scope : Scopes) return Boolean
   --# global in Dict;
   --#        in LexTokenManager.State;
   is
      Current : Iterator;
   begin
      Current := First_Loop (CompilationUnit => GetEnclosingCompilationUnit (Scope));
      loop
         exit when IsNullIterator (Current);
         exit when LexTokenManager.Lex_String_Case_Insensitive_Compare
           (Lex_Str1 => GetSimpleName (CurrentSymbol (Current)),
            Lex_Str2 => Name) =
           LexTokenManager.Str_Eq;
         Current := NextSymbol (Current);
      end loop;
      return not IsNullIterator (Current);
   end Is_Loop_Name;

   --------------------------------------------------------------------------------
   --  Is_Refined_Own_Variable
   --------------------------------------------------------------------------------

   function IsRefinedOwnVariable (Variable : Symbol) return Boolean is

      function Is_Refined_Own_Variable (The_Variable : RawDict.Variable_Info_Ref) return Boolean
      --# global in Dict;
      is
      begin
         return Is_Own_Variable (The_Variable => The_Variable)
           and then Own_Variable_Has_Constituents (The_Variable => The_Variable);
      end Is_Refined_Own_Variable;

   begin -- IsRefinedOwnVariable
      return RawDict.GetSymbolDiscriminant (Variable) = Variable_Symbol
        and then Is_Refined_Own_Variable (The_Variable => RawDict.Get_Variable_Info_Ref (Item => Variable)); -- GAA External
   end IsRefinedOwnVariable;

   --------------------------------------------------------------------------------
   --  Is_Procedure
   --------------------------------------------------------------------------------

   function Is_Procedure (The_Subprogram : RawDict.Subprogram_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Subprogram_Return_Type (The_Subprogram => The_Subprogram) = RawDict.Null_Type_Info_Ref;
   end Is_Procedure;

   --------------------------------------------------------------------------------

   function IsProcedure (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Subprogram_Symbol
        and then Is_Procedure (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => TheSymbol)); -- GAA External
   end IsProcedure;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function GetLoop (CompilationUnit : Symbol;
                     Number          : Positive) return Symbol is
      TheLoop : Iterator;
   begin
      TheLoop := First_Loop (CompilationUnit => CompilationUnit);
      for No in Positive range 1 .. Number - 1 loop
         TheLoop := NextSymbol (TheLoop);
      end loop;
      return CurrentSymbol (TheLoop);
   end GetLoop;

   --------------------------------------------------------------------------------
   --  Get_Abstraction
   --------------------------------------------------------------------------------

   function Get_Subprogram_Abstraction (The_Subprogram : RawDict.Subprogram_Info_Ref;
                                        Scope          : Scopes) return Abstractions
   --# global Dict;
   is
      Declared_Scope : Scopes;
      Abstraction    : Abstractions;
   begin
      if RawDict.Get_Subprogram_Has_Second_Annotation (The_Subprogram => The_Subprogram) then
         Declared_Scope := Get_Subprogram_Scope (The_Subprogram => The_Subprogram);
         if (Get_Visibility (Scope => Declared_Scope) = Visible or else Get_Visibility (Scope => Declared_Scope) = Privat)
           and then IsLocal (Scope, GetLocalScope (Declared_Scope)) then
            Abstraction := IsRefined;
         else
            Abstraction := IsAbstract;
         end if;
      else
         Abstraction := IsAbstract;
      end if;
      return Abstraction;
   end Get_Subprogram_Abstraction;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Abstraction (The_Task_Type : RawDict.Type_Info_Ref;
                                       Scope         : Scopes) return Abstractions
   --# global Dict;
   is
      Declared_Scope : Scopes;
      Abstraction    : Abstractions;
   begin
      if RawDict.Get_Task_Type_Has_Second_Annotation (The_Task_Type => The_Task_Type) then
         Declared_Scope := Get_Type_Scope (Type_Mark => The_Task_Type);
         if (Get_Visibility (Scope => Declared_Scope) = Visible or else Get_Visibility (Scope => Declared_Scope) = Privat)
           and then IsLocal (Scope, GetLocalScope (Declared_Scope)) then
            Abstraction := IsRefined;
         else
            Abstraction := IsAbstract;
         end if;
      else
         Abstraction := IsAbstract;
      end if;
      return Abstraction;
   end Get_Task_Type_Abstraction;

   --------------------------------------------------------------------------------

   function GetAbstraction (Subprogram : Symbol;
                            Scope      : Scopes) return Abstractions is
      Abstraction : Abstractions;
   begin
      case RawDict.GetSymbolDiscriminant (Subprogram) is
         when Subprogram_Symbol =>
            Abstraction :=
              Get_Subprogram_Abstraction (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Subprogram), -- GAA External
                                          Scope          => Scope);
         when Type_Symbol =>
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Subprogram)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.GetAbstraction");
            Abstraction :=
              Get_Task_Type_Abstraction (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => Subprogram), -- GAA External
                                         Scope         => Scope);
         when others => -- non-exec code
            Abstraction := IsAbstract;
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.GetAbstraction");
      end case;
      return Abstraction;
   end GetAbstraction;

   --------------------------------------------------------------------------------

   -- Similar to preceding function but is concerned with whether there is a second
   -- constraint rather than a second flow annotation.  If there is a second flow
   -- anno then it is assumed there is a second constraint (even if the user has just
   -- left it as True).  If there is NO second flow anno then we still need to check
   -- whether there is a second constraint (to cover the private type refinement case)
   function GetConstraintAbstraction (Subprogram : Symbol;
                                      Scope      : Scopes) return Abstractions is
      DeclaredScope : Scopes;
      Abstraction   : Abstractions;
   begin
      Abstraction := GetAbstraction (Subprogram, Scope);
      if Abstraction = IsAbstract and then RawDict.GetSymbolDiscriminant (Subprogram) = Subprogram_Symbol then
         -- No second flow anno, so check for second constraint.  Second half of guard protects against
         -- looking for a second proof annotation on a task or protected type
         if RawDict.Get_Subprogram_Has_Second_Constraint
           (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Subprogram)) then -- GAA External
            DeclaredScope := Get_Subprogram_Scope
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Subprogram)); -- GAA External
            if (Get_Visibility (Scope => DeclaredScope) = Visible or else Get_Visibility (Scope => DeclaredScope) = Privat)
              and then IsLocal (Scope, GetLocalScope (DeclaredScope)) then
               Abstraction := IsRefined;
            else
               Abstraction := IsAbstract;
            end if;
         else
            Abstraction := IsAbstract;
         end if;
      end if;
      return Abstraction;
   end GetConstraintAbstraction;

   --------------------------------------------------------------------------------

   procedure AdjustTypeUpperBound (TypeMark : Symbol;
                                   NewBound : LexTokenManager.Lex_String) is
   begin
      RawDict.Set_Type_Upper (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark), -- GAA External
                              Upper     => NewBound);
   end AdjustTypeUpperBound;

   --------------------------------------------------------------------------------

   procedure AdjustTypeLowerBound (TypeMark : Symbol;
                                   NewBound : LexTokenManager.Lex_String) is
   begin
      RawDict.Set_Type_Lower (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark), -- GAA External
                              Lower     => NewBound);
   end AdjustTypeLowerBound;

   --------------------------------------------------------------------------------

   procedure AdjustTypeErrorBound (TypeMark : Symbol;
                                   NewBound : LexTokenManager.Lex_String) is
   begin
      RawDict.Set_Type_Error_Bound (Type_Mark   => RawDict.Get_Type_Info_Ref (TypeMark), -- GAA External
                                    Error_Bound => NewBound);
   end AdjustTypeErrorBound;

   --------------------------------------------------------------------------------

   function GetMainProgramPriority return  LexTokenManager.Lex_String is
   begin
      return Dict.Main.The_Priority; -- NullString if not PriorityGiven or invalid value
   end GetMainProgramPriority;

   --------------------------------------------------------------------------------

   function IsFunction (TheSymbol : Symbol) return Boolean is
   begin
      return Is_Subprogram (TheSymbol) and then Get_Type (The_Symbol => TheSymbol) /= RawDict.Null_Type_Info_Ref;
   end IsFunction;

   --------------------------------------------------------------------------------

   function IsProtectedFunction (TheSymbol : Symbol) return Boolean is
   begin
      return IsFunction (TheSymbol)
        and then RawDict.GetSymbolDiscriminant (GetRegion (GetScope (TheSymbol))) = Type_Symbol
        and then Is_Protected_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => GetRegion (GetScope (TheSymbol))));
   end IsProtectedFunction;

   --------------------------------------------------------------------------------

   function IsEntry (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Subprogram_Symbol
        and then RawDict.Get_Subprogram_Is_Entry
        (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => TheSymbol)); -- GAA External
   end IsEntry;

   --------------------------------------------------------------------------------

   function IsProgramUnit (Sym : Symbol) return Boolean is
   begin
      return Sym /= NullSymbol
        and then (RawDict.GetSymbolDiscriminant (Sym) = Package_Symbol
                    or else (RawDict.GetSymbolDiscriminant (Sym) = Type_Symbol
                               and then (Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Sym)) -- GAA External
                                           or else Is_Protected_Type
                                           (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Sym)))) -- GAA External
                    or else (RawDict.GetSymbolDiscriminant (Sym) = Subprogram_Symbol
                               and then (RawDict.Get_Subprogram_Is_Entry
                                           (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Sym)) -- GAA External
                                           or else RawDict.Get_Subprogram_Generic_Unit
                                           (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Sym)) /= -- GAA External
                                           RawDict.Null_Generic_Unit_Info_Ref))
                    or else Is_Subprogram (Sym));
   end IsProgramUnit;

   --------------------------------------------------------------------------------

   function GetSubprogramEntryBarrier (Subprogram : Symbol) return Symbol is
      Result : Symbol := NullSymbol;
   begin
      if RawDict.GetSymbolDiscriminant (Subprogram) = Subprogram_Symbol
        and then RawDict.Get_Subprogram_Is_Entry
        (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Subprogram)) then -- GAA External
         Result := RawDict.Get_Subprogram_Entry_Barrier
           (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Subprogram)); -- GAA External
      end if;
      return Result;
   end GetSubprogramEntryBarrier;

   --------------------------------------------------------------------------------

   function IsInterruptHandler (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Subprogram_Symbol
        and then RawDict.Get_Subprogram_Is_Interrupt_Handler
        (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => TheSymbol)); -- GAA External
   end IsInterruptHandler;

   --------------------------------------------------------------------------------

   function IsAdaFunction (TheSymbol : Symbol) return Boolean is
   begin
      return IsFunction (TheSymbol)
        and then RawDict.GetSymbolDiscriminant (TheSymbol) = Subprogram_Symbol
        and then Get_Subprogram_Context (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => TheSymbol)) = -- GAA External
        ProgramContext;
   end IsAdaFunction;

   --------------------------------------------------------------------------------

   function IsAnUncheckedConversion (TheSymbol : Symbol) return Boolean is
   begin
      return IsAdaFunction (TheSymbol)
        and then RawDict.Get_Subprogram_Instantiation_Of
        (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (TheSymbol)) = -- GAA External
        Dict.Subprograms.Unchecked_Conversion;
   end IsAnUncheckedConversion;

   --------------------------------------------------------------------------------

   function UsesUncheckedConversion (TheUnit : Symbol) return Boolean is
      Result : Boolean;
   begin
      case RawDict.GetSymbolDiscriminant (TheUnit) is
         when Subprogram_Symbol =>
            -- covers procedures, functions and entries
            Result := RawDict.Get_Subprogram_Uses_Unchecked_Conversion
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => TheUnit)); -- GAA External
         when Type_Symbol =>
            -- covers task bodies
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheUnit)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.UsesUncheckedConversion");
            Result := RawDict.Get_Task_Type_Uses_Unchecked_Conversion
              (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => TheUnit)); -- GAA External
         when Package_Symbol =>
            -- a package could only use U_C in elabotration code where it would be illegal
            Result := False;
         when others => -- non-exec code
            Result := False;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.UsesUncheckedConversion");
      end case;
      return Result;
   end UsesUncheckedConversion;

   --------------------------------------------------------------------------------

   function AssignsFromExternal (TheUnit : Symbol) return Boolean is
      Result : Boolean;
   begin
      case RawDict.GetSymbolDiscriminant (TheUnit) is
         when Subprogram_Symbol =>
            -- covers procedures, functions and entries
            Result := RawDict.Get_Subprogram_Assigns_From_External
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => TheUnit)); -- GAA External
         when Type_Symbol =>
            -- covers task bodies
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheUnit)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.AssignsFromExternal");
            Result := RawDict.Get_Task_Type_Assigns_From_External
              (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => TheUnit)); -- GAA External
         when Package_Symbol =>
            -- a package could only read external variable in elabotration code
            -- where it would be illegal
            Result := False;
         when others => -- non-exec code
            Result := False;
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.AssignsFromExternal");
      end case;
      return Result;
   end AssignsFromExternal;

   --------------------------------------------------------------------------------

   function IsObject (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Constant_Symbol or else Is_Variable (TheSymbol);
   end IsObject;

   --------------------------------------------------------------------------------

   function PrefixAllowed (Prefix : Symbol;
                           Scope  : Scopes) return Boolean is
      Result : Boolean;
   begin
      case RawDict.GetSymbolDiscriminant (Prefix) is
         when Package_Symbol =>
            Result := RawDict.Get_Package_Info_Ref (Item => Prefix) /= Get_Enclosing_Package (Scope => Scope); -- GAA External
         when Type_Symbol =>
            Result := RawDict.Get_Type_Discriminant (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Prefix)) = -- GAA External
              Record_Type_Item;
         when others =>
            if IsObject (Prefix) then
               Result := RawDict.Get_Type_Discriminant (Type_Mark => Get_Type (The_Symbol => Prefix)) = Record_Type_Item
                 or else RawDict.Get_Type_Discriminant (Type_Mark => Get_Type (The_Symbol => Prefix)) =
                 Protected_Type_Item;
            else
               Result := False;
            end if;
      end case;
      return Result;
   end PrefixAllowed;

   --------------------------------------------------------------------------------

   function PrefixRequired (Item  : Symbol;
                            Scope : Scopes) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (GetRegion (GetScope (Item))) = Package_Symbol
        and then RawDict.Get_Package_Info_Ref (Item => GetRegion (GetScope (Item))) /= Get_Enclosing_Package (Scope => Scope); -- GAA External
   end PrefixRequired;

   --------------------------------------------------------------------------------

   function GetPrefix (Item : Symbol) return LexTokenManager.Lex_String is
   begin
      return GetSimpleName (GetRegion (GetScope (Item)));
   end GetPrefix;

   --------------------------------------------------------------------------------

   function GetEnclosingProtectedRegion (Scope : Scopes) return Symbol is
      CurrentRegion : Symbol;
      CurrentScope  : Scopes;
      Result        : Symbol;
   begin
      Result       := NullSymbol;
      CurrentScope := Scope;
      loop
         CurrentRegion := GetRegion (CurrentScope);
         if RawDict.GetSymbolDiscriminant (CurrentRegion) = Type_Symbol
           and then RawDict.Get_Type_Discriminant (Type_Mark => RawDict.Get_Type_Info_Ref (Item => CurrentRegion)) =
           Protected_Type_Item then
            Result := CurrentRegion;
            exit;
         end if;
         exit when IsLibraryLevel (CurrentScope);
         CurrentScope := GetEnclosingScope (CurrentScope);
      end loop;
      return Result;
   end GetEnclosingProtectedRegion;

   --------------------------------------------------------------------------------

   function IsOrIsInProtectedScope (Scope : Scopes) return Boolean is
   begin
      return GetEnclosingProtectedRegion (Scope) /= NullSymbol;
   end IsOrIsInProtectedScope;

   --------------------------------------------------------------------------------
   --  First_Protected_Element
   --------------------------------------------------------------------------------

   function First_Protected_Element (The_Protected_Type : RawDict.Type_Info_Ref) return Iterator
   --# global in Dict;
   is
      The_Declaration    : RawDict.Declaration_Info_Ref;
      Protected_Elements : Iterator := NullIterator;
   begin
      The_Declaration := RawDict.Get_Protected_Type_First_Private_Declaration (The_Protected_Type => The_Protected_Type);
      if The_Declaration /= RawDict.Null_Declaration_Info_Ref then
         Protected_Elements := Iterator'
           (ProtectedElementIterator,
            IsAbstract,
            RawDict.Get_Declaration_Item (The_Declaration => The_Declaration),
            RawDict.Get_Declaration_Symbol (The_Declaration));
      end if;
      return Protected_Elements;
   end First_Protected_Element;

   --------------------------------------------------------------------------------

   function FirstProtectedElement (The_Protected_Type : Symbol) return Iterator is
   begin
      return First_Protected_Element (The_Protected_Type => RawDict.Get_Type_Info_Ref (The_Protected_Type)); -- GAA External
   end FirstProtectedElement;

   --------------------------------------------------------------------------------
   --  Get_Protected_Type_Own_Variable
   --------------------------------------------------------------------------------

   function Get_Protected_Type_Own_Variable (The_Protected_Type : RawDict.Type_Info_Ref) return RawDict.Variable_Info_Ref
   --# global in Dict;
   is
   begin
      return RawDict.Get_Own_Variable_Variable
        (The_Own_Variable => RawDict.Get_Protected_Type_Own_Variable (The_Protected_Type => The_Protected_Type));
   end Get_Protected_Type_Own_Variable;

   --------------------------------------------------------------------------------

   function GetProtectedTypeOwnVariable (TheProtectedType : Symbol) return Symbol is
   begin
      return RawDict.Get_Variable_Symbol -- GAA External
        (Get_Protected_Type_Own_Variable
           (The_Protected_Type => RawDict.Get_Type_Info_Ref (TheProtectedType))); -- GAA External
   end GetProtectedTypeOwnVariable;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   -- Trace routines for dictionary look up operations.  These are invisible to the
   -- flow analyser. They are enabled by -debug=l (for Lookup)

   procedure TraceMsg (Msg : in String)
   --# derives null from Msg;
   is
      --# hide TraceMsg;
   begin
      if CommandLineData.Content.Debug.Lookup_Trace then
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output, Msg, 0);
      end if;
   end TraceMsg;

   procedure Trace_Lex_Str (Msg : in String;
                            L   : in LexTokenManager.Lex_String)
   --# derives null from L,
   --#                   Msg;
   is
      --# hide Trace_Lex_Str;
   begin
      if CommandLineData.Content.Debug.Lookup_Trace then
         SPARK_IO.Put_String (SPARK_IO.Standard_Output, Msg, 0);
         if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => L,
                                                                 Lex_Str2 => LexTokenManager.Null_String) =
           LexTokenManager.Str_Eq then
            SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "Null String", 0);
         else
            E_Strings.Put_Line (File  => SPARK_IO.Standard_Output,
                                E_Str => LexTokenManager.Lex_String_To_String (Lex_Str => L));
         end if;
      end if;
   end Trace_Lex_Str;

   procedure Trace_Sym (Msg   : in String;
                        Sym   : in Symbol;
                        Scope : in Scopes)
   --# derives null from Msg,
   --#                   Scope,
   --#                   Sym;
   is
      --# hide Trace_Sym;
   begin
      if CommandLineData.Content.Debug.Lookup_Trace then
         SPARK_IO.Put_String (SPARK_IO.Standard_Output, Msg, 0);
         if Sym = NullSymbol then
            SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "Null Symbol", 0);
         else
            E_Strings.Put_String
              (File  => SPARK_IO.Standard_Output,
               E_Str => GetAnyPrefixNeeded (Sym       => Sym,
                                            Scope     => Scope,
                                            Separator => "."));
            SPARK_IO.Put_Char (SPARK_IO.Standard_Output, '.');
            E_Strings.Put_Line (File  => SPARK_IO.Standard_Output,
                                E_Str => GenerateSimpleName (Item      => Sym,
                                                             Separator => "."));
         end if;
      end if;
   end Trace_Sym;

   --------------------------------------------------------------------------------

   procedure LookupScope
     (Name          : in     LexTokenManager.Lex_String;
      Stop_At       : in     LexTokenManager.Lex_String;
      Scope         : in     Scopes;
      Calling_Scope : in     Scopes;
      Context       : in     Contexts;
      Item          :    out Symbol;
      Is_Visible    :    out Boolean)
   --# global in Dict;
   --#        in LexTokenManager.State;
   --# derives Is_Visible,
   --#         Item       from Calling_Scope,
   --#                         Context,
   --#                         Dict,
   --#                         LexTokenManager.State,
   --#                         Name,
   --#                         Scope,
   --#                         Stop_At;
      is separate;

   --------------------------------------------------------------------------------

   function LookupImmediateScope
     (Name    : LexTokenManager.Lex_String;
      Scope   : Scopes;
      Context : Contexts)
     return    Symbol
   is
      Item      : Symbol;
      IsVisible : Boolean;
   begin
      Trace_Lex_Str (Msg => "In LookupImmediateScope, seeking ",
                     L   => Name);
      LookupScope
        (Name          => Name,
         Stop_At       => LexTokenManager.Null_String,
         Scope         => Scope,
         Calling_Scope => Scope,
         Context       => Context,
         Item          => Item,
         Is_Visible    => IsVisible);
      if not IsVisible then
         Item := NullSymbol;
      end if;
      return Item;
   end LookupImmediateScope;

   --------------------------------------------------------------------------------

   function IsDirectlyDefined
     (Name    : LexTokenManager.Lex_String;
      Scope   : Scopes;
      Context : Contexts)
     return    Boolean
   is
   begin
      return Is_Loop_Name (Name  => Name,
                           Scope => Scope)
        or else LookupImmediateScope (Name    => Name,
                                      Scope   => Scope,
                                      Context => Context) /= NullSymbol;
   end IsDirectlyDefined;

   --------------------------------------------------------------------------------
   --  Is_Withed_Locally
   --------------------------------------------------------------------------------

   function Is_Withed_Locally (The_Withed_Symbol : Symbol;
                               Scope             : Scopes) return Boolean is
      Region                : Symbol;
      The_Generic_Unit      : RawDict.Generic_Unit_Info_Ref;
      The_Context_Clause    : RawDict.Context_Clause_Info_Ref;
      The_Withed_Package    : RawDict.Package_Info_Ref;
      The_Withed_Subprogram : RawDict.Subprogram_Info_Ref;
   begin
      Region := GetRegion (Scope);
      case Get_Visibility (Scope => Scope) is
         when Visible | Privat =>
            case RawDict.GetSymbolDiscriminant (Region) is
               when Package_Symbol =>
                  The_Context_Clause := RawDict.Get_Package_Visible_With_Clauses
                    (The_Package => RawDict.Get_Package_Info_Ref (Item => Region));
               when Type_Symbol =>
                  SystemErrors.RT_Assert
                    (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Region))
                       or else Is_Protected_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Region)),
                     Sys_Err => SystemErrors.Invalid_Symbol_Table,
                     Msg     => "in Dictionary.Is_Withed_Locally");
                  The_Context_Clause := RawDict.Get_Package_Visible_With_Clauses
                    (The_Package => Get_Enclosing_Package (Scope => Scope));
               when Generic_Unit_Symbol =>
                  The_Generic_Unit := RawDict.Get_Generic_Unit_Info_Ref (Item => Region);
                  case RawDict.Get_Generic_Unit_Kind (The_Generic_Unit => The_Generic_Unit) is
                     when Generic_Of_Package =>
                        The_Context_Clause := RawDict.Get_Package_Visible_With_Clauses
                          (The_Package => RawDict.Get_Generic_Unit_Owning_Package (The_Generic_Unit => The_Generic_Unit));
                     when Generic_Of_Subprogram =>
                        The_Context_Clause := RawDict.Get_Subprogram_With_Clauses
                          (The_Subprogram => RawDict.Get_Generic_Unit_Owning_Subprogram (The_Generic_Unit => The_Generic_Unit));
                  end case;
               when others => -- non-exec code
                  The_Context_Clause := RawDict.Null_Context_Clause_Info_Ref;
                  SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                            Msg     => "in Dictionary.Is_Withed_Locally");
            end case;
         when Local =>
            case RawDict.GetSymbolDiscriminant (Region) is
               when Package_Symbol =>
                  The_Context_Clause := RawDict.Get_Package_Local_With_Clauses
                    (The_Package => RawDict.Get_Package_Info_Ref (Item => Region));
               when Subprogram_Symbol =>
                  The_Context_Clause := RawDict.Get_Subprogram_With_Clauses
                    (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Region));
               when Type_Symbol =>
                  case RawDict.Get_Type_Discriminant (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Region)) is
                     when Protected_Type_Item =>
                        The_Context_Clause := RawDict.Get_Protected_Type_With_Clauses
                          (The_Protected_Type => RawDict.Get_Type_Info_Ref (Item => Region));
                     when Task_Type_Item =>
                        The_Context_Clause := RawDict.Get_Task_Type_With_Clauses
                          (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => Region));
                     when others => -- non-exec code
                        The_Context_Clause := RawDict.Null_Context_Clause_Info_Ref;
                        SystemErrors.Fatal_Error
                          (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                           Msg     => "in Dictionary.Is_Withed_Locally");
                  end case;
               when LoopSymbol =>
                  The_Context_Clause := RawDict.Null_Context_Clause_Info_Ref;
               when others => -- non-exec code
                  The_Context_Clause := RawDict.Null_Context_Clause_Info_Ref;
                  SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                            Msg     => "in Dictionary.Is_Withed_Locally");
            end case;
      end case;

      case RawDict.GetSymbolDiscriminant (The_Withed_Symbol) is
         when Package_Symbol =>
            The_Withed_Package := RawDict.Get_Package_Info_Ref (Item => The_Withed_Symbol);
            while The_Context_Clause /= RawDict.Null_Context_Clause_Info_Ref
              and then (RawDict.Get_Context_Clause_Is_Subprogram (The_Context_Clause => The_Context_Clause)
                          or else RawDict.Get_Context_Clause_Package (The_Context_Clause => The_Context_Clause) /= The_Withed_Package) loop
               The_Context_Clause := RawDict.Get_Next_Context_Clause (The_Context_Clause => The_Context_Clause);
            end loop;
         when Subprogram_Symbol =>
            The_Withed_Subprogram := RawDict.Get_Subprogram_Info_Ref (Item => The_Withed_Symbol);
            while The_Context_Clause /= RawDict.Null_Context_Clause_Info_Ref
              and then (not RawDict.Get_Context_Clause_Is_Subprogram (The_Context_Clause => The_Context_Clause)
                          or else RawDict.Get_Context_Clause_Subprogram (The_Context_Clause => The_Context_Clause) /= The_Withed_Subprogram) loop
               The_Context_Clause := RawDict.Get_Next_Context_Clause (The_Context_Clause => The_Context_Clause);
            end loop;
         when others => -- non-exec code
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.Is_Withed_Locally");
      end case;
      return The_Context_Clause /= RawDict.Null_Context_Clause_Info_Ref;
   end Is_Withed_Locally;

   --------------------------------------------------------------------------------
   --  Is_Withed
   --------------------------------------------------------------------------------

   function Is_Withed (The_Withed_Symbol : Symbol;
                       Scope             : Scopes) return Boolean is
      Region                : Symbol;
      The_Context_Clause    : RawDict.Context_Clause_Info_Ref;
      The_Withed_Package    : RawDict.Package_Info_Ref;
      The_Withed_Subprogram : RawDict.Subprogram_Info_Ref;
      Result                : Boolean;
   begin
      Region := GetRegion (Scope);
      Result := Is_Withed_Locally (The_Withed_Symbol => The_Withed_Symbol,
                                   Scope             => Scope);
      if not Result
        and then RawDict.GetSymbolDiscriminant (Region) = Package_Symbol
        and then Get_Visibility (Scope => Scope) = Local then
         The_Context_Clause := RawDict.Get_Package_Visible_With_Clauses
           (The_Package => RawDict.Get_Package_Info_Ref (Item => Region));

         case RawDict.GetSymbolDiscriminant (The_Withed_Symbol) is
            when Package_Symbol =>
               The_Withed_Package := RawDict.Get_Package_Info_Ref (Item => The_Withed_Symbol);
               while The_Context_Clause /= RawDict.Null_Context_Clause_Info_Ref
                 and then (RawDict.Get_Context_Clause_Is_Subprogram (The_Context_Clause => The_Context_Clause)
                             or else RawDict.Get_Context_Clause_Package (The_Context_Clause => The_Context_Clause) /= The_Withed_Package) loop
                  The_Context_Clause := RawDict.Get_Next_Context_Clause (The_Context_Clause => The_Context_Clause);
               end loop;
            when Subprogram_Symbol =>
               The_Withed_Subprogram := RawDict.Get_Subprogram_Info_Ref (Item => The_Withed_Symbol);
               while The_Context_Clause /= RawDict.Null_Context_Clause_Info_Ref
                 and then (not RawDict.Get_Context_Clause_Is_Subprogram (The_Context_Clause => The_Context_Clause)
                             or else RawDict.Get_Context_Clause_Subprogram (The_Context_Clause => The_Context_Clause) /= The_Withed_Subprogram) loop
                  The_Context_Clause := RawDict.Get_Next_Context_Clause (The_Context_Clause => The_Context_Clause);
               end loop;
            when others => -- non-exec code
               SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                         Msg     => "in Dictionary.Is_Withed");
         end case;
         Result := The_Context_Clause /= RawDict.Null_Context_Clause_Info_Ref;
      end if;
      return Result;
   end Is_Withed;

   --------------------------------------------------------------------------------
   --  Is_Used_Locally
   --------------------------------------------------------------------------------

   function Is_Used_Locally (Type_Mark : RawDict.Type_Info_Ref;
                             Scope     : Scopes) return Boolean
   --# global Dict;
   is
      Region              : Symbol;
      The_Generic_Unit    : RawDict.Generic_Unit_Info_Ref;
      The_Use_Type_Clause : RawDict.Use_Type_Clause_Info_Ref;
   begin
      Region := GetRegion (Scope);
      case Get_Visibility (Scope => Scope) is
         when Visible | Privat =>
            case RawDict.GetSymbolDiscriminant (Region) is
               when Package_Symbol =>
                  The_Use_Type_Clause := RawDict.Get_Package_Visible_Use_Type_Clauses
                    (The_Package => RawDict.Get_Package_Info_Ref (Item => Region));
               when Type_Symbol =>
                  case RawDict.Get_Type_Discriminant (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Region)) is
                     when Protected_Type_Item =>
                        The_Use_Type_Clause := RawDict.Get_Package_Visible_Use_Type_Clauses
                          (The_Package => Get_Enclosing_Package (Scope => Scope));
                     when Task_Type_Item =>
                        The_Use_Type_Clause := RawDict.Get_Package_Visible_Use_Type_Clauses
                          (The_Package => Get_Enclosing_Package (Scope => Scope));
                     when others => -- non-exec code
                        The_Use_Type_Clause := RawDict.Null_Use_Type_Clause_Info_Ref;
                        SystemErrors.Fatal_Error
                          (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                           Msg     => "in Dictionary.Is_Used_Locally");
                  end case;
               when Generic_Unit_Symbol =>
                  The_Generic_Unit := RawDict.Get_Generic_Unit_Info_Ref (Item => Region);
                  case RawDict.Get_Generic_Unit_Kind (The_Generic_Unit => The_Generic_Unit) is
                     when Generic_Of_Package =>
                        The_Use_Type_Clause := RawDict.Get_Package_Visible_Use_Type_Clauses
                          (The_Package => RawDict.Get_Generic_Unit_Owning_Package (The_Generic_Unit => The_Generic_Unit));
                     when Generic_Of_Subprogram =>
                        The_Use_Type_Clause := RawDict.Get_Subprogram_Use_Type_Clauses
                          (The_Subprogram => RawDict.Get_Generic_Unit_Owning_Subprogram (The_Generic_Unit => The_Generic_Unit));
                  end case;
               when others => -- non-exec code
                  The_Use_Type_Clause := RawDict.Null_Use_Type_Clause_Info_Ref;
                  SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                            Msg     => "in Dictionary.Is_Used_Locally");
            end case;
         when Local =>
            case RawDict.GetSymbolDiscriminant (Region) is
               when Package_Symbol =>
                  The_Use_Type_Clause := RawDict.Get_Package_Local_Use_Type_Clauses
                    (The_Package => RawDict.Get_Package_Info_Ref (Item => Region));
               when Subprogram_Symbol =>
                  The_Use_Type_Clause := RawDict.Get_Subprogram_Use_Type_Clauses
                    (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Region));
               when Type_Symbol =>
                  case RawDict.Get_Type_Discriminant (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Region)) is
                     when Protected_Type_Item =>
                        The_Use_Type_Clause := RawDict.Get_Protected_Type_Use_Type_Clauses
                          (The_Protected_Type => RawDict.Get_Type_Info_Ref (Item => Region));
                     when Task_Type_Item =>
                        The_Use_Type_Clause := RawDict.Get_Task_Type_Use_Type_Clauses
                          (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => Region));
                     when others => -- non-exec code
                        The_Use_Type_Clause := RawDict.Null_Use_Type_Clause_Info_Ref;
                        SystemErrors.Fatal_Error
                          (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                           Msg     => "in Dictionary.Is_Used_Locally");
                  end case;
               when others => -- non-exec code
                  The_Use_Type_Clause := RawDict.Null_Use_Type_Clause_Info_Ref;
                  SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                            Msg     => "in Dictionary.Is_Used_Locally");
            end case;
      end case;

      while The_Use_Type_Clause /= RawDict.Null_Use_Type_Clause_Info_Ref
        and then RawDict.Get_Use_Type_Clause_Type (The_Use_Type_Clause => The_Use_Type_Clause) /= Type_Mark loop
         The_Use_Type_Clause := RawDict.Get_Next_Use_Type_Clause (The_Use_Type_Clause => The_Use_Type_Clause);
      end loop;
      return The_Use_Type_Clause /= RawDict.Null_Use_Type_Clause_Info_Ref;
   end Is_Used_Locally;

   --------------------------------------------------------------------------------

   function IsUsedLocally (TheType : Symbol;
                           Scope   : Scopes) return Boolean is
   begin
      return Is_Used_Locally (Type_Mark => RawDict.Get_Type_Info_Ref (TheType), -- GAA External
                              Scope     => Scope);
   end IsUsedLocally;

   --------------------------------------------------------------------------------
   --  Is_Embedded_Package
   --------------------------------------------------------------------------------

   function Is_Embedded_Package (The_Package : RawDict.Package_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return GetRegion (Get_Package_Scope (The_Package => The_Package)) /=
        RawDict.Get_Package_Symbol (Get_Predefined_Package_Standard);
   end Is_Embedded_Package;

   --------------------------------------------------------------------------------

   function IsEmbeddedPackage (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Package_Symbol
        and then Is_Embedded_Package (The_Package => RawDict.Get_Package_Info_Ref (Item => TheSymbol)); -- GAA External
   end IsEmbeddedPackage;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function Is_Descendent_Of_Private_Child (Candidate, The_Package : RawDict.Package_Info_Ref) return Boolean
   --# global in Dict;
   is
      Current_Package, Next_Package : RawDict.Package_Info_Ref;
   begin
      Current_Package := Candidate;
      loop
         Next_Package := RawDict.Get_Package_Parent (The_Package => Current_Package);
         exit when Next_Package = The_Package or else Next_Package = RawDict.Null_Package_Info_Ref;
         Current_Package := Next_Package;
      end loop;
      return Next_Package = The_Package and then RawDict.Get_Package_Is_Private (The_Package => Current_Package);
   end Is_Descendent_Of_Private_Child;

   --------------------------------------------------------------------------------

   function Is_Global_Variable
     (Abstraction : Abstractions;
      Subprogram  : Symbol;
      Variable    : Symbol)
     return        Boolean
   is
      The_Global_Variable : RawDict.Global_Variable_Info_Ref;
   begin
      case RawDict.GetSymbolDiscriminant (Subprogram) is
         when Subprogram_Symbol =>
            The_Global_Variable := RawDict.Get_Subprogram_First_Global_Variable
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Subprogram), -- GAA External
               Abstraction    => Abstraction);
         when Type_Symbol =>
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Subprogram)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.Is_Global_Variable");
            The_Global_Variable := RawDict.Get_Task_Type_First_Global_Variable
              (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => Subprogram), -- GAA External
               Abstraction   => Abstraction);
         when others => -- non-exec code
            The_Global_Variable := RawDict.Null_Global_Variable_Info_Ref;
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.Is_Global_Variable");
      end case;

      case RawDict.GetSymbolDiscriminant (Variable) is
         when Subprogram_Parameter_Symbol =>
            while The_Global_Variable /= RawDict.Null_Global_Variable_Info_Ref
              and then (RawDict.Get_Kind_Of_Global_Variable (The_Global_Variable => The_Global_Variable) =
                          RawDict.Subprogram_Variable_Item
                          or else RawDict.Get_Kind_Of_Global_Variable (The_Global_Variable => The_Global_Variable) =
                          RawDict.Task_Type_Variable_Item
                          or else RawDict.Get_Global_Variable_Parameter (The_Global_Variable => The_Global_Variable) /=
                          RawDict.Get_Subprogram_Parameter_Info_Ref (Item => Variable)) loop -- GAA External
               The_Global_Variable := RawDict.Get_Next_Global_Variable (The_Global_Variable => The_Global_Variable);
            end loop;
         when Variable_Symbol =>
            while The_Global_Variable /= RawDict.Null_Global_Variable_Info_Ref
              and then (RawDict.Get_Kind_Of_Global_Variable (The_Global_Variable => The_Global_Variable) =
                          RawDict.Subprogram_Parameter_Item
                          or else RawDict.Get_Global_Variable_Variable (The_Global_Variable => The_Global_Variable) /=
                          RawDict.Get_Variable_Info_Ref (Item => Variable)) loop -- GAA External
               The_Global_Variable := RawDict.Get_Next_Global_Variable (The_Global_Variable => The_Global_Variable);
            end loop;
         when others => -- non-exec code
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.Is_Global_Variable");
      end case;
      return The_Global_Variable /= RawDict.Null_Global_Variable_Info_Ref;
   end Is_Global_Variable;

   --------------------------------------------------------------------------------

   function LookupItem
     (Name              : LexTokenManager.Lex_String;
      Scope             : Scopes;
      Context           : Contexts;
      Full_Package_Name : Boolean)
     return              Symbol
      is separate;

   --------------------------------------------------------------------------------
   --  Is_Renamed
   --------------------------------------------------------------------------------

   function Is_Renamed_Local (The_Subprogram : RawDict.Subprogram_Info_Ref;
                              Scope          : Scopes) return Boolean
   --# global in Dict;
      is separate;

   --------------------------------------------------------------------------------

   function Is_Renamed (Subprogram : Symbol;
                        Scope      : Scopes) return Boolean is
   begin
      return Is_Renamed_Local (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Subprogram), -- GAA External
                               Scope          => Scope);
   end Is_Renamed;

   --------------------------------------------------------------------------------
   --  Is_Callable
   --------------------------------------------------------------------------------

   function Is_Callable
     (The_Subprogram : RawDict.Subprogram_Info_Ref;
      Prefix_Needed  : Boolean;
      Scope          : Scopes)
     return         Boolean
   --# global in Dict;
      is separate;

   --------------------------------------------------------------------------------

   function IsCallable
     (Subprogram   : Symbol;
      PrefixNeeded : Boolean;
      Scope        : Scopes)
     return         Boolean is
   begin
      return Is_Callable (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Subprogram), -- GAA External
                          Prefix_Needed  => PrefixNeeded,
                          Scope          => Scope);
   end IsCallable;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function IsDefined
     (Name              : LexTokenManager.Lex_String;
      Scope             : Scopes;
      Context           : Contexts;
      Full_Package_Name : Boolean)
     return              Boolean
   is
   begin
      return Is_Loop_Name (Name  => Name,
                           Scope => Scope)
        or else LookupItem (Name              => Name,
                            Scope             => Scope,
                            Context           => Context,
                            Full_Package_Name => Full_Package_Name) /=
        NullSymbol;
   end IsDefined;

   --------------------------------------------------------------------------------
   --  Unary_Operator_Is_Defined
   --------------------------------------------------------------------------------

   function UnaryOperatorIsDefined (Name    : SP_Symbols.SP_Symbol;
                                    Operand : Symbol) return Boolean is

      function Unary_Operator_Is_Defined (Name      : SP_Symbols.SP_Symbol;
                                          Type_Mark : RawDict.Type_Info_Ref) return Boolean
      --# global in CommandLineData.Content;
      --#        in Dict;
      is
         Result : Boolean;
      begin
         case Name is
            when SP_Symbols.plus | SP_Symbols.minus | SP_Symbols.RWabs =>
               -- Unary arithmetic operators on modular types are not
               -- allowed in SPARK at present.  "abs" and "+" are just no-op
               -- for modular type, so they're useless.  Unary "-" is a potential
               -- danger, since "-1" for a modular type is confusing and can always
               -- be written as T'Last or 16#FFFF# or similar.
               --
               -- For time type Time_Span, unary "abs" and "-" are defined but "+" is not.
               Result := Type_Is_Integer (Type_Mark => Type_Mark)
                 or else Type_Is_Real (Type_Mark => Type_Mark)
                 or else (CommandLineData.Ravenscar_Selected
                            and then Type_Mark = Get_Predefined_Time_Span_Type
                            and then Name /= SP_Symbols.plus)
                 or else Type_Mark = Get_Unknown_Type_Mark;
            when SP_Symbols.RWnot =>
               -- Note - unary "not" is allowed for modular types.
               Result := Type_Is_Boolean (Type_Mark => Type_Mark)
                 or else Type_Is_Modular (Type_Mark => Type_Mark)
                 or else Type_Mark = Get_Universal_Integer_Type
                 or else Type_Is_Boolean_Array (Type_Mark => Type_Mark)
                 or else Type_Mark = Get_Unknown_Type_Mark;
            when others =>
               Result := False;
         end case;
         return Result;
      end Unary_Operator_Is_Defined;

   begin -- UnaryOperatorIsDefined
      return Unary_Operator_Is_Defined (Name      => Name,
                                        Type_Mark => RawDict.Get_Type_Info_Ref (Operand)); -- GAA External
   end UnaryOperatorIsDefined;

   --------------------------------------------------------------------------------
   --  Get_Predefined_String_Type
   --------------------------------------------------------------------------------

   function Get_Predefined_String_Type return  RawDict.Type_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Types.The_Predefined_Types (Predefined_String_Type);
   end Get_Predefined_String_Type;

   --------------------------------------------------------------------------------

   function GetPredefinedStringType return Symbol is
   begin
      return RawDict.Get_Type_Symbol (Get_Predefined_String_Type); -- GAA External
   end GetPredefinedStringType;

   --------------------------------------------------------------------------------

   function IsPredefinedStringType (TheSymbol : Symbol) return Boolean is
   begin
      return RawDict.Get_Type_Info_Ref (TheSymbol) = Get_Predefined_String_Type; -- GAA External
   end IsPredefinedStringType;

   --------------------------------------------------------------------------------
   --  Get_Predefined_Natural_Subtype
   --------------------------------------------------------------------------------

   function Get_Predefined_Natural_Subtype return  RawDict.Type_Info_Ref
   --# global in Dict;
   is
   begin
      return Dict.Types.The_Predefined_Types (Predefined_Natural_Subtype);
   end Get_Predefined_Natural_Subtype;

   --------------------------------------------------------------------------------

   function GetPredefinedNaturalSubtype return Symbol is
   begin
      return RawDict.Get_Type_Symbol (Get_Predefined_Natural_Subtype); -- GAA External
   end GetPredefinedNaturalSubtype;

   --------------------------------------------------------------------------------
   --  Get_Binary_Operator_Type
   --------------------------------------------------------------------------------

   function Get_Binary_Operator_Type_Local
     (Name           : SP_Symbols.SP_Symbol;
      The_Left_Type  : RawDict.Type_Info_Ref;
      The_Right_Type : RawDict.Type_Info_Ref)
     return RawDict.Type_Info_Ref
   --# global in CommandLineData.Content;
   --#        in Dict;
      is separate;

   --------------------------------------------------------------------------------

   function Get_Binary_Operator_Type
     (Name  : SP_Symbols.SP_Symbol;
      Left  : Symbol;
      Right : Symbol)
     return  Symbol
   is
   begin
      return RawDict.Get_Type_Symbol -- GAA External
        (Get_Binary_Operator_Type_Local
           (Name           => Name,
            The_Left_Type  => RawDict.Get_Type_Info_Ref (Left), -- GAA External
            The_Right_Type => RawDict.Get_Type_Info_Ref (Right))); -- GAA External
   end Get_Binary_Operator_Type;

   --------------------------------------------------------------------------------
   --  Defined_In_Package_Standard
   --------------------------------------------------------------------------------

   function Defined_In_Package_Standard (Type_Mark : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
      Result : Boolean;
   begin
      if Type_Mark = Get_Unknown_Type_Mark then
         Result := False;
      else
         Result := Get_Type_Scope (Type_Mark => Type_Mark) =
           Set_Visibility (The_Visibility => Visible,
                           The_Unit       => RawDict.Get_Package_Symbol (Get_Predefined_Package_Standard));
      end if;
      return Result;
   end Defined_In_Package_Standard;

   --------------------------------------------------------------------------------

   function DefinedInPackageStandard (TheSymbol : Symbol) return Boolean is
   begin
      return Defined_In_Package_Standard (Type_Mark => RawDict.Get_Type_Info_Ref (TheSymbol)); -- GAA External
   end DefinedInPackageStandard;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function IsPredefined (TheSymbol : Symbol) return Boolean is
      Result : Boolean;
      Scope  : Scopes;
   begin
      if RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then RawDict.Get_Type_Info_Ref (Item => TheSymbol) = Get_Unknown_Type_Mark then -- GAA External
         Result := False;
      else
         Scope := GetScope (TheSymbol);
         case CommandLineData.Content.Language_Profile is
            when CommandLineData.SPARK83 =>
               Result := Scope = Set_Visibility
                 (The_Visibility => Visible,
                  The_Unit       => RawDict.Get_Package_Symbol (Get_Predefined_Package_Standard))
                 or else Scope = Set_Visibility
                 (The_Visibility => Visible,
                  The_Unit       => RawDict.Get_Package_Symbol (Get_Predefined_Package_ASCII));
            when CommandLineData.SPARK95_Onwards =>
               Result :=
                 Scope = Set_Visibility
                 (The_Visibility => Visible,
                  The_Unit       => RawDict.Get_Package_Symbol (Get_Predefined_Package_Standard))
                 or else Scope = Set_Visibility
                 (The_Visibility => Visible,
                  The_Unit       => RawDict.Get_Package_Symbol (Get_Predefined_Package_ASCII))
                 or else Scope = Set_Visibility
                 (The_Visibility => Visible,
                  The_Unit       => RawDict.Get_Package_Symbol (Get_Predefined_Package_Ada))
                 or else Scope = Set_Visibility
                 (The_Visibility => Visible,
                  The_Unit       => RawDict.Get_Package_Symbol (Dict.Packages.Package_Ada_Characters))
                 or else Scope = Set_Visibility
                 (The_Visibility => Visible,
                  The_Unit       => RawDict.Get_Package_Symbol (Dict.Packages.Package_Ada_Characters_Latin1))
                 or else (CommandLineData.Ravenscar_Selected
                            and then (Scope = Set_Visibility
                                        (The_Visibility => Visible,
                                         The_Unit       => RawDict.Get_Package_Symbol (Dict.Packages.Package_Synchronous_Task_Control))
                                        or else Scope = Set_Visibility
                                        (The_Visibility => Visible,
                                         The_Unit       => RawDict.Get_Package_Symbol (Dict.Packages.Package_Real_Time))
                                        or else Scope = Set_Visibility
                                        (The_Visibility => Visible,
                                         The_Unit       => RawDict.Get_Package_Symbol (Dict.Packages.Package_Interrupts))));
         end case;
      end if;
      return Result;
   end IsPredefined;

   --------------------------------------------------------------------------------
   --  Operator_Is_Visible
   --------------------------------------------------------------------------------

   package body Operator_Is_Visible is separate;

   --------------------------------------------------------------------------------

   function UnaryOperatorIsVisible
     (Name    : SP_Symbols.SP_Symbol;
      Operand : Symbol;
      Scope   : Scopes)
     return    Boolean is
   begin
      return Operator_Is_Visible.Unary_Operator_Is_Visible
        (Name      => Name,
         Type_Mark => RawDict.Get_Type_Info_Ref (Operand), -- GAA External
         Scope     => Scope);
   end UnaryOperatorIsVisible;

   --------------------------------------------------------------------------------

   function BinaryOperatorIsVisible
     (Name  : SP_Symbols.SP_Symbol;
      Left  : Symbol;
      Right : Symbol;
      Scope : Scopes)
     return  Boolean is
   begin
      return Operator_Is_Visible.Binary_Operator_Is_Visible
        (Name           => Name,
         The_Left_Type  => RawDict.Get_Type_Info_Ref (Left), -- GAA External
         The_Right_Type => RawDict.Get_Type_Info_Ref (Right), -- GAA External
         Scope          => Scope);
   end BinaryOperatorIsVisible;

   --------------------------------------------------------------------------------
   --  Is_Scalar_Type_Mark
   --------------------------------------------------------------------------------

   function Is_Scalar_Type_Mark (Type_Mark : RawDict.Type_Info_Ref;
                                 Scope     : Scopes) return Boolean
   --# global in Dict;
   is
   begin
      return Type_Is_Scalar (Type_Mark => Type_Mark)
        and then not Type_Is_Private_Here (Type_Mark => Type_Mark,
                                           Scope     => Scope);
   end Is_Scalar_Type_Mark;

   --------------------------------------------------------------------------------

   function IsScalarTypeMark (TheSymbol : Symbol;
                              Scope     : Scopes) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Scalar_Type_Mark (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol), -- GAA External
                                      Scope     => Scope);
   end IsScalarTypeMark;

   --------------------------------------------------------------------------------
   --  Is_Discrete_Type_Mark
   --------------------------------------------------------------------------------

   function IsDiscreteTypeMark (TheSymbol : Symbol;
                                Scope     : Scopes) return Boolean is

      function Is_Discrete_Type_Mark (Type_Mark : RawDict.Type_Info_Ref;
                                      Scope     : Scopes) return Boolean
      --# global in Dict;
      is
      begin
         return Type_Is_Discrete (Type_Mark => Type_Mark)
           and then not Type_Is_Private_Here (Type_Mark => Type_Mark,
                                              Scope     => Scope);
      end Is_Discrete_Type_Mark;

   begin -- IsDiscreteTypeMark
      return RawDict.GetSymbolDiscriminant (TheSymbol) = Type_Symbol
        and then Is_Discrete_Type_Mark (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheSymbol), -- GAA External
                                        Scope     => Scope);
   end IsDiscreteTypeMark;

   --------------------------------------------------------------------------------
   --  Is_Constrained_Array_Type_Mark
   --------------------------------------------------------------------------------

   function Is_Constrained_Array_Type_Mark (TheSymbol : Symbol;
                                            Scope     : Scopes) return Boolean is

      function Is_Constrained_Array_Type_Mark_Local (Type_Mark : RawDict.Type_Info_Ref;
                                                     Scope     : Scopes) return Boolean
      --# global in Dict;
      is
      begin
         return Is_Array_Type_Mark (Type_Mark => Type_Mark,
                                    Scope     => Scope)
           and then RawDict.Get_Type_Constrained (Type_Mark => Type_Mark);
      end Is_Constrained_Array_Type_Mark_Local;

   begin -- Is_Constrained_Array_Type_Mark
      return Is_Constrained_Array_Type_Mark_Local (Type_Mark => RawDict.Get_Type_Info_Ref (TheSymbol), -- GAA External
                                                   Scope     => Scope);
   end Is_Constrained_Array_Type_Mark;

   --------------------------------------------------------------------------------
   --  Is_Unconstrained_Array_Type_Mark
   --------------------------------------------------------------------------------

   function Is_Unconstrained_Array_Type_Mark (TheSymbol : Symbol;
                                              Scope     : Scopes) return Boolean is

      function Is_Unconstrained_Array_Type_Mark_Local (Type_Mark : RawDict.Type_Info_Ref;
                                                       Scope     : Scopes) return Boolean
      --# global in Dict;
      is
      begin
         return Is_Array_Type_Mark (Type_Mark => Type_Mark,
                                    Scope     => Scope)
           and then not RawDict.Get_Type_Constrained (Type_Mark => Type_Mark);
      end Is_Unconstrained_Array_Type_Mark_Local;

   begin -- Is_Unconstrained_Array_Type_Mark
      return Is_Unconstrained_Array_Type_Mark_Local (Type_Mark => RawDict.Get_Type_Info_Ref (TheSymbol), -- GAA External
                                                     Scope     => Scope);
   end Is_Unconstrained_Array_Type_Mark;

   --------------------------------------------------------------------------------
   --  Attribute_Is_Visible
   --------------------------------------------------------------------------------

   function Attribute_Is_Visible
     (Name      : LexTokenManager.Lex_String;
      Prefix    : PrefixSort;
      Type_Mark : RawDict.Type_Info_Ref;
      Scope     : Scopes)
     return     Boolean
   --# global in CommandLineData.Content;
   --#        in Dict;
   --#        in LexTokenManager.State;
      is separate;

   --------------------------------------------------------------------------------

   function AttributeIsVisible
     (Name     : LexTokenManager.Lex_String;
      Prefix   : PrefixSort;
      TypeMark : Symbol;
      Scope    : Scopes)
     return     Boolean is
   begin
      return Attribute_Is_Visible
        (Name      => Name,
         Prefix    => Prefix,
         Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark), -- GAA External
         Scope     => Scope);
   end AttributeIsVisible;

   --------------------------------------------------------------------------------
   --  Attribute_Is_Visible_But_Obsolete
   --------------------------------------------------------------------------------

   function Attribute_Is_Visible_But_Obsolete_Local
     (Name      : LexTokenManager.Lex_String;
      Prefix    : PrefixSort;
      Type_Mark : RawDict.Type_Info_Ref;
      Scope     : Scopes)
     return     Boolean
   --# global in CommandLineData.Content;
   --#        in Dict;
   --#        in LexTokenManager.State;
      is separate;

   --------------------------------------------------------------------------------

   function Attribute_Is_Visible_But_Obsolete
     (Name     : LexTokenManager.Lex_String;
      Prefix   : PrefixSort;
      TypeMark : Symbol;
      Scope    : Scopes)
     return     Boolean is
   begin
      return Attribute_Is_Visible_But_Obsolete_Local
        (Name      => Name,
         Prefix    => Prefix,
         Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark), -- GAA External
         Scope     => Scope);
   end Attribute_Is_Visible_But_Obsolete;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   procedure Write_Location (File : in SPARK_IO.File_Type;
                             Loc  : in Location)
   --# global in     Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                File,
   --#                                Loc;
   is

      procedure Write_Token_Position (File : in SPARK_IO.File_Type;
                                      Pos  : in LexTokenManager.Token_Position)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                File,
      --#                                Pos;
      is

         procedure Write_Line_Number (File  : in SPARK_IO.File_Type;
                                      Value : in LexTokenManager.Line_Numbers)
         --# global in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                File,
         --#                                Value;
         is
         begin
            Write_Integer (File, Integer (Value));
         end Write_Line_Number;

      begin -- Write_Token_Position
         Write_String (File, "line ");
         Write_Line_Number (File  => File,
                            Value => Pos.Start_Line_No);
         Write_String (File, " column ");
         Write_Integer (File, Pos.Start_Pos);
      end Write_Token_Position;

   begin -- Write_Location
      E_Strings.Put_String (File  => File,
                            E_Str => Dict.File_Name);
      Write_Space (File => File);
      Write_Token_Position (File => File,
                            Pos  => Loc.Start_Position);
      Write_String (File, " to ");
      Write_Token_Position (File => File,
                            Pos  => Loc.End_Position);
   end Write_Location;

   --------------------------------------------------------------------------------

   procedure Write_Reference
     (Discriminant    : in ReferenceDiscriminant;
      Item            : in Symbol;
      CompilationUnit : in Symbol;
      Reference       : in Location)
   --# global in     Dict;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CompilationUnit,
   --#                                Dict,
   --#                                Discriminant,
   --#                                Item,
   --#                                LexTokenManager.State,
   --#                                Reference;
   is

      procedure Write_Discriminant (File         : in SPARK_IO.File_Type;
                                    Discriminant : in ReferenceDiscriminant)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Discriminant,
      --#                                File;
      is
      begin
         case Discriminant is
            when ReadRef =>
               Write_String (File, "read");
            when WriteRef =>
               Write_String (File, "write");
            when CallRef =>
               Write_String (File, "call");
            when OtherRef =>
               Write_String (File, "other");
         end case;
      end Write_Discriminant;

   begin -- Write_Reference
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         Write_Discriminant (File         => Dict.TemporaryFile,
                             Discriminant => Discriminant);
         Write_String (Dict.TemporaryFile, " reference to ");
         Write_Name (File => Dict.TemporaryFile,
                     Item => Item);
         Write_String (Dict.TemporaryFile, " in ");
         Write_Name (File => Dict.TemporaryFile,
                     Item => CompilationUnit);
         Write_String (Dict.TemporaryFile, " is at ");
         Write_Location (File => Dict.TemporaryFile,
                         Loc  => Reference);
         Write_Line (Dict.TemporaryFile, " ;");
      end if;
   end Write_Reference;

   --------------------------------------------------------------------------------

   procedure AddOtherReference (Item, CompilationUnit : in Symbol;
                                Reference             : in Location) is
   begin
      Write_Reference (Discriminant    => OtherRef,
                       Item            => Item,
                       CompilationUnit => CompilationUnit,
                       Reference       => Reference);
   end AddOtherReference;

   --------------------------------------------------------------------------------
   --  Write_Declaration
   --------------------------------------------------------------------------------

   procedure Write_Constant_Declaration
     (The_Constant : in RawDict.Constant_Info_Ref;
      Is_Deferred  : in Boolean;
      Declaration  : in Location)
   --# global in     Dict;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Declaration,
   --#                                Dict,
   --#                                Is_Deferred,
   --#                                LexTokenManager.State,
   --#                                The_Constant;
   is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         if Is_Deferred then
            Write_String (Dict.TemporaryFile, "deferred ");
         end if;
         Write_String (Dict.TemporaryFile, "constant declaration of ");
         Write_Name (File => Dict.TemporaryFile,
                     Item => RawDict.Get_Constant_Symbol (The_Constant));
         Write_String (Dict.TemporaryFile, " is at ");
         Write_Location (File => Dict.TemporaryFile,
                         Loc  => Declaration);
         Write_Line (Dict.TemporaryFile, " ;");
      end if;
   end Write_Constant_Declaration;

   --------------------------------------------------------------------------------

   procedure Write_Variable_Declaration (The_Variable : in RawDict.Variable_Info_Ref;
                                         Declaration  : in Location)
   --# global in     Dict;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                The_Variable;
   is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         Write_String (Dict.TemporaryFile, "variable declaration of ");
         Write_Name (File => Dict.TemporaryFile,
                     Item => RawDict.Get_Variable_Symbol (The_Variable));
         Write_String (Dict.TemporaryFile, " is at ");
         Write_Location (File => Dict.TemporaryFile,
                         Loc  => Declaration);
         Write_Line (Dict.TemporaryFile, " ;");
      end if;
   end Write_Variable_Declaration;

   --------------------------------------------------------------------------------

   procedure Write_Type_Declaration
     (Type_Mark    : in RawDict.Type_Info_Ref;
      Discriminant : in DeclarationDiscriminant;
      Declaration  : in Location)
   --# global in     Dict;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Declaration,
   --#                                Dict,
   --#                                Discriminant,
   --#                                LexTokenManager.State,
   --#                                Type_Mark;
   is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         case Discriminant is
            when TypeDeclaration =>
               Write_String (Dict.TemporaryFile, "type declaration of ");
            when TypeAnnouncement =>
               Write_String (Dict.TemporaryFile, "type announcement of ");
            when PrivateTypeDeclaration =>
               Write_String (Dict.TemporaryFile, "private type declaration of ");
         end case;
         Write_Name (File => Dict.TemporaryFile,
                     Item => RawDict.Get_Type_Symbol (Type_Mark));
         Write_String (Dict.TemporaryFile, " is at ");
         Write_Location (File => Dict.TemporaryFile,
                         Loc  => Declaration);
         Write_Line (Dict.TemporaryFile, " ;");
      end if;
   end Write_Type_Declaration;

   --------------------------------------------------------------------------------

   procedure Write_Subprogram_Declaration
     (The_Subprogram : in RawDict.Subprogram_Info_Ref;
      Is_Renaming    : in Boolean;
      Scope          : in Scopes;
      Declaration    : in Location)
   --# global in     Dict;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Declaration,
   --#                                Dict,
   --#                                Is_Renaming,
   --#                                LexTokenManager.State,
   --#                                Scope,
   --#                                The_Subprogram;
   is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         if Is_Renaming then
            Write_String (Dict.TemporaryFile, "subprogram ");
            Write_Name (File => Dict.TemporaryFile,
                        Item => RawDict.Get_Subprogram_Symbol (The_Subprogram));
            Write_String (Dict.TemporaryFile, " is renamed in ");
            Write_Scope (Dict.TemporaryFile, Scope);
         else
            Write_String (Dict.TemporaryFile, "subprogram specification of ");
            Write_Name (File => Dict.TemporaryFile,
                        Item => RawDict.Get_Subprogram_Symbol (The_Subprogram));
            Write_String (Dict.TemporaryFile, " is");
         end if;
         Write_String (Dict.TemporaryFile, " at ");
         Write_Location (File => Dict.TemporaryFile,
                         Loc  => Declaration);
         Write_Line (Dict.TemporaryFile, " ;");
      end if;
   end Write_Subprogram_Declaration;

   --------------------------------------------------------------------------------
   --  Add_Deferred_Constant
   --------------------------------------------------------------------------------

   procedure Add_Deferred_Constant_Local
     (Name           : in     LexTokenManager.Lex_String;
      Type_Mark      : in     RawDict.Type_Info_Ref;
      Type_Reference : in     Location;
      Comp_Unit      : in     ContextManager.UnitDescriptors;
      Declaration    : in     Location;
      The_Package    : in     RawDict.Package_Info_Ref;
      The_Constant   :    out RawDict.Constant_Info_Ref)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         The_Constant      from Comp_Unit,
   --#                                Declaration,
   --#                                Dict,
   --#                                Name,
   --#                                The_Package,
   --#                                Type_Mark &
   --#         SPARK_IO.File_Sys from *,
   --#                                Comp_Unit,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                The_Package,
   --#                                Type_Mark,
   --#                                Type_Reference;
   --# post The_Constant /= RawDict.Null_Constant_Info_Ref;
   is
      The_Declaration : RawDict.Declaration_Info_Ref;
   begin
      Add_Declaration
        (Comp_Unit       => Comp_Unit,
         Loc             => Declaration,
         Scope           => Set_Visibility
           (The_Visibility => Visible,
            The_Unit       => RawDict.Get_Package_Symbol (The_Package)),
         Context         => ProgramContext,
         The_Declaration => The_Declaration);
      RawDict.Create_Constant
        (Name            => Name,
         Type_Mark       => Type_Mark,
         Static          => False,
         The_Declaration => The_Declaration,
         Is_Deferred     => True,
         Comp_Unit       => Comp_Unit,
         Loc             => Declaration.Start_Position,
         The_Constant    => The_Constant);
      if Type_Mark /= Get_Unknown_Type_Mark then
         AddOtherReference (RawDict.Get_Type_Symbol (Type_Mark),
                            RawDict.Get_Package_Symbol (The_Package),
                            Type_Reference);
      end if;
      Write_Constant_Declaration (The_Constant => The_Constant,
                                  Is_Deferred  => True,
                                  Declaration  => Declaration);
   end Add_Deferred_Constant_Local;

   --------------------------------------------------------------------------------

   procedure Add_Deferred_Constant
     (Name           : in     LexTokenManager.Lex_String;
      Type_Mark      : in     Symbol;
      Type_Reference : in     Location;
      Comp_Unit      : in     ContextManager.UnitDescriptors;
      Declaration    : in     Location;
      The_Package    : in     Symbol;
      TheConstant    :    out Symbol) is
      The_Constant : RawDict.Constant_Info_Ref;
   begin
      Add_Deferred_Constant_Local
        (Name           => Name,
         Type_Mark      => RawDict.Get_Type_Info_Ref (Type_Mark), -- GAA External
         Type_Reference => Type_Reference,
         Comp_Unit      => Comp_Unit,
         Declaration    => Declaration,
         The_Package    => RawDict.Get_Package_Info_Ref (The_Package), -- GAA External
         The_Constant   => The_Constant);
      TheConstant := RawDict.Get_Constant_Symbol (The_Constant); -- GAA External
   end Add_Deferred_Constant;

   --------------------------------------------------------------------------------
   --  Add_Constant
   --------------------------------------------------------------------------------

   procedure Add_Constant_Local (Name              : in     LexTokenManager.Lex_String;
                                 Type_Mark         : in     RawDict.Type_Info_Ref;
                                 Static            : in     Boolean;
                                 The_Declaration   : in     RawDict.Declaration_Info_Ref;
                                 Comp_Unit         : in     ContextManager.UnitDescriptors;
                                 Declaration       : in     Location;
                                 Value             : in     LexTokenManager.Lex_String;
                                 Exp_Is_Wellformed : in     Boolean;
                                 Exp_Node          : in     ExaminerConstants.RefType;
                                 The_Constant      : in out RawDict.Constant_Info_Ref)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Comp_Unit,
   --#                                Declaration,
   --#                                Exp_Is_Wellformed,
   --#                                Exp_Node,
   --#                                Name,
   --#                                Static,
   --#                                The_Constant,
   --#                                The_Declaration,
   --#                                Type_Mark,
   --#                                Value &
   --#         SPARK_IO.File_Sys from *,
   --#                                Comp_Unit,
   --#                                Declaration,
   --#                                Dict,
   --#                                Exp_Is_Wellformed,
   --#                                Exp_Node,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Static,
   --#                                The_Constant,
   --#                                The_Declaration,
   --#                                Type_Mark,
   --#                                Value &
   --#         The_Constant      from *,
   --#                                Comp_Unit,
   --#                                Declaration,
   --#                                Dict,
   --#                                Name,
   --#                                Static,
   --#                                The_Declaration,
   --#                                Type_Mark;
   --# post The_Constant /= RawDict.Null_Constant_Info_Ref;
   is
   begin
      if The_Constant = RawDict.Null_Constant_Info_Ref then
         RawDict.Create_Constant
           (Name            => Name,
            Type_Mark       => Type_Mark,
            Static          => Static,
            The_Declaration => The_Declaration,
            Is_Deferred     => False,
            Comp_Unit       => Comp_Unit,
            Loc             => Declaration.Start_Position,
            The_Constant    => The_Constant);
      else
         RawDict.Set_Constant_Static (The_Constant => The_Constant,
                                      Static       => Static);
         RawDict.Set_Constant_Declaration (The_Constant    => The_Constant,
                                           The_Declaration => The_Declaration);
      end if;
      RawDict.Set_Constant_Value (The_Constant => The_Constant,
                                  Value        => Value);
      RawDict.Set_Constant_Exp_Node (The_Constant      => The_Constant,
                                     Exp_Is_Wellformed => Exp_Is_Wellformed,
                                     Exp_Node          => Exp_Node);
      Write_Constant_Declaration (The_Constant => The_Constant,
                                  Is_Deferred  => False,
                                  Declaration  => Declaration);
   end Add_Constant_Local;

   --------------------------------------------------------------------------------

   procedure Add_Constant (Name              : in     LexTokenManager.Lex_String;
                           The_Type          : in     Symbol;
                           Static            : in     Boolean;
                           Comp_Unit         : in     ContextManager.UnitDescriptors;
                           Declaration       : in     Location;
                           Value             : in     LexTokenManager.Lex_String;
                           Exp_Is_Wellformed : in     Boolean;
                           Exp_Node          : in     ExaminerConstants.RefType;
                           Constant_Sym      : in out Symbol) is
      The_Constant : RawDict.Constant_Info_Ref;
   begin
      The_Constant := RawDict.Get_Constant_Info_Ref (Constant_Sym); -- GAA External
      Add_Constant_Local
        (Name              => Name,
         Type_Mark         => RawDict.Get_Type_Info_Ref (The_Type), -- GAA External
         Static            => Static,
         The_Declaration   => RawDict.Null_Declaration_Info_Ref,
         Comp_Unit         => Comp_Unit,
         Declaration       => Declaration,
         Value             => Value,
         Exp_Is_Wellformed => Exp_Is_Wellformed,
         Exp_Node          => Exp_Node,
         The_Constant      => The_Constant);
      Constant_Sym := RawDict.Get_Constant_Symbol (The_Constant); -- GAA External
   end Add_Constant;

   --------------------------------------------------------------------------------
   --  Add_Constant_Declaration
   --------------------------------------------------------------------------------

   procedure Add_Constant_Declaration
     (Name              : in     LexTokenManager.Lex_String;
      Type_Mark         : in     Symbol;
      Type_Reference    : in     Location;
      Value             : in     LexTokenManager.Lex_String;
      Exp_Is_Wellformed : in     Boolean;
      Exp_Node          : in     ExaminerConstants.RefType;
      Static            : in     Boolean;
      Comp_Unit         : in     ContextManager.UnitDescriptors;
      Declaration       : in     Location;
      Scope             : in     Scopes;
      Context           : in     Contexts;
      TheConstant       :    out Symbol) is
      The_Constant : RawDict.Constant_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Constant_Declaration_Local
        (Name              : in     LexTokenManager.Lex_String;
         Type_Mark         : in     RawDict.Type_Info_Ref;
         Type_Reference    : in     Location;
         Value             : in     LexTokenManager.Lex_String;
         Exp_Is_Wellformed : in     Boolean;
         Exp_Node          : in     ExaminerConstants.RefType;
         Static            : in     Boolean;
         Comp_Unit         : in     ContextManager.UnitDescriptors;
         Declaration       : in     Location;
         Scope             : in     Scopes;
         Context           : in     Contexts;
         The_Constant      :    out RawDict.Constant_Info_Ref)
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                CommandLineData.Content,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Exp_Is_Wellformed,
      --#                                Exp_Node,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope,
      --#                                Static,
      --#                                Type_Mark,
      --#                                Value &
      --#         SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                Exp_Is_Wellformed,
      --#                                Exp_Node,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope,
      --#                                Static,
      --#                                Type_Mark,
      --#                                Type_Reference,
      --#                                Value &
      --#         The_Constant      from CommandLineData.Content,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope,
      --#                                Static,
      --#                                Type_Mark;
      --# post The_Constant /= RawDict.Null_Constant_Info_Ref;
      is
         Region          : Symbol;
         The_Declaration : RawDict.Declaration_Info_Ref;
      begin
         Region := GetRegion (Scope);
         if (Type_Is_Private (Type_Mark => Get_Root_Type (Type_Mark => Type_Mark))
               or else CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83)
           and then Get_Visibility (Scope => Scope) = Privat
           and then RawDict.GetSymbolDiscriminant (Region) = Package_Symbol
           and then Context = ProgramContext then
            The_Constant := RawDict.Get_Constant_Info_Ref
              (LookupImmediateScope (Name, Set_Visibility (The_Visibility => Visible,
                                                           The_Unit       => Region), Context));
         else
            The_Constant := RawDict.Null_Constant_Info_Ref;
         end if;

         Add_Declaration
           (Comp_Unit       => Comp_Unit,
            Loc             => Declaration,
            Scope           => Scope,
            Context         => Context,
            The_Declaration => The_Declaration);

         Add_Constant_Local
           (Name              => Name,
            Type_Mark         => Type_Mark,
            Static            => Static,
            The_Declaration   => The_Declaration,
            Comp_Unit         => Comp_Unit,
            Declaration       => Declaration,
            Value             => Value,
            Exp_Is_Wellformed => Exp_Is_Wellformed,
            Exp_Node          => Exp_Node,
            The_Constant      => The_Constant);

         if Type_Mark /= Get_Unknown_Type_Mark then
            AddOtherReference (RawDict.Get_Type_Symbol (Type_Mark), Region, Type_Reference);
         end if;
      end Add_Constant_Declaration_Local;

   begin -- Add_Constant_Declaration
      Add_Constant_Declaration_Local
        (Name              => Name,
         Type_Mark         => RawDict.Get_Type_Info_Ref (Type_Mark), -- GAA External
         Type_Reference    => Type_Reference,
         Value             => Value,
         Exp_Is_Wellformed => Exp_Is_Wellformed,
         Exp_Node          => Exp_Node,
         Static            => Static,
         Comp_Unit         => Comp_Unit,
         Declaration       => Declaration,
         Scope             => Scope,
         Context           => Context,
         The_Constant      => The_Constant);
      TheConstant := RawDict.Get_Constant_Symbol (The_Constant); -- GAA External
   end Add_Constant_Declaration;

   --------------------------------------------------------------------------------
   --  Promote_Deferred_To_Full_Constant
   --------------------------------------------------------------------------------

   -- This procedure is only use in 1 special case - to promote the
   -- standard declaration of System.Default_Bit_Order from a deferred
   -- constant to a normal constant in the case where the full declaration
   -- is supplied in the configuration file.  (Some compilers, particularly GNAT,
   -- do give such a full constant declaration)
   procedure Promote_Deferred_To_Full_Constant
     (Constant_Sym : in Symbol;
      Comp_Unit    : in ContextManager.UnitDescriptors;
      Declaration  : in Location;
      Value        : in LexTokenManager.Lex_String;
      Exp_Node     : in ExaminerConstants.RefType;
      The_Package  : in Symbol) is

      procedure Promote_Deferred_To_Full_Constant_Local
        (The_Constant : in RawDict.Constant_Info_Ref;
         Comp_Unit    : in ContextManager.UnitDescriptors;
         Declaration  : in Location;
         Value        : in LexTokenManager.Lex_String;
         Exp_Node     : in ExaminerConstants.RefType;
         The_Package  : in Symbol)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   Comp_Unit,
      --#                   Declaration,
      --#                   Exp_Node,
      --#                   The_Constant,
      --#                   The_Package,
      --#                   Value;
      is
         The_Declaration : RawDict.Declaration_Info_Ref;
      begin
         Add_Declaration
           (Comp_Unit       => Comp_Unit,
            Loc             => Declaration,
            Scope           => Set_Visibility (The_Visibility => Visible,
                                               The_Unit       => The_Package),
            Context         => ProgramContext,
            The_Declaration => The_Declaration);

         RawDict.Set_Declaration_Item (The_Declaration => The_Declaration,
                                       Item            => RawDict.Get_Constant_Symbol (The_Constant));
         RawDict.Set_Constant_Value (The_Constant => The_Constant,
                                     Value        => Value);
         RawDict.Set_Constant_Exp_Node (The_Constant      => The_Constant,
                                        Exp_Is_Wellformed => True,
                                        Exp_Node          => Exp_Node);
         RawDict.Set_Constant_Static (The_Constant => The_Constant,
                                      Static       => True);
         RawDict.Set_Constant_Declaration (The_Constant    => The_Constant,
                                           The_Declaration => The_Declaration);
         RawDict.Set_Constant_Deferred_Declaration
           (The_Constant    => The_Constant,
            The_Declaration => RawDict.Null_Declaration_Info_Ref);
      end Promote_Deferred_To_Full_Constant_Local;

   begin -- Promote_Deferred_To_Full_Constant
      Promote_Deferred_To_Full_Constant_Local
        (The_Constant => RawDict.Get_Constant_Info_Ref (Constant_Sym), -- GAA External
         Comp_Unit    => Comp_Unit,
         Declaration  => Declaration,
         Value        => Value,
         Exp_Node     => Exp_Node,
         The_Package  => The_Package);
   end Promote_Deferred_To_Full_Constant;

   --------------------------------------------------------------------------------
   --  Add_Constant_Rule_Policy
   --------------------------------------------------------------------------------

   procedure AddConstantRulePolicy
     (TheConstant   : in     Symbol;
      Comp_Unit     : in     ContextManager.UnitDescriptors;
      Declaration   : in     Location;
      TheScope      : in     Scopes;
      ThePolicy     : in     Rule_Policies;
      TheRulePolicy :    out Symbol) is
      The_Rule_Policy : RawDict.Rule_Policy_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Constant_Rule_Policy
        (The_Constant    : in     RawDict.Constant_Info_Ref;
         Comp_Unit       : in     ContextManager.UnitDescriptors;
         Declaration     : in     Location;
         Scope           : in     Scopes;
         The_Policy      : in     Rule_Policies;
         The_Rule_Policy :    out RawDict.Rule_Policy_Info_Ref)
      --# global in out Dict;
      --# derives Dict            from *,
      --#                              Comp_Unit,
      --#                              Declaration,
      --#                              Scope,
      --#                              The_Constant,
      --#                              The_Policy &
      --#         The_Rule_Policy from Comp_Unit,
      --#                              Declaration,
      --#                              Dict,
      --#                              Scope,
      --#                              The_Policy;
      --# post The_Rule_Policy /= RawDict.Null_Rule_Policy_Info_Ref;
      is
      begin
         RawDict.Create_Rule_Policy
           (Scope           => Scope,
            Value           => The_Policy,
            Comp_Unit       => Comp_Unit,
            Loc             => Declaration.Start_Position,
            The_Rule_Policy => The_Rule_Policy);

         if RawDict.Get_Constant_First_Rule_Policy (The_Constant => The_Constant) = RawDict.Null_Rule_Policy_Info_Ref then
            RawDict.Set_Constant_First_Rule_Policy (The_Constant    => The_Constant,
                                                    The_Rule_Policy => The_Rule_Policy);
         else
            RawDict.Set_Next_Rule_Policy
              (The_Rule_Policy => RawDict.Get_Constant_Last_Rule_Policy (The_Constant => The_Constant),
               Next            => The_Rule_Policy);
         end if;
         RawDict.Set_Constant_Last_Rule_Policy (The_Constant    => The_Constant,
                                                The_Rule_Policy => The_Rule_Policy);
      end Add_Constant_Rule_Policy;

   begin -- AddConstantRulePolicy
      Add_Constant_Rule_Policy (The_Constant    => RawDict.Get_Constant_Info_Ref (TheConstant), -- GAA External
                                Comp_Unit       => Comp_Unit,
                                Declaration     => Declaration,
                                Scope           => TheScope,
                                The_Policy      => ThePolicy,
                                The_Rule_Policy => The_Rule_Policy);
      TheRulePolicy := RawDict.Get_Rule_Policy_Symbol (The_Rule_Policy); -- GAA External
   end AddConstantRulePolicy;

   -------------------------------------------------------------------------------
   --  Add_Variable_Declaration
   -------------------------------------------------------------------------------

   procedure Add_Variable_Declaration_Local
     (The_Variable    : in     RawDict.Variable_Info_Ref;
      Type_Mark       : in     RawDict.Type_Info_Ref;
      Initialized     : in     Boolean;
      Is_Aliased      : in     Boolean;
      Exp_Node        : in     ExaminerConstants.RefType;
      Type_Reference  : in     Location;
      Comp_Unit       : in     ContextManager.UnitDescriptors;
      Declaration     : in     Location;
      Scope           : in     Scopes;
      The_Declaration :    out RawDict.Declaration_Info_Ref)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Comp_Unit,
   --#                                Declaration,
   --#                                Exp_Node,
   --#                                Initialized,
   --#                                Is_Aliased,
   --#                                Scope,
   --#                                The_Variable,
   --#                                Type_Mark &
   --#         SPARK_IO.File_Sys from *,
   --#                                Comp_Unit,
   --#                                Declaration,
   --#                                Dict,
   --#                                Exp_Node,
   --#                                Initialized,
   --#                                Is_Aliased,
   --#                                LexTokenManager.State,
   --#                                Scope,
   --#                                The_Variable,
   --#                                Type_Mark,
   --#                                Type_Reference &
   --#         The_Declaration   from Comp_Unit,
   --#                                Declaration,
   --#                                Dict,
   --#                                Scope,
   --#                                The_Variable;
   --# post The_Declaration /= RawDict.Null_Declaration_Info_Ref;
   is
   begin
      if RawDict.Get_Symbol_Compilation_Unit (Item => RawDict.Get_Variable_Symbol (The_Variable)) = Comp_Unit then
         --  There was a usage of the variable before its actual
         --  declaration -> update the declaration position. The
         --  declaration must be in the same compilation unit as the
         --  usage (in the package specification) to prevent to get a
         --  cross-reference from the package specification to the
         --  package body.
         RawDict.Set_Symbol_Location (Item     => RawDict.Get_Variable_Symbol (The_Variable),
                                      Location => Declaration.Start_Position);
      end if;

      Add_Declaration
        (Comp_Unit       => Comp_Unit,
         Loc             => Declaration,
         Scope           => Scope,
         Context         => ProgramContext,
         The_Declaration => The_Declaration);

      RawDict.Set_Variable_Declaration (The_Variable    => The_Variable,
                                        The_Declaration => The_Declaration);
      RawDict.Set_Variable_Exp_Node (The_Variable => The_Variable,
                                     Exp_Node     => Exp_Node);
      RawDict.Set_Variable_Type (The_Variable => The_Variable,
                                 Type_Mark    => Type_Mark);

      if Is_Aliased then
         RawDict.Set_Variable_Is_Aliased (The_Variable => The_Variable);
      end if;

      if Initialized then
         RawDict.Set_Variable_Initialized (The_Variable => The_Variable);
      end if;

      if Type_Mark /= Get_Unknown_Type_Mark then
         AddOtherReference (RawDict.Get_Type_Symbol (Type_Mark), GetRegion (Scope), Type_Reference);
      end if;

      Write_Variable_Declaration (The_Variable => The_Variable,
                                  Declaration  => Declaration);
   end Add_Variable_Declaration_Local;

   -------------------------------------------------------------------------------

   procedure Add_Variable_Declaration
     (Variable_Sym       : in     Symbol;
      The_Type           : in     Symbol;
      Initialized        : in     Boolean;
      Is_Aliased         : in     Boolean;
      Exp_Node           : in     ExaminerConstants.RefType;
      Type_Reference     : in     Location;
      Comp_Unit          : in     ContextManager.UnitDescriptors;
      Declaration        : in     Location;
      Scope              : in     Scopes;
      Declaration_Symbol :    out Symbol) is
      The_Declaration : RawDict.Declaration_Info_Ref;
   begin
      Add_Variable_Declaration_Local
        (The_Variable    => RawDict.Get_Variable_Info_Ref (Variable_Sym), -- GAA External
         Type_Mark       => RawDict.Get_Type_Info_Ref (The_Type), -- GAA External
         Initialized     => Initialized,
         Is_Aliased      => Is_Aliased,
         Exp_Node        => Exp_Node,
         Type_Reference  => Type_Reference,
         Comp_Unit       => Comp_Unit,
         Declaration     => Declaration,
         Scope           => Scope,
         The_Declaration => The_Declaration);
      Declaration_Symbol := RawDict.Get_Declaration_Symbol (The_Declaration); -- GAA External
   end Add_Variable_Declaration;

   -------------------------------------------------------------------------------
   --  Add_Variable
   -------------------------------------------------------------------------------

   procedure Add_Variable
     (Name               : in     LexTokenManager.Lex_String;
      The_Type           : in     Symbol;
      Initialized        : in     Boolean;
      Is_Aliased         : in     Boolean;
      Exp_Node           : in     ExaminerConstants.RefType;
      Type_Reference     : in     Location;
      Comp_Unit          : in     ContextManager.UnitDescriptors;
      Declaration        : in     Location;
      Scope              : in     Scopes;
      Declaration_Symbol :    out Symbol;
      Variable_Symbol    :    out Symbol) is
      The_Declaration : RawDict.Declaration_Info_Ref;
      The_Variable    : RawDict.Variable_Info_Ref;

      -------------------------------------------------------------------------------

      procedure Add_Variable_Local
        (Name            : in     LexTokenManager.Lex_String;
         Type_Mark       : in     RawDict.Type_Info_Ref;
         Initialized     : in     Boolean;
         Is_Aliased      : in     Boolean;
         Exp_Node        : in     ExaminerConstants.RefType;
         Type_Reference  : in     Location;
         Comp_Unit       : in     ContextManager.UnitDescriptors;
         Declaration     : in     Location;
         Scope           : in     Scopes;
         The_Declaration :    out RawDict.Declaration_Info_Ref;
         The_Variable    :    out RawDict.Variable_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Declaration,
      --#                                Exp_Node,
      --#                                Initialized,
      --#                                Is_Aliased,
      --#                                Name,
      --#                                Scope,
      --#                                Type_Mark &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Declaration,
      --#                                Dict,
      --#                                Exp_Node,
      --#                                Initialized,
      --#                                Is_Aliased,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope,
      --#                                Type_Mark,
      --#                                Type_Reference &
      --#         The_Declaration   from Comp_Unit,
      --#                                Declaration,
      --#                                Dict,
      --#                                Name,
      --#                                Scope &
      --#         The_Variable      from Comp_Unit,
      --#                                Declaration,
      --#                                Dict,
      --#                                Name;
      --# post The_Declaration /= RawDict.Null_Declaration_Info_Ref and
      --#   The_Variable /= RawDict.Null_Variable_Info_Ref;
      is
      begin
         RawDict.Create_Variable
           (Name         => Name,
            Comp_Unit    => Comp_Unit,
            Loc          => Declaration.Start_Position,
            The_Variable => The_Variable);

         Add_Variable_Declaration_Local
           (The_Variable    => The_Variable,
            Type_Mark       => Type_Mark,
            Initialized     => Initialized,
            Is_Aliased      => Is_Aliased,
            Exp_Node        => Exp_Node,
            Type_Reference  => Type_Reference,
            Comp_Unit       => Comp_Unit,
            Declaration     => Declaration,
            Scope           => Scope,
            The_Declaration => The_Declaration);
      end Add_Variable_Local;

   begin -- Add_Variable
      Add_Variable_Local
        (Name            => Name,
         Type_Mark       => RawDict.Get_Type_Info_Ref (The_Type), -- GAA External
         Initialized     => Initialized,
         Is_Aliased      => Is_Aliased,
         Exp_Node        => Exp_Node,
         Type_Reference  => Type_Reference,
         Comp_Unit       => Comp_Unit,
         Declaration     => Declaration,
         Scope           => Scope,
         The_Declaration => The_Declaration,
         The_Variable    => The_Variable);
      Declaration_Symbol := RawDict.Get_Declaration_Symbol (The_Declaration); -- GAA External
      Variable_Symbol    := RawDict.Get_Variable_Symbol (The_Variable); -- GAA External
   end Add_Variable;

   -------------------------------------------------------------------------------
   -------------------------------------------------------------------------------

   procedure AddVariableAddressClause (Variable : in Symbol) is
   begin
      RawDict.Set_Variable_Has_Address_Clause (The_Variable => RawDict.Get_Variable_Info_Ref (Variable)); -- GAA External
   end AddVariableAddressClause;

   -------------------------------------------------------------------------------
   --  Add_Variable_Pragma_Import
   -------------------------------------------------------------------------------

   procedure AddVariablePragmaImport (Variable : in Symbol) is

      procedure Add_Variable_Pragma_Import (The_Variable : in RawDict.Variable_Info_Ref)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   The_Variable;
      is
      begin
         RawDict.Set_Variable_Has_Pragma_Import (The_Variable => The_Variable);
         -- Ada LRM states that variable with pragma import is deemed to be initialized by external code segment
         RawDict.Set_Variable_Initialized (The_Variable => The_Variable);  -- so we can mark it as initialized
      end Add_Variable_Pragma_Import;

   begin -- AddVariablePragmaImport
      Add_Variable_Pragma_Import (The_Variable => RawDict.Get_Variable_Info_Ref (Variable)); -- GAA External
   end AddVariablePragmaImport;

   -------------------------------------------------------------------------------
   -------------------------------------------------------------------------------

   procedure AddTypeSizeAttribute (TypeMark : in Symbol;
                                   SizeVal  : in LexTokenManager.Lex_String) is
   begin
      RawDict.Set_Type_Size_Attribute (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark), -- GAA External
                                       Size_Val  => SizeVal);
   end AddTypeSizeAttribute;

   -------------------------------------------------------------------------------

   procedure SetVariableMarkedValid (Variable : in Symbol;
                                     Val      : in Boolean) is
   begin
      RawDict.Set_Variable_Marked_Valid (The_Variable => RawDict.Get_Variable_Info_Ref (Variable), -- GAA External
                                         Val          => Val);
   end SetVariableMarkedValid;

   -------------------------------------------------------------------------------

   procedure SetSubcomponentMarkedValid (Subcomponent : in Symbol;
                                         Val          : in Boolean) is
   begin
      RawDict.Set_Subcomponent_Marked_Valid
        (The_Subcomponent => RawDict.Get_Subcomponent_Info_Ref (Subcomponent), -- GAA External
         Marked_Valid     => Val);
   end SetSubcomponentMarkedValid;

   -------------------------------------------------------------------------------

   procedure SetTypeAtomic (TypeMark : in Symbol) is
   begin
      RawDict.Set_Type_Atomic (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark)); -- GAA External
   end SetTypeAtomic;

   -------------------------------------------------------------------------------
   --  Add_Record_Subcomponent
   -------------------------------------------------------------------------------

   procedure Add_Record_Subcomponent
     (Prefix               : in     Symbol;
      The_Record_Component : in     RawDict.Record_Component_Info_Ref;
      Comp_Unit            : in     ContextManager.UnitDescriptors;
      The_Subcomponent     :    out RawDict.Subcomponent_Info_Ref)
   --# global in out Dict;
   --# derives Dict,
   --#         The_Subcomponent from Comp_Unit,
   --#                               Dict,
   --#                               Prefix,
   --#                               The_Record_Component;
   --# post The_Subcomponent /= RawDict.Null_Subcomponent_Info_Ref;
      is separate;

   -------------------------------------------------------------------------------

   procedure AddRecordSubcomponent
     (Prefix       : in     Symbol;
      Component    : in     Symbol;
      Comp_Unit    : in     ContextManager.UnitDescriptors;
      Subcomponent :    out Symbol) is
      The_Subcomponent : RawDict.Subcomponent_Info_Ref;
   begin
      Add_Record_Subcomponent (Prefix               => Prefix,
                               The_Record_Component => RawDict.Get_Record_Component_Info_Ref (Component), -- GAA External
                               Comp_Unit            => Comp_Unit,
                               The_Subcomponent     => The_Subcomponent);
      Subcomponent := RawDict.Get_Subcomponent_Symbol (The_Subcomponent); -- GAA External
   end AddRecordSubcomponent;

   -------------------------------------------------------------------------------
   --  Add_Quantified_Variable
   -------------------------------------------------------------------------------

   procedure AddQuantifiedVariable
     (Name          : in     LexTokenManager.Lex_String;
      Comp_Unit     : in     ContextManager.UnitDescriptors;
      Declaration   : in     Location;
      TypeMark      : in     Symbol;
      TheConstraint : in     Symbol;
      Region        : in     Symbol;
      Variable      :    out Symbol) is
      The_Quantified_Variable  : RawDict.Quantified_Variable_Info_Ref;

      -------------------------------------------------------------------------------

      procedure Add_Quantified_Variable
        (Name                     : in     LexTokenManager.Lex_String;
         Comp_Unit                : in     ContextManager.UnitDescriptors;
         Declaration              : in     Location;
         Type_Mark                : in     RawDict.Type_Info_Ref;
         The_Parameter_Constraint : in     RawDict.Parameter_Constraint_Info_Ref;
         Region                   : in     Symbol;
         The_Quantified_Variable  :    out RawDict.Quantified_Variable_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict,
      --#         The_Quantified_Variable from Comp_Unit,
      --#                                      Declaration,
      --#                                      Dict,
      --#                                      Name,
      --#                                      Region,
      --#                                      The_Parameter_Constraint,
      --#                                      Type_Mark &
      --#         SPARK_IO.File_Sys       from *,
      --#                                      Comp_Unit,
      --#                                      Declaration,
      --#                                      Dict,
      --#                                      LexTokenManager.State,
      --#                                      Name,
      --#                                      Region,
      --#                                      The_Parameter_Constraint,
      --#                                      Type_Mark;
      --# post The_Quantified_Variable /= RawDict.Null_Quantified_Variable_Info_Ref;
      is
      begin
         RawDict.Create_Quantified_Variable
           (Name                     => Name,
            Type_Mark                => Type_Mark,
            The_Parameter_Constraint => The_Parameter_Constraint,
            Region                   => Region,
            Comp_Unit                => Comp_Unit,
            Loc                      => Declaration.Start_Position,
            The_Quantified_Variable  => The_Quantified_Variable);
         if Type_Mark /= RawDict.Null_Type_Info_Ref then
            if Type_Mark /= Get_Unknown_Type_Mark then
               AddOtherReference (RawDict.Get_Type_Symbol (Type_Mark), Region, Declaration);
            end if;
         elsif The_Parameter_Constraint /= RawDict.Null_Parameter_Constraint_Info_Ref then
            AddOtherReference (RawDict.Get_Parameter_Constraint_Symbol (The_Parameter_Constraint), Region, Declaration);
         end if;
      end Add_Quantified_Variable;

   begin -- AddQuantifiedVariable
      Add_Quantified_Variable
        (Name                     => Name,
         Comp_Unit                => Comp_Unit,
         Declaration              => Declaration,
         Type_Mark                => RawDict.Get_Type_Info_Ref (TypeMark), -- GAA External
         The_Parameter_Constraint => RawDict.Get_Parameter_Constraint_Info_Ref (TheConstraint), -- GAA External
         Region                   => Region,
         The_Quantified_Variable  => The_Quantified_Variable);
      Variable := RawDict.Get_Quantified_Variable_Symbol (The_Quantified_Variable); -- GAA External
   end AddQuantifiedVariable;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   procedure Add_Record_Type_Mark
     (Type_Mark      : in RawDict.Type_Info_Ref;
      Is_Tagged_Type : in Boolean;
      Extends        : in RawDict.Type_Info_Ref)
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Extends,
   --#                   Is_Tagged_Type,
   --#                   Type_Mark;
   is
   begin
      RawDict.Set_Type_Discriminant (Type_Mark    => Type_Mark,
                                     Discriminant => Record_Type_Item);
      RawDict.Set_Type_Static (Type_Mark => Type_Mark,
                               Static    => False);
      RawDict.Set_Type_Is_Tagged (Type_Mark => Type_Mark,
                                  Is_Tagged => Is_Tagged_Type);
      RawDict.Set_Type_Extends (Type_Mark => Type_Mark,
                                Root_Type => Extends);
   end Add_Record_Type_Mark;

   --------------------------------------------------------------------------------
   --  Add_Type_Announcement
   --------------------------------------------------------------------------------

   procedure Add_Type_Announcement
     (Name        : in     LexTokenManager.Lex_String;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      The_Package : in     Symbol;
      The_Type    :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Type_Announcement_Local
        (Name        : in     LexTokenManager.Lex_String;
         Comp_Unit   : in     ContextManager.UnitDescriptors;
         Declaration : in     Location;
         The_Package : in     Symbol;
         Type_Mark   :    out RawDict.Type_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict,
      --#         Type_Mark         from Comp_Unit,
      --#                                Declaration,
      --#                                Dict,
      --#                                Name,
      --#                                The_Package &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                The_Package;
      --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
      is
         The_Declaration : RawDict.Declaration_Info_Ref;
      begin
         Add_Declaration
           (Comp_Unit       => Comp_Unit,
            Loc             => Declaration,
            Scope           => Set_Visibility (The_Visibility => Visible,
                                               The_Unit       => The_Package),
            Context         => ProofContext,
            The_Declaration => The_Declaration);
         RawDict.Create_Type
           (Name            => Name,
            The_Declaration => The_Declaration,
            Is_Private      => False,
            Is_Announcement => True,
            Comp_Unit       => Comp_Unit,
            Loc             => Declaration.Start_Position,
            Type_Mark       => Type_Mark);
         Write_Type_Declaration (Type_Mark    => Type_Mark,
                                 Discriminant => TypeAnnouncement,
                                 Declaration  => Declaration);
      end Add_Type_Announcement_Local;

   begin -- Add_Type_Announcement
      Add_Type_Announcement_Local
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Declaration,
         The_Package => The_Package,
         Type_Mark   => Type_Mark);
      The_Type := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Type_Announcement;

   --------------------------------------------------------------------------------
   --  Add_Private_Type
   --------------------------------------------------------------------------------

   procedure Add_Private_Type_Local
     (Name           : in     LexTokenManager.Lex_String;
      Comp_Unit      : in     ContextManager.UnitDescriptors;
      Declaration    : in     Location;
      The_Package    : in     RawDict.Package_Info_Ref;
      Is_Limited     : in     Boolean;
      Is_Tagged_Type : in     Boolean;
      Extends        : in     RawDict.Type_Info_Ref;
      Type_Mark      :    out RawDict.Type_Info_Ref)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys from *,
   --#                                Comp_Unit,
   --#                                Declaration,
   --#                                Dict,
   --#                                Extends,
   --#                                Is_Limited,
   --#                                Is_Tagged_Type,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                The_Package &
   --#         Type_Mark         from Comp_Unit,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                The_Package;
   --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
   is
      The_Declaration : RawDict.Declaration_Info_Ref;
   begin
      Type_Mark := RawDict.Get_Type_Info_Ref
        (LookupImmediateScope
           (Name,
            Set_Visibility (The_Visibility => Visible,
                            The_Unit       => RawDict.Get_Package_Symbol (The_Package)),
            ProofContext));
      Add_Declaration
        (Comp_Unit       => Comp_Unit,
         Loc             => Declaration,
         Scope           => Set_Visibility
           (The_Visibility => Visible,
            The_Unit       => RawDict.Get_Package_Symbol (The_Package)),
         Context         => ProgramContext,
         The_Declaration => The_Declaration);

      if Type_Mark = RawDict.Null_Type_Info_Ref then
         RawDict.Create_Type
           (Name            => Name,
            The_Declaration => The_Declaration,
            Is_Private      => True,
            Is_Announcement => False,
            Comp_Unit       => Comp_Unit,
            Loc             => Declaration.Start_Position,
            Type_Mark       => Type_Mark);
      elsif RawDict.Get_Symbol_Compilation_Unit (Item => RawDict.Get_Type_Symbol (Type_Mark)) = Comp_Unit then
         --  There was a usage of the type before its actual
         --  declaration -> update the declaration position. The
         --  declaration must be in the same compilation unit as the
         --  usage (in the package specification) to prevent to get a
         --  cross-reference from the package specification to the
         --  package body.
         RawDict.Set_Symbol_Location (Item     => RawDict.Get_Type_Symbol (Type_Mark),
                                      Location => Declaration.Start_Position);
         RawDict.Set_Type_Private_Type_Declaration (Type_Mark       => Type_Mark,
                                                    The_Declaration => The_Declaration);
         RawDict.Set_Declaration_Item (The_Declaration => The_Declaration,
                                       Item            => RawDict.Get_Type_Symbol (Type_Mark));
      end if;

      RawDict.Set_Type_Extends (Type_Mark => Type_Mark,
                                Root_Type => Extends);

      if Is_Limited then
         RawDict.Set_Type_Limited (Type_Mark  => Type_Mark,
                                   Is_Limited => Sometimes);
         RawDict.Set_Type_Limited_Private (Type_Mark => Type_Mark);
      else
         RawDict.Set_Type_Limited (Type_Mark  => Type_Mark,
                                   Is_Limited => Never);
      end if;

      if Is_Tagged_Type then
         RawDict.Set_Package_Declares_Tagged_Type (The_Package => The_Package);
      end if;

      RawDict.Set_Type_Is_Tagged (Type_Mark => Type_Mark,
                                  Is_Tagged => Is_Tagged_Type);
      Write_Type_Declaration (Type_Mark    => Type_Mark,
                              Discriminant => PrivateTypeDeclaration,
                              Declaration  => Declaration);
   end Add_Private_Type_Local;

   --------------------------------------------------------------------------------

   procedure Add_Private_Type
     (Name           : in     LexTokenManager.Lex_String;
      Comp_Unit      : in     ContextManager.UnitDescriptors;
      Declaration    : in     Location;
      The_Package    : in     Symbol;
      Is_Limited     : in     Boolean;
      Is_Tagged_Type : in     Boolean;
      Extends        : in     Symbol;
      The_Type       :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;
   begin
      Add_Private_Type_Local
        (Name           => Name,
         Comp_Unit      => Comp_Unit,
         Declaration    => Declaration,
         The_Package    => RawDict.Get_Package_Info_Ref (The_Package), -- GAA External
         Is_Limited     => Is_Limited,
         Is_Tagged_Type => Is_Tagged_Type,
         Extends        => RawDict.Get_Type_Info_Ref (Extends), -- GAA External
         Type_Mark      => Type_Mark);
      The_Type := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Private_Type;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   procedure Add_Type_Declaration
     (Name        : in     LexTokenManager.Lex_String;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      Scope       : in     Scopes;
      Context     : in     Contexts;
      Type_Mark   :    out RawDict.Type_Info_Ref)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys from *,
   --#                                Comp_Unit,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Scope &
   --#         Type_Mark         from Comp_Unit,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Scope;
   --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
   is
      Region          : Symbol;
      The_Declaration : RawDict.Declaration_Info_Ref;
   begin
      Region := GetRegion (Scope);
      if RawDict.GetSymbolDiscriminant (Region) = Package_Symbol and then Context = ProgramContext then
         -------------------------------------------------------------
         -- If declaring a type within a package in ProgramContext,
         -- then that type may have already been declared by way of a
         -- private type declaration, a type-annoucement or both.
         -------------------------------------------------------------
         Type_Mark := RawDict.Get_Type_Info_Ref
           (LookupImmediateScope (Name, Set_Visibility (The_Visibility => Visible,
                                                        The_Unit       => Region), ProofContext));
      else
         Type_Mark := RawDict.Null_Type_Info_Ref;
      end if;

      Add_Declaration
        (Comp_Unit       => Comp_Unit,
         Loc             => Declaration,
         Scope           => Scope,
         Context         => Context,
         The_Declaration => The_Declaration);

      if Type_Mark = RawDict.Null_Type_Info_Ref then
         ------------------------------------------------------------
         -- Type does not already exist in Dictionary, so create it
         ------------------------------------------------------------
         RawDict.Create_Type
           (Name            => Name,
            The_Declaration => The_Declaration,
            Is_Private      => False,
            Is_Announcement => False,
            Comp_Unit       => Comp_Unit,
            Loc             => Declaration.Start_Position,
            Type_Mark       => Type_Mark);
      else
         ------------------------------------------------------------
         -- Type already exists in the Dictionary, so it MUST be
         -- either private, announced or both.
         ------------------------------------------------------------
         SystemErrors.RT_Assert
           (C       => RawDict.Get_Type_Announcement (Type_Mark => Type_Mark) /= RawDict.Null_Declaration_Info_Ref
              or else Type_Is_Private (Type_Mark => Type_Mark),
            Sys_Err => SystemErrors.Assertion_Failure,
            Msg     => "Add_Type_Declaration : Re-declaration of a non-announced, non-private type");

         RawDict.Set_Type_Declaration (Type_Mark       => Type_Mark,
                                       The_Declaration => The_Declaration);

         if RawDict.Get_Symbol_Compilation_Unit (Item => RawDict.Get_Type_Symbol (Type_Mark)) = Comp_Unit then
            --  There was a usage of the type before its actual
            --  declaration -> update the declaration position. The
            --  declaration must be in the same compilation unit as
            --  the usage (in the package specification) to prevent
            --  to get a cross-reference from the package
            --  specification to the package body.
            RawDict.Set_Symbol_Location (Item     => RawDict.Get_Type_Symbol (Type_Mark),
                                         Location => Declaration.Start_Position);
         end if;
      end if;

      Write_Type_Declaration (Type_Mark    => Type_Mark,
                              Discriminant => TypeDeclaration,
                              Declaration  => Declaration);
   end Add_Type_Declaration;

   --------------------------------------------------------------------------------
   --  Add_Enumeration_Type
   --------------------------------------------------------------------------------

   procedure Add_Enumeration_Type
     (Name        : in     LexTokenManager.Lex_String;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      Scope       : in     Scopes;
      Context     : in     Contexts;
      The_Type    :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Enumeration_Type_Local
        (Name        : in     LexTokenManager.Lex_String;
         Comp_Unit   : in     ContextManager.UnitDescriptors;
         Declaration : in     Location;
         Scope       : in     Scopes;
         Context     : in     Contexts;
         Type_Mark   :    out RawDict.Type_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict,
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope &
      --#         Type_Mark         from Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope;
      --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
      is
      begin
         Add_Type_Declaration
           (Name        => Name,
            Comp_Unit   => Comp_Unit,
            Declaration => Declaration,
            Scope       => Scope,
            Context     => Context,
            Type_Mark   => Type_Mark);
         Add_Enumeration_Type_Mark (Type_Mark => Type_Mark,
                                    Static    => True);
      end Add_Enumeration_Type_Local;

   begin -- Add_Enumeration_Type
      Add_Enumeration_Type_Local
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Declaration,
         Scope       => Scope,
         Context     => Context,
         Type_Mark   => Type_Mark);
      The_Type := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Enumeration_Type;

   --------------------------------------------------------------------------------
   --  Add_Protected_Type
   --------------------------------------------------------------------------------

   procedure Add_Protected_Type
     (Name        : in     LexTokenManager.Lex_String;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      Scope       : in     Scopes;
      Context     : in     Contexts;
      Mode        : in     Modes;
      Constrained : in     Boolean;
      The_Type    :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Protected_Type_Local
        (Name        : in     LexTokenManager.Lex_String;
         Comp_Unit   : in     ContextManager.UnitDescriptors;
         Declaration : in     Location;
         Scope       : in     Scopes;
         Context     : in     Contexts;
         Mode        : in     Modes;
         Constrained : in     Boolean;
         Type_Mark   :    out RawDict.Type_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Constrained,
      --#                                Context,
      --#                                Declaration,
      --#                                LexTokenManager.State,
      --#                                Mode,
      --#                                Name,
      --#                                Scope &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope &
      --#         Type_Mark         from Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope;
      --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
      is
         The_Variable     : RawDict.Variable_Info_Ref;
         The_Own_Variable : RawDict.Own_Variable_Info_Ref;

         --------------------------------------------------------------------------------

         procedure Add_Protected_Type_Mark
           (Type_Mark   : in RawDict.Type_Info_Ref;
            Comp_Unit   : in ContextManager.UnitDescriptors;
            Declaration : in Location)
         --# global in out Dict;
         --# derives Dict from *,
         --#                   Comp_Unit,
         --#                   Declaration,
         --#                   Type_Mark;
         is
            Protected_Info_Sym : Symbol;
         begin
            RawDict.CreateProtectedInfo (Comp_Unit => Comp_Unit,
                                         Loc       => Declaration.Start_Position,
                                         InfoSym   => Protected_Info_Sym);
            RawDict.Set_Type_Ancillary_Fields (Type_Mark       => Type_Mark,
                                               The_Declaration => Protected_Info_Sym);
            RawDict.Set_Type_Discriminant (Type_Mark    => Type_Mark,
                                           Discriminant => Protected_Type_Item);
            RawDict.Set_Type_Static (Type_Mark => Type_Mark,
                                     Static    => False);
            RawDict.Set_Type_Equality_Defined (Type_Mark        => Type_Mark,
                                               Equality_Defined => False);
            RawDict.Set_Type_Limited (Type_Mark  => Type_Mark,
                                      Is_Limited => Always);
            RawDict.Set_Type_Atomic (Type_Mark => Type_Mark);
         end Add_Protected_Type_Mark;

         --------------------------------------------------------------------------------

         procedure Add_Access_Type
           (Accessed_Type : in RawDict.Type_Info_Ref;
            Comp_Unit     : in ContextManager.UnitDescriptors;
            Declaration   : in Location)
         --# global in out Dict;
         --# derives Dict from *,
         --#                   Accessed_Type,
         --#                   Comp_Unit,
         --#                   Declaration;
         is
            New_Access_Type : RawDict.Type_Info_Ref;
         begin
            -- Create a type record with the name "Access" to ensure we never look it up
            -- by mistake (access being reserved word).
            RawDict.Create_Type
              (Name            => LexTokenManager.Access_Token,
               The_Declaration => RawDict.Null_Declaration_Info_Ref,
               Is_Private      => False,
               Is_Announcement => False,
               Comp_Unit       => Comp_Unit,
               Loc             => Declaration.Start_Position,
               Type_Mark       => New_Access_Type);
            -- mark it as an access type
            RawDict.Set_Type_Discriminant (Type_Mark    => New_Access_Type,
                                           Discriminant => Access_Type_Item);
            -- set other fields to "safe" values
            RawDict.Set_Type_Static (Type_Mark => New_Access_Type,
                                     Static    => False);
            RawDict.Set_Type_Equality_Defined (Type_Mark        => New_Access_Type,
                                               Equality_Defined => False);
            RawDict.Set_Type_Limited (Type_Mark  => New_Access_Type,
                                      Is_Limited => Always);
            -- link type and access type together
            RawDict.Set_Type_Accesses (Type_Mark  => Accessed_Type,
                                       The_Access => New_Access_Type);
            RawDict.Set_Type_Accesses (Type_Mark  => New_Access_Type,
                                       The_Access => Accessed_Type);
         end Add_Access_Type;

      begin -- Add_Protected_Type_Local
         Add_Type_Declaration
           (Name        => Name,
            Comp_Unit   => Comp_Unit,
            Declaration => Declaration,
            Scope       => Scope,
            Context     => Context,
            Type_Mark   => Type_Mark);
         Add_Protected_Type_Mark (Type_Mark   => Type_Mark,
                                  Comp_Unit   => Comp_Unit,
                                  Declaration => Declaration);

         -- We also declare an abstract own variable with the same name as the protected type
         -- at this point.  The existence of such a variable simplifies the use of the protected
         -- type name in the annotations of its operations
         RawDict.Create_Variable
           (Name         => Name,
            Comp_Unit    => Comp_Unit,
            Loc          => Declaration.Start_Position,
            The_Variable => The_Variable);
         RawDict.Create_Own_Variable
           (Variable         => The_Variable,
            Owner            => RawDict.Get_Type_Symbol (Type_Mark),
            Comp_Unit        => Comp_Unit,
            Loc              => Declaration.Start_Position,
            The_Own_Variable => The_Own_Variable);
         RawDict.Set_Variable_Own_Variable (The_Variable => The_Variable,
                                            Own_Variable => The_Own_Variable);
         RawDict.Set_Protected_Type_Own_Variable (The_Protected_Type => Type_Mark,
                                                  Own_Variable       => The_Own_Variable);
         RawDict.Set_Type_Constrained (Type_Mark   => Type_Mark,
                                       Constrained => Constrained);

         -- the type of this fictitious abstract own variable is the type itself
         RawDict.Set_Variable_Type (The_Variable => The_Variable,
                                    Type_Mark    => Type_Mark);
         RawDict.Set_Variable_Abstract_Type (The_Variable       => The_Variable,
                                             Abstract_Type_Mark => Type_Mark);
         RawDict.Set_Own_Variable_Typed (The_Own_Variable => The_Own_Variable);

         -- set some sensible values for the new own variable
         RawDict.Set_Own_Variable_Announced (The_Own_Variable => The_Own_Variable);
         RawDict.Set_Own_Variable_Initialized (The_Own_Variable => The_Own_Variable);
         RawDict.Set_Own_Variable_Mode (The_Own_Variable => The_Own_Variable,
                                        Mode             => Mode);
         RawDict.Set_Own_Variable_Protected (The_Own_Variable => The_Own_Variable,
                                             Is_Protected     => True);

         -- implicitly declare an access type to go with the PT
         Add_Access_Type (Accessed_Type => Type_Mark,
                          Comp_Unit     => Comp_Unit,
                          Declaration   => Declaration);
      end Add_Protected_Type_Local;

   begin -- Add_Protected_Type
      Add_Protected_Type_Local
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Declaration,
         Scope       => Scope,
         Context     => Context,
         Mode        => Mode,
         Constrained => Constrained,
         Type_Mark   => Type_Mark);
      The_Type := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Protected_Type;

   --------------------------------------------------------------------------------
   --  Add_Task_Type
   --------------------------------------------------------------------------------

   procedure Add_Task_Type
     (Name        : in     LexTokenManager.Lex_String;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      Scope       : in     Scopes;
      Context     : in     Contexts;
      Constrained : in     Boolean;
      The_Type    :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Task_Type_Local
        (Name        : in     LexTokenManager.Lex_String;
         Comp_Unit   : in     ContextManager.UnitDescriptors;
         Declaration : in     Location;
         Scope       : in     Scopes;
         Context     : in     Contexts;
         Constrained : in     Boolean;
         Type_Mark   :    out RawDict.Type_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Constrained,
      --#                                Context,
      --#                                Declaration,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope &
      --#         Type_Mark         from Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope;
      --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
      is

         procedure Add_Task_Type_Mark
           (Type_Mark   : in RawDict.Type_Info_Ref;
            Comp_Unit   : in ContextManager.UnitDescriptors;
            Declaration : in Location)
         --# global in out Dict;
         --# derives Dict from *,
         --#                   Comp_Unit,
         --#                   Declaration,
         --#                   Type_Mark;
         is
            Task_Info_Sym : Symbol;
         begin
            RawDict.CreateTaskInfo (Comp_Unit => Comp_Unit,
                                    Loc       => Declaration.Start_Position,
                                    InfoSym   => Task_Info_Sym);
            RawDict.Set_Type_Ancillary_Fields (Type_Mark       => Type_Mark,
                                               The_Declaration => Task_Info_Sym);
            RawDict.Set_Type_Discriminant (Type_Mark    => Type_Mark,
                                           Discriminant => Task_Type_Item);
            RawDict.Set_Type_Static (Type_Mark => Type_Mark,
                                     Static    => False);
            RawDict.Set_Type_Equality_Defined (Type_Mark        => Type_Mark,
                                               Equality_Defined => False);
            RawDict.Set_Type_Limited (Type_Mark  => Type_Mark,
                                      Is_Limited => Always);
         end Add_Task_Type_Mark;

      begin -- Add_Task_Type_Local
         Add_Type_Declaration
           (Name        => Name,
            Comp_Unit   => Comp_Unit,
            Declaration => Declaration,
            Scope       => Scope,
            Context     => Context,
            Type_Mark   => Type_Mark);
         Add_Task_Type_Mark (Type_Mark   => Type_Mark,
                             Comp_Unit   => Comp_Unit,
                             Declaration => Declaration);
         RawDict.Set_Type_Constrained (Type_Mark   => Type_Mark,
                                       Constrained => Constrained);
      end Add_Task_Type_Local;

   begin -- Add_Task_Type
      Add_Task_Type_Local
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Declaration,
         Scope       => Scope,
         Context     => Context,
         Constrained => Constrained,
         Type_Mark   => Type_Mark);
      The_Type := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Task_Type;

   --------------------------------------------------------------------------------
   --  Add_Task_Or_Protected_Subtype
   --------------------------------------------------------------------------------

   procedure Add_Task_Or_Protected_Subtype
     (Name        : in     LexTokenManager.Lex_String;
      Parent      : in     Symbol;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      Scope       : in     Scopes;
      Context     : in     Contexts;
      The_Subtype :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Task_Or_Protected_Subtype_Local
        (Name        : in     LexTokenManager.Lex_String;
         Parent_Type : in     RawDict.Type_Info_Ref;
         Comp_Unit   : in     ContextManager.UnitDescriptors;
         Declaration : in     Location;
         Scope       : in     Scopes;
         Context     : in     Contexts;
         Type_Mark   :    out RawDict.Type_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict,
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Parent_Type,
      --#                                Scope &
      --#         Type_Mark         from Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope;
      --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
      is
         Extra_Info : Symbol;
      begin
         SystemErrors.RT_Assert
           (C       => Is_Task_Type (Type_Mark => Parent_Type) or else Is_Protected_Type (Type_Mark => Parent_Type),
            Sys_Err => SystemErrors.Precondition_Failure,
            Msg     => "In call to AddTaskOrProtectedSubtype");

         -- add a "basic" type record making sure we use any existing announced type
         Add_Type_Declaration
           (Name        => Name,
            Comp_Unit   => Comp_Unit,
            Declaration => Declaration,
            Scope       => Scope,
            Context     => Context,
            Type_Mark   => Type_Mark);
         -- set its fields to suitable values:
         -- subtype is same sort of thing as its parent
         RawDict.Set_Type_Discriminant
           (Type_Mark    => Type_Mark,
            Discriminant => RawDict.Get_Type_Discriminant (Type_Mark => Parent_Type));
         -- make types as limited as possible
         RawDict.Set_Type_Static (Type_Mark => Type_Mark,
                                  Static    => False);
         RawDict.Set_Type_Equality_Defined (Type_Mark        => Type_Mark,
                                            Equality_Defined => False);
         RawDict.Set_Type_Limited (Type_Mark  => Type_Mark,
                                   Is_Limited => Always);
         if Type_Is_Atomic (Parent_Type) then
            RawDict.Set_Type_Atomic (Type_Mark => Type_Mark);
         end if;
         -- link subtype to parent
         RawDict.Set_Type_Parent (Type_Mark => Type_Mark,
                                  Parent    => Parent_Type);
         -- a legal subtype must have had its discriminants constrained
         RawDict.Set_Type_Constrained (Type_Mark   => Type_Mark,
                                       Constrained => True);

         -- create subtype info record and hook it to Ancillary_Fields of subtype
         RawDict.CreateSubtype (Comp_Unit  => Comp_Unit,
                                Loc        => Declaration.Start_Position,
                                TheSubtype => Extra_Info);
         RawDict.Set_Type_Ancillary_Fields (Type_Mark       => Type_Mark,
                                            The_Declaration => Extra_Info);
         Write_Type_Declaration (Type_Mark    => Type_Mark,
                                 Discriminant => TypeDeclaration,
                                 Declaration  => Declaration);
      end Add_Task_Or_Protected_Subtype_Local;

   begin -- Add_Task_Or_Protected_Subtype
      Add_Task_Or_Protected_Subtype_Local
        (Name        => Name,
         Parent_Type => RawDict.Get_Type_Info_Ref (Parent), -- GAA External
         Comp_Unit   => Comp_Unit,
         Declaration => Declaration,
         Scope       => Scope,
         Context     => Context,
         Type_Mark   => Type_Mark);
      The_Subtype := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Task_Or_Protected_Subtype;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   procedure Add_Representation_Clause (The_Type : in Symbol;
                                        Clause   : in Location) is

      procedure Write_Representation_Clause (Type_Mark : in RawDict.Type_Info_Ref;
                                             Clause    : in Location)
      --# global in     Dict;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Clause,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Type_Mark;
      is
      begin
         if SPARK_IO.Is_Open (Dict.TemporaryFile) then
            Write_String (Dict.TemporaryFile, "representation clause of ");
            Write_Name (File => Dict.TemporaryFile,
                        Item => RawDict.Get_Type_Symbol (Type_Mark));
            Write_String (Dict.TemporaryFile, " is at ");
            Write_Location (File => Dict.TemporaryFile,
                            Loc  => Clause);
            Write_Line (Dict.TemporaryFile, " ;");
         end if;
      end Write_Representation_Clause;

   begin -- Add_Representation_Clause
      Write_Representation_Clause (Type_Mark => RawDict.Get_Type_Info_Ref (The_Type), -- GAA External
                                   Clause    => Clause);
   end Add_Representation_Clause;

   --------------------------------------------------------------------------------

   procedure AddEnumerationLiteralRepresentation (Literal : in Symbol;
                                                  Code    : in Integer) is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         Write_String (Dict.TemporaryFile, "enumeration literal representation of ");
         Write_Name (File => Dict.TemporaryFile,
                     Item => Literal);
         Write_String (Dict.TemporaryFile, " is ");
         Write_Integer (Dict.TemporaryFile, Code);
         Write_Line (Dict.TemporaryFile, " ;");
      end if;
   end AddEnumerationLiteralRepresentation;

   --------------------------------------------------------------------------------
   --  Add_Integer_Type
   --------------------------------------------------------------------------------

   procedure Add_Integer_Type_Local
     (Name        : in     LexTokenManager.Lex_String;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      Lower       : in     LexTokenManager.Lex_String;
      Upper       : in     LexTokenManager.Lex_String;
      Scope       : in     Scopes;
      Context     : in     Contexts;
      Type_Mark   :    out RawDict.Type_Info_Ref)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Comp_Unit,
   --#                                Context,
   --#                                Declaration,
   --#                                LexTokenManager.State,
   --#                                Lower,
   --#                                Name,
   --#                                Scope,
   --#                                Upper &
   --#         SPARK_IO.File_Sys from *,
   --#                                Comp_Unit,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Scope &
   --#         Type_Mark         from Comp_Unit,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Scope;
   --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
   is
   begin
      Add_Type_Declaration
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Declaration,
         Scope       => Scope,
         Context     => Context,
         Type_Mark   => Type_Mark);
      Add_Integer_Type_Mark (Type_Mark => Type_Mark,
                             Static    => True,
                             Lower     => Lower,
                             Upper     => Upper);
      RawDict.Set_Type_Derived (Type_Mark  => Type_Mark,
                                Is_Derived => True);
   end Add_Integer_Type_Local;

   --------------------------------------------------------------------------------

   procedure Add_Integer_Type
     (Name        : in     LexTokenManager.Lex_String;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      Lower       : in     LexTokenManager.Lex_String;
      Upper       : in     LexTokenManager.Lex_String;
      Scope       : in     Scopes;
      Context     : in     Contexts;
      The_Type    :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;
   begin
      Add_Integer_Type_Local
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Declaration,
         Lower       => Lower,
         Upper       => Upper,
         Scope       => Scope,
         Context     => Context,
         Type_Mark   => Type_Mark);
      The_Type := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Integer_Type;

   --------------------------------------------------------------------------------
   --  Add_Predef_Integer_Type
   --------------------------------------------------------------------------------

   procedure Add_Predef_Integer_Type
     (Name        : in LexTokenManager.Lex_String;
      Comp_Unit   : in ContextManager.UnitDescriptors;
      Declaration : in Location;
      Lower       : in LexTokenManager.Lex_String;
      Upper       : in LexTokenManager.Lex_String;
      Scope       : in Scopes;
      Context     : in Contexts) is
      Type_Mark : RawDict.Type_Info_Ref;
   begin
      Add_Type_Declaration
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Declaration,
         Scope       => Scope,
         Context     => Context,
         Type_Mark   => Type_Mark);
      Add_Integer_Type_Mark (Type_Mark => Type_Mark,
                             Static    => True,
                             Lower     => Lower,
                             Upper     => Upper);
   end Add_Predef_Integer_Type;

   --------------------------------------------------------------------------------
   --  Add_Modular_Type
   --------------------------------------------------------------------------------

   procedure Add_Modular_Type
     (Name        : in     LexTokenManager.Lex_String;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      Modulus     : in     LexTokenManager.Lex_String;
      Scope       : in     Scopes;
      Context     : in     Contexts;
      The_Type    :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Modular_Type_Local
        (Name        : in     LexTokenManager.Lex_String;
         Comp_Unit   : in     ContextManager.UnitDescriptors;
         Declaration : in     Location;
         Modulus     : in     LexTokenManager.Lex_String;
         Scope       : in     Scopes;
         Context     : in     Contexts;
         Type_Mark   :    out RawDict.Type_Info_Ref)
      --# global in out Dict;
      --#        in out LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict                  from *,
      --#                                    Comp_Unit,
      --#                                    Context,
      --#                                    Declaration,
      --#                                    LexTokenManager.State,
      --#                                    Modulus,
      --#                                    Name,
      --#                                    Scope &
      --#         LexTokenManager.State from *,
      --#                                    Modulus &
      --#         SPARK_IO.File_Sys     from *,
      --#                                    Comp_Unit,
      --#                                    Context,
      --#                                    Declaration,
      --#                                    Dict,
      --#                                    LexTokenManager.State,
      --#                                    Name,
      --#                                    Scope &
      --#         Type_Mark             from Comp_Unit,
      --#                                    Context,
      --#                                    Declaration,
      --#                                    Dict,
      --#                                    LexTokenManager.State,
      --#                                    Name,
      --#                                    Scope;
      --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
      is
         Upper       : LexTokenManager.Lex_String;
         Upper_Value : Maths.Value;
         Ok          : Maths.ErrorCode;
      begin
         Add_Type_Declaration
           (Name        => Name,
            Comp_Unit   => Comp_Unit,
            Declaration => Declaration,
            Scope       => Scope,
            Context     => Context,
            Type_Mark   => Type_Mark);
         -- T'Last is (T'Modulus - 1), so...
         Maths.Subtract (Maths.ValueRep (Modulus), Maths.OneInteger,
                         -- to get
                         Upper_Value, Ok);

         if Ok = Maths.NoError then
            Maths.StorageRep (Upper_Value, Upper);
         else
            Upper := LexTokenManager.Null_String;
         end if;

         Add_Modular_Type_Mark
           (Type_Mark => Type_Mark,
            Static    => True,
            Lower     => LexTokenManager.Zero_Value,
            Upper     => Upper,
            Modulus   => Modulus);
      end Add_Modular_Type_Local;

   begin -- Add_Modular_Type
      Add_Modular_Type_Local
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Declaration,
         Modulus     => Modulus,
         Scope       => Scope,
         Context     => Context,
         Type_Mark   => Type_Mark);
      The_Type := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Modular_Type;

   --------------------------------------------------------------------------------
   --  Add_Abstract_Proof_Type
   --------------------------------------------------------------------------------

   procedure Add_Abstract_Proof_Type_Local
     (Name        : in     LexTokenManager.Lex_String;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      Scope       : in     Scopes;
      Type_Mark   :    out RawDict.Type_Info_Ref)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys from *,
   --#                                Comp_Unit,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Scope &
   --#         Type_Mark         from Comp_Unit,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Scope;
   --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
   is
   begin
      Add_Type_Declaration
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Declaration,
         Scope       => Scope,
         Context     => ProgramContext,
         Type_Mark   => Type_Mark);
      RawDict.Set_Type_Discriminant (Type_Mark    => Type_Mark,
                                     Discriminant => Abstract_Proof_Type_Item);
   end Add_Abstract_Proof_Type_Local;

   --------------------------------------------------------------------------------

   procedure Add_Abstract_Proof_Type
     (Name        : in     LexTokenManager.Lex_String;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      Scope       : in     Scopes;
      The_Type    :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;
   begin
      Add_Abstract_Proof_Type_Local
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Declaration,
         Scope       => Scope,
         Type_Mark   => Type_Mark);
      The_Type := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Abstract_Proof_Type;

   --------------------------------------------------------------------------------
   --  Add_Default_Abstract_Proof_Type
   --------------------------------------------------------------------------------

   procedure Add_Default_Abstract_Proof_Type
     (Name        : in     LexTokenManager.Lex_String;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      Scope       : in     Scopes;
      The_Type    :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Default_Abstract_Proof_Type_Local
        (Name        : in     LexTokenManager.Lex_String;
         Comp_Unit   : in     ContextManager.UnitDescriptors;
         Declaration : in     Location;
         Scope       : in     Scopes;
         Type_Mark   :    out RawDict.Type_Info_Ref)
      --# global in out Dict;
      --#        in out LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict,
      --#         SPARK_IO.File_Sys     from *,
      --#                                    Comp_Unit,
      --#                                    Declaration,
      --#                                    Dict,
      --#                                    LexTokenManager.State,
      --#                                    Name,
      --#                                    Scope &
      --#         LexTokenManager.State from *,
      --#                                    Name &
      --#         Type_Mark             from Comp_Unit,
      --#                                    Declaration,
      --#                                    Dict,
      --#                                    LexTokenManager.State,
      --#                                    Name,
      --#                                    Scope;
      --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
      is
         E_Name    : E_Strings.T;
         Type_Name : LexTokenManager.Lex_String;
      begin
         -- Create name for type by appending to __type to own variable name (e.g. state__type)
         E_Name := LexTokenManager.Lex_String_To_String (Lex_Str => Name);
         E_Strings.Append_String (E_Str => E_Name,
                                  Str   => "__type");
         LexTokenManager.Insert_Examiner_String (Str     => E_Name,
                                                 Lex_Str => Type_Name);
         Add_Abstract_Proof_Type_Local
           (Name        => Type_Name,
            Comp_Unit   => Comp_Unit,
            Declaration => Declaration,
            Scope       => Scope,
            Type_Mark   => Type_Mark);
      end Add_Default_Abstract_Proof_Type_Local;

   begin -- Add_Default_Abstract_Proof_Type
      Add_Default_Abstract_Proof_Type_Local
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Declaration,
         Scope       => Scope,
         Type_Mark   => Type_Mark);
      The_Type := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Default_Abstract_Proof_Type;

   --------------------------------------------------------------------------------
   --  Add_Floating_Point_Type
   --------------------------------------------------------------------------------

   procedure Add_Floating_Point_Type
     (Name        : in     LexTokenManager.Lex_String;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      Lower       : in     LexTokenManager.Lex_String;
      Upper       : in     LexTokenManager.Lex_String;
      Error_Bound : in     LexTokenManager.Lex_String;
      Scope       : in     Scopes;
      Context     : in     Contexts;
      The_Type    :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Floating_Point_Type_Local
        (Name        : in     LexTokenManager.Lex_String;
         Comp_Unit   : in     ContextManager.UnitDescriptors;
         Declaration : in     Location;
         Lower       : in     LexTokenManager.Lex_String;
         Upper       : in     LexTokenManager.Lex_String;
         Error_Bound : in     LexTokenManager.Lex_String;
         Scope       : in     Scopes;
         Context     : in     Contexts;
         Type_Mark   :    out RawDict.Type_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Error_Bound,
      --#                                LexTokenManager.State,
      --#                                Lower,
      --#                                Name,
      --#                                Scope,
      --#                                Upper &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope &
      --#         Type_Mark         from Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope;
      --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
      is
      begin
         Add_Type_Declaration
           (Name        => Name,
            Comp_Unit   => Comp_Unit,
            Declaration => Declaration,
            Scope       => Scope,
            Context     => Context,
            Type_Mark   => Type_Mark);
         Add_Floating_Point_Type_Mark (Type_Mark => Type_Mark,
                                       Static    => True);
         RawDict.Set_Type_Lower (Type_Mark => Type_Mark,
                                 Lower     => Lower);
         RawDict.Set_Type_Upper (Type_Mark => Type_Mark,
                                 Upper     => Upper);
         RawDict.Set_Type_Error_Bound (Type_Mark   => Type_Mark,
                                       Error_Bound => Error_Bound);
         RawDict.Set_Type_Derived (Type_Mark  => Type_Mark,
                                   Is_Derived => True);
      end Add_Floating_Point_Type_Local;

   begin -- Add_Floating_Point_Type
      Add_Floating_Point_Type_Local
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Declaration,
         Lower       => Lower,
         Upper       => Upper,
         Error_Bound => Error_Bound,
         Scope       => Scope,
         Context     => Context,
         Type_Mark   => Type_Mark);
      The_Type := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Floating_Point_Type;

   --------------------------------------------------------------------------------
   --  Add_Predef_Floating_Point_Type
   --------------------------------------------------------------------------------

   procedure Add_Predef_Floating_Point_Type
     (Name        : in LexTokenManager.Lex_String;
      Comp_Unit   : in ContextManager.UnitDescriptors;
      Declaration : in Location;
      Lower       : in LexTokenManager.Lex_String;
      Upper       : in LexTokenManager.Lex_String;
      Error_Bound : in LexTokenManager.Lex_String;
      Scope       : in Scopes;
      Context     : in Contexts) is
      Type_Mark : RawDict.Type_Info_Ref;
   begin
      Add_Type_Declaration
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Declaration,
         Scope       => Scope,
         Context     => Context,
         Type_Mark   => Type_Mark);
      Add_Floating_Point_Type_Mark (Type_Mark => Type_Mark,
                                    Static    => True);
      RawDict.Set_Type_Lower (Type_Mark => Type_Mark,
                              Lower     => Lower);
      RawDict.Set_Type_Upper (Type_Mark => Type_Mark,
                              Upper     => Upper);
      RawDict.Set_Type_Error_Bound (Type_Mark   => Type_Mark,
                                    Error_Bound => Error_Bound);
   end Add_Predef_Floating_Point_Type;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   procedure SetTypeIsWellformed (TypeMark   : in Symbol;
                                  Wellformed : in Boolean) is
   begin
      RawDict.Set_Type_Wellformed (Type_Mark  => RawDict.Get_Type_Info_Ref (TypeMark), -- GAA External
                                   Wellformed => Wellformed);
   end SetTypeIsWellformed;

   --------------------------------------------------------------------------------

   procedure SetBaseType (TypeMark, BaseType : in Symbol) is
   begin
      RawDict.Set_Type_Base_Type
        (Type_Mark => RawDict.Get_Type_Info_Ref (TypeMark), -- GAA External
         Base_Type => RawDict.Get_Type_Info_Ref (BaseType)); -- GAA External
   end SetBaseType;

   --------------------------------------------------------------------------------
   --  Add_Fixed_Point_Type
   --------------------------------------------------------------------------------

   procedure Add_Fixed_Point_Type
     (Name        : in     LexTokenManager.Lex_String;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      Lower       : in     LexTokenManager.Lex_String;
      Upper       : in     LexTokenManager.Lex_String;
      Error_Bound : in     LexTokenManager.Lex_String;
      Scope       : in     Scopes;
      Context     : in     Contexts;
      The_Type    :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Fixed_Point_Type_Local
        (Name        : in     LexTokenManager.Lex_String;
         Comp_Unit   : in     ContextManager.UnitDescriptors;
         Declaration : in     Location;
         Lower       : in     LexTokenManager.Lex_String;
         Upper       : in     LexTokenManager.Lex_String;
         Error_Bound : in     LexTokenManager.Lex_String;
         Scope       : in     Scopes;
         Context     : in     Contexts;
         Type_Mark   :    out RawDict.Type_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Error_Bound,
      --#                                LexTokenManager.State,
      --#                                Lower,
      --#                                Name,
      --#                                Scope,
      --#                                Upper &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope &
      --#         Type_Mark         from Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope;
      --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
      is
      begin
         Add_Type_Declaration
           (Name        => Name,
            Comp_Unit   => Comp_Unit,
            Declaration => Declaration,
            Scope       => Scope,
            Context     => Context,
            Type_Mark   => Type_Mark);
         Add_Fixed_Point_Type_Mark (Type_Mark => Type_Mark,
                                    Static    => True);
         RawDict.Set_Type_Lower (Type_Mark => Type_Mark,
                                 Lower     => Lower);
         RawDict.Set_Type_Upper (Type_Mark => Type_Mark,
                                 Upper     => Upper);
         RawDict.Set_Type_Error_Bound (Type_Mark   => Type_Mark,
                                       Error_Bound => Error_Bound);
      end Add_Fixed_Point_Type_Local;

   begin -- Add_Fixed_Point_Type
      Add_Fixed_Point_Type_Local
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Declaration,
         Lower       => Lower,
         Upper       => Upper,
         Error_Bound => Error_Bound,
         Scope       => Scope,
         Context     => Context,
         Type_Mark   => Type_Mark);
      The_Type := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Fixed_Point_Type;

   --------------------------------------------------------------------------------
   --  Add_Array_Type
   --------------------------------------------------------------------------------

   procedure Add_Array_Type
     (Name                     : in     LexTokenManager.Lex_String;
      Comp_Unit                : in     ContextManager.UnitDescriptors;
      Declaration              : in     Location;
      Scope                    : in     Scopes;
      Context                  : in     Contexts;
      Constrained              : in     Boolean;
      Component_Type           : in     Symbol;
      Component_Type_Reference : in     Location;
      The_Type                 :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Array_Type_Local
        (Name                     : in     LexTokenManager.Lex_String;
         Comp_Unit                : in     ContextManager.UnitDescriptors;
         Declaration              : in     Location;
         Scope                    : in     Scopes;
         Context                  : in     Contexts;
         Constrained              : in     Boolean;
         The_Component_Type       : in     RawDict.Type_Info_Ref;
         Component_Type_Reference : in     Location;
         Type_Mark                :    out RawDict.Type_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Constrained,
      --#                                Context,
      --#                                Declaration,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope,
      --#                                The_Component_Type &
      --#         SPARK_IO.File_Sys from *,
      --#                                Component_Type_Reference,
      --#                                Comp_Unit,
      --#                                Constrained,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope,
      --#                                The_Component_Type &
      --#         Type_Mark         from Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope;
      --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
      is
      begin
         Add_Type_Declaration
           (Name        => Name,
            Comp_Unit   => Comp_Unit,
            Declaration => Declaration,
            Scope       => Scope,
            Context     => Context,
            Type_Mark   => Type_Mark);
         Add_Array_Type_Mark (Type_Mark      => Type_Mark,
                              Component_Type => The_Component_Type,
                              Static         => False);
         RawDict.Set_Type_Constrained (Type_Mark   => Type_Mark,
                                       Constrained => Constrained);

         if The_Component_Type /= Get_Unknown_Type_Mark then
            AddOtherReference (RawDict.Get_Type_Symbol (The_Component_Type), GetRegion (Scope), Component_Type_Reference);
         end if;
      end Add_Array_Type_Local;

   begin -- Add_Array_Type
      Add_Array_Type_Local
        (Name                     => Name,
         Comp_Unit                => Comp_Unit,
         Declaration              => Declaration,
         Scope                    => Scope,
         Context                  => Context,
         Constrained              => Constrained,
         The_Component_Type       => RawDict.Get_Type_Info_Ref (Component_Type), -- GAA External
         Component_Type_Reference => Component_Type_Reference,
         Type_Mark                => Type_Mark);
      The_Type := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Array_Type;

   --------------------------------------------------------------------------------
   --  Add_Array_Index
   --------------------------------------------------------------------------------

   procedure Add_Array_Index
     (The_Array_Type  : in     RawDict.Type_Info_Ref;
      Index_Type      : in     RawDict.Type_Info_Ref;
      Comp_Unit       : in     ContextManager.UnitDescriptors;
      Declaration     : in     Location;
      The_Array_Index :    out RawDict.Array_Index_Info_Ref)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Comp_Unit,
   --#                                Declaration,
   --#                                Index_Type,
   --#                                The_Array_Type &
   --#         SPARK_IO.File_Sys from *,
   --#                                Comp_Unit,
   --#                                Declaration,
   --#                                Dict,
   --#                                Index_Type,
   --#                                LexTokenManager.State,
   --#                                The_Array_Type &
   --#         The_Array_Index   from Comp_Unit,
   --#                                Declaration,
   --#                                Dict,
   --#                                Index_Type;
   --# post The_Array_Index /= RawDict.Null_Array_Index_Info_Ref;
   is
      Current : RawDict.Array_Index_Info_Ref;
   begin
      RawDict.Create_Array_Index
        (Index_Type      => Index_Type,
         Comp_Unit       => Comp_Unit,
         Loc             => Declaration.Start_Position,
         The_Array_Index => The_Array_Index);

      Current := RawDict.Get_Type_Last_Array_Index (Type_Mark => The_Array_Type);
      if Current = RawDict.Null_Array_Index_Info_Ref then
         RawDict.Set_Type_First_Array_Index (Type_Mark   => The_Array_Type,
                                             Array_Index => The_Array_Index);
      else
         RawDict.Set_Next_Array_Index (The_Array_Index => Current,
                                       Next            => The_Array_Index);
      end if;
      RawDict.Set_Type_Last_Array_Index (Type_Mark   => The_Array_Type,
                                         Array_Index => The_Array_Index);
      if Index_Type /= Get_Unknown_Type_Mark then
         AddOtherReference (RawDict.Get_Type_Symbol (Index_Type),
                            GetRegion (Get_Type_Scope (The_Array_Type)),
                            Declaration);
      end if;
   end Add_Array_Index;

   procedure AddArrayIndex
     (TheArrayType  : in     Symbol;
      IndexType     : in     Symbol;
      Comp_Unit     : in     ContextManager.UnitDescriptors;
      Declaration   : in     Location;
      TheArrayIndex :    out Symbol) is
      The_Array_Index : RawDict.Array_Index_Info_Ref;
   begin
      Add_Array_Index
        (The_Array_Type  => RawDict.Get_Type_Info_Ref (TheArrayType), -- GAA External
         Index_Type      => RawDict.Get_Type_Info_Ref (IndexType), -- GAA External
         Comp_Unit       => Comp_Unit,
         Declaration     => Declaration,
         The_Array_Index => The_Array_Index);
      TheArrayIndex := RawDict.Get_Array_Index_Symbol (The_Array_Index); -- GAA External
   end AddArrayIndex;

   --------------------------------------------------------------------------------
   --  Add_Record_Type
   --------------------------------------------------------------------------------

   procedure Add_Record_Type
     (Name           : in     LexTokenManager.Lex_String;
      Is_Tagged_Type : in     Boolean;
      Extends        : in     Symbol;
      Comp_Unit      : in     ContextManager.UnitDescriptors;
      Declaration    : in     Location;
      Scope          : in     Scopes;
      Context        : in     Contexts;
      The_Type       :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Record_Type_Local
        (Name           : in     LexTokenManager.Lex_String;
         Is_Tagged_Type : in     Boolean;
         Extends        : in     RawDict.Type_Info_Ref;
         Comp_Unit      : in     ContextManager.UnitDescriptors;
         Declaration    : in     Location;
         Scope          : in     Scopes;
         Context        : in     Contexts;
         Type_Mark      :    out RawDict.Type_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Extends,
      --#                                Is_Tagged_Type,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope &
      --#         Type_Mark         from Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope;
      --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
      is
      begin
         Add_Type_Declaration
           (Name        => Name,
            Comp_Unit   => Comp_Unit,
            Declaration => Declaration,
            Scope       => Scope,
            Context     => Context,
            Type_Mark   => Type_Mark);
         Add_Record_Type_Mark
           (Type_Mark      => Type_Mark,
            Is_Tagged_Type => Is_Tagged_Type,
            Extends        => Extends);

         -- mark package as containing a tagged type if necessary
         if Is_Tagged_Type then
            RawDict.Set_Package_Declares_Tagged_Type
              (The_Package => RawDict.Get_Package_Info_Ref (GetRegion (Scope)));
         end if;
      end Add_Record_Type_Local;

   begin -- Add_Record_Type
      Add_Record_Type_Local
        (Name           => Name,
         Is_Tagged_Type => Is_Tagged_Type,
         Extends        => RawDict.Get_Type_Info_Ref (Extends), -- GAA External
         Comp_Unit      => Comp_Unit,
         Declaration    => Declaration,
         Scope          => Scope,
         Context        => Context,
         Type_Mark      => Type_Mark);
      The_Type := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Record_Type;

   --------------------------------------------------------------------------------
   --  Add_Record_Component
   --------------------------------------------------------------------------------

   procedure Add_Record_Component
     (Name                   : in LexTokenManager.Lex_String;
      Comp_Unit              : in ContextManager.UnitDescriptors;
      Declaration            : in Location;
      The_Record_Type        : in RawDict.Type_Info_Ref;
      The_Component_Type     : in RawDict.Type_Info_Ref;
      InheritedField         : in Boolean;
      ComponentTypeReference : in Location)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Comp_Unit,
   --#                                Declaration,
   --#                                InheritedField,
   --#                                Name,
   --#                                The_Component_Type,
   --#                                The_Record_Type &
   --#         SPARK_IO.File_Sys from *,
   --#                                ComponentTypeReference,
   --#                                Comp_Unit,
   --#                                Declaration,
   --#                                Dict,
   --#                                InheritedField,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                The_Component_Type,
   --#                                The_Record_Type;
      is separate;

   --------------------------------------------------------------------------------

   procedure AddRecordComponent
     (Name                   : in LexTokenManager.Lex_String;
      Comp_Unit              : in ContextManager.UnitDescriptors;
      Declaration            : in Location;
      TheRecordType          : in Symbol;
      TheComponentType       : in Symbol;
      InheritedField         : in Boolean;
      ComponentTypeReference : in Location) is
   begin
      Add_Record_Component
        (Name                   => Name,
         Comp_Unit              => Comp_Unit,
         Declaration            => Declaration,
         The_Record_Type        => RawDict.Get_Type_Info_Ref (TheRecordType), -- GAA External
         The_Component_Type     => RawDict.Get_Type_Info_Ref (TheComponentType), -- GAA External
         InheritedField         => InheritedField,
         ComponentTypeReference => ComponentTypeReference);
   end AddRecordComponent;

   --------------------------------------------------------------------------------
   --  Add_Generic_Type
   --------------------------------------------------------------------------------

   procedure Add_Generic_Type_Local
     (Name        : in     LexTokenManager.Lex_String;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      Scope       : in     Scopes;
      Type_Mark   :    out RawDict.Type_Info_Ref)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys from *,
   --#                                Comp_Unit,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Scope &
   --#         Type_Mark         from Comp_Unit,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Scope;
   --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
   is
   begin
      Add_Type_Declaration
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Declaration,
         Scope       => Scope,
         Context     => ProgramContext,
         Type_Mark   => Type_Mark);
      RawDict.Set_Type_Discriminant (Type_Mark    => Type_Mark,
                                     Discriminant => Generic_Type_Item);
   end Add_Generic_Type_Local;

   --------------------------------------------------------------------------------

   procedure Add_Generic_Type
     (Name        : in     LexTokenManager.Lex_String;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      Scope       : in     Scopes;
      The_Type    :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;
   begin
      Add_Generic_Type_Local
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Declaration,
         Scope       => Scope,
         Type_Mark   => Type_Mark);
      The_Type := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Generic_Type;

   --------------------------------------------------------------------------------
   --  Set_Generic_Private_Type
   --------------------------------------------------------------------------------

   procedure Set_Generic_Private_Type_Local (Type_Mark  : in RawDict.Type_Info_Ref;
                                             Is_Limited : in Boolean)
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Is_Limited,
   --#                   Type_Mark;
   is
   begin
      RawDict.Set_Type_Kind_Of_Generic (Type_Mark       => Type_Mark,
                                        Kind_Of_Generic => Generic_Private_Type);
      if Is_Limited then
         RawDict.Set_Type_Limited (Type_Mark  => Type_Mark,
                                   Is_Limited => Sometimes);
         RawDict.Set_Type_Limited_Private (Type_Mark => Type_Mark);
      else
         RawDict.Set_Type_Limited (Type_Mark  => Type_Mark,
                                   Is_Limited => Never);
      end if;
   end Set_Generic_Private_Type_Local;

   --------------------------------------------------------------------------------

   procedure Set_Generic_Private_Type (The_Type   : in Symbol;
                                       Is_Limited : in Boolean) is
   begin
      Set_Generic_Private_Type_Local (Type_Mark  => RawDict.Get_Type_Info_Ref (The_Type), -- GAA External
                                      Is_Limited => Is_Limited);
   end Set_Generic_Private_Type;

   --------------------------------------------------------------------------------
   --  Set_Generic_Discrete_Type
   --------------------------------------------------------------------------------

   procedure Set_Generic_Discrete_Type (The_Type : in Symbol) is
   begin
      RawDict.Set_Type_Kind_Of_Generic (Type_Mark       => RawDict.Get_Type_Info_Ref (The_Type), -- GAA External
                                        Kind_Of_Generic => Generic_Discrete_Type);
   end Set_Generic_Discrete_Type;

   --------------------------------------------------------------------------------
   --  Set_Generic_Integer_Type
   --------------------------------------------------------------------------------

   procedure Set_Generic_Integer_Type (The_Type : in Symbol) is
   begin
      RawDict.Set_Type_Kind_Of_Generic (Type_Mark       => RawDict.Get_Type_Info_Ref (The_Type), -- GAA External
                                        Kind_Of_Generic => Generic_Integer_Type);
   end Set_Generic_Integer_Type;

   --------------------------------------------------------------------------------
   --  Set_Generic_Modular_Type
   --------------------------------------------------------------------------------

   procedure Set_Generic_Modular_Type (The_Type : in Symbol) is
   begin
      RawDict.Set_Type_Kind_Of_Generic (Type_Mark       => RawDict.Get_Type_Info_Ref (The_Type), -- GAA External
                                        Kind_Of_Generic => Generic_Modular_Type);
   end Set_Generic_Modular_Type;

   --------------------------------------------------------------------------------
   --  Set_Generic_Floating_Point_Type
   --------------------------------------------------------------------------------

   procedure Set_Generic_Floating_Point_Type (The_Type : in Symbol) is
   begin
      RawDict.Set_Type_Kind_Of_Generic (Type_Mark       => RawDict.Get_Type_Info_Ref (The_Type), -- GAA External
                                        Kind_Of_Generic => Generic_Floating_Point_Type);
   end Set_Generic_Floating_Point_Type;

   --------------------------------------------------------------------------------
   --  Set_Generic_Fixed_Point_Type
   --------------------------------------------------------------------------------

   procedure Set_Generic_Fixed_Point_Type (The_Type : in Symbol) is
   begin
      RawDict.Set_Type_Kind_Of_Generic (Type_Mark       => RawDict.Get_Type_Info_Ref (The_Type), -- GAA External
                                        Kind_Of_Generic => Generic_Fixed_Point_Type);
   end Set_Generic_Fixed_Point_Type;

   --------------------------------------------------------------------------------
   --  Set_Generic_Array_Type
   --------------------------------------------------------------------------------

   procedure Set_Generic_Array_Type (The_Type : in Symbol) is

      procedure Set_Generic_Array_Type_Local (Type_Mark : in RawDict.Type_Info_Ref)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   Type_Mark;
      is
      begin
         RawDict.Set_Type_Discriminant (Type_Mark    => Type_Mark,
                                        Discriminant => Generic_Type_Item);
         RawDict.Set_Type_Kind_Of_Generic (Type_Mark       => Type_Mark,
                                           Kind_Of_Generic => Generic_Array_Type);
      end Set_Generic_Array_Type_Local;

   begin -- Set_Generic_Array_Type
      Set_Generic_Array_Type_Local (Type_Mark => RawDict.Get_Type_Info_Ref (The_Type)); -- GAA External
   end Set_Generic_Array_Type;

   --------------------------------------------------------------------------------
   --  Add_Generic_Object
   --------------------------------------------------------------------------------

   procedure Add_Generic_Object
     (Name           : in     LexTokenManager.Lex_String;
      Comp_Unit      : in     ContextManager.UnitDescriptors;
      Declaration    : in     Location;
      Scope          : in     Scopes;
      The_Type       : in     Symbol;
      The_Object_Sym :    out Symbol) is
      The_Object : RawDict.Constant_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Generic_Object_Local
        (Name        : in     LexTokenManager.Lex_String;
         Comp_Unit   : in     ContextManager.UnitDescriptors;
         Declaration : in     Location;
         Scope       : in     Scopes;
         Type_Mark   : in     RawDict.Type_Info_Ref;
         The_Object  :    out RawDict.Constant_Info_Ref)
      --# global in out Dict;
      --# derives Dict,
      --#         The_Object from Comp_Unit,
      --#                         Declaration,
      --#                         Dict,
      --#                         Name,
      --#                         Scope,
      --#                         Type_Mark;
      --# post The_Object /= RawDict.Null_Constant_Info_Ref;
      is
         The_Declaration : RawDict.Declaration_Info_Ref;
      begin
         Add_Declaration
           (Comp_Unit       => Comp_Unit,
            Loc             => Declaration,
            Scope           => Scope,
            Context         => ProgramContext,
            The_Declaration => The_Declaration);
         RawDict.Create_Constant
           (Name            => Name,
            Type_Mark       => Type_Mark,
            Static          => False,
            The_Declaration => The_Declaration,
            Is_Deferred     => False,
            Comp_Unit       => Comp_Unit,
            Loc             => Declaration.Start_Position,
            The_Constant    => The_Object);
      end Add_Generic_Object_Local;

   begin -- Add_Generic_Object
      Add_Generic_Object_Local
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Declaration,
         Scope       => Scope,
         Type_Mark   => RawDict.Get_Type_Info_Ref (The_Type), -- GAA External
         The_Object  => The_Object);
      The_Object_Sym := RawDict.Get_Constant_Symbol (The_Object); -- GAA External
   end Add_Generic_Object;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   procedure AddRecordComponentRepresentation
     (Component        : in Symbol;
      Clause           : in Location;
      RelativeAddress  : in Natural;
      FirstBitPosition : in Natural;
      LastBitPosition  : in Natural) is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         Write_String (Dict.TemporaryFile, "record component representation of ");
         Write_Name (File => Dict.TemporaryFile,
                     Item => Component);
         Write_Space (File => Dict.TemporaryFile);
         Write_Integer (Dict.TemporaryFile, RelativeAddress);
         Write_String (Dict.TemporaryFile, " range ");
         Write_Integer (Dict.TemporaryFile, FirstBitPosition);
         Write_String (Dict.TemporaryFile, " ... ");
         Write_Integer (Dict.TemporaryFile, LastBitPosition);
         Write_String (Dict.TemporaryFile, " is at ");
         Write_Location (File => Dict.TemporaryFile,
                         Loc  => Clause);
         Write_Line (Dict.TemporaryFile, " ;");
      end if;
   end AddRecordComponentRepresentation;

   --------------------------------------------------------------------------------

   procedure AddAlignmentClause (TheType : in Symbol;
                                 Clause  : in Location) is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         Write_String (Dict.TemporaryFile, "alignment clause of ");
         Write_Name (File => Dict.TemporaryFile,
                     Item => TheType);
         Write_String (Dict.TemporaryFile, " is at ");
         Write_Location (File => Dict.TemporaryFile,
                         Loc  => Clause);
         Write_Line (Dict.TemporaryFile, " ;");
      end if;
   end AddAlignmentClause;

   --------------------------------------------------------------------------------

   procedure AddLoop
     (Scope         : in     Scopes;
      Comp_Unit     : in     ContextManager.UnitDescriptors;
      LoopStatement : in     Location;
      TheLoop       :    out Symbol) is separate;

   --------------------------------------------------------------------------------

   procedure AddLoopName (Name    : in LexTokenManager.Lex_String;
                          TheLoop : in Symbol) is
   begin
      RawDict.SetLoopName (Name, TheLoop);
   end AddLoopName;

   --------------------------------------------------------------------------------
   --  AddLoopParameter
   --------------------------------------------------------------------------------

   procedure AddLoopParameter
     (TheLoop       : in Symbol;
      Comp_Unit     : in ContextManager.UnitDescriptors;
      Declaration   : in Location;
      Name          : in LexTokenManager.Lex_String;
      TypeMark      : in Symbol;
      StaticRange   : in Boolean;
      IsReverse     : in Boolean;
      TypeReference : in Location) is

      procedure Add_Loop_Parameter
        (The_Loop       : in Symbol;
         Comp_Unit      : in ContextManager.UnitDescriptors;
         Declaration    : in Location;
         Name           : in LexTokenManager.Lex_String;
         Type_Mark      : in RawDict.Type_Info_Ref;
         Static_Range   : in Boolean;
         Is_Reverse     : in Boolean;
         Type_Reference : in Location)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Declaration,
      --#                                Is_Reverse,
      --#                                Name,
      --#                                Static_Range,
      --#                                The_Loop,
      --#                                Type_Mark &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Declaration,
      --#                                Dict,
      --#                                Is_Reverse,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Static_Range,
      --#                                The_Loop,
      --#                                Type_Mark,
      --#                                Type_Reference;
      is
         Loop_Parameter : Symbol;
      begin
         RawDict.CreateLoopParameter
           (Name           => Name,
            Type_Mark      => Type_Mark,
            TheLoop        => The_Loop,
            HasStaticRange => Static_Range,
            IsReverse      => Is_Reverse,
            Comp_Unit      => Comp_Unit,
            Loc            => Declaration.Start_Position,
            LoopParameter  => Loop_Parameter);
         RawDict.SetLoopParameter (The_Loop, Loop_Parameter);

         if Type_Mark /= Get_Unknown_Type_Mark then
            AddOtherReference (RawDict.Get_Type_Symbol (Type_Mark), The_Loop, Type_Reference);
         end if;
      end Add_Loop_Parameter;

   begin -- AddLoopParameter
      Add_Loop_Parameter
        (The_Loop       => TheLoop,
         Comp_Unit      => Comp_Unit,
         Declaration    => Declaration,
         Name           => Name,
         Type_Mark      => RawDict.Get_Type_Info_Ref (TypeMark), -- GAA External
         Static_Range   => StaticRange,
         Is_Reverse     => IsReverse,
         Type_Reference => TypeReference);
   end AddLoopParameter;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   procedure MarkLoopHasExits (TheLoop : in Symbol) is
   begin
      if RawDict.GetSymbolDiscriminant (TheLoop) = LoopSymbol then
         RawDict.SetLoopHasExits (TheLoop);
      end if;
      -- if it is not a loop then the exit statement must be illegally placed so we do nothing
   end MarkLoopHasExits;

   --------------------------------------------------------------------------------

   procedure SetLoopExitExpn (ForLoop : in Symbol;
                              Expn    : in Natural) is
   begin
      RawDict.SetLoopExitExpn (ForLoop, Expn);
   end SetLoopExitExpn;

   --------------------------------------------------------------------------------

   procedure SetLoopEntryExpn (ForLoop : in Symbol;
                               Expn    : in Natural) is
   begin
      RawDict.SetLoopEntryExpn (ForLoop, Expn);
   end SetLoopEntryExpn;

   --------------------------------------------------------------------------------

   function Loop_Var_Only_An_Import (OriginalVariable : Symbol;
                                     TheLoop          : Symbol) return Boolean
   --# global in Dict;
   is
      CurrentRegion  : Symbol;
      IsOnlyAnImport : Boolean := False;
      Stop           : Boolean := False;
   begin
      CurrentRegion := RawDict.GetLoopRegion (TheLoop);
      while not Stop loop
         case RawDict.GetSymbolDiscriminant (CurrentRegion) is
            when Subprogram_Symbol =>
               --# accept Flow, 41, "Stable expression expected here";
               case RawDict.GetSymbolDiscriminant (OriginalVariable) is
                  when Variable_Symbol =>
                     IsOnlyAnImport :=
                       (Is_Imported_Subprogram_Variable
                          (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => CurrentRegion),
                           Abstraction    => IsAbstract,
                           The_Variable   => RawDict.Get_Variable_Info_Ref (Item => OriginalVariable))
                          or else Is_Imported_Subprogram_Variable
                          (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => CurrentRegion),
                           Abstraction    => IsRefined,
                           The_Variable   => RawDict.Get_Variable_Info_Ref (Item => OriginalVariable)))
                       and then not (Is_Exported_Subprogram_Variable
                                       (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => CurrentRegion),
                                        Abstraction    => IsAbstract,
                                        The_Variable   => RawDict.Get_Variable_Info_Ref (Item => OriginalVariable))
                                       or else Is_Exported_Subprogram_Variable
                                       (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => CurrentRegion),
                                        Abstraction    => IsRefined,
                                        The_Variable   => RawDict.Get_Variable_Info_Ref (Item => OriginalVariable)));
                  when Subprogram_Parameter_Symbol =>
                     IsOnlyAnImport :=
                       (Is_Imported_Subprogram_Subprogram_Parameter
                          (The_Subprogram             => RawDict.Get_Subprogram_Info_Ref (Item => CurrentRegion),
                           Abstraction                => IsAbstract,
                           The_Subprogram_Parameter   => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => OriginalVariable),
                           Is_Implicit_Proof_Function => False)
                          or else Is_Imported_Subprogram_Subprogram_Parameter
                          (The_Subprogram             => RawDict.Get_Subprogram_Info_Ref (Item => CurrentRegion),
                           Abstraction                => IsRefined,
                           The_Subprogram_Parameter   => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => OriginalVariable),
                           Is_Implicit_Proof_Function => False))
                       and then not (Is_Exported_Subprogram_Subprogram_Parameter
                                       (The_Subprogram             => RawDict.Get_Subprogram_Info_Ref (Item => CurrentRegion),
                                        Abstraction                => IsAbstract,
                                        The_Subprogram_Parameter   => RawDict.Get_Subprogram_Parameter_Info_Ref
                                          (Item => OriginalVariable),
                                        Is_Implicit_Proof_Function => False)
                                       or else Is_Exported_Subprogram_Subprogram_Parameter
                                       (The_Subprogram             => RawDict.Get_Subprogram_Info_Ref (Item => CurrentRegion),
                                        Abstraction                => IsRefined,
                                        The_Subprogram_Parameter   => RawDict.Get_Subprogram_Parameter_Info_Ref
                                          (Item => OriginalVariable),
                                        Is_Implicit_Proof_Function => False));
                  when others => -- non-exec code
                     IsOnlyAnImport := False;
                     SystemErrors.Fatal_Error
                       (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                        Msg     => "in Dictionary.Loop_Var_Only_An_Import");
               end case;
               --# end accept;
               Stop := True;
            when ImplicitProofFunctionSymbol =>
               --# accept Flow, 41, "Stable expression expected here";
               case RawDict.GetSymbolDiscriminant (OriginalVariable) is
                  when Variable_Symbol =>
                     IsOnlyAnImport :=
                       (Is_Imported_Subprogram_Variable
                          (The_Subprogram => RawDict.GetImplicitProofFunctionAdaFunction (CurrentRegion),
                           Abstraction    => IsAbstract,
                           The_Variable   => RawDict.Get_Variable_Info_Ref (Item => OriginalVariable))
                          or else Is_Imported_Subprogram_Variable
                          (The_Subprogram => RawDict.GetImplicitProofFunctionAdaFunction (CurrentRegion),
                           Abstraction    => IsRefined,
                           The_Variable   => RawDict.Get_Variable_Info_Ref (Item => OriginalVariable)))
                       and then not (Is_Exported_Subprogram_Variable
                                       (The_Subprogram => RawDict.GetImplicitProofFunctionAdaFunction (CurrentRegion),
                                        Abstraction    => IsAbstract,
                                        The_Variable   => RawDict.Get_Variable_Info_Ref (Item => OriginalVariable))
                                       or else Is_Exported_Subprogram_Variable
                                       (The_Subprogram => RawDict.GetImplicitProofFunctionAdaFunction (CurrentRegion),
                                        Abstraction    => IsRefined,
                                        The_Variable   => RawDict.Get_Variable_Info_Ref (Item => OriginalVariable)));
                  when Subprogram_Parameter_Symbol =>
                     IsOnlyAnImport :=
                       (Is_Imported_Subprogram_Subprogram_Parameter
                          (The_Subprogram             => RawDict.GetImplicitProofFunctionAdaFunction (CurrentRegion),
                           Abstraction                => IsAbstract,
                           The_Subprogram_Parameter   => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => OriginalVariable),
                           Is_Implicit_Proof_Function => True)
                          or else Is_Imported_Subprogram_Subprogram_Parameter
                          (The_Subprogram             => RawDict.GetImplicitProofFunctionAdaFunction (CurrentRegion),
                           Abstraction                => IsRefined,
                           The_Subprogram_Parameter   => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => OriginalVariable),
                           Is_Implicit_Proof_Function => True))
                       and then not (Is_Exported_Subprogram_Subprogram_Parameter
                                       (The_Subprogram             => RawDict.GetImplicitProofFunctionAdaFunction (CurrentRegion),
                                        Abstraction                => IsAbstract,
                                        The_Subprogram_Parameter   => RawDict.Get_Subprogram_Parameter_Info_Ref
                                          (Item => OriginalVariable),
                                        Is_Implicit_Proof_Function => True)
                                       or else Is_Exported_Subprogram_Subprogram_Parameter
                                       (The_Subprogram             => RawDict.GetImplicitProofFunctionAdaFunction (CurrentRegion),
                                        Abstraction                => IsRefined,
                                        The_Subprogram_Parameter   => RawDict.Get_Subprogram_Parameter_Info_Ref
                                          (Item => OriginalVariable),
                                        Is_Implicit_Proof_Function => True));
                  when others => -- non-exec code
                     IsOnlyAnImport := False;
                     SystemErrors.Fatal_Error
                       (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                        Msg     => "in Dictionary.Loop_Var_Only_An_Import");
               end case;
               --# end accept;
               Stop := True;
            when Type_Symbol =>
               SystemErrors.RT_Assert
                 (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => CurrentRegion)),
                  Sys_Err => SystemErrors.Invalid_Symbol_Table,
                  Msg     => "in Dictionary.Loop_Var_Only_An_Import");
               IsOnlyAnImport :=
                 (Is_Imported_Task_Type_Variable
                    (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => CurrentRegion),
                     Abstraction   => IsAbstract,
                     The_Variable  => RawDict.Get_Variable_Info_Ref (OriginalVariable))
                    or else Is_Imported_Task_Type_Variable
                    (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => CurrentRegion),
                     Abstraction   => IsRefined,
                     The_Variable  => RawDict.Get_Variable_Info_Ref (OriginalVariable)))
                 and then not (Is_Exported_Task_Type_Variable
                                 (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => CurrentRegion),
                                  Abstraction   => IsAbstract,
                                  The_Variable  => RawDict.Get_Variable_Info_Ref (OriginalVariable))
                                 or else Is_Exported_Task_Type_Variable
                                 (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => CurrentRegion),
                                  Abstraction   => IsRefined,
                                  The_Variable  => RawDict.Get_Variable_Info_Ref (OriginalVariable)));
               Stop           := True;
            when others =>
               CurrentRegion := RawDict.GetLoopRegion (CurrentRegion);
         end case;
      end loop;
      return IsOnlyAnImport;
   end Loop_Var_Only_An_Import;

   --------------------------------------------------------------------------------

   procedure IdempotentCreateLoopOnEntryVariable
     (OriginalVariable : in     Symbol;
      TheLoop          : in     Symbol;
      OnEntryVariable  :    out Symbol) is
      Found                : Boolean := False;
      OnEntryVariableToTry : Symbol;
      NewOnEntryVariable   : Symbol;
   begin
      -- Before creating a new variable, check whether the standard variable
      -- itself is acceptable in this context due to non-modification.
      if (RawDict.GetSymbolDiscriminant (OriginalVariable) = Variable_Symbol
            or else RawDict.GetSymbolDiscriminant (OriginalVariable) = Subprogram_Parameter_Symbol)
        and then Loop_Var_Only_An_Import (OriginalVariable => OriginalVariable,
                                          TheLoop          => TheLoop) then
         OnEntryVariable := OriginalVariable;
      else

         -- Search linked list of OnEntryVariables associated with TheLoop.  If an entry matching
         -- OriginalVariable is found then current search posiiton; this is the
         -- required OnEntryVariable symbol.  Otherwise, create a new entry and link it in to list.

         OnEntryVariableToTry := RawDict.GetLoopOnEntryVars (TheLoop);
         while OnEntryVariableToTry /= NullSymbol loop
            Found := RawDict.GetLoopEntryVariableOriginalVar (OnEntryVariableToTry) = OriginalVariable;
            exit when Found;

            OnEntryVariableToTry := RawDict.GetLoopEntryVariableNext (OnEntryVariableToTry);
         end loop;

         if Found then
            -- Return existing var
            OnEntryVariable := OnEntryVariableToTry;
         else
            RawDict.CreateLoopEntryVariable
              (OriginalVar       => OriginalVariable,
               TheLoop           => TheLoop,
               Comp_Unit         => ContextManager.NullUnit,
               Loc               => LexTokenManager.Null_Token_Position,
               LoopEntryVariable => NewOnEntryVariable);
            -- Insert it at the head of the list
            RawDict.SetLoopEntryVariableNext (NewOnEntryVariable, RawDict.GetLoopOnEntryVars (TheLoop));
            RawDict.SetLoopOnEntryVars (TheLoop, NewOnEntryVariable);
            -- Return newly created var
            OnEntryVariable := NewOnEntryVariable;
         end if;
      end if;
   end IdempotentCreateLoopOnEntryVariable;

   --------------------------------------------------------------------------------

   function GetLoopOnEntryVariable (OriginalVariable : Symbol;
                                    TheLoop          : Symbol) return Symbol is
      Result               : Symbol := NullSymbol; -- Default result for error case only
      OnEntryVariableToTry : Symbol;

   begin
      if (RawDict.GetSymbolDiscriminant (OriginalVariable) = Variable_Symbol
            or else RawDict.GetSymbolDiscriminant (OriginalVariable) = Subprogram_Parameter_Symbol)
        and then Loop_Var_Only_An_Import (OriginalVariable => OriginalVariable,
                                          TheLoop          => TheLoop) then
         Result := OriginalVariable;
      else
         OnEntryVariableToTry := RawDict.GetLoopOnEntryVars (TheLoop);
         while OnEntryVariableToTry /= NullSymbol loop
            if RawDict.GetLoopEntryVariableOriginalVar (OnEntryVariableToTry) = OriginalVariable then
               Result := OnEntryVariableToTry;
               exit;
            end if;
            OnEntryVariableToTry := RawDict.GetLoopEntryVariableNext (OnEntryVariableToTry);
         end loop;
      end if;
      return Result;
   end GetLoopOnEntryVariable;

   --------------------------------------------------------------------------------

   procedure Add_Subtype_Declaration
     (Name                  : in     LexTokenManager.Lex_String;
      Is_Full_Range_Subtype : in     Boolean;
      Parent                : in     RawDict.Type_Info_Ref;
      Parent_Reference      : in     Location;
      Comp_Unit             : in     ContextManager.UnitDescriptors;
      Declaration           : in     Location;
      Scope                 : in     Scopes;
      Context               : in     Contexts;
      Type_Mark             :    out RawDict.Type_Info_Ref)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Comp_Unit,
   --#                                Context,
   --#                                Declaration,
   --#                                Is_Full_Range_Subtype,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Parent,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Comp_Unit,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                Is_Full_Range_Subtype,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Parent,
   --#                                Parent_Reference,
   --#                                Scope &
   --#         Type_Mark         from Comp_Unit,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Scope;
   --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
   is
   begin
      Add_Type_Declaration (Name        => Name,
                            Comp_Unit   => Comp_Unit,
                            Declaration => Declaration,
                            Scope       => Scope,
                            Context     => Context,
                            Type_Mark   => Type_Mark);

      if Is_Full_Range_Subtype then
         RawDict.Set_Type_Is_Full_Range_Subtype (Type_Mark => Type_Mark);
      end if;

      RawDict.Set_Type_Parent (Type_Mark => Type_Mark,
                               Parent    => Parent);

      if Parent /= Get_Unknown_Type_Mark then
         AddOtherReference (RawDict.Get_Type_Symbol (Parent), GetRegion (Scope), Parent_Reference);
      end if;
   end Add_Subtype_Declaration;

   --------------------------------------------------------------------------------
   --  Add_Full_Range_Subtype
   --------------------------------------------------------------------------------

   procedure Add_Full_Range_Subtype
     (Name             : in     LexTokenManager.Lex_String;
      Parent           : in     Symbol;
      Parent_Reference : in     Location;
      Comp_Unit        : in     ContextManager.UnitDescriptors;
      Declaration      : in     Location;
      Scope            : in     Scopes;
      Context          : in     Contexts;
      The_Subtype      :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;
   begin
      Add_Subtype_Declaration
        (Name                  => Name,
         Is_Full_Range_Subtype => True,
         Parent                => RawDict.Get_Type_Info_Ref (Parent), -- GAA External
         Parent_Reference      => Parent_Reference,
         Comp_Unit             => Comp_Unit,
         Declaration           => Declaration,
         Scope                 => Scope,
         Context               => Context,
         Type_Mark             => Type_Mark);
      The_Subtype := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Full_Range_Subtype;

   --------------------------------------------------------------------------------
   --  Add_Enumeration_Subtype
   --------------------------------------------------------------------------------

   procedure Add_Enumeration_Subtype
     (Name             : in     LexTokenManager.Lex_String;
      Static           : in     Boolean;
      Parent           : in     Symbol;
      Parent_Reference : in     Location;
      Lower            : in     LexTokenManager.Lex_String;
      Upper            : in     LexTokenManager.Lex_String;
      Comp_Unit        : in     ContextManager.UnitDescriptors;
      Declaration      : in     Location;
      Scope            : in     Scopes;
      Context          : in     Contexts;
      The_Subtype      :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Enumeration_Subtype_Local
        (Name             : in     LexTokenManager.Lex_String;
         Static           : in     Boolean;
         Parent           : in     RawDict.Type_Info_Ref;
         Parent_Reference : in     Location;
         Lower            : in     LexTokenManager.Lex_String;
         Upper            : in     LexTokenManager.Lex_String;
         Comp_Unit        : in     ContextManager.UnitDescriptors;
         Declaration      : in     Location;
         Scope            : in     Scopes;
         Context          : in     Contexts;
         Type_Mark        :    out RawDict.Type_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                LexTokenManager.State,
      --#                                Lower,
      --#                                Name,
      --#                                Parent,
      --#                                Scope,
      --#                                Static,
      --#                                Upper &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Parent,
      --#                                Parent_Reference,
      --#                                Scope &
      --#         Type_Mark         from Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope;
      --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
      is
      begin
         Add_Subtype_Declaration
           (Name                  => Name,
            Is_Full_Range_Subtype => False,
            Parent                => Parent,
            Parent_Reference      => Parent_Reference,
            Comp_Unit             => Comp_Unit,
            Declaration           => Declaration,
            Scope                 => Scope,
            Context               => Context,
            Type_Mark             => Type_Mark);
         Add_Enumeration_Type_Mark (Type_Mark => Type_Mark,
                                    Static    => Static);
         RawDict.Set_Type_Lower (Type_Mark => Type_Mark,
                                 Lower     => Lower);
         RawDict.Set_Type_Upper (Type_Mark => Type_Mark,
                                 Upper     => Upper);
      end Add_Enumeration_Subtype_Local;

   begin -- Add_Enumeration_Subtype
      Add_Enumeration_Subtype_Local
        (Name             => Name,
         Static           => Static,
         Parent           => RawDict.Get_Type_Info_Ref (Parent), -- GAA External
         Parent_Reference => Parent_Reference,
         Lower            => Lower,
         Upper            => Upper,
         Comp_Unit        => Comp_Unit,
         Declaration      => Declaration,
         Scope            => Scope,
         Context          => Context,
         Type_Mark        => Type_Mark);
      The_Subtype := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Enumeration_Subtype;

   --------------------------------------------------------------------------------
   --  Add_Integer_Subtype
   --------------------------------------------------------------------------------

   procedure Add_Integer_Subtype
     (Name             : in     LexTokenManager.Lex_String;
      Static           : in     Boolean;
      Parent           : in     Symbol;
      Parent_Reference : in     Location;
      Lower            : in     LexTokenManager.Lex_String;
      Upper            : in     LexTokenManager.Lex_String;
      Comp_Unit        : in     ContextManager.UnitDescriptors;
      Declaration      : in     Location;
      Scope            : in     Scopes;
      Context          : in     Contexts;
      The_Subtype      :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Integer_Subtype_Local
        (Name             : in     LexTokenManager.Lex_String;
         Static           : in     Boolean;
         Parent           : in     RawDict.Type_Info_Ref;
         Parent_Reference : in     Location;
         Lower            : in     LexTokenManager.Lex_String;
         Upper            : in     LexTokenManager.Lex_String;
         Comp_Unit        : in     ContextManager.UnitDescriptors;
         Declaration      : in     Location;
         Scope            : in     Scopes;
         Context          : in     Contexts;
         Type_Mark        :    out RawDict.Type_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                LexTokenManager.State,
      --#                                Lower,
      --#                                Name,
      --#                                Parent,
      --#                                Scope,
      --#                                Static,
      --#                                Upper &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Parent,
      --#                                Parent_Reference,
      --#                                Scope &
      --#         Type_Mark         from Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope;
      --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
      is
      begin
         Add_Subtype_Declaration
           (Name                  => Name,
            Is_Full_Range_Subtype => False,
            Parent                => Parent,
            Parent_Reference      => Parent_Reference,
            Comp_Unit             => Comp_Unit,
            Declaration           => Declaration,
            Scope                 => Scope,
            Context               => Context,
            Type_Mark             => Type_Mark);
         Add_Integer_Type_Mark (Type_Mark => Type_Mark,
                                Static    => Static,
                                Lower     => Lower,
                                Upper     => Upper);
      end Add_Integer_Subtype_Local;

   begin -- Add_Integer_Subtype
      Add_Integer_Subtype_Local
        (Name             => Name,
         Static           => Static,
         Parent           => RawDict.Get_Type_Info_Ref (Parent), -- GAA External
         Parent_Reference => Parent_Reference,
         Lower            => Lower,
         Upper            => Upper,
         Comp_Unit        => Comp_Unit,
         Declaration      => Declaration,
         Scope            => Scope,
         Context          => Context,
         Type_Mark        => Type_Mark);
      The_Subtype := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Integer_Subtype;

   --------------------------------------------------------------------------------
   --  Add_Modular_Subtype
   --------------------------------------------------------------------------------

   procedure Add_Modular_Subtype
     (Name             : in     LexTokenManager.Lex_String;
      Parent           : in     Symbol;
      Parent_Reference : in     Location;
      Lower            : in     LexTokenManager.Lex_String;
      Upper            : in     LexTokenManager.Lex_String;
      Comp_Unit        : in     ContextManager.UnitDescriptors;
      Declaration      : in     Location;
      Scope            : in     Scopes;
      Context          : in     Contexts;
      The_Subtype      :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Modular_Subtype_Local
        (Name             : in     LexTokenManager.Lex_String;
         Parent           : in     RawDict.Type_Info_Ref;
         Parent_Reference : in     Location;
         Lower            : in     LexTokenManager.Lex_String;
         Upper            : in     LexTokenManager.Lex_String;
         Comp_Unit        : in     ContextManager.UnitDescriptors;
         Declaration      : in     Location;
         Scope            : in     Scopes;
         Context          : in     Contexts;
         Type_Mark        :    out RawDict.Type_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                LexTokenManager.State,
      --#                                Lower,
      --#                                Name,
      --#                                Parent,
      --#                                Scope,
      --#                                Upper &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Parent,
      --#                                Parent_Reference,
      --#                                Scope &
      --#         Type_Mark         from Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope;
      --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
      is
      begin
         Add_Subtype_Declaration
           (Name                  => Name,
            Is_Full_Range_Subtype => False,
            Parent                => Parent,
            Parent_Reference      => Parent_Reference,
            Comp_Unit             => Comp_Unit,
            Declaration           => Declaration,
            Scope                 => Scope,
            Context               => Context,
            Type_Mark             => Type_Mark);
         Add_Modular_Type_Mark
           (Type_Mark => Type_Mark,
            Static    => True,
            Lower     => Lower,
            Upper     => Upper,
            Modulus   => Get_Scalar_Attribute_Value (False, LexTokenManager.Modulus_Token, Parent));
      end Add_Modular_Subtype_Local;

   begin -- Add_Modular_Subtype
      Add_Modular_Subtype_Local
        (Name             => Name,
         Parent           => RawDict.Get_Type_Info_Ref (Parent), -- GAA External
         Parent_Reference => Parent_Reference,
         Lower            => Lower,
         Upper            => Upper,
         Comp_Unit        => Comp_Unit,
         Declaration      => Declaration,
         Scope            => Scope,
         Context          => Context,
         Type_Mark        => Type_Mark);
      The_Subtype := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Modular_Subtype;

   --------------------------------------------------------------------------------
   --  Add_Floating_Point_Subtype
   --------------------------------------------------------------------------------

   procedure Add_Floating_Point_Subtype
     (Name             : in     LexTokenManager.Lex_String;
      Static           : in     Boolean;
      Parent           : in     Symbol;
      Parent_Reference : in     Location;
      Lower            : in     LexTokenManager.Lex_String;
      Upper            : in     LexTokenManager.Lex_String;
      Error_Bound      : in     LexTokenManager.Lex_String;
      Comp_Unit        : in     ContextManager.UnitDescriptors;
      Declaration      : in     Location;
      Scope            : in     Scopes;
      Context          : in     Contexts;
      The_Subtype      :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Floating_Point_Subtype_Local
        (Name             : in     LexTokenManager.Lex_String;
         Static           : in     Boolean;
         Parent           : in     RawDict.Type_Info_Ref;
         Parent_Reference : in     Location;
         Lower            : in     LexTokenManager.Lex_String;
         Upper            : in     LexTokenManager.Lex_String;
         Error_Bound      : in     LexTokenManager.Lex_String;
         Comp_Unit        : in     ContextManager.UnitDescriptors;
         Declaration      : in     Location;
         Scope            : in     Scopes;
         Context          : in     Contexts;
         Type_Mark        :    out RawDict.Type_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Error_Bound,
      --#                                LexTokenManager.State,
      --#                                Lower,
      --#                                Name,
      --#                                Parent,
      --#                                Scope,
      --#                                Static,
      --#                                Upper &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Parent,
      --#                                Parent_Reference,
      --#                                Scope &
      --#         Type_Mark         from Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope;
      --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
      is
      begin
         Add_Subtype_Declaration
           (Name                  => Name,
            Is_Full_Range_Subtype => False,
            Parent                => Parent,
            Parent_Reference      => Parent_Reference,
            Comp_Unit             => Comp_Unit,
            Declaration           => Declaration,
            Scope                 => Scope,
            Context               => Context,
            Type_Mark             => Type_Mark);
         Add_Floating_Point_Type_Mark (Type_Mark => Type_Mark,
                                       Static    => Static);
         RawDict.Set_Type_Lower (Type_Mark => Type_Mark,
                                 Lower     => Lower);
         RawDict.Set_Type_Upper (Type_Mark => Type_Mark,
                                 Upper     => Upper);
         RawDict.Set_Type_Error_Bound (Type_Mark   => Type_Mark,
                                       Error_Bound => Error_Bound);
      end Add_Floating_Point_Subtype_Local;

   begin -- Add_Floating_Point_Subtype
      Add_Floating_Point_Subtype_Local
        (Name             => Name,
         Static           => Static,
         Parent           => RawDict.Get_Type_Info_Ref (Parent), -- GAA External
         Parent_Reference => Parent_Reference,
         Lower            => Lower,
         Upper            => Upper,
         Error_Bound      => Error_Bound,
         Comp_Unit        => Comp_Unit,
         Declaration      => Declaration,
         Scope            => Scope,
         Context          => Context,
         Type_Mark        => Type_Mark);
      The_Subtype := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Floating_Point_Subtype;

   --------------------------------------------------------------------------------
   --  Add_Fixed_Point_Subtype
   --------------------------------------------------------------------------------

   procedure Add_Fixed_Point_Subtype
     (Name             : in     LexTokenManager.Lex_String;
      Static           : in     Boolean;
      Parent           : in     Symbol;
      Parent_Reference : in     Location;
      Lower            : in     LexTokenManager.Lex_String;
      Upper            : in     LexTokenManager.Lex_String;
      Error_Bound      : in     LexTokenManager.Lex_String;
      Comp_Unit        : in     ContextManager.UnitDescriptors;
      Declaration      : in     Location;
      Scope            : in     Scopes;
      Context          : in     Contexts;
      The_Subtype      :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Fixed_Point_Subtype_Local
        (Name             : in     LexTokenManager.Lex_String;
         Static           : in     Boolean;
         Parent           : in     RawDict.Type_Info_Ref;
         Parent_Reference : in     Location;
         Lower            : in     LexTokenManager.Lex_String;
         Upper            : in     LexTokenManager.Lex_String;
         Error_Bound      : in     LexTokenManager.Lex_String;
         Comp_Unit        : in     ContextManager.UnitDescriptors;
         Declaration      : in     Location;
         Scope            : in     Scopes;
         Context          : in     Contexts;
         Type_Mark        :    out RawDict.Type_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Error_Bound,
      --#                                LexTokenManager.State,
      --#                                Lower,
      --#                                Name,
      --#                                Parent,
      --#                                Scope,
      --#                                Static,
      --#                                Upper &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Parent,
      --#                                Parent_Reference,
      --#                                Scope &
      --#         Type_Mark         from Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope;
      --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
      is
      begin
         Add_Subtype_Declaration
           (Name                  => Name,
            Is_Full_Range_Subtype => False,
            Parent                => Parent,
            Parent_Reference      => Parent_Reference,
            Comp_Unit             => Comp_Unit,
            Declaration           => Declaration,
            Scope                 => Scope,
            Context               => Context,
            Type_Mark             => Type_Mark);
         Add_Fixed_Point_Type_Mark (Type_Mark => Type_Mark,
                                    Static    => Static);
         RawDict.Set_Type_Lower (Type_Mark => Type_Mark,
                                 Lower     => Lower);
         RawDict.Set_Type_Upper (Type_Mark => Type_Mark,
                                 Upper     => Upper);
         RawDict.Set_Type_Error_Bound (Type_Mark   => Type_Mark,
                                       Error_Bound => Error_Bound);
      end Add_Fixed_Point_Subtype_Local;

   begin -- Add_Fixed_Point_Subtype
      Add_Fixed_Point_Subtype_Local
        (Name             => Name,
         Static           => Static,
         Parent           => RawDict.Get_Type_Info_Ref (Parent), -- GAA External
         Parent_Reference => Parent_Reference,
         Lower            => Lower,
         Upper            => Upper,
         Error_Bound      => Error_Bound,
         Comp_Unit        => Comp_Unit,
         Declaration      => Declaration,
         Scope            => Scope,
         Context          => Context,
         Type_Mark        => Type_Mark);
      The_Subtype := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Fixed_Point_Subtype;

   --------------------------------------------------------------------------------
   --  Add_Array_Subtype
   --------------------------------------------------------------------------------

   procedure Add_Array_Subtype
     (Name             : in     LexTokenManager.Lex_String;
      Parent           : in     Symbol;
      Parent_Reference : in     Location;
      Comp_Unit        : in     ContextManager.UnitDescriptors;
      Declaration      : in     Location;
      Scope            : in     Scopes;
      Context          : in     Contexts;
      Static           : in     Boolean;
      The_Subtype      :    out Symbol) is
      Type_Mark : RawDict.Type_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Array_Subtype_Local
        (Name             : in     LexTokenManager.Lex_String;
         Parent           : in     RawDict.Type_Info_Ref;
         Parent_Reference : in     Location;
         Comp_Unit        : in     ContextManager.UnitDescriptors;
         Declaration      : in     Location;
         Scope            : in     Scopes;
         Context          : in     Contexts;
         Static           : in     Boolean;
         Type_Mark        :    out RawDict.Type_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Parent,
      --#                                Scope,
      --#                                Static &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Parent,
      --#                                Parent_Reference,
      --#                                Scope &
      --#         Type_Mark         from Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope;
      --# post Type_Mark /= RawDict.Null_Type_Info_Ref;
      is
      begin
         Add_Subtype_Declaration
           (Name                  => Name,
            Is_Full_Range_Subtype => False,
            Parent                => Parent,
            Parent_Reference      => Parent_Reference,
            Comp_Unit             => Comp_Unit,
            Declaration           => Declaration,
            Scope                 => Scope,
            Context               => Context,
            Type_Mark             => Type_Mark);
         Add_Array_Type_Mark
           (Type_Mark      => Type_Mark,
            Component_Type => Get_Array_Component (Type_Mark => RawDict.Get_Type_Parent (Type_Mark => Type_Mark)),
            Static         => Static);
         RawDict.Set_Type_Constrained (Type_Mark   => Type_Mark,
                                       Constrained => True);
      end Add_Array_Subtype_Local;

   begin -- Add_Array_Subtype
      Add_Array_Subtype_Local
        (Name             => Name,
         Parent           => RawDict.Get_Type_Info_Ref (Parent), -- GAA External
         Parent_Reference => Parent_Reference,
         Comp_Unit        => Comp_Unit,
         Declaration      => Declaration,
         Scope            => Scope,
         Context          => Context,
         Static           => Static,
         Type_Mark        => Type_Mark);
      The_Subtype := RawDict.Get_Type_Symbol (Type_Mark); -- GAA External
   end Add_Array_Subtype;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   procedure Write_Proof_Statement (Discriminant : in ProofStatementDiscriminant;
                                    Statement    : in Location)
   --# global in     Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                Discriminant,
   --#                                Statement;
   is

      procedure Write_Discriminant (File         : in SPARK_IO.File_Type;
                                    Discriminant : in ProofStatementDiscriminant)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Discriminant,
      --#                                File;
      is
      begin
         case Discriminant is
            when AssertStatement =>
               Write_String (File, "assert statement");
            when CheckStatement =>
               Write_String (File, "check statement");
         end case;
      end Write_Discriminant;

   begin -- Write_Proof_Statement
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         Write_Discriminant (File         => Dict.TemporaryFile,
                             Discriminant => Discriminant);
         Write_String (Dict.TemporaryFile, " at ");
         Write_Location (File => Dict.TemporaryFile,
                         Loc  => Statement);
         Write_Line (Dict.TemporaryFile, " ;");
      end if;
   end Write_Proof_Statement;

   --------------------------------------------------------------------------------

   procedure AddAssertStatement (Statement : in Location) is
   begin
      Write_Proof_Statement (Discriminant => AssertStatement,
                             Statement    => Statement);
   end AddAssertStatement;

   --------------------------------------------------------------------------------

   procedure AddCheckStatement (Statement : in Location) is
   begin
      Write_Proof_Statement (Discriminant => CheckStatement,
                             Statement    => Statement);
   end AddCheckStatement;

   --------------------------------------------------------------------------------
   --  Add_Precondition
   --------------------------------------------------------------------------------

   procedure AddPrecondition
     (Abstraction  : in Abstractions;
      Subprogram   : in Symbol;
      Predicate    : in ExaminerConstants.RefType;
      Precondition : in Location) is

      procedure Add_Precondition
        (Abstraction    : in Abstractions;
         The_Subprogram : in RawDict.Subprogram_Info_Ref;
         Predicate      : in ExaminerConstants.RefType;
         Precondition   : in Location)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Abstraction,
      --#                                Predicate,
      --#                                The_Subprogram &
      --#         SPARK_IO.File_Sys from *,
      --#                                Abstraction,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Precondition,
      --#                                Predicate,
      --#                                The_Subprogram;
      is
      begin
         -- mark subprog as having second proof constraint (independent of whether it has second flow anno)
         if Abstraction = IsRefined then
            RawDict.Set_Subprogram_Has_Second_Constraint (The_Subprogram => The_Subprogram);
         end if;

         RawDict.Set_Subprogram_Precondition
           (The_Subprogram => The_Subprogram,
            Abstraction    => Abstraction,
            Precondition   => Predicate);

         if SPARK_IO.Is_Open (Dict.TemporaryFile) then
            if Abstraction = IsRefined then
               Write_String (Dict.TemporaryFile, "refined ");
            end if;
            Write_String (Dict.TemporaryFile, "precondition of ");
            Write_Name (File => Dict.TemporaryFile,
                        Item => RawDict.Get_Subprogram_Symbol (The_Subprogram));
            Write_String (Dict.TemporaryFile, " is at ");
            Write_Location (File => Dict.TemporaryFile,
                            Loc  => Precondition);
            Write_Line (Dict.TemporaryFile, " ;");
         end if;
      end Add_Precondition;

   begin -- AddPrecondition
      Add_Precondition (Abstraction    => Abstraction,
                        The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Subprogram), -- GAA External
                        Predicate      => Predicate,
                        Precondition   => Precondition);
   end AddPrecondition;

   --------------------------------------------------------------------------------
   --  Add_Postcondition
   --------------------------------------------------------------------------------

   procedure AddPostcondition
     (Abstraction   : in Abstractions;
      Subprogram    : in Symbol;
      Predicate     : in ExaminerConstants.RefType;
      Postcondition : in Location) is

      procedure Add_Postcondition
        (Abstraction    : in Abstractions;
         The_Subprogram : in RawDict.Subprogram_Info_Ref;
         Predicate      : in ExaminerConstants.RefType;
         Postcondition  : in Location)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Abstraction,
      --#                                Predicate,
      --#                                The_Subprogram &
      --#         SPARK_IO.File_Sys from *,
      --#                                Abstraction,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Postcondition,
      --#                                Predicate,
      --#                                The_Subprogram;
      is
      begin
         -- mark subprog as having second proof constraint (independent of whether it has second flow anno)
         if Abstraction = IsRefined then
            RawDict.Set_Subprogram_Has_Second_Constraint (The_Subprogram => The_Subprogram);
         end if;

         RawDict.Set_Subprogram_Postcondition
           (The_Subprogram => The_Subprogram,
            Abstraction    => Abstraction,
            Postcondition  => Predicate);

         if SPARK_IO.Is_Open (Dict.TemporaryFile) then
            if Abstraction = IsRefined then
               Write_String (Dict.TemporaryFile, "refined ");
            end if;
            Write_String (Dict.TemporaryFile, "postcondition of ");
            Write_Name (File => Dict.TemporaryFile,
                        Item => RawDict.Get_Subprogram_Symbol (The_Subprogram));
            Write_String (Dict.TemporaryFile, " is at ");
            Write_Location (File => Dict.TemporaryFile,
                            Loc  => Postcondition);
            Write_Line (Dict.TemporaryFile, " ;");
         end if;
      end Add_Postcondition;

   begin -- AddPostcondition
      Add_Postcondition (Abstraction    => Abstraction,
                         The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Subprogram), -- GAA External
                         Predicate      => Predicate,
                         Postcondition   => Postcondition);
   end AddPostcondition;

   --------------------------------------------------------------------------------
   --  Add_Subprogram
   --------------------------------------------------------------------------------

   procedure Add_Subprogram
     (Name           : in     LexTokenManager.Lex_String;
      Comp_Unit      : in     ContextManager.UnitDescriptors;
      Specification  : in     Location;
      Scope          : in     Scopes;
      Context        : in     Contexts;
      The_Subprogram :    out RawDict.Subprogram_Info_Ref)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         The_Subprogram    from Comp_Unit,
   --#                                Context,
   --#                                Dict,
   --#                                Name,
   --#                                Scope,
   --#                                Specification &
   --#         SPARK_IO.File_Sys from *,
   --#                                Comp_Unit,
   --#                                Context,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Scope,
   --#                                Specification;
   --# post The_Subprogram /= RawDict.Null_Subprogram_Info_Ref;
   is
      The_Declaration : RawDict.Declaration_Info_Ref;
   begin
      Add_Declaration
        (Comp_Unit       => Comp_Unit,
         Loc             => Specification,
         Scope           => Scope,
         Context         => Context,
         The_Declaration => The_Declaration);
      RawDict.Create_Subprogram
        (Name            => Name,
         The_Declaration => The_Declaration,
         Comp_Unit       => Comp_Unit,
         Loc             => Specification.Start_Position,
         The_Subprogram  => The_Subprogram);
      Write_Subprogram_Declaration
        (The_Subprogram => The_Subprogram,
         Is_Renaming    => False,
         Scope          => Scope,
         Declaration    => Specification);
   end Add_Subprogram;

   --------------------------------------------------------------------------------

   procedure AddSubprogram
     (Name          : in     LexTokenManager.Lex_String;
      Comp_Unit     : in     ContextManager.UnitDescriptors;
      Specification : in     Location;
      Scope         : in     Scopes;
      Context       : in     Contexts;
      Subprogram    :    out Symbol) is
      The_Subprogram : RawDict.Subprogram_Info_Ref;
   begin
      Add_Subprogram
        (Name           => Name,
         Comp_Unit      => Comp_Unit,
         Specification  => Specification,
         Scope          => Scope,
         Context        => Context,
         The_Subprogram => The_Subprogram);
      Subprogram := RawDict.Get_Subprogram_Symbol (The_Subprogram); -- GAA External
   end AddSubprogram;

   --------------------------------------------------------------------------------
   --  Actual_Of_Generic_Formal_Type
   --------------------------------------------------------------------------------

   function Actual_Of_Generic_Formal_Type
     (The_Generic_Formal_Type : RawDict.Type_Info_Ref;
      Actual_Subprogram       : RawDict.Subprogram_Info_Ref;
      Not_Found_Expected      : Boolean)
     return RawDict.Type_Info_Ref
   --# global in Dict;
   --# return Result => (Not_Found_Expected or Result /= RawDict.Null_Type_Info_Ref);
   is
      Current_Generic_Association : RawDict.Generic_Association_Info_Ref;
      Result                      : RawDict.Type_Info_Ref := RawDict.Null_Type_Info_Ref;
   begin
      -- now that we have the actual generic formal type we can look for it in the linked list of parameters
      Current_Generic_Association := RawDict.Get_Subprogram_First_Generic_Association (The_Subprogram => Actual_Subprogram);
      while Current_Generic_Association /= RawDict.Null_Generic_Association_Info_Ref loop
         if not RawDict.Get_Generic_Association_Is_Object (The_Generic_Association => Current_Generic_Association)
           and then RawDict.Get_Generic_Association_Formal_Type (The_Generic_Association => Current_Generic_Association) =
           The_Generic_Formal_Type then
            -- match found
            Result := RawDict.Get_Generic_Association_Actual_Type (The_Generic_Association => Current_Generic_Association);
            exit;
         end if;
         Current_Generic_Association := RawDict.Get_Next_Generic_Association
           (The_Generic_Association => Current_Generic_Association);
      end loop;
      SystemErrors.RT_Assert
        (C       => Not_Found_Expected or else Result /= RawDict.Null_Type_Info_Ref,
         Sys_Err => SystemErrors.Invalid_Symbol_Table,
         Msg     => "in Dictionary.Actual_Of_Generic_Formal_Type");
      --# accept Flow, 35, Not_Found_Expected, "Importation of the initial value is ineffective OK" &
      --#        Flow, 50, Not_Found_Expected, "Value is not derived from the imported value OK";
      return Result;
   end Actual_Of_Generic_Formal_Type;

   --------------------------------------------------------------------------------

   function ActualOfGenericFormalType (TheGenericFormalSym : Symbol;
                                       ActualSubprogramSym : Symbol) return Symbol is
   begin
      return RawDict.Get_Type_Symbol -- GAA External
        (Actual_Of_Generic_Formal_Type
           (The_Generic_Formal_Type => RawDict.Get_Type_Info_Ref (TheGenericFormalSym), -- GAA External
            Actual_Subprogram       => RawDict.Get_Subprogram_Info_Ref (ActualSubprogramSym), -- GAA External
            Not_Found_Expected      => False));
   end ActualOfGenericFormalType;

   --------------------------------------------------------------------------------
   --  Actual_Of_Generic_Formal_Object
   --------------------------------------------------------------------------------

   function ActualOfGenericFormalObject (TheGenericFormalSym : Symbol;
                                         ActualSubprogramSym : Symbol) return Symbol is

      function Actual_Of_Generic_Formal_Object
        (The_Generic_Formal_Object : RawDict.Constant_Info_Ref;
         Actual_Subprogram         : RawDict.Subprogram_Info_Ref)
        return RawDict.Constant_Info_Ref
      --# global in Dict;
      --# return Result => (Result /= RawDict.Null_Constant_Info_Ref);
      is
         Current_Generic_Association : RawDict.Generic_Association_Info_Ref;
         Result                      : RawDict.Constant_Info_Ref := RawDict.Null_Constant_Info_Ref;
      begin
         -- now that we have the actual generic formal object we can look for it in the linked list of parameters
         Current_Generic_Association := RawDict.Get_Subprogram_First_Generic_Association (The_Subprogram => Actual_Subprogram);
         while Current_Generic_Association /= RawDict.Null_Generic_Association_Info_Ref loop
            if RawDict.Get_Generic_Association_Is_Object (The_Generic_Association => Current_Generic_Association)
              and then RawDict.Get_Generic_Association_Formal_Object (The_Generic_Association => Current_Generic_Association) =
              The_Generic_Formal_Object then
               -- match found
               Result := RawDict.Get_Generic_Association_Actual_Object (The_Generic_Association => Current_Generic_Association);
               exit;
            end if;
            Current_Generic_Association := RawDict.Get_Next_Generic_Association
              (The_Generic_Association => Current_Generic_Association);
         end loop;
         SystemErrors.RT_Assert
           (C       => Result /= RawDict.Null_Constant_Info_Ref,
            Sys_Err => SystemErrors.Invalid_Symbol_Table,
            Msg     => "in Dictionary.Actual_Of_Generic_Formal_Object");
         return Result;
      end Actual_Of_Generic_Formal_Object;

   begin -- ActualOfGenericFormalObject
      return RawDict.Get_Constant_Symbol -- GAA External
        (Actual_Of_Generic_Formal_Object
           (The_Generic_Formal_Object => RawDict.Get_Constant_Info_Ref (TheGenericFormalSym), -- GAA External
            Actual_Subprogram         => RawDict.Get_Subprogram_Info_Ref (ActualSubprogramSym))); -- GAA External
   end ActualOfGenericFormalObject;

   --------------------------------------------------------------------------------
   --  Get_Generic_Of_Instantiation
   --------------------------------------------------------------------------------

   function Get_Generic_Of_Instantiation (The_Subprogram : Symbol) return RawDict.Subprogram_Info_Ref
   --# global in Dict;
   is
      Result : RawDict.Subprogram_Info_Ref;
   begin
      case RawDict.GetSymbolDiscriminant (The_Subprogram) is
         when Subprogram_Symbol =>
            Result := RawDict.Get_Subprogram_Instantiation_Of
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => The_Subprogram));
         when ImplicitProofFunctionSymbol =>
            Result := RawDict.Get_Subprogram_Instantiation_Of
              (The_Subprogram => RawDict.GetImplicitProofFunctionAdaFunction (The_Subprogram));
         when others => -- non-exec code
            Result := RawDict.Null_Subprogram_Info_Ref;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.Get_Generic_Of_Instantiation");
      end case;
      return Result;
   end Get_Generic_Of_Instantiation;

   --------------------------------------------------------------------------------

   function GetGenericOfInstantiation (PackageOrSubProgram : Symbol) return Symbol is
   begin
      return RawDict.Get_Subprogram_Symbol (Get_Generic_Of_Instantiation (The_Subprogram => PackageOrSubProgram)); -- GAA External
   end GetGenericOfInstantiation;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function IsInstantiation (PackageOrSubProgram : Symbol) return Boolean is
   begin
      return Get_Generic_Of_Instantiation (The_Subprogram => PackageOrSubProgram) /= RawDict.Null_Subprogram_Info_Ref;
   end IsInstantiation;

   --------------------------------------------------------------------------------

   procedure SetSubprogramSignatureNotWellformed (Abstraction : in Abstractions;
                                                  Subprogram  : in Symbol) is
   begin
      case RawDict.GetSymbolDiscriminant (Subprogram) is
         when Subprogram_Symbol =>
            RawDict.Set_Subprogram_Signature_Not_Wellformed
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Subprogram), -- GAA External
               Abstraction    => Abstraction);
         when Type_Symbol =>
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Subprogram)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.SetSubprogramSignatureNotWellformed");
            RawDict.Set_Task_Type_Signature_Not_Wellformed
              (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => Subprogram), -- GAA External
               Abstraction   => Abstraction);
         when others => -- non-exec code
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.SetSubprogramSignatureNotWellformed");
      end case;
   end SetSubprogramSignatureNotWellformed;

   --------------------------------------------------------------------------------

   procedure SetSubprogramIsEntry (Subprogram : in Symbol) is
   begin
      RawDict.Set_Subprogram_Is_Entry (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Subprogram)); -- GAA External
   end SetSubprogramIsEntry;

   --------------------------------------------------------------------------------

   procedure Set_Package_Generic_Unit (Pack_Sym     : in Symbol;
                                       Generic_Unit : in Symbol) is
   begin
      RawDict.Set_Package_Generic_Unit
        (The_Package      => RawDict.Get_Package_Info_Ref (Pack_Sym), -- GAA External
         The_Generic_Unit => RawDict.Get_Generic_Unit_Info_Ref (Generic_Unit)); -- GAA External
   end Set_Package_Generic_Unit;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Generic_Unit (Subprogram   : in Symbol;
                                          Generic_Unit : in Symbol) is
   begin
      RawDict.Set_Subprogram_Generic_Unit
        (The_Subprogram   => RawDict.Get_Subprogram_Info_Ref (Subprogram), -- GAA External
         The_Generic_Unit => RawDict.Get_Generic_Unit_Info_Ref (Generic_Unit)); -- GAA External
   end Set_Subprogram_Generic_Unit;

   --------------------------------------------------------------------------------

   procedure Set_Generic_Unit_Owning_Package (Generic_Unit : in Symbol;
                                              Pack_Sym     : in Symbol) is
   begin
      RawDict.Set_Generic_Unit_Owning_Package
        (The_Generic_Unit => RawDict.Get_Generic_Unit_Info_Ref (Generic_Unit), -- GAA External
         The_Package      => RawDict.Get_Package_Info_Ref (Pack_Sym)); -- GAA External
   end Set_Generic_Unit_Owning_Package;

   --------------------------------------------------------------------------------

   procedure Set_Generic_Unit_Owning_Subprogram (Generic_Unit : in Symbol;
                                                 Subprogram   : in Symbol) is
   begin
      RawDict.Set_Generic_Unit_Owning_Subprogram
        (The_Generic_Unit => RawDict.Get_Generic_Unit_Info_Ref (Generic_Unit), -- GAA External
         The_Subprogram   => RawDict.Get_Subprogram_Info_Ref (Subprogram)); -- GAA External
   end Set_Generic_Unit_Owning_Subprogram;

   --------------------------------------------------------------------------------

   procedure SetHasDerivesAnnotation (Task_Or_Proc : in Symbol) is
   begin
      case RawDict.GetSymbolDiscriminant (Task_Or_Proc) is
         when Subprogram_Symbol =>
            RawDict.Set_Subprogram_Has_Derives_Annotation
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Task_Or_Proc)); -- GAA External
         when ImplicitProofFunctionSymbol =>
            RawDict.Set_Subprogram_Has_Derives_Annotation
              (The_Subprogram => RawDict.GetImplicitProofFunctionAdaFunction (Task_Or_Proc));
         when Type_Symbol =>
            -- The only other possibility is a task.
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Task_Or_Proc)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.SetHasDerivesAnnotation");
            RawDict.Set_Task_Type_Has_Derives_Annotation
              (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => Task_Or_Proc)); -- GAA External
         when others => -- non-exec code
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.SetHasDerivesAnnotation");
      end case;
   end SetHasDerivesAnnotation;

   --------------------------------------------------------------------------------

   procedure SetSubprogramEntryBarrier (Subprogram, TheBarrier : in Symbol) is
   begin
      RawDict.Set_Subprogram_Entry_Barrier
        (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Subprogram), -- GAA External
         The_Barrier    => TheBarrier);
   end SetSubprogramEntryBarrier;

   --------------------------------------------------------------------------------

   procedure SetIsInterruptHandler (Subprogram : in Symbol) is
   begin
      RawDict.Set_Subprogram_Is_Interrupt_Handler (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Subprogram)); -- GAA External
   end SetIsInterruptHandler;

   --------------------------------------------------------------------------------

   procedure SetProtectedTypeElementsHidden (TheProtectedType : in Symbol) is
   begin
      RawDict.Set_Protected_Type_Elements_Hidden
        (The_Protected_Type => RawDict.Get_Type_Info_Ref (TheProtectedType)); -- GAA External
   end SetProtectedTypeElementsHidden;

   --------------------------------------------------------------------------------

   procedure SetProtectedTypeEntry (TheProtectedType, TheEntry : in Symbol) is
   begin
      RawDict.Set_Protected_Type_The_Entry
        (The_Protected_Type => RawDict.Get_Type_Info_Ref (TheProtectedType), -- GAA External
         The_Entry          => TheEntry);
   end SetProtectedTypeEntry;

   --------------------------------------------------------------------------------
   --  Set_Type_Has_Pragma
   --------------------------------------------------------------------------------

   procedure SetTypeHasPragma (TheProtectedOrTaskType : in Symbol;
                               ThePragma              : in RavenscarPragmas) is

      procedure Set_Type_Has_Pragma (The_Protected_Or_Task_Type : in RawDict.Type_Info_Ref;
                                     The_Pragma                 : in RavenscarPragmas)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   The_Pragma,
      --#                   The_Protected_Or_Task_Type;
      is
      begin
         case RawDict.Get_Type_Discriminant (Type_Mark => The_Protected_Or_Task_Type) is
            when Protected_Type_Item =>
               RawDict.Set_Protected_Type_Has_Pragma (The_Protected_Type => The_Protected_Or_Task_Type,
                                                      The_Pragma         => The_Pragma);
            when Task_Type_Item =>
               RawDict.Set_Task_Type_Has_Pragma (The_Task_Type => The_Protected_Or_Task_Type,
                                                 The_Pragma    => The_Pragma);
            when others => -- non-exec code
               SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                         Msg     => "in Dictionary.SetTypeHasPragma");
         end case;
      end Set_Type_Has_Pragma;

   begin -- SetTypeHasPragma
      Set_Type_Has_Pragma
        (The_Protected_Or_Task_Type => RawDict.Get_Type_Info_Ref (TheProtectedOrTaskType), -- GAA External
         The_Pragma                 => ThePragma);
   end SetTypeHasPragma;

   --------------------------------------------------------------------------------
   --  Set_Type_Pragma_Value
   --------------------------------------------------------------------------------

   procedure SetTypePragmaValue
     (TheProtectedOrTaskType : in Symbol;
      ThePragma              : in RavenscarPragmasWithValue;
      TheValue               : in LexTokenManager.Lex_String) is

      procedure Set_Type_Pragma_Value
        (The_Protected_Or_Task_Type : in RawDict.Type_Info_Ref;
         The_Pragma                 : in RavenscarPragmasWithValue;
         The_Value                  : in LexTokenManager.Lex_String)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   The_Pragma,
      --#                   The_Protected_Or_Task_Type,
      --#                   The_Value;
      is
      begin
         case RawDict.Get_Type_Discriminant (Type_Mark => The_Protected_Or_Task_Type) is
            when Protected_Type_Item =>
               RawDict.Set_Protected_Type_Pragma_Value
                 (The_Protected_Type => The_Protected_Or_Task_Type,
                  The_Pragma         => The_Pragma,
                  The_Value          => The_Value);
            when Task_Type_Item =>
               RawDict.Set_Task_Type_Pragma_Value
                 (The_Task_Type => The_Protected_Or_Task_Type,
                  The_Pragma    => The_Pragma,
                  The_Value     => The_Value);
            when others => -- non-exec code
               SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                         Msg     => "in Dictionary.SetTypePragmaValue");
         end case;
      end Set_Type_Pragma_Value;

   begin -- SetTypePragmaValue
      Set_Type_Pragma_Value
        (The_Protected_Or_Task_Type => RawDict.Get_Type_Info_Ref (TheProtectedOrTaskType), -- GAA External
         The_Pragma                 => ThePragma,
         The_Value                  => TheValue);
   end SetTypePragmaValue;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   procedure SetMainProgramPriority (TheValue : in LexTokenManager.Lex_String) is
   begin
      Dict.Main.Priority_Given := True;
      Dict.Main.The_Priority   := TheValue;
      null;
   end SetMainProgramPriority;

   --------------------------------------------------------------------------------

   function IsPredefinedSuspendUntilTrueOperation (TheProcedure : Symbol) return Boolean is
   begin
      return RawDict.Get_Subprogram_Info_Ref (TheProcedure) = Dict.Subprograms.STC_Suspend_Until_True; -- GAA External
   end IsPredefinedSuspendUntilTrueOperation;

   --------------------------------------------------------------------------------

   function IsPredefinedRealTimeClockOperation (TheProcedure : Symbol) return Boolean is
   begin
      return RawDict.GetSymbolDiscriminant (TheProcedure) = Subprogram_Symbol
        and then RawDict.Get_Subprogram_Info_Ref (TheProcedure) = Dict.Subprograms.Ada_Real_Time_Clock; -- GAA External
   end IsPredefinedRealTimeClockOperation;

   --------------------------------------------------------------------------------
   --  Add_Return_Type
   --------------------------------------------------------------------------------

   procedure Add_Return_Type
     (The_Function   : in RawDict.Subprogram_Info_Ref;
      Type_Mark      : in RawDict.Type_Info_Ref;
      Comp_Unit      : in ContextManager.UnitDescriptors;
      Type_Reference : in Location)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Comp_Unit,
   --#                                The_Function,
   --#                                Type_Mark,
   --#                                Type_Reference &
   --#         SPARK_IO.File_Sys from *,
   --#                                Comp_Unit,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                The_Function,
   --#                                Type_Mark,
   --#                                Type_Reference;
   is
      The_Proof_Function : Symbol;
   begin
      RawDict.Set_Subprogram_Return_Type (The_Subprogram => The_Function,
                                          Type_Mark      => Type_Mark);
      RawDict.Set_Subprogram_Has_Derives_Annotation (The_Subprogram => The_Function);
      if Get_Subprogram_Context (The_Subprogram => The_Function) = ProgramContext then
         RawDict.CreateImplicitProofFunction
           (Ada_Function  => The_Function,
            Comp_Unit     => Comp_Unit,
            Loc           => Type_Reference.Start_Position,
            ProofFunction => The_Proof_Function);
         RawDict.Set_Subprogram_Implicit_Proof_Function
           (The_Subprogram     => The_Function,
            Abstraction        => IsAbstract,
            The_Proof_Function => The_Proof_Function);
      end if;

      if Type_Mark /= Get_Unknown_Type_Mark then
         AddOtherReference (RawDict.Get_Type_Symbol (Type_Mark), RawDict.Get_Subprogram_Symbol (The_Function), Type_Reference);
      end if;
   end Add_Return_Type;

   -------------------------------------------------------------------------------

   procedure AddReturnType
     (TheFunction   : in Symbol;
      TypeMark      : in Symbol;
      Comp_Unit     : in ContextManager.UnitDescriptors;
      TypeReference : in Location) is
   begin
      Add_Return_Type
        (The_Function   => RawDict.Get_Subprogram_Info_Ref (TheFunction), -- GAA External
         Type_Mark      => RawDict.Get_Type_Info_Ref (TypeMark), -- GAA External
         Comp_Unit      => Comp_Unit,
         Type_Reference => TypeReference);
   end AddReturnType;

   -------------------------------------------------------------------------------
   --  Add_Implicit_Return_Variable
   -------------------------------------------------------------------------------

   procedure AddImplicitReturnVariable
     (Abstraction : in     Abstractions;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      Name        : in     LexTokenManager.Lex_String;
      TheFunction : in     Symbol;
      Variable    :    out Symbol) is
      The_Implicit_Return_Variable : RawDict.Implicit_Return_Variable_Info_Ref;

      -------------------------------------------------------------------------------

      procedure Add_Implicit_Return_Variable
        (Abstraction                  : in     Abstractions;
         Comp_Unit                    : in     ContextManager.UnitDescriptors;
         Declaration                  : in     Location;
         Name                         : in     LexTokenManager.Lex_String;
         The_Function                 : in     RawDict.Subprogram_Info_Ref;
         The_Implicit_Return_Variable :    out RawDict.Implicit_Return_Variable_Info_Ref)
      --# global in out Dict;
      --# derives Dict                         from *,
      --#                                           Abstraction,
      --#                                           Comp_Unit,
      --#                                           Declaration,
      --#                                           Name,
      --#                                           The_Function &
      --#         The_Implicit_Return_Variable from Comp_Unit,
      --#                                           Declaration,
      --#                                           Dict,
      --#                                           Name,
      --#                                           The_Function;
      --# post The_Implicit_Return_Variable /= RawDict.Null_Implicit_Return_Variable_Info_Ref;
      is
      begin
         RawDict.Create_Implicit_Return_Variable
           (Name                         => Name,
            The_Function                 => The_Function,
            Comp_Unit                    => Comp_Unit,
            Loc                          => Declaration.Start_Position,
            The_Implicit_Return_Variable => The_Implicit_Return_Variable);
         RawDict.Set_Subprogram_Implicit_Return_Variable
           (The_Subprogram               => The_Function,
            Abstraction                  => Abstraction,
            The_Implicit_Return_Variable => The_Implicit_Return_Variable);
      end Add_Implicit_Return_Variable;

   begin -- AddImplicitReturnVariable
      Add_Implicit_Return_Variable
        (Abstraction                  => Abstraction,
         Comp_Unit                    => Comp_Unit,
         Declaration                  => Declaration,
         Name                         => Name,
         The_Function                 => RawDict.Get_Subprogram_Info_Ref (TheFunction), -- GAA External
         The_Implicit_Return_Variable => The_Implicit_Return_Variable);
      Variable := RawDict.Get_Implicit_Return_Variable_Symbol (The_Implicit_Return_Variable); -- GAA External
   end AddImplicitReturnVariable;

   --------------------------------------------------------------------------------
   --  Add_Subprogram_Parameter
   --------------------------------------------------------------------------------

   procedure Add_Subprogram_Parameter
     (Name           : in LexTokenManager.Lex_String;
      The_Subprogram : in RawDict.Subprogram_Info_Ref;
      Type_Mark      : in RawDict.Type_Info_Ref;
      Type_Reference : in Location;
      Mode           : in Modes;
      Comp_Unit      : in ContextManager.UnitDescriptors;
      Specification  : in Location)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Comp_Unit,
   --#                                Mode,
   --#                                Name,
   --#                                Specification,
   --#                                The_Subprogram,
   --#                                Type_Mark &
   --#         SPARK_IO.File_Sys from *,
   --#                                Comp_Unit,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Mode,
   --#                                Name,
   --#                                Specification,
   --#                                The_Subprogram,
   --#                                Type_Mark,
   --#                                Type_Reference;
      is separate;

   --------------------------------------------------------------------------------

   procedure AddSubprogramParameter
     (Name          : in LexTokenManager.Lex_String;
      Subprogram    : in Symbol;
      TypeMark      : in Symbol;
      TypeReference : in Location;
      Mode          : in Modes;
      Comp_Unit     : in ContextManager.UnitDescriptors;
      Specification : in Location) is
   begin
      Add_Subprogram_Parameter
        (Name           => Name,
         The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Subprogram), -- GAA External
         Type_Mark      => RawDict.Get_Type_Info_Ref (TypeMark), -- GAA External
         Type_Reference => TypeReference,
         Mode           => Mode,
         Comp_Unit      => Comp_Unit,
         Specification  => Specification);
   end AddSubprogramParameter;

   --------------------------------------------------------------------------------
   --  Add_Generic_Formal_Parameter
   --------------------------------------------------------------------------------

   procedure Add_Generic_Formal_Parameter_Local
     (Comp_Unit             : in     ContextManager.UnitDescriptors;
      Declaration           : in     Location;
      The_Generic_Unit      : in     RawDict.Generic_Unit_Info_Ref;
      Type_Mark             : in     RawDict.Type_Info_Ref;
      The_Object            : in     RawDict.Constant_Info_Ref;
      The_Generic_Parameter :    out RawDict.Generic_Parameter_Info_Ref)
   --# global in out Dict;
   --# derives Dict,
   --#         The_Generic_Parameter from Comp_Unit,
   --#                                    Declaration,
   --#                                    Dict,
   --#                                    The_Generic_Unit,
   --#                                    The_Object,
   --#                                    Type_Mark;
      is separate;

   --------------------------------------------------------------------------------

   procedure Add_Generic_Formal_Parameter
     (Comp_Unit    : in ContextManager.UnitDescriptors;
      Declaration  : in Location;
      Generic_Unit : in Symbol;
      The_Type     : in Symbol;
      The_Object   : in Symbol) is
      The_Generic_Parameter : RawDict.Generic_Parameter_Info_Ref;
      pragma Unreferenced (The_Generic_Parameter);
   begin
      --# accept Flow, 10, The_Generic_Parameter, "Expected ineffective assignment to The_Generic_Parameter";
      Add_Generic_Formal_Parameter_Local
        (Comp_Unit             => Comp_Unit,
         Declaration           => Declaration,
         The_Generic_Unit      => RawDict.Get_Generic_Unit_Info_Ref (Generic_Unit), -- GAA External
         Type_Mark             => RawDict.Get_Type_Info_Ref (The_Type), -- GAA External
         The_Object            => RawDict.Get_Constant_Info_Ref (The_Object), -- GAA External
         The_Generic_Parameter => The_Generic_Parameter);
      --# end accept;
      --# accept Flow, 33, The_Generic_Parameter, "Expected The_Generic_Parameter to be neither referenced nor exported";
   end Add_Generic_Formal_Parameter;

   --------------------------------------------------------------------------------
   --  Add_Generic_Type_Association
   --------------------------------------------------------------------------------

   procedure AddGenericTypeAssociation
     (SubprogramOrPackage : in Symbol;
      Comp_Unit           : in ContextManager.UnitDescriptors;
      Declaration         : in Location;
      FormalSym           : in Symbol;
      ActualSym           : in Symbol) is

      procedure Add_Generic_Type_Association
        (The_Subprogram : in RawDict.Subprogram_Info_Ref;
         Comp_Unit      : in ContextManager.UnitDescriptors;
         Declaration    : in Location;
         Formal_Type    : in RawDict.Type_Info_Ref;
         Actual_Type    : in RawDict.Type_Info_Ref)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   Actual_Type,
      --#                   Comp_Unit,
      --#                   Declaration,
      --#                   Formal_Type,
      --#                   The_Subprogram;
      is
         The_Generic_Association, Previous : RawDict.Generic_Association_Info_Ref;
      begin
         RawDict.Create_Generic_Type_Association
           (Formal_Type             => Formal_Type,
            Actual_Type             => Actual_Type,
            Comp_Unit               => Comp_Unit,
            Loc                     => Declaration.Start_Position,
            The_Generic_Association => The_Generic_Association);

         Previous := RawDict.Get_Subprogram_Last_Generic_Association (The_Subprogram => The_Subprogram);
         if Previous = RawDict.Null_Generic_Association_Info_Ref then
            RawDict.Set_Subprogram_First_Generic_Association (The_Subprogram          => The_Subprogram,
                                                              The_Generic_Association => The_Generic_Association);
         else
            RawDict.Set_Next_Generic_Association (The_Generic_Association => Previous,
                                                  Next                    => The_Generic_Association);
         end if;
         RawDict.Set_Subprogram_Last_Generic_Association (The_Subprogram          => The_Subprogram,
                                                          The_Generic_Association => The_Generic_Association);
      end Add_Generic_Type_Association;

   begin -- AddGenericTypeAssociation
      Add_Generic_Type_Association
        (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (SubprogramOrPackage), -- GAA External
         Comp_Unit      => Comp_Unit,
         Declaration    => Declaration,
         Formal_Type    => RawDict.Get_Type_Info_Ref (FormalSym), -- GAA External
         Actual_Type    => RawDict.Get_Type_Info_Ref (ActualSym)); -- GAA External
   end AddGenericTypeAssociation;

   --------------------------------------------------------------------------------
   --  Add_Generic_Object_Association
   --------------------------------------------------------------------------------

   procedure AddGenericObjectAssociation
     (SubprogramOrPackage : in Symbol;
      Comp_Unit           : in ContextManager.UnitDescriptors;
      Declaration         : in Location;
      FormalSym           : in Symbol;
      ActualSym           : in Symbol) is

      procedure Add_Generic_Object_Association
        (The_Subprogram : in RawDict.Subprogram_Info_Ref;
         Comp_Unit      : in ContextManager.UnitDescriptors;
         Declaration    : in Location;
         Formal_Object  : in RawDict.Generic_Parameter_Info_Ref;
         Actual_Object  : in RawDict.Constant_Info_Ref)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   Actual_Object,
      --#                   Comp_Unit,
      --#                   Declaration,
      --#                   Formal_Object,
      --#                   The_Subprogram;
      is
         The_Generic_Association, Previous : RawDict.Generic_Association_Info_Ref;
      begin
         RawDict.Create_Generic_Object_Association
           (Formal_Object           => RawDict.Get_Generic_Parameter_Object (The_Generic_Parameter => Formal_Object),
            Actual_Object           => Actual_Object,
            Comp_Unit               => Comp_Unit,
            Loc                     => Declaration.Start_Position,
            The_Generic_Association => The_Generic_Association);

         Previous := RawDict.Get_Subprogram_Last_Generic_Association (The_Subprogram => The_Subprogram);
         if Previous = RawDict.Null_Generic_Association_Info_Ref then
            RawDict.Set_Subprogram_First_Generic_Association (The_Subprogram          => The_Subprogram,
                                                              The_Generic_Association => The_Generic_Association);
         else
            RawDict.Set_Next_Generic_Association (The_Generic_Association => Previous,
                                                  Next                    => The_Generic_Association);
         end if;
         RawDict.Set_Subprogram_Last_Generic_Association (The_Subprogram          => The_Subprogram,
                                                          The_Generic_Association => The_Generic_Association);
      end Add_Generic_Object_Association;

   begin -- AddGenericObjectAssociation
      Add_Generic_Object_Association
        (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (SubprogramOrPackage), -- GAA External
         Comp_Unit      => Comp_Unit,
         Declaration    => Declaration,
         Formal_Object  => RawDict.Get_Generic_Parameter_Info_Ref (FormalSym), -- GAA External
         Actual_Object  => RawDict.Get_Constant_Info_Ref (ActualSym)); -- GAA External
   end AddGenericObjectAssociation;

   --------------------------------------------------------------------------------
   --  Add_Known_Discriminant
   --------------------------------------------------------------------------------

   procedure AddKnownDiscriminant
     (Name                : in LexTokenManager.Lex_String;
      Comp_Unit           : in ContextManager.UnitDescriptors;
      Declaration         : in Location;
      ProtectedOrTaskType : in Symbol;
      TypeMark            : in Symbol) is

      procedure Add_Known_Discriminant
        (Name                       : in LexTokenManager.Lex_String;
         Comp_Unit                  : in ContextManager.UnitDescriptors;
         Declaration                : in Location;
         The_Protected_Or_Task_Type : in RawDict.Type_Info_Ref;
         Type_Mark                  : in RawDict.Type_Info_Ref)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   Comp_Unit,
      --#                   Declaration,
      --#                   Name,
      --#                   The_Protected_Or_Task_Type,
      --#                   Type_Mark;
      is
         Discriminant, Previous : Symbol;
      begin
         RawDict.CreateKnownDiscriminant
           (Name              => Name,
            Protected_Type    => The_Protected_Or_Task_Type,
            Type_Mark         => Type_Mark,
            Comp_Unit         => Comp_Unit,
            Loc               => Declaration.Start_Position,
            KnownDiscriminant => Discriminant);
         case RawDict.Get_Type_Discriminant (Type_Mark => The_Protected_Or_Task_Type) is
            when Protected_Type_Item =>
               Previous := RawDict.Get_Protected_Type_Last_Discriminant (The_Protected_Type => The_Protected_Or_Task_Type);
               if Previous = NullSymbol then
                  RawDict.Set_Protected_Type_First_Discriminant
                    (The_Protected_Type => The_Protected_Or_Task_Type,
                     Discriminant       => Discriminant);
               else
                  RawDict.SetNextDiscriminant (Previous, Discriminant);
               end if;
               RawDict.Set_Protected_Type_Last_Discriminant
                 (The_Protected_Type => The_Protected_Or_Task_Type,
                  Discriminant       => Discriminant);
            when Task_Type_Item =>
               Previous := RawDict.Get_Task_Type_Last_Discriminant (The_Task_Type => The_Protected_Or_Task_Type);
               if Previous = NullSymbol then
                  RawDict.Set_Task_Type_First_Discriminant
                    (The_Task_Type => The_Protected_Or_Task_Type,
                     Discriminant  => Discriminant);
               else
                  RawDict.SetNextDiscriminant (Previous, Discriminant);
               end if;
               RawDict.Set_Task_Type_Last_Discriminant (The_Task_Type => The_Protected_Or_Task_Type,
                                                        Discriminant  => Discriminant);
            when others => -- non-exec code
               SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                         Msg     => "in Dictionary.AddKnownDiscriminant");
         end case;
      end Add_Known_Discriminant;

   begin -- AddKnownDiscriminant
      Add_Known_Discriminant
        (Name                       => Name,
         Comp_Unit                  => Comp_Unit,
         Declaration                => Declaration,
         The_Protected_Or_Task_Type => RawDict.Get_Type_Info_Ref (ProtectedOrTaskType), -- GAA External
         Type_Mark                  => RawDict.Get_Type_Info_Ref (TypeMark)); -- GAA External
   end AddKnownDiscriminant;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   procedure SetDiscriminantSetsPriority (TheDiscriminant : in Symbol) is
   begin
      RawDict.SetDiscriminantSetsPriority (TheDiscriminant);
   end SetDiscriminantSetsPriority;

   --------------------------------------------------------------------------------
   --  Add_Discriminant_Constraint_Static_Value
   --------------------------------------------------------------------------------

   procedure AddDiscriminantConstraintStaticValue
     (ProtectedOrTaskSubtype : in Symbol;
      Comp_Unit              : in ContextManager.UnitDescriptors;
      Declaration            : in Location;
      TheValue               : in LexTokenManager.Lex_String) is

      procedure Add_Discriminant_Constraint_Static_Value
        (The_Protected_Or_Task_Subtype : in RawDict.Type_Info_Ref;
         Comp_Unit                     : in ContextManager.UnitDescriptors;
         Declaration                   : in Location;
         The_Value                     : in LexTokenManager.Lex_String)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   Comp_Unit,
      --#                   Declaration,
      --#                   The_Protected_Or_Task_Subtype,
      --#                   The_Value;
      is
         DiscriminantConstraint : Symbol;
         SubtypeInfo            : Symbol;
         Previous               : Symbol;
      begin
         -- create DiscrimianntConstraint record
         RawDict.CreateDiscriminantConstraint
           (Comp_Unit                 => Comp_Unit,
            Loc                       => Declaration.Start_Position,
            TheDiscriminantConstraint => DiscriminantConstraint);
         RawDict.SetDiscriminantConstraintStaticValue (DiscriminantConstraint, The_Value);
         -- link to subtype
         SubtypeInfo := RawDict.Get_Type_Ancillary_Fields (Type_Mark => The_Protected_Or_Task_Subtype);

         -- assuming precondition met then SubtypeInfo is symbol of record containing priority, linked
         -- list of constraints etc.
         Previous := RawDict.GetSubtypeInfoLastConstraint (SubtypeInfo);
         if Previous = NullSymbol then
            RawDict.SetSubtypeInfoFirstConstraint (SubtypeInfo, DiscriminantConstraint);
         else
            RawDict.SetNextDiscriminantConstraint (Previous, DiscriminantConstraint);
         end if;
         RawDict.SetSubtypeInfoLastConstraint (SubtypeInfo, DiscriminantConstraint);
      end Add_Discriminant_Constraint_Static_Value;

   begin -- AddDiscriminantConstraintStaticValue
      Add_Discriminant_Constraint_Static_Value
        (The_Protected_Or_Task_Subtype => RawDict.Get_Type_Info_Ref (ProtectedOrTaskSubtype), -- GAA External
         Comp_Unit                     => Comp_Unit,
         Declaration                   => Declaration,
         The_Value                     => TheValue);
   end AddDiscriminantConstraintStaticValue;

   --------------------------------------------------------------------------------
   --  Add_Discriminant_Constraint_Accessed_Object
   --------------------------------------------------------------------------------

   procedure AddDiscriminantConstraintAccessedObject
     (ProtectedOrTaskSubtype : in Symbol;
      Comp_Unit              : in ContextManager.UnitDescriptors;
      Declaration            : in Location;
      TheObject              : in Symbol) is

      procedure Add_Discriminant_Constraint_Accessed_Object
        (The_Protected_Or_Task_Subtype : in RawDict.Type_Info_Ref;
         Comp_Unit                     : in ContextManager.UnitDescriptors;
         Declaration                   : in Location;
         The_Object                    : in Symbol)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   Comp_Unit,
      --#                   Declaration,
      --#                   The_Object,
      --#                   The_Protected_Or_Task_Subtype;
      is
         DiscriminantConstraint : Symbol;
         SubtypeInfo            : Symbol;
         Previous               : Symbol;
      begin
         -- create DiscrimianntConstraint record
         RawDict.CreateDiscriminantConstraint
           (Comp_Unit                 => Comp_Unit,
            Loc                       => Declaration.Start_Position,
            TheDiscriminantConstraint => DiscriminantConstraint);
         RawDict.SetDiscriminantConstraintAccessedObject (DiscriminantConstraint, The_Object);
         -- link to subtype
         SubtypeInfo := RawDict.Get_Type_Ancillary_Fields (Type_Mark => The_Protected_Or_Task_Subtype);

         -- assuming precondition met then SubtypeInfo is symbol of record containing priority, linked
         -- list of constraints etc.
         Previous := RawDict.GetSubtypeInfoLastConstraint (SubtypeInfo);
         if Previous = NullSymbol then
            RawDict.SetSubtypeInfoFirstConstraint (SubtypeInfo, DiscriminantConstraint);
         else
            RawDict.SetNextDiscriminantConstraint (Previous, DiscriminantConstraint);
         end if;
         RawDict.SetSubtypeInfoLastConstraint (SubtypeInfo, DiscriminantConstraint);
      end Add_Discriminant_Constraint_Accessed_Object;

   begin -- AddDiscriminantConstraintAccessedObject
      Add_Discriminant_Constraint_Accessed_Object
        (The_Protected_Or_Task_Subtype => RawDict.Get_Type_Info_Ref (ProtectedOrTaskSubtype), -- GAA External
         Comp_Unit                     => Comp_Unit,
         Declaration                   => Declaration,
         The_Object                    => TheObject);
   end AddDiscriminantConstraintAccessedObject;

   --------------------------------------------------------------------------------
   --  Set_Subtype_Priority
   --------------------------------------------------------------------------------

   procedure SetSubtypePriority (ProtectedOrTaskSubtype : in Symbol;
                                 ThePriority            : in LexTokenManager.Lex_String) is

      procedure Set_Subtype_Priority (The_Protected_Or_Task_Subtype : in RawDict.Type_Info_Ref;
                                      The_Priority            : in LexTokenManager.Lex_String)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   The_Priority,
      --#                   The_Protected_Or_Task_Subtype;
      is
         SubtypeInfo : Symbol;
      begin
         SubtypeInfo := RawDict.Get_Type_Ancillary_Fields (Type_Mark => The_Protected_Or_Task_Subtype);
         RawDict.SetSubtypeInfoPriority (SubtypeInfo, The_Priority);
      end Set_Subtype_Priority;

   begin -- SetSubtypePriority
      Set_Subtype_Priority
        (The_Protected_Or_Task_Subtype => RawDict.Get_Type_Info_Ref (ProtectedOrTaskSubtype), -- GAA External
         The_Priority                  => ThePriority);
   end SetSubtypePriority;

   --------------------------------------------------------------------------------
   --  Has_Body_Stub
   --------------------------------------------------------------------------------

   function Has_Package_Body_Stub (The_Package : RawDict.Package_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Package_Body (The_Package => The_Package) /= RawDict.Null_Declaration_Info_Ref
        and then not RawDict.Get_Package_Has_Proper_Body (The_Package => The_Package);
   end Has_Package_Body_Stub;

   --------------------------------------------------------------------------------

   function Has_Protected_Type_Body_Stub (The_Protected_Type : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Protected_Type_Body (The_Protected_Type => The_Protected_Type) /= RawDict.Null_Declaration_Info_Ref
        and then not RawDict.Get_Protected_Type_Has_Proper_Body (The_Protected_Type => The_Protected_Type);
   end Has_Protected_Type_Body_Stub;

   --------------------------------------------------------------------------------

   function Has_Task_Type_Body_Stub (The_Task_Type : RawDict.Type_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Task_Type_Body (The_Task_Type => The_Task_Type) /= RawDict.Null_Declaration_Info_Ref
        and then not RawDict.Get_Task_Type_Has_Proper_Body (The_Task_Type => The_Task_Type);
   end Has_Task_Type_Body_Stub;

   --------------------------------------------------------------------------------

   function Has_Subprogram_Body_Stub (The_Subprogram : RawDict.Subprogram_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.Get_Subprogram_Body (The_Subprogram => The_Subprogram) /= RawDict.Null_Declaration_Info_Ref
        and then not RawDict.Get_Subprogram_Has_Proper_Body (The_Subprogram => The_Subprogram);
   end Has_Subprogram_Body_Stub;

   --------------------------------------------------------------------------------

   function HasBodyStub (CompilationUnit : Symbol) return Boolean is
      Result : Boolean;
   begin
      case RawDict.GetSymbolDiscriminant (CompilationUnit) is
         when Package_Symbol =>
            Result := Has_Package_Body_Stub (The_Package => RawDict.Get_Package_Info_Ref (Item => CompilationUnit)); -- GAA External
         when Type_Symbol =>
            case RawDict.Get_Type_Discriminant (Type_Mark => RawDict.Get_Type_Info_Ref (Item => CompilationUnit)) is -- GAA External
               when Protected_Type_Item =>
                  Result := Has_Protected_Type_Body_Stub
                    (The_Protected_Type => RawDict.Get_Type_Info_Ref (Item => CompilationUnit)); -- GAA External
               when Task_Type_Item =>
                  Result := Has_Task_Type_Body_Stub
                    (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => CompilationUnit)); -- GAA External
               when others => -- non-exec code
                  Result := False;
                  SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                            Msg     => "in Dictionary.HasBodyStub");
            end case;
         when Subprogram_Symbol =>
            Result := Has_Subprogram_Body_Stub
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => CompilationUnit)); -- GAA External
         when Variable_Symbol =>
            Result := False;
         when others => -- non-exec code
            Result := False;
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.HasBodyStub");
      end case;
      return Result;
   end HasBodyStub;

   --------------------------------------------------------------------------------
   --  Add_Body
   --------------------------------------------------------------------------------

   procedure Add_Subprogram_Body
     (The_Subprogram : in RawDict.Subprogram_Info_Ref;
      Comp_Unit      : in ContextManager.UnitDescriptors;
      The_Body       : in Location;
      Hidden         : in Boolean)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Comp_Unit,
   --#                                Hidden,
   --#                                The_Body,
   --#                                The_Subprogram &
   --#         SPARK_IO.File_Sys from *,
   --#                                Comp_Unit,
   --#                                Dict,
   --#                                Hidden,
   --#                                LexTokenManager.State,
   --#                                The_Body,
   --#                                The_Subprogram;
   is
      Scope           : Scopes;
      The_Declaration : RawDict.Declaration_Info_Ref;
   begin
      if not Has_Subprogram_Body_Stub (The_Subprogram => The_Subprogram) then
         Scope := Get_Subprogram_Scope (The_Subprogram => The_Subprogram);
         if Get_Visibility (Scope => Scope) = Visible or else Get_Visibility (Scope => Scope) = Privat then
            Scope := GetLocalScope (Scope);
         end if;
         Add_Declaration
           (Comp_Unit       => Comp_Unit,
            Loc             => The_Body,
            Scope           => Scope,
            Context         => ProgramContext,
            The_Declaration => The_Declaration);
         RawDict.Set_Declaration_Item (The_Declaration => The_Declaration,
                                       Item            => RawDict.Get_Subprogram_Symbol (The_Subprogram));
         RawDict.Set_Subprogram_Body (The_Subprogram => The_Subprogram,
                                      The_Body       => The_Declaration);
      end if;
      RawDict.Set_Subprogram_Has_Proper_Body (The_Subprogram => The_Subprogram,
                                              Is_Hidden      => Hidden);
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         if Hidden then
            Write_String (Dict.TemporaryFile, "hidden ");
         end if;
         Write_String (Dict.TemporaryFile, "body of ");
         Write_Name (File => Dict.TemporaryFile,
                     Item => RawDict.Get_Subprogram_Symbol (The_Subprogram));
         Write_String (Dict.TemporaryFile, " is at ");
         Write_Location (File => Dict.TemporaryFile,
                         Loc  => The_Body);
         Write_Line (Dict.TemporaryFile, " ;");
      end if;
   end Add_Subprogram_Body;

   --------------------------------------------------------------------------------

   procedure AddBody
     (CompilationUnit : in Symbol;
      Comp_Unit       : in ContextManager.UnitDescriptors;
      TheBody         : in Location;
      Hidden          : in Boolean) is

      procedure Add_Package_Body
        (The_Package : in RawDict.Package_Info_Ref;
         Comp_Unit   : in ContextManager.UnitDescriptors;
         The_Body    : in Location;
         Hidden      : in Boolean)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                The_Body,
      --#                                The_Package &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Dict,
      --#                                Hidden,
      --#                                LexTokenManager.State,
      --#                                The_Body,
      --#                                The_Package;
      is
         Scope           : Scopes;
         The_Declaration : RawDict.Declaration_Info_Ref;
      begin
         if not Has_Package_Body_Stub (The_Package => The_Package) then
            Scope := Get_Package_Scope (The_Package => The_Package);
            if RawDict.Get_Package_Parent (The_Package => The_Package) /= RawDict.Null_Package_Info_Ref then -- adding child package body
               RawDict.Create_Declaration
                 (Context         => ProgramContext,
                  Scope           => Scope,
                  Comp_Unit       => Comp_Unit,
                  Loc             => The_Body.Start_Position,
                  The_Declaration => The_Declaration);
            else
               if Get_Visibility (Scope => Scope) = Visible or else Get_Visibility (Scope => Scope) = Privat then
                  Scope := GetLocalScope (Scope);
               end if;
               Add_Declaration
                 (Comp_Unit       => Comp_Unit,
                  Loc             => The_Body,
                  Scope           => Scope,
                  Context         => ProgramContext,
                  The_Declaration => The_Declaration);
            end if;
            RawDict.Set_Declaration_Item
              (The_Declaration => The_Declaration,
               Item            => RawDict.Get_Package_Symbol (The_Package));
            RawDict.Set_Package_Body (The_Package => The_Package,
                                      The_Body    => The_Declaration);
         end if;
         RawDict.Set_Package_Has_Proper_Body (The_Package => The_Package);
         if SPARK_IO.Is_Open (Dict.TemporaryFile) then
            if Hidden then
               Write_String (Dict.TemporaryFile, "hidden ");
            end if;
            Write_String (Dict.TemporaryFile, "body of ");
            Write_Name (File => Dict.TemporaryFile,
                        Item => RawDict.Get_Package_Symbol (The_Package));
            Write_String (Dict.TemporaryFile, " is at ");
            Write_Location (File => Dict.TemporaryFile,
                            Loc  => The_Body);
            Write_Line (Dict.TemporaryFile, " ;");
         end if;
      end Add_Package_Body;

      --------------------------------------------------------------------------------

      procedure Add_Protected_Type_Body
        (The_Protected_Type : in RawDict.Type_Info_Ref;
         Comp_Unit          : in ContextManager.UnitDescriptors;
         The_Body           : in Location;
         Hidden             : in Boolean)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                The_Body,
      --#                                The_Protected_Type &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Dict,
      --#                                Hidden,
      --#                                LexTokenManager.State,
      --#                                The_Body,
      --#                                The_Protected_Type;
      is
         Scope           : Scopes;
         The_Declaration : RawDict.Declaration_Info_Ref;
      begin
         if not Has_Protected_Type_Body_Stub (The_Protected_Type => The_Protected_Type) then
            Scope := Get_Type_Scope (Type_Mark => The_Protected_Type);
            if Get_Visibility (Scope => Scope) = Visible or else Get_Visibility (Scope => Scope) = Privat then
               Scope := GetLocalScope (Scope);
            end if;
            Add_Declaration
              (Comp_Unit       => Comp_Unit,
               Loc             => The_Body,
               Scope           => Scope,
               Context         => ProgramContext,
               The_Declaration => The_Declaration);
            RawDict.Set_Declaration_Item
              (The_Declaration => The_Declaration,
               Item            => RawDict.Get_Type_Symbol (The_Protected_Type));
            RawDict.Set_Protected_Type_Body (The_Protected_Type => The_Protected_Type,
                                             The_Body           => The_Declaration);
         end if;
         RawDict.Set_Protected_Type_Has_Proper_Body (The_Protected_Type => The_Protected_Type);
         if SPARK_IO.Is_Open (Dict.TemporaryFile) then
            if Hidden then
               Write_String (Dict.TemporaryFile, "hidden ");
            end if;
            Write_String (Dict.TemporaryFile, "body of ");
            Write_Name (File => Dict.TemporaryFile,
                        Item => RawDict.Get_Type_Symbol (The_Protected_Type));
            Write_String (Dict.TemporaryFile, " is at ");
            Write_Location (File => Dict.TemporaryFile,
                            Loc  => The_Body);
            Write_Line (Dict.TemporaryFile, " ;");
         end if;
      end Add_Protected_Type_Body;

      --------------------------------------------------------------------------------

      procedure Add_Task_Type_Body
        (The_Task_Type : in RawDict.Type_Info_Ref;
         Comp_Unit     : in ContextManager.UnitDescriptors;
         The_Body      : in Location;
         Hidden        : in Boolean)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Hidden,
      --#                                The_Body,
      --#                                The_Task_Type &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Dict,
      --#                                Hidden,
      --#                                LexTokenManager.State,
      --#                                The_Body,
      --#                                The_Task_Type;
      is
         Scope           : Scopes;
         The_Declaration : RawDict.Declaration_Info_Ref;
      begin
         if not Has_Task_Type_Body_Stub (The_Task_Type => The_Task_Type) then
            Scope := Get_Type_Scope (Type_Mark => The_Task_Type);
            if Get_Visibility (Scope => Scope) = Visible or else Get_Visibility (Scope => Scope) = Privat then
               Scope := GetLocalScope (Scope);
            end if;
            Add_Declaration
              (Comp_Unit       => Comp_Unit,
               Loc             => The_Body,
               Scope           => Scope,
               Context         => ProgramContext,
               The_Declaration => The_Declaration);
            RawDict.Set_Declaration_Item (The_Declaration => The_Declaration,
                                          Item            => RawDict.Get_Type_Symbol (The_Task_Type));
            RawDict.Set_Task_Type_Body (The_Task_Type => The_Task_Type,
                                        The_Body      => The_Declaration);
         end if;
         RawDict.Set_Task_Type_Has_Proper_Body (The_Task_Type => The_Task_Type,
                                                Is_Hidden     => Hidden);
         if SPARK_IO.Is_Open (Dict.TemporaryFile) then
            if Hidden then
               Write_String (Dict.TemporaryFile, "hidden ");
            end if;
            Write_String (Dict.TemporaryFile, "body of ");
            Write_Name (File => Dict.TemporaryFile,
                        Item => RawDict.Get_Type_Symbol (The_Task_Type));
            Write_String (Dict.TemporaryFile, " is at ");
            Write_Location (File => Dict.TemporaryFile,
                            Loc  => The_Body);
            Write_Line (Dict.TemporaryFile, " ;");
         end if;
      end Add_Task_Type_Body;

   begin -- AddBody
      case RawDict.GetSymbolDiscriminant (CompilationUnit) is
         when Package_Symbol =>
            Add_Package_Body (The_Package => RawDict.Get_Package_Info_Ref (Item => CompilationUnit), -- GAA External
                              Comp_Unit   => Comp_Unit,
                              The_Body    => TheBody,
                              Hidden      => Hidden);
         when Type_Symbol =>
            case RawDict.Get_Type_Discriminant (Type_Mark => RawDict.Get_Type_Info_Ref (Item => CompilationUnit)) is -- GAA External
               when Protected_Type_Item =>
                  Add_Protected_Type_Body
                    (The_Protected_Type => RawDict.Get_Type_Info_Ref (Item => CompilationUnit), -- GAA External
                     Comp_Unit          => Comp_Unit,
                     The_Body           => TheBody,
                     Hidden             => Hidden);
               when Task_Type_Item =>
                  Add_Task_Type_Body
                    (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => CompilationUnit), -- GAA External
                     Comp_Unit     => Comp_Unit,
                     The_Body      => TheBody,
                     Hidden        => Hidden);
               when others => -- non-exec code
                  SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                            Msg     => "in Dictionary.AddBody");
            end case;
         when Subprogram_Symbol =>
            Add_Subprogram_Body
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => CompilationUnit), -- GAA External
               Comp_Unit      => Comp_Unit,
               The_Body       => TheBody,
               Hidden         => Hidden);
         when others => -- non-exec code
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.AddBody");
      end case;
   end AddBody;

   --------------------------------------------------------------------------------
   --  Add_Body_Stub
   --------------------------------------------------------------------------------

   procedure AddBodyStub (CompilationUnit : in Symbol;
                          Comp_Unit       : in ContextManager.UnitDescriptors;
                          BodyStub        : in Location) is

      procedure Add_Package_Body_Stub
        (The_Package : in RawDict.Package_Info_Ref;
         Comp_Unit   : in ContextManager.UnitDescriptors;
         Body_Stub   : in Location)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Body_Stub,
      --#                                Comp_Unit,
      --#                                The_Package &
      --#         SPARK_IO.File_Sys from *,
      --#                                Body_Stub,
      --#                                Comp_Unit,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                The_Package;
      is
         Scope           : Scopes;
         The_Declaration : RawDict.Declaration_Info_Ref;
      begin
         Scope := Get_Package_Scope (The_Package => The_Package);
         if Get_Visibility (Scope => Scope) = Visible or else Get_Visibility (Scope => Scope) = Privat then
            Scope := GetLocalScope (Scope);
         end if;
         Add_Declaration
           (Comp_Unit       => Comp_Unit,
            Loc             => Body_Stub,
            Scope           => Scope,
            Context         => ProgramContext,
            The_Declaration => The_Declaration);
         RawDict.Set_Declaration_Item
           (The_Declaration => The_Declaration,
            Item            => RawDict.Get_Package_Symbol (The_Package));
         RawDict.Set_Package_Body (The_Package => The_Package,
                                   The_Body    => The_Declaration);
         if SPARK_IO.Is_Open (Dict.TemporaryFile) then
            Write_String (Dict.TemporaryFile, "body stub of ");
            Write_Name (File => Dict.TemporaryFile,
                        Item => RawDict.Get_Package_Symbol (The_Package));
            Write_String (Dict.TemporaryFile, " is at ");
            Write_Location (File => Dict.TemporaryFile,
                            Loc  => Body_Stub);
            Write_Line (Dict.TemporaryFile, " ;");
         end if;
      end Add_Package_Body_Stub;

      --------------------------------------------------------------------------------

      procedure Add_Protected_Type_Body_Stub
        (The_Protected_Type : in RawDict.Type_Info_Ref;
         Comp_Unit          : in ContextManager.UnitDescriptors;
         Body_Stub          : in Location)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Body_Stub,
      --#                                Comp_Unit,
      --#                                The_Protected_Type &
      --#         SPARK_IO.File_Sys from *,
      --#                                Body_Stub,
      --#                                Comp_Unit,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                The_Protected_Type;
      is
         Scope           : Scopes;
         The_Declaration : RawDict.Declaration_Info_Ref;
      begin
         Scope := Get_Type_Scope (Type_Mark => The_Protected_Type);
         if Get_Visibility (Scope => Scope) = Visible or else Get_Visibility (Scope => Scope) = Privat then
            Scope := GetLocalScope (Scope);
         end if;
         Add_Declaration
           (Comp_Unit       => Comp_Unit,
            Loc             => Body_Stub,
            Scope           => Scope,
            Context         => ProgramContext,
            The_Declaration => The_Declaration);
         RawDict.Set_Declaration_Item (The_Declaration => The_Declaration,
                                       Item            => RawDict.Get_Type_Symbol (The_Protected_Type));
         RawDict.Set_Protected_Type_Body (The_Protected_Type => The_Protected_Type,
                                          The_Body           => The_Declaration);
         if SPARK_IO.Is_Open (Dict.TemporaryFile) then
            Write_String (Dict.TemporaryFile, "body stub of ");
            Write_Name (File => Dict.TemporaryFile,
                        Item => RawDict.Get_Type_Symbol (The_Protected_Type));
            Write_String (Dict.TemporaryFile, " is at ");
            Write_Location (File => Dict.TemporaryFile,
                            Loc  => Body_Stub);
            Write_Line (Dict.TemporaryFile, " ;");
         end if;
      end Add_Protected_Type_Body_Stub;

      --------------------------------------------------------------------------------

      procedure Add_Task_Type_Body_Stub
        (The_Task_Type : in RawDict.Type_Info_Ref;
         Comp_Unit     : in ContextManager.UnitDescriptors;
         Body_Stub     : in Location)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Body_Stub,
      --#                                Comp_Unit,
      --#                                The_Task_Type &
      --#         SPARK_IO.File_Sys from *,
      --#                                Body_Stub,
      --#                                Comp_Unit,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                The_Task_Type;
      is
         Scope           : Scopes;
         The_Declaration : RawDict.Declaration_Info_Ref;
      begin
         Scope := Get_Type_Scope (Type_Mark => The_Task_Type);
         if Get_Visibility (Scope => Scope) = Visible or else Get_Visibility (Scope => Scope) = Privat then
            Scope := GetLocalScope (Scope);
         end if;
         Add_Declaration
           (Comp_Unit       => Comp_Unit,
            Loc             => Body_Stub,
            Scope           => Scope,
            Context         => ProgramContext,
            The_Declaration => The_Declaration);
         RawDict.Set_Declaration_Item (The_Declaration => The_Declaration,
                                       Item            => RawDict.Get_Type_Symbol (The_Task_Type));
         RawDict.Set_Task_Type_Body (The_Task_Type => The_Task_Type,
                                     The_Body      => The_Declaration);
         if SPARK_IO.Is_Open (Dict.TemporaryFile) then
            Write_String (Dict.TemporaryFile, "body stub of ");
            Write_Name (File => Dict.TemporaryFile,
                        Item => RawDict.Get_Type_Symbol (The_Task_Type));
            Write_String (Dict.TemporaryFile, " is at ");
            Write_Location (File => Dict.TemporaryFile,
                            Loc  => Body_Stub);
            Write_Line (Dict.TemporaryFile, " ;");
         end if;
      end Add_Task_Type_Body_Stub;

      --------------------------------------------------------------------------------

      procedure Add_Subprogram_Body_Stub
        (The_Subprogram : in RawDict.Subprogram_Info_Ref;
         Comp_Unit      : in ContextManager.UnitDescriptors;
         Body_Stub      : in Location)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Body_Stub,
      --#                                Comp_Unit,
      --#                                The_Subprogram &
      --#         SPARK_IO.File_Sys from *,
      --#                                Body_Stub,
      --#                                Comp_Unit,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                The_Subprogram;
      is
         Scope           : Scopes;
         The_Declaration : RawDict.Declaration_Info_Ref;
      begin
         Scope := Get_Subprogram_Scope (The_Subprogram => The_Subprogram);
         if Get_Visibility (Scope => Scope) = Visible or else Get_Visibility (Scope => Scope) = Privat then
            Scope := GetLocalScope (Scope);
         end if;
         Add_Declaration
           (Comp_Unit       => Comp_Unit,
            Loc             => Body_Stub,
            Scope           => Scope,
            Context         => ProgramContext,
            The_Declaration => The_Declaration);
         RawDict.Set_Declaration_Item (The_Declaration => The_Declaration,
                                       Item            => RawDict.Get_Subprogram_Symbol (The_Subprogram));
         RawDict.Set_Subprogram_Body (The_Subprogram => The_Subprogram,
                                      The_Body       => The_Declaration);
         if SPARK_IO.Is_Open (Dict.TemporaryFile) then
            Write_String (Dict.TemporaryFile, "body stub of ");
            Write_Name (File => Dict.TemporaryFile,
                        Item => RawDict.Get_Subprogram_Symbol (The_Subprogram));
            Write_String (Dict.TemporaryFile, " is at ");
            Write_Location (File => Dict.TemporaryFile,
                            Loc  => Body_Stub);
            Write_Line (Dict.TemporaryFile, " ;");
         end if;
      end Add_Subprogram_Body_Stub;

   begin -- AddBodyStub
      case RawDict.GetSymbolDiscriminant (CompilationUnit) is
         when Package_Symbol =>
            Add_Package_Body_Stub (The_Package => RawDict.Get_Package_Info_Ref (Item => CompilationUnit), -- GAA External
                                   Comp_Unit   => Comp_Unit,
                                   Body_Stub   => BodyStub);
         when Type_Symbol =>
            case RawDict.Get_Type_Discriminant (Type_Mark => RawDict.Get_Type_Info_Ref (Item => CompilationUnit)) is -- GAA External
               when Protected_Type_Item =>
                  Add_Protected_Type_Body_Stub
                    (The_Protected_Type => RawDict.Get_Type_Info_Ref (Item => CompilationUnit), -- GAA External
                     Comp_Unit          => Comp_Unit,
                     Body_Stub          => BodyStub);
               when Task_Type_Item =>
                  Add_Task_Type_Body_Stub
                    (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => CompilationUnit), -- GAA External
                     Comp_Unit     => Comp_Unit,
                     Body_Stub     => BodyStub);
               when others => -- non-exec code
                  SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                            Msg     => "in Dictionary.AddBodyStub");
            end case;
         when Subprogram_Symbol =>
            Add_Subprogram_Body_Stub
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => CompilationUnit), -- GAA External
               Comp_Unit      => Comp_Unit,
               Body_Stub      => BodyStub);
         when others => -- non-exec code
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.AddBodyStub");
      end case;
   end AddBodyStub;

   --------------------------------------------------------------------------------
   --  Add_Subprogram_Instantiation
   --------------------------------------------------------------------------------

   procedure AddSubprogramInstantiation
     (Name          : in     LexTokenManager.Lex_String;
      Comp_Unit     : in     ContextManager.UnitDescriptors;
      Declaration   : in     Location;
      TheGeneric    : in     Symbol;
      Specification : in     Location;
      Scope         : in     Scopes;
      Context       : in     Contexts;
      Subprogram    :    out Symbol) is
      The_Subprogram  : RawDict.Subprogram_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Subprogram_Instantiation
        (Name                   : in     LexTokenManager.Lex_String;
         Comp_Unit              : in     ContextManager.UnitDescriptors;
         Declaration            : in     Location;
         The_Generic_Subprogram : in     RawDict.Subprogram_Info_Ref;
         Specification          : in     Location;
         Scope                  : in     Scopes;
         Context                : in     Contexts;
         The_Subprogram         :    out RawDict.Subprogram_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Name,
      --#                                Scope,
      --#                                Specification,
      --#                                The_Generic_Subprogram &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Context,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Scope,
      --#                                Specification,
      --#                                The_Generic_Subprogram &
      --#         The_Subprogram    from Comp_Unit,
      --#                                Context,
      --#                                Dict,
      --#                                Name,
      --#                                Scope,
      --#                                Specification;
      --# post The_Subprogram /= RawDict.Null_Subprogram_Info_Ref;
      is
      begin
         Add_Subprogram
           (Name           => Name,
            Comp_Unit      => Comp_Unit,
            Specification  => Specification,
            Scope          => Scope,
            Context        => Context,
            The_Subprogram => The_Subprogram);
         RawDict.Set_Subprogram_Instantiation_Of
           (The_Subprogram => The_Subprogram,
            The_Generic    => The_Generic_Subprogram);
         if RawDict.Get_Subprogram_Has_Derives_Annotation (The_Subprogram => The_Generic_Subprogram) then
            RawDict.Set_Subprogram_Has_Derives_Annotation  (The_Subprogram => The_Subprogram);
         end if;
         -- Mark as having hidden body so Examiner won't demand a body later on
         Add_Subprogram_Body (The_Subprogram => The_Subprogram,
                              Comp_Unit      => Comp_Unit,
                              The_Body       => Declaration,
                              Hidden         => True);
      end Add_Subprogram_Instantiation;

   begin -- AddSubprogramInstantiation
      Add_Subprogram_Instantiation
        (Name                   => Name,
         Comp_Unit              => Comp_Unit,
         Declaration            => Declaration,
         The_Generic_Subprogram => RawDict.Get_Subprogram_Info_Ref (TheGeneric), -- GAA External
         Specification          => Specification,
         Scope                  => Scope,
         Context                => Context,
         The_Subprogram         => The_Subprogram);
      Subprogram := RawDict.Get_Subprogram_Symbol (The_Subprogram); -- GAA External
   end AddSubprogramInstantiation;

   --------------------------------------------------------------------------------
   --  Add_Main_Program
   --------------------------------------------------------------------------------

   procedure AddMainProgram (Subprogram : in Symbol;
                             Annotation : in Location) is

      procedure Add_Main_Program (The_Subprogram : in RawDict.Subprogram_Info_Ref;
                                  Annotation     : in Location)
      --# global in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                The_Subprogram &
      --#         SPARK_IO.File_Sys from *,
      --#                                Annotation,
      --#                                Dict,
      --#                                The_Subprogram;
      is
      begin
         Dict.Main.Subprogram := The_Subprogram;
         if SPARK_IO.Is_Open (Dict.TemporaryFile) then
            Write_String (Dict.TemporaryFile, "main program annotation is at ");
            Write_Location (File => Dict.TemporaryFile,
                            Loc  => Annotation);
            Write_Line (Dict.TemporaryFile, " ;");
         end if;
      end Add_Main_Program;

   begin -- AddMainProgram
      Add_Main_Program
        (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Subprogram), -- GAA External
         Annotation     => Annotation);
   end AddMainProgram;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   procedure AddDependencyRelation (Abstraction        : in Abstractions;
                                    TheProcedure       : in Symbol;
                                    DependencyRelation : in Location) is
   begin
      if Abstraction = IsRefined then
         case RawDict.GetSymbolDiscriminant (TheProcedure) is
            when Subprogram_Symbol =>
               RawDict.Set_Subprogram_Has_Second_Annotation
                 (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => TheProcedure)); -- GAA External
            when Type_Symbol =>
               SystemErrors.RT_Assert
                 (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheProcedure)), -- GAA External
                  Sys_Err => SystemErrors.Invalid_Symbol_Table,
                  Msg     => "in Dictionary.AddDependencyRelation");
               RawDict.Set_Task_Type_Has_Second_Annotation
                 (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => TheProcedure)); -- GAA External
            when others => -- non-exec code
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                  Msg     => "in Dictionary.AddDependencyRelation");
         end case;
      end if;
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         if Abstraction = IsRefined then
            Write_String (Dict.TemporaryFile, "refined ");
         end if;
         Write_String (Dict.TemporaryFile, "dependency relation of ");
         Write_Name (File => Dict.TemporaryFile,
                     Item => TheProcedure);
         Write_String (Dict.TemporaryFile, " is at ");
         Write_Location (File => Dict.TemporaryFile,
                         Loc  => DependencyRelation);
         Write_Line (Dict.TemporaryFile, " ;");
      end if;
   end AddDependencyRelation;

   --------------------------------------------------------------------------------

   procedure Add_Renaming_Declaration
     (The_Subprogram : in RawDict.Subprogram_Info_Ref;
      The_Operator   : in RawDict.Operator_Info_Ref;
      Comp_Unit      : in ContextManager.UnitDescriptors;
      Declaration    : in Location;
      Scope          : in Scopes)
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Comp_Unit,
   --#                   Declaration,
   --#                   Scope,
   --#                   The_Operator,
   --#                   The_Subprogram;
      is separate;

   --------------------------------------------------------------------------------
   --  Rename_Subprogram
   --------------------------------------------------------------------------------

   procedure RenameSubprogram
     (Subprogram          : in Symbol;
      SubprogramReference : in Location;
      Comp_Unit           : in ContextManager.UnitDescriptors;
      Declaration         : in Location;
      Scope               : in Scopes) is

      procedure Rename_Subprogram
        (The_Subprogram       : in RawDict.Subprogram_Info_Ref;
         Subprogram_Reference : in Location;
         Comp_Unit            : in ContextManager.UnitDescriptors;
         Declaration          : in Location;
         Scope                : in Scopes)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Declaration,
      --#                                Scope,
      --#                                The_Subprogram &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Scope,
      --#                                Subprogram_Reference,
      --#                                The_Subprogram;
      is
      begin
         Add_Renaming_Declaration
           (The_Subprogram  => The_Subprogram,
            The_Operator    => RawDict.Null_Operator_Info_Ref,
            Comp_Unit       => Comp_Unit,
            Declaration     => Declaration,
            Scope           => Scope);
         Write_Subprogram_Declaration
           (The_Subprogram => The_Subprogram,
            Is_Renaming    => True,
            Scope          => Scope,
            Declaration    => Declaration);
         AddOtherReference (RawDict.Get_Subprogram_Symbol (The_Subprogram), GetRegion (Scope), Subprogram_Reference);
      end Rename_Subprogram;

   begin -- RenameSubprogram
      Rename_Subprogram
        (The_Subprogram       => RawDict.Get_Subprogram_Info_Ref (Subprogram), -- GAA External
         Subprogram_Reference => SubprogramReference,
         Comp_Unit            => Comp_Unit,
         Declaration          => Declaration,
         Scope                => Scope);
   end RenameSubprogram;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   procedure WriteOperatorRenamingDeclaration (The_Operator : in RawDict.Operator_Info_Ref;
                                               Scope        : in Scopes)
   --# global in     Dict;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Scope,
   --#                                The_Operator;
      is separate;

   --------------------------------------------------------------------------------
   --  Rename_Unary_Operator
   --------------------------------------------------------------------------------

   procedure RenameUnaryOperator
     (Name        : in     SP_Symbols.SP_Symbol;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      Operand     : in     Symbol;
      Scope       : in     Scopes;
      Op_Sym      :    out Symbol) is
      The_Operator : RawDict.Operator_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Rename_Unary_Operator
        (Name         : in     SP_Symbols.SP_Symbol;
         Comp_Unit    : in     ContextManager.UnitDescriptors;
         Declaration  : in     Location;
         Operand      : in     RawDict.Type_Info_Ref;
         Scope        : in     Scopes;
         The_Operator :    out RawDict.Operator_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Declaration,
      --#                                Name,
      --#                                Operand,
      --#                                Scope &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Operand,
      --#                                Scope &
      --#         The_Operator      from Comp_Unit,
      --#                                Declaration,
      --#                                Dict,
      --#                                Name,
      --#                                Operand;
      --# post The_Operator /= RawDict.Null_Operator_Info_Ref;
      is
      begin
         RawDict.Create_Unary_Operator
           (Name         => Name,
            Operand      => Operand,
            Comp_Unit    => Comp_Unit,
            Loc          => Declaration.Start_Position,
            The_Operator => The_Operator);
         Add_Renaming_Declaration
           (The_Subprogram  => RawDict.Null_Subprogram_Info_Ref,
            The_Operator    => The_Operator,
            Comp_Unit       => Comp_Unit,
            Declaration     => Declaration,
            Scope           => Scope);
         WriteOperatorRenamingDeclaration (The_Operator => The_Operator,
                                           Scope        => Scope);
      end Rename_Unary_Operator;

   begin -- RenameUnaryOperator
      Rename_Unary_Operator
        (Name         => Name,
         Comp_Unit    => Comp_Unit,
         Declaration  => Declaration,
         Operand      => RawDict.Get_Type_Info_Ref (Operand), -- GAA External
         Scope        => Scope,
         The_Operator => The_Operator);
      Op_Sym := RawDict.Get_Operator_Symbol (The_Operator); -- GAA External
   end RenameUnaryOperator;

   --------------------------------------------------------------------------------
   --  RenameBinaryOperator
   --------------------------------------------------------------------------------

   procedure RenameBinaryOperator
     (Name        : in     SP_Symbols.SP_Symbol;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      Left        : in     Symbol;
      Right       : in     Symbol;
      Scope       : in     Scopes;
      Op_Sym      :    out Symbol) is
      The_Operator : RawDict.Operator_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Rename_Binary_Operator
        (Name         : in     SP_Symbols.SP_Symbol;
         Comp_Unit    : in     ContextManager.UnitDescriptors;
         Declaration  : in     Location;
         Left         : in     RawDict.Type_Info_Ref;
         Right        : in     RawDict.Type_Info_Ref;
         Scope        : in     Scopes;
         The_Operator :    out RawDict.Operator_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Declaration,
      --#                                Left,
      --#                                Name,
      --#                                Right,
      --#                                Scope &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Declaration,
      --#                                Dict,
      --#                                Left,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Right,
      --#                                Scope &
      --#         The_Operator      from Comp_Unit,
      --#                                Declaration,
      --#                                Dict,
      --#                                Left,
      --#                                Name,
      --#                                Right;
      --# post The_Operator /= RawDict.Null_Operator_Info_Ref;
      is
      begin
         RawDict.Create_Binary_Operator
           (Name         => Name,
            Left         => Left,
            Right        => Right,
            Comp_Unit    => Comp_Unit,
            Loc          => Declaration.Start_Position,
            The_Operator => The_Operator);
         Add_Renaming_Declaration
           (The_Subprogram  => RawDict.Null_Subprogram_Info_Ref,
            The_Operator    => The_Operator,
            Comp_Unit       => Comp_Unit,
            Declaration     => Declaration,
            Scope           => Scope);
         WriteOperatorRenamingDeclaration (The_Operator => The_Operator,
                                           Scope        => Scope);
      end Rename_Binary_Operator;

   begin -- RenameBinaryOperator
      Rename_Binary_Operator
        (Name         => Name,
         Comp_Unit    => Comp_Unit,
         Declaration  => Declaration,
         Left         => RawDict.Get_Type_Info_Ref (Left), -- GAA External
         Right        => RawDict.Get_Type_Info_Ref (Right), -- GAA External
         Scope        => Scope,
         The_Operator => The_Operator);
      Op_Sym := RawDict.Get_Operator_Symbol (The_Operator); -- GAA External
   end RenameBinaryOperator;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   procedure AddGlobalAnnotation
     (Abstraction : in Abstractions;
      Subprogram  : in Symbol;
      Comp_Unit   : in ContextManager.UnitDescriptors;
      Annotation  : in Location) is
      The_Subprogram     : RawDict.Subprogram_Info_Ref;
      The_Proof_Function : Symbol;
   begin
      if Abstraction = IsRefined and then RawDict.GetSymbolDiscriminant (Subprogram) = Subprogram_Symbol then
         The_Subprogram := RawDict.Get_Subprogram_Info_Ref (Item => Subprogram); -- GAA External
         if not RawDict.Get_Subprogram_Has_Second_Annotation (The_Subprogram => The_Subprogram) then
            RawDict.Set_Subprogram_Has_Second_Annotation (The_Subprogram => The_Subprogram);
            if IsFunction (Subprogram) and then Get_Subprogram_Context (The_Subprogram => The_Subprogram) = ProgramContext then
               RawDict.CreateImplicitProofFunction
                 (Ada_Function  => The_Subprogram,
                  Comp_Unit     => Comp_Unit,
                  Loc           => Annotation.Start_Position,
                  ProofFunction => The_Proof_Function);
               RawDict.Set_Subprogram_Implicit_Proof_Function
                 (The_Subprogram     => The_Subprogram,
                  Abstraction        => IsRefined,
                  The_Proof_Function => The_Proof_Function);
            end if;
         end if;
      end if;
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         if Abstraction = IsRefined then
            Write_String (Dict.TemporaryFile, "refined ");
         end if;
         Write_String (Dict.TemporaryFile, "global annotation of ");
         Write_Name (File => Dict.TemporaryFile,
                     Item => Subprogram);
         Write_String (Dict.TemporaryFile, " is at ");
         Write_Location (File => Dict.TemporaryFile,
                         Loc  => Annotation);
         Write_Line (Dict.TemporaryFile, " ;");
      end if;
   end AddGlobalAnnotation;

   --------------------------------------------------------------------------------
   --  Add_Global_Variable
   --------------------------------------------------------------------------------

   procedure Add_Subprogram_Global_Variable
     (Abstraction         : in     Abstractions;
      The_Subprogram      : in     RawDict.Subprogram_Info_Ref;
      The_Variable        : in     RawDict.Variable_Info_Ref;
      Mode                : in     Modes;
      Prefix_Needed       : in     Boolean;
      Comp_Unit           : in     ContextManager.UnitDescriptors;
      Variable_Reference  : in     Location;
      The_Global_Variable :    out RawDict.Global_Variable_Info_Ref)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         The_Global_Variable from Abstraction,
   --#                                  Comp_Unit,
   --#                                  Dict,
   --#                                  Mode,
   --#                                  Prefix_Needed,
   --#                                  The_Subprogram,
   --#                                  The_Variable,
   --#                                  Variable_Reference &
   --#         SPARK_IO.File_Sys   from *,
   --#                                  Abstraction,
   --#                                  Comp_Unit,
   --#                                  Dict,
   --#                                  LexTokenManager.State,
   --#                                  Mode,
   --#                                  Prefix_Needed,
   --#                                  The_Subprogram,
   --#                                  The_Variable,
   --#                                  Variable_Reference;
   --# post The_Global_Variable /= RawDict.Null_Global_Variable_Info_Ref;
   is
   begin
      RawDict.Create_Global_Variable
        (Mode                 => Mode,
         Prefix_Needed        => Prefix_Needed,
         The_Subprogram       => The_Subprogram,
         The_Task_Type        => RawDict.Null_Type_Info_Ref,
         Subprogram_Parameter => RawDict.Null_Subprogram_Parameter_Info_Ref,
         Variable             => The_Variable,
         Next_Subprogram      => RawDict.Get_Variable_Global_References (The_Variable => The_Variable,
                                                                         Abstraction  => Abstraction),
         Comp_Unit            => Comp_Unit,
         Loc                  => Variable_Reference.Start_Position,
         The_Global_Variable  => The_Global_Variable);

      RawDict.Set_Variable_Global_References
        (The_Variable => The_Variable,
         Abstraction  => Abstraction,
         Reference    => The_Global_Variable);

      if RawDict.Get_Subprogram_First_Global_Variable (The_Subprogram => The_Subprogram,
                                                       Abstraction    => Abstraction) =
        RawDict.Null_Global_Variable_Info_Ref then
         RawDict.Set_Subprogram_First_Global_Variable
           (The_Subprogram      => The_Subprogram,
            Abstraction         => Abstraction,
            The_Global_Variable => The_Global_Variable);
      else
         RawDict.Set_Next_Global_Variable
           (The_Global_Variable => RawDict.Get_Subprogram_Last_Global_Variable (The_Subprogram => The_Subprogram,
                                                                                Abstraction    => Abstraction),
            Next                => The_Global_Variable);
      end if;
      RawDict.Set_Subprogram_Last_Global_Variable
        (The_Subprogram      => The_Subprogram,
         Abstraction         => Abstraction,
         The_Global_Variable => The_Global_Variable);
      AddOtherReference
        (RawDict.Get_Variable_Symbol (The_Variable),
         RawDict.Get_Subprogram_Symbol (The_Subprogram),
         Variable_Reference);
   end Add_Subprogram_Global_Variable;

   --------------------------------------------------------------------------------

   procedure AddGlobalVariable
     (Abstraction         : in     Abstractions;
      Subprogram          : in     Symbol;
      Variable            : in     Symbol;
      Mode                : in     Modes;
      PrefixNeeded        : in     Boolean;
      Comp_Unit           : in     ContextManager.UnitDescriptors;
      VariableReference   : in     Location;
      Global_Variable_Sym :    out Symbol) is
      The_Global_Variable : RawDict.Global_Variable_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Subprogram_Global_Parameter
        (Abstraction              : in     Abstractions;
         The_Subprogram           : in     RawDict.Subprogram_Info_Ref;
         The_Subprogram_Parameter : in     RawDict.Subprogram_Parameter_Info_Ref;
         Mode                     : in     Modes;
         Prefix_Needed            : in     Boolean;
         Comp_Unit                : in     ContextManager.UnitDescriptors;
         Variable_Reference       : in     Location;
         The_Global_Variable      :    out RawDict.Global_Variable_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict,
      --#         The_Global_Variable from Abstraction,
      --#                                  Comp_Unit,
      --#                                  Dict,
      --#                                  Mode,
      --#                                  Prefix_Needed,
      --#                                  The_Subprogram,
      --#                                  The_Subprogram_Parameter,
      --#                                  Variable_Reference &
      --#         SPARK_IO.File_Sys   from *,
      --#                                  Abstraction,
      --#                                  Comp_Unit,
      --#                                  Dict,
      --#                                  LexTokenManager.State,
      --#                                  Mode,
      --#                                  Prefix_Needed,
      --#                                  The_Subprogram,
      --#                                  The_Subprogram_Parameter,
      --#                                  Variable_Reference;
      --# post The_Global_Variable /= RawDict.Null_Global_Variable_Info_Ref;
      is
      begin
         RawDict.Create_Global_Variable
           (Mode                 => Mode,
            Prefix_Needed        => Prefix_Needed,
            The_Subprogram       => The_Subprogram,
            The_Task_Type        => RawDict.Null_Type_Info_Ref,
            Subprogram_Parameter => The_Subprogram_Parameter,
            Variable             => RawDict.Null_Variable_Info_Ref,
            Next_Subprogram      => RawDict.Get_Subprogram_Parameter_Global_References
              (The_Subprogram_Parameter => The_Subprogram_Parameter,
               Abstraction              => Abstraction),
            Comp_Unit            => Comp_Unit,
            Loc                  => Variable_Reference.Start_Position,
            The_Global_Variable  => The_Global_Variable);

         RawDict.Set_Subprogram_Parameter_Global_References
           (The_Subprogram_Parameter => The_Subprogram_Parameter,
            Abstraction              => Abstraction,
            Reference                => The_Global_Variable);

         if RawDict.Get_Subprogram_First_Global_Variable (The_Subprogram => The_Subprogram,
                                                          Abstraction    => Abstraction) =
           RawDict.Null_Global_Variable_Info_Ref then
            RawDict.Set_Subprogram_First_Global_Variable
              (The_Subprogram      => The_Subprogram,
               Abstraction         => Abstraction,
               The_Global_Variable => The_Global_Variable);
         else
            RawDict.Set_Next_Global_Variable
              (The_Global_Variable => RawDict.Get_Subprogram_Last_Global_Variable (The_Subprogram => The_Subprogram,
                                                                                   Abstraction    => Abstraction),
               Next                => The_Global_Variable);
         end if;
         RawDict.Set_Subprogram_Last_Global_Variable
           (The_Subprogram      => The_Subprogram,
            Abstraction         => Abstraction,
            The_Global_Variable => The_Global_Variable);
         AddOtherReference
           (RawDict.Get_Subprogram_Parameter_Symbol (The_Subprogram_Parameter),
            RawDict.Get_Subprogram_Symbol (The_Subprogram),
            Variable_Reference);
      end Add_Subprogram_Global_Parameter;

      --------------------------------------------------------------------------------

      procedure Add_Task_Type_Global_Variable
        (Abstraction         : in     Abstractions;
         The_Task_Type       : in     RawDict.Type_Info_Ref;
         The_Variable        : in     RawDict.Variable_Info_Ref;
         Mode                : in     Modes;
         Prefix_Needed       : in     Boolean;
         Comp_Unit           : in     ContextManager.UnitDescriptors;
         Variable_Reference  : in     Location;
         The_Global_Variable :    out RawDict.Global_Variable_Info_Ref)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict,
      --#         The_Global_Variable from Abstraction,
      --#                                  Comp_Unit,
      --#                                  Dict,
      --#                                  Mode,
      --#                                  Prefix_Needed,
      --#                                  The_Task_Type,
      --#                                  The_Variable,
      --#                                  Variable_Reference &
      --#         SPARK_IO.File_Sys   from *,
      --#                                  Abstraction,
      --#                                  Comp_Unit,
      --#                                  Dict,
      --#                                  LexTokenManager.State,
      --#                                  Mode,
      --#                                  Prefix_Needed,
      --#                                  The_Task_Type,
      --#                                  The_Variable,
      --#                                  Variable_Reference;
      --# post The_Global_Variable /= RawDict.Null_Global_Variable_Info_Ref;
      is
      begin
         RawDict.Create_Global_Variable
           (Mode                 => Mode,
            Prefix_Needed        => Prefix_Needed,
            The_Subprogram       => RawDict.Null_Subprogram_Info_Ref,
            The_Task_Type        => The_Task_Type,
            Subprogram_Parameter => RawDict.Null_Subprogram_Parameter_Info_Ref,
            Variable             => The_Variable,
            Next_Subprogram      => RawDict.Get_Variable_Global_References (The_Variable => The_Variable,
                                                                            Abstraction  => Abstraction),
            Comp_Unit            => Comp_Unit,
            Loc                  => Variable_Reference.Start_Position,
            The_Global_Variable  => The_Global_Variable);

         RawDict.Set_Variable_Global_References
           (The_Variable => The_Variable,
            Abstraction  => Abstraction,
            Reference    => The_Global_Variable);

         if RawDict.Get_Task_Type_First_Global_Variable (The_Task_Type => The_Task_Type,
                                                         Abstraction   => Abstraction) =
           RawDict.Null_Global_Variable_Info_Ref then
            RawDict.Set_Task_Type_First_Global_Variable
              (The_Task_Type       => The_Task_Type,
               Abstraction         => Abstraction,
               The_Global_Variable => The_Global_Variable);
         else
            RawDict.Set_Next_Global_Variable
              (The_Global_Variable => RawDict.Get_Task_Type_Last_Global_Variable (The_Task_Type => The_Task_Type,
                                                                                  Abstraction   => Abstraction),
               Next                => The_Global_Variable);
         end if;
         RawDict.Set_Task_Type_Last_Global_Variable
           (The_Task_Type       => The_Task_Type,
            Abstraction         => Abstraction,
            The_Global_Variable => The_Global_Variable);
         AddOtherReference
           (RawDict.Get_Variable_Symbol (The_Variable),
            RawDict.Get_Type_Symbol (The_Task_Type),
            Variable_Reference);
      end Add_Task_Type_Global_Variable;

   begin -- AddGlobalVariable
      case RawDict.GetSymbolDiscriminant (Subprogram) is
         when Subprogram_Symbol =>
            case RawDict.GetSymbolDiscriminant (Variable) is
               when Variable_Symbol =>
                  Add_Subprogram_Global_Variable
                    (Abstraction         => Abstraction,
                     The_Subprogram      => RawDict.Get_Subprogram_Info_Ref (Item => Subprogram), -- GAA External
                     The_Variable        => RawDict.Get_Variable_Info_Ref (Item => Variable), -- GAA External
                     Mode                => Mode,
                     Prefix_Needed       => PrefixNeeded,
                     Comp_Unit           => Comp_Unit,
                     Variable_Reference  => VariableReference,
                     The_Global_Variable => The_Global_Variable);
               when Subprogram_Parameter_Symbol =>
                  Add_Subprogram_Global_Parameter
                    (Abstraction              => Abstraction,
                     The_Subprogram           => RawDict.Get_Subprogram_Info_Ref (Item => Subprogram), -- GAA External
                     The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => Variable), -- GAA External
                     Mode                     => Mode,
                     Prefix_Needed            => PrefixNeeded,
                     Comp_Unit                => Comp_Unit,
                     Variable_Reference       => VariableReference,
                     The_Global_Variable      => The_Global_Variable);
               when others => -- non-exec code
                  The_Global_Variable := RawDict.Null_Global_Variable_Info_Ref;
                  SystemErrors.Fatal_Error
                    (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                     Msg     => "in Dictionary.AddGlobalVariable");
            end case;
         when Type_Symbol =>
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Subprogram)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.AddGlobalVariable");
            Add_Task_Type_Global_Variable
              (Abstraction         => Abstraction,
               The_Task_Type       => RawDict.Get_Type_Info_Ref (Item => Subprogram), -- GAA External
               The_Variable        => RawDict.Get_Variable_Info_Ref (Item => Variable), -- GAA External
               Mode                => Mode,
               Prefix_Needed       => PrefixNeeded,
               Comp_Unit           => Comp_Unit,
               Variable_Reference  => VariableReference,
               The_Global_Variable => The_Global_Variable);
         when others => -- non-exec code
            The_Global_Variable := RawDict.Null_Global_Variable_Info_Ref;
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.AddGlobalVariable");
      end case;
      Global_Variable_Sym := RawDict.Get_Global_Variable_Symbol (The_Global_Variable); -- GAA External
   end AddGlobalVariable;

   --------------------------------------------------------------------------------
   --  Get_Global_Variable
   --------------------------------------------------------------------------------

   function Get_Subprogram_Variable_Global_Variable
     (The_Subprogram : RawDict.Subprogram_Info_Ref;
      Abstraction    : Abstractions;
      The_Variable   : RawDict.Variable_Info_Ref)
     return           RawDict.Global_Variable_Info_Ref
   --# global in Dict;
   is
      The_Global_Variable : RawDict.Global_Variable_Info_Ref;
   begin
      The_Global_Variable := RawDict.Get_Variable_Global_References (The_Variable => The_Variable,
                                                                     Abstraction  => Abstraction);
      loop
         exit when (RawDict.Get_Kind_Of_Global_Variable (The_Global_Variable => The_Global_Variable) =
                      RawDict.Subprogram_Parameter_Item
                      or else RawDict.Get_Kind_Of_Global_Variable (The_Global_Variable => The_Global_Variable) =
                      RawDict.Subprogram_Variable_Item)
           and then RawDict.Get_Global_Variable_Subprogram (The_Global_Variable => The_Global_Variable) = The_Subprogram;
         The_Global_Variable := RawDict.Get_Global_Variable_Next_Subprogram (The_Global_Variable => The_Global_Variable);
      end loop;
      return The_Global_Variable;
   end Get_Subprogram_Variable_Global_Variable;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Parameter_Global_Variable
     (The_Subprogram           : RawDict.Subprogram_Info_Ref;
      Abstraction              : Abstractions;
      The_Subprogram_Parameter : RawDict.Subprogram_Parameter_Info_Ref)
     return                     RawDict.Global_Variable_Info_Ref
   --# global in Dict;
   is
      The_Global_Variable : RawDict.Global_Variable_Info_Ref;
   begin
      The_Global_Variable :=
        RawDict.Get_Subprogram_Parameter_Global_References
        (The_Subprogram_Parameter => The_Subprogram_Parameter,
         Abstraction              => Abstraction);
      loop
         exit when (RawDict.Get_Kind_Of_Global_Variable (The_Global_Variable => The_Global_Variable) =
                      RawDict.Subprogram_Parameter_Item
                      or else RawDict.Get_Kind_Of_Global_Variable (The_Global_Variable => The_Global_Variable) =
                      RawDict.Subprogram_Variable_Item)
           and then RawDict.Get_Global_Variable_Subprogram (The_Global_Variable => The_Global_Variable) = The_Subprogram;
         The_Global_Variable := RawDict.Get_Global_Variable_Next_Subprogram (The_Global_Variable => The_Global_Variable);
      end loop;
      return The_Global_Variable;
   end Get_Subprogram_Parameter_Global_Variable;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Variable_Global_Variable
     (The_Task_Type : RawDict.Type_Info_Ref;
      Abstraction   : Abstractions;
      The_Variable  : RawDict.Variable_Info_Ref)
     return          RawDict.Global_Variable_Info_Ref
   --# global in Dict;
   is
      The_Global_Variable : RawDict.Global_Variable_Info_Ref;
   begin
      The_Global_Variable := RawDict.Get_Variable_Global_References (The_Variable => The_Variable,
                                                                     Abstraction  => Abstraction);
      loop
         exit when RawDict.Get_Kind_Of_Global_Variable (The_Global_Variable => The_Global_Variable) =
           RawDict.Task_Type_Variable_Item
           and then RawDict.Get_Global_Variable_Task_Type (The_Global_Variable => The_Global_Variable) = The_Task_Type;
         The_Global_Variable := RawDict.Get_Global_Variable_Next_Subprogram (The_Global_Variable => The_Global_Variable);
      end loop;
      return The_Global_Variable;
   end Get_Task_Type_Variable_Global_Variable;

   --------------------------------------------------------------------------------
   --  Get_Global_Mode
   --------------------------------------------------------------------------------

   function Get_Subprogram_Variable_Global_Mode
     (The_Subprogram : RawDict.Subprogram_Info_Ref;
      Abstraction    : Abstractions;
      The_Variable   : RawDict.Variable_Info_Ref)
     return           Modes
   --# global in Dict;
   is
   begin
      return RawDict.Get_Global_Variable_Mode
        (The_Global_Variable => Get_Subprogram_Variable_Global_Variable
           (The_Subprogram => The_Subprogram,
            Abstraction    => Abstraction,
            The_Variable   => The_Variable));
   end Get_Subprogram_Variable_Global_Mode;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Parameter_Global_Mode
     (The_Subprogram           : RawDict.Subprogram_Info_Ref;
      Abstraction              : Abstractions;
      The_Subprogram_Parameter : RawDict.Subprogram_Parameter_Info_Ref)
     return                     Modes
   --# global in Dict;
   is
   begin
      return RawDict.Get_Global_Variable_Mode
        (The_Global_Variable => Get_Subprogram_Parameter_Global_Variable
           (The_Subprogram           => The_Subprogram,
            Abstraction              => Abstraction,
            The_Subprogram_Parameter => The_Subprogram_Parameter));
   end Get_Subprogram_Parameter_Global_Mode;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Variable_Global_Mode
     (The_Task_Type : RawDict.Type_Info_Ref;
      Abstraction   : Abstractions;
      The_Variable  : RawDict.Variable_Info_Ref)
     return          Modes
   --# global in Dict;
   is
   begin
      return RawDict.Get_Global_Variable_Mode
        (The_Global_Variable => Get_Task_Type_Variable_Global_Variable
           (The_Task_Type => The_Task_Type,
            Abstraction   => Abstraction,
            The_Variable  => The_Variable));
   end Get_Task_Type_Variable_Global_Mode;

   --------------------------------------------------------------------------------

   function GetGlobalMode
     (Abstraction : Abstractions;
      Subprogram  : Symbol;
      Variable    : Symbol)
     return        Modes
   is
      Result : Modes;
   begin
      case RawDict.GetSymbolDiscriminant (Subprogram) is
         when Subprogram_Symbol =>
            case RawDict.GetSymbolDiscriminant (Variable) is
               when Variable_Symbol =>
                  Result := Get_Subprogram_Variable_Global_Mode
                    (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => Subprogram), -- GAA External
                     Abstraction    => Abstraction,
                     The_Variable   => RawDict.Get_Variable_Info_Ref (Item => Variable)); -- GAA External
               when Subprogram_Parameter_Symbol =>
                  Result := Get_Subprogram_Parameter_Global_Mode
                    (The_Subprogram           => RawDict.Get_Subprogram_Info_Ref (Item => Subprogram), -- GAA External
                     Abstraction              => Abstraction,
                     The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => Variable)); -- GAA External
               when others => -- non-exec code
                  Result := InvalidMode;
                  SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                            Msg     => "in Dictionary.GetGlobalMode");
            end case;
         when Type_Symbol =>
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => Subprogram)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.GetGlobalMode");
            Result := Get_Task_Type_Variable_Global_Mode
              (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => Subprogram), -- GAA External
               Abstraction   => Abstraction,
               The_Variable  => RawDict.Get_Variable_Info_Ref (Item => Variable)); -- GAA External
         when others => -- non-exec code
            Result := InvalidMode;
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.GetGlobalMode");
      end case;
      return Result;
   end GetGlobalMode;

   --------------------------------------------------------------------------------
   --  Get_Own_Variable_Mode
   --------------------------------------------------------------------------------

   function Get_Own_Variable_Mode (The_Variable : RawDict.Variable_Info_Ref) return Modes
   --# global in Dict;
   is
   begin
      return RawDict.Get_Own_Variable_Mode (The_Own_Variable => RawDict.Get_Variable_Own_Variable (The_Variable => The_Variable));
   end Get_Own_Variable_Mode;

   --------------------------------------------------------------------------------

   function GetOwnVariableMode (Variable : Symbol) return Modes is
   begin
      return Get_Own_Variable_Mode (The_Variable => RawDict.Get_Variable_Info_Ref (Variable)); -- GAA External
   end GetOwnVariableMode;

   --------------------------------------------------------------------------------
   --  Get_Own_Variable_Protected
   --------------------------------------------------------------------------------

   function Get_Own_Variable_Protected (The_Variable : RawDict.Variable_Info_Ref) return Boolean
   --# global in Dict;
   is
   begin
      SystemErrors.RT_Assert
        (C       => Is_Own_Variable (The_Variable => The_Variable),
         Sys_Err => SystemErrors.Precondition_Failure,
         Msg     => "In call to Get_Own_Variable_Protected");
      return RawDict.Get_Own_Variable_Protected
        (The_Own_Variable => RawDict.Get_Variable_Own_Variable (The_Variable => The_Variable));
   end Get_Own_Variable_Protected;

   --------------------------------------------------------------------------------

   function GetOwnVariableProtected (Variable : Symbol) return Boolean is
   begin
      return Get_Own_Variable_Protected (The_Variable => RawDict.Get_Variable_Info_Ref (Variable)); -- GAA External
   end GetOwnVariableProtected;

   --------------------------------------------------------------------------------
   --  Get_Own_Variable_Is_Interrupt_Stream
   --------------------------------------------------------------------------------

   function GetOwnVariableIsInterruptStream (Variable : Symbol) return Boolean is

      function Get_Own_Variable_Is_Interrupt_Stream (The_Variable : RawDict.Variable_Info_Ref) return Boolean
      --# global in Dict;
      is
      begin
         SystemErrors.RT_Assert
           (C       => Is_Own_Variable (The_Variable => The_Variable),
            Sys_Err => SystemErrors.Precondition_Failure,
            Msg     => "in call to Get_OwnVariable_Is_Interrupt_Stream");
         return RawDict.Get_Own_Variable_Is_Interrupt_Stream
           (The_Own_Variable => RawDict.Get_Variable_Own_Variable (The_Variable => The_Variable));
      end Get_Own_Variable_Is_Interrupt_Stream;

   begin -- GetOwnVariableIsInterruptStream
      return Get_Own_Variable_Is_Interrupt_Stream (The_Variable => RawDict.Get_Variable_Info_Ref (Variable)); -- GAA External
   end GetOwnVariableIsInterruptStream;

   ------------------------------------------------------------------------------
   --  Get_Own_Variable_Type_Here
   ------------------------------------------------------------------------------

   -- This function extends the Dictionary.Get_Type (which does not use the
   -- scope to determine the type of a variable) for own variables. An own
   -- variable has different type depending on viewpoint (scope), abstract vs
   -- concrete view. The right type is determined by
   --  1. Scope of viewpoint of the own variable (local, child package, or
   --     outside)
   --  2. Announcement of the own variable type
   --  3. Declaration scope of the corresponding concrete (Ada) variable

   function GetOwnVariableTypeHere (OwnVariable : Symbol;
                                    Scope       : Scopes) return Symbol is

      function Get_Own_Variable_Type_Here (Own_Variable : RawDict.Variable_Info_Ref;
                                           Scope        : Scopes) return RawDict.Type_Info_Ref
      --# global in Dict;
      is
         Decl_Scope       : Scopes;
         Result_Type_Mark : RawDict.Type_Info_Ref;
      begin
         Decl_Scope := Get_Variable_Scope (The_Variable => Own_Variable);
         if Variable_Is_Declared (The_Variable => Own_Variable)
           and then (Get_Visibility (Scope => Decl_Scope) = Visible
                       or else IsLocal (Scope, Decl_Scope)
                       or else (RawDict.GetSymbolDiscriminant (GetRegion (Decl_Scope)) = Package_Symbol
                                  and then Is_Private_Seeing_Descendent
                                  (Scope       => Scope,
                                   The_Package => RawDict.Get_Package_Info_Ref (GetRegion (Decl_Scope))))) then
            Result_Type_Mark := RawDict.Get_Variable_Type (The_Variable => Own_Variable);
         else
            Result_Type_Mark := RawDict.Get_Variable_Abstract_Type (The_Variable => Own_Variable);
         end if;
         return Result_Type_Mark;
      end Get_Own_Variable_Type_Here;

   begin -- GetOwnVariableTypeHere
      return RawDict.Get_Type_Symbol -- GAA External
        (Get_Own_Variable_Type_Here (Own_Variable => RawDict.Get_Variable_Info_Ref (OwnVariable), -- GAA External
                                     Scope        => Scope));
   end GetOwnVariableTypeHere;

   --------------------------------------------------------------------------------
   --  Get_Constituent_Mode
   --------------------------------------------------------------------------------

   function Get_Constituent_Mode (The_Variable : RawDict.Variable_Info_Ref) return Modes
   --# global in Dict;
   is
   begin
      return RawDict.Get_Constituent_Mode (The_Constituent => RawDict.Get_Variable_Constituent (The_Variable => The_Variable));
   end Get_Constituent_Mode;

   --------------------------------------------------------------------------------

   function GetConstituentMode (Variable : Symbol) return Modes is
   begin
      return Get_Constituent_Mode (The_Variable => RawDict.Get_Variable_Info_Ref (Variable)); -- GAA External
   end GetConstituentMode;

   --------------------------------------------------------------------------------
   --  Add_Export
   --------------------------------------------------------------------------------

   procedure Add_Subprogram_Export_Variable
     (The_Subprogram   : in RawDict.Subprogram_Info_Ref;
      Abstraction      : in Abstractions;
      The_Export       : in RawDict.Variable_Info_Ref;
      Export_Reference : in Location;
      Annotation       : in Location)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Abstraction,
   --#                                The_Export,
   --#                                The_Subprogram &
   --#         SPARK_IO.File_Sys from *,
   --#                                Abstraction,
   --#                                Annotation,
   --#                                Dict,
   --#                                Export_Reference,
   --#                                LexTokenManager.State,
   --#                                The_Export,
   --#                                The_Subprogram;
   is
   begin
      RawDict.Set_Global_Variable_Exported
        (The_Global_Variable => Get_Subprogram_Variable_Global_Variable
           (The_Subprogram => The_Subprogram,
            Abstraction    => Abstraction,
            The_Variable   => The_Export));

      AddOtherReference (RawDict.Get_Variable_Symbol (The_Export),
                         RawDict.Get_Subprogram_Symbol (The_Subprogram),
                         Export_Reference);

      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         if Abstraction = IsRefined then
            Write_String (Dict.TemporaryFile, "refined ");
         end if;
         Write_String (Dict.TemporaryFile, "export of ");
         Write_Name (File => Dict.TemporaryFile,
                     Item => RawDict.Get_Variable_Symbol (The_Export));
         Write_String (Dict.TemporaryFile, " from ");
         Write_Name (File => Dict.TemporaryFile,
                     Item => RawDict.Get_Subprogram_Symbol (The_Subprogram));
         Write_String (Dict.TemporaryFile, " is at ");
         Write_Location (File => Dict.TemporaryFile,
                         Loc  => Annotation);
         Write_Line (Dict.TemporaryFile, " ;");
      end if;
   end Add_Subprogram_Export_Variable;

   --------------------------------------------------------------------------------

   procedure Add_Subprogram_Export_Parameter
     (The_Subprogram   : in RawDict.Subprogram_Info_Ref;
      Abstraction      : in Abstractions;
      The_Export       : in RawDict.Subprogram_Parameter_Info_Ref;
      Export_Reference : in Location;
      Annotation       : in Location)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Abstraction,
   --#                                The_Export,
   --#                                The_Subprogram &
   --#         SPARK_IO.File_Sys from *,
   --#                                Abstraction,
   --#                                Annotation,
   --#                                Dict,
   --#                                Export_Reference,
   --#                                LexTokenManager.State,
   --#                                The_Export,
   --#                                The_Subprogram;
   is
   begin
      if RawDict.Get_Subprogram_Parameter_Subprogram (The_Subprogram_Parameter => The_Export) = The_Subprogram then
         RawDict.Set_Subprogram_Parameter_Exported
           (The_Subprogram_Parameter => The_Export,
            Abstraction              => Abstraction);
      else
         RawDict.Set_Global_Variable_Exported
           (The_Global_Variable => Get_Subprogram_Parameter_Global_Variable
              (The_Subprogram           => The_Subprogram,
               Abstraction              => Abstraction,
               The_Subprogram_Parameter => The_Export));
      end if;

      AddOtherReference (RawDict.Get_Subprogram_Parameter_Symbol (The_Export),
                         RawDict.Get_Subprogram_Symbol (The_Subprogram),
                         Export_Reference);

      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         if Abstraction = IsRefined then
            Write_String (Dict.TemporaryFile, "refined ");
         end if;
         Write_String (Dict.TemporaryFile, "export of ");
         Write_Name (File => Dict.TemporaryFile,
                     Item => RawDict.Get_Subprogram_Parameter_Symbol (The_Export));
         Write_String (Dict.TemporaryFile, " from ");
         Write_Name (File => Dict.TemporaryFile,
                     Item => RawDict.Get_Subprogram_Symbol (The_Subprogram));
         Write_String (Dict.TemporaryFile, " is at ");
         Write_Location (File => Dict.TemporaryFile,
                         Loc  => Annotation);
         Write_Line (Dict.TemporaryFile, " ;");
      end if;
   end Add_Subprogram_Export_Parameter;

   --------------------------------------------------------------------------------

   procedure AddExport
     (Abstraction     : in Abstractions;
      TheProcedure    : in Symbol;
      TheExport       : in Symbol;
      ExportReference : in Location;
      Annotation      : in Location) is

      procedure Add_Task_Type_Export
        (The_Task_Type    : in RawDict.Type_Info_Ref;
         Abstraction      : in Abstractions;
         The_Export       : in RawDict.Variable_Info_Ref;
         Export_Reference : in Location;
         Annotation       : in Location)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Abstraction,
      --#                                The_Export,
      --#                                The_Task_Type &
      --#         SPARK_IO.File_Sys from *,
      --#                                Abstraction,
      --#                                Annotation,
      --#                                Dict,
      --#                                Export_Reference,
      --#                                LexTokenManager.State,
      --#                                The_Export,
      --#                                The_Task_Type;
      is
      begin
         RawDict.Set_Global_Variable_Exported
           (The_Global_Variable => Get_Task_Type_Variable_Global_Variable
              (The_Task_Type => The_Task_Type,
               Abstraction   => Abstraction,
               The_Variable  => The_Export));

         AddOtherReference (RawDict.Get_Variable_Symbol (The_Export), RawDict.Get_Type_Symbol (The_Task_Type), Export_Reference);

         if SPARK_IO.Is_Open (Dict.TemporaryFile) then
            if Abstraction = IsRefined then
               Write_String (Dict.TemporaryFile, "refined ");
            end if;
            Write_String (Dict.TemporaryFile, "export of ");
            Write_Name (File => Dict.TemporaryFile,
                        Item => RawDict.Get_Variable_Symbol (The_Export));
            Write_String (Dict.TemporaryFile, " from ");
            Write_Name (File => Dict.TemporaryFile,
                        Item => RawDict.Get_Type_Symbol (The_Task_Type));
            Write_String (Dict.TemporaryFile, " is at ");
            Write_Location (File => Dict.TemporaryFile,
                            Loc  => Annotation);
            Write_Line (Dict.TemporaryFile, " ;");
         end if;
      end Add_Task_Type_Export;

   begin -- AddExport
      case RawDict.GetSymbolDiscriminant (TheProcedure) is
         when Subprogram_Symbol =>
            case RawDict.GetSymbolDiscriminant (TheExport) is
               when Variable_Symbol =>
                  Add_Subprogram_Export_Variable
                    (The_Subprogram   => RawDict.Get_Subprogram_Info_Ref (Item => TheProcedure), -- GAA External
                     Abstraction      => Abstraction,
                     The_Export       => RawDict.Get_Variable_Info_Ref (Item => TheExport), -- GAA External
                     Export_Reference => ExportReference,
                     Annotation       => Annotation);
               when Subprogram_Parameter_Symbol =>
                  Add_Subprogram_Export_Parameter
                    (The_Subprogram   => RawDict.Get_Subprogram_Info_Ref (Item => TheProcedure), -- GAA External
                     Abstraction      => Abstraction,
                     The_Export       => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => TheExport), -- GAA External
                     Export_Reference => ExportReference,
                     Annotation       => Annotation);
               when others => -- non-exec code
                  SystemErrors.Fatal_Error
                    (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                     Msg     => "in Dictionary.AddExport");
            end case;
         when Type_Symbol =>
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheProcedure)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.AddExport");
            Add_Task_Type_Export
              (The_Task_Type    => RawDict.Get_Type_Info_Ref (Item => TheProcedure), -- GAA External
               Abstraction      => Abstraction,
               The_Export       => RawDict.Get_Variable_Info_Ref (TheExport), -- GAA External
               Export_Reference => ExportReference,
               Annotation       => Annotation);
         when others => -- non-exec code
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.AddExport");
      end case;
   end AddExport;

   --------------------------------------------------------------------------------
   --  Force_Import
   --------------------------------------------------------------------------------

   procedure Force_Subprogram_Import
     (The_Subprogram   : in RawDict.Subprogram_Info_Ref;
      Abstraction      : in Abstractions;
      The_Import       : in Symbol;
      Import_Reference : in Location)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Abstraction,
   --#                                The_Import,
   --#                                The_Subprogram &
   --#         SPARK_IO.File_Sys from *,
   --#                                Abstraction,
   --#                                Dict,
   --#                                Import_Reference,
   --#                                LexTokenManager.State,
   --#                                The_Import,
   --#                                The_Subprogram;
   is
   begin
      if RawDict.GetSymbolDiscriminant (The_Import) = Subprogram_Parameter_Symbol
        and then RawDict.Get_Subprogram_Parameter_Subprogram
        (The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => The_Import)) =
        The_Subprogram then
         RawDict.Set_Subprogram_Parameter_Imported
           (The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => The_Import),
            Abstraction              => Abstraction);
      else
         case RawDict.GetSymbolDiscriminant (The_Import) is
            when Variable_Symbol =>
               RawDict.Set_Global_Variable_Imported
                 (The_Global_Variable => Get_Subprogram_Variable_Global_Variable
                    (The_Subprogram => The_Subprogram,
                     Abstraction    => Abstraction,
                     The_Variable   => RawDict.Get_Variable_Info_Ref (Item => The_Import)));
            when Subprogram_Parameter_Symbol =>
               RawDict.Set_Global_Variable_Imported
                 (The_Global_Variable => Get_Subprogram_Parameter_Global_Variable
                    (The_Subprogram           => The_Subprogram,
                     Abstraction              => Abstraction,
                     The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => The_Import)));
            when others => -- non-exec code
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                  Msg     => "in Dictionary.Force_Subprogram_Import");
         end case;
      end if;
      AddOtherReference (The_Import, RawDict.Get_Subprogram_Symbol (The_Subprogram), Import_Reference);
   end Force_Subprogram_Import;

   --------------------------------------------------------------------------------

   procedure Force_Task_Type_Import
     (The_Task_Type    : in RawDict.Type_Info_Ref;
      Abstraction      : in Abstractions;
      The_Import       : in RawDict.Variable_Info_Ref;
      Import_Reference : in Location)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Abstraction,
   --#                                The_Import,
   --#                                The_Task_Type &
   --#         SPARK_IO.File_Sys from *,
   --#                                Abstraction,
   --#                                Dict,
   --#                                Import_Reference,
   --#                                LexTokenManager.State,
   --#                                The_Import,
   --#                                The_Task_Type;
   is
   begin
      RawDict.Set_Global_Variable_Imported
        (The_Global_Variable => Get_Task_Type_Variable_Global_Variable
           (The_Task_Type => The_Task_Type,
            Abstraction   => Abstraction,
            The_Variable  => The_Import));
      AddOtherReference (RawDict.Get_Variable_Symbol (The_Import), RawDict.Get_Type_Symbol (The_Task_Type), Import_Reference);
   end Force_Task_Type_Import;

   --------------------------------------------------------------------------------

   procedure ForceImport
     (Abstraction     : in Abstractions;
      TheProcedure    : in Symbol;
      TheImport       : in Symbol;
      ImportReference : in Location) is
   begin
      case RawDict.GetSymbolDiscriminant (TheProcedure) is
         when Subprogram_Symbol =>
            Force_Subprogram_Import
              (The_Subprogram   => RawDict.Get_Subprogram_Info_Ref (Item => TheProcedure), -- GAA External
               Abstraction      => Abstraction,
               The_Import       => TheImport,
               Import_Reference => ImportReference);
         when Type_Symbol =>
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheProcedure)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.ForceImport");
            Force_Task_Type_Import
              (The_Task_Type    => RawDict.Get_Type_Info_Ref (Item => TheProcedure), -- GAA External
               Abstraction      => Abstraction,
               The_Import       => RawDict.Get_Variable_Info_Ref (TheImport),
               Import_Reference => ImportReference);
         when others => -- non-exec code
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.ForceImport");
      end case;
   end ForceImport;

   --------------------------------------------------------------------------------
   --  Add_Dependency
   --------------------------------------------------------------------------------

   procedure Add_Subprogram_Dependency
     (Abstraction      : in Abstractions;
      Comp_Unit        : in ContextManager.UnitDescriptors;
      The_Subprogram   : in RawDict.Subprogram_Info_Ref;
      The_Export       : in Symbol;
      The_Import       : in Symbol;
      Import_Reference : in Location)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Abstraction,
   --#                                Comp_Unit,
   --#                                Import_Reference,
   --#                                The_Export,
   --#                                The_Import,
   --#                                The_Subprogram &
   --#         SPARK_IO.File_Sys from *,
   --#                                Abstraction,
   --#                                Comp_Unit,
   --#                                Dict,
   --#                                Import_Reference,
   --#                                LexTokenManager.State,
   --#                                The_Export,
   --#                                The_Import,
   --#                                The_Subprogram;
   is
      The_Dependency      : RawDict.Dependency_Info_Ref;
      The_Global_Variable : RawDict.Global_Variable_Info_Ref;
   begin
      case RawDict.GetSymbolDiscriminant (The_Import) is
         when Variable_Symbol =>
            RawDict.Create_Dependency
              (The_Import_Parameter => RawDict.Null_Subprogram_Parameter_Info_Ref,
               The_Import_Variable  => RawDict.Get_Variable_Info_Ref (Item => The_Import),
               Comp_Unit            => Comp_Unit,
               Loc                  => Import_Reference.Start_Position,
               The_Dependency       => The_Dependency);
         when Subprogram_Parameter_Symbol =>
            RawDict.Create_Dependency
              (The_Import_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => The_Import),
               The_Import_Variable  => RawDict.Null_Variable_Info_Ref,
               Comp_Unit            => Comp_Unit,
               Loc                  => Import_Reference.Start_Position,
               The_Dependency       => The_Dependency);
         when others =>
            The_Dependency := RawDict.Null_Dependency_Info_Ref;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.Add_Subprogram_Dependency");
      end case;
      if RawDict.GetSymbolDiscriminant (The_Export) = Subprogram_Parameter_Symbol
        and then RawDict.Get_Subprogram_Parameter_Subprogram
        (The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => The_Export)) =
        The_Subprogram then
         RawDict.Set_Next_Dependency
           (The_Dependency => The_Dependency,
            Next           => RawDict.Get_Subprogram_Parameter_Dependencies
              (The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => The_Export),
               Abstraction              => Abstraction));
         RawDict.Set_Subprogram_Parameter_Dependencies
           (The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => The_Export),
            Abstraction              => Abstraction,
            Dependency               => The_Dependency);
         Force_Subprogram_Import
           (The_Subprogram   => The_Subprogram,
            Abstraction      => Abstraction,
            The_Import       => The_Import,
            Import_Reference => Import_Reference);
      else
         case RawDict.GetSymbolDiscriminant (The_Export) is
            when Variable_Symbol =>
               The_Global_Variable :=
                 Get_Subprogram_Variable_Global_Variable
                 (The_Subprogram => The_Subprogram,
                  Abstraction    => Abstraction,
                  The_Variable   => RawDict.Get_Variable_Info_Ref (Item => The_Export));
            when Subprogram_Parameter_Symbol =>
               The_Global_Variable :=
                 Get_Subprogram_Parameter_Global_Variable
                 (The_Subprogram           => The_Subprogram,
                  Abstraction              => Abstraction,
                  The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => The_Export));
            when others => -- non-exec code
               The_Global_Variable := RawDict.Null_Global_Variable_Info_Ref;
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                  Msg     => "in Dictionary.Add_Subprogram_Dependency");
         end case;
         RawDict.Set_Next_Dependency
           (The_Dependency => The_Dependency,
            Next           => RawDict.Get_Global_Variable_Dependencies
              (The_Global_Variable => The_Global_Variable,
               Abstraction         => Abstraction));
         RawDict.Set_Global_Variable_Dependencies
           (The_Global_Variable => The_Global_Variable,
            Abstraction         => Abstraction,
            Dependency          => The_Dependency);
         Force_Subprogram_Import
           (The_Subprogram   => The_Subprogram,
            Abstraction      => Abstraction,
            The_Import       => The_Import,
            Import_Reference => Import_Reference);
      end if;
   end Add_Subprogram_Dependency;

   --------------------------------------------------------------------------------

   procedure AddDependency
     (Abstraction     : in Abstractions;
      Comp_Unit       : in ContextManager.UnitDescriptors;
      TheProcedure    : in Symbol;
      TheExport       : in Symbol;
      TheImport       : in Symbol;
      ImportReference : in Location) is

      procedure Add_Task_Type_Dependency
        (Abstraction      : in Abstractions;
         Comp_Unit        : in ContextManager.UnitDescriptors;
         The_Task_Type    : in RawDict.Type_Info_Ref;
         The_Export       : in RawDict.Variable_Info_Ref;
         The_Import       : in RawDict.Variable_Info_Ref;
         Import_Reference : in Location)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Abstraction,
      --#                                Comp_Unit,
      --#                                Import_Reference,
      --#                                The_Export,
      --#                                The_Import,
      --#                                The_Task_Type &
      --#         SPARK_IO.File_Sys from *,
      --#                                Abstraction,
      --#                                Comp_Unit,
      --#                                Dict,
      --#                                Import_Reference,
      --#                                LexTokenManager.State,
      --#                                The_Export,
      --#                                The_Import,
      --#                                The_Task_Type;
      is
         The_Dependency      : RawDict.Dependency_Info_Ref;
         The_Global_Variable : RawDict.Global_Variable_Info_Ref;
      begin
         RawDict.Create_Dependency
           (The_Import_Parameter => RawDict.Null_Subprogram_Parameter_Info_Ref,
            The_Import_Variable  => The_Import,
            Comp_Unit            => Comp_Unit,
            Loc                  => Import_Reference.Start_Position,
            The_Dependency       => The_Dependency);
         The_Global_Variable :=
           Get_Task_Type_Variable_Global_Variable
           (The_Task_Type => The_Task_Type,
            Abstraction   => Abstraction,
            The_Variable  => The_Export);
         RawDict.Set_Next_Dependency
           (The_Dependency => The_Dependency,
            Next           => RawDict.Get_Global_Variable_Dependencies
              (The_Global_Variable => The_Global_Variable,
               Abstraction         => Abstraction));
         RawDict.Set_Global_Variable_Dependencies
           (The_Global_Variable => The_Global_Variable,
            Abstraction         => Abstraction,
            Dependency          => The_Dependency);
         Force_Task_Type_Import
           (The_Task_Type    => The_Task_Type,
            Abstraction      => Abstraction,
            The_Import       => The_Import,
            Import_Reference => Import_Reference);
      end Add_Task_Type_Dependency;

   begin -- AddDependency
      case RawDict.GetSymbolDiscriminant (TheProcedure) is
         when Subprogram_Symbol =>
            Add_Subprogram_Dependency
              (Abstraction      => Abstraction,
               Comp_Unit        => Comp_Unit,
               The_Subprogram   => RawDict.Get_Subprogram_Info_Ref (Item => TheProcedure), -- GAA External
               The_Export       => TheExport,
               The_Import       => TheImport,
               Import_Reference => ImportReference);
         when Type_Symbol =>
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheProcedure)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.AddDependency");
            Add_Task_Type_Dependency
              (Abstraction      => Abstraction,
               Comp_Unit        => Comp_Unit,
               The_Task_Type    => RawDict.Get_Type_Info_Ref (Item => TheProcedure), -- GAA External
               The_Export       => RawDict.Get_Variable_Info_Ref (TheExport),
               The_Import       => RawDict.Get_Variable_Info_Ref (TheImport),
               Import_Reference => ImportReference);
         when others => -- non-exec code
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.AddDependency");
      end case;
   end AddDependency;

   --------------------------------------------------------------------------------
   --  Add_Virtual_Element
   --------------------------------------------------------------------------------

   procedure AddVirtualElement
     (ToProtectedType   : in Symbol;
      Comp_Unit         : in ContextManager.UnitDescriptors;
      Declaration       : in Location;
      TheVirtualElement : in Symbol;
      TheOwner          : in Symbol) is

      procedure Add_Virtual_Element
        (The_Protected_Type  : in RawDict.Type_Info_Ref;
         Comp_Unit           : in ContextManager.UnitDescriptors;
         Declaration         : in Location;
         The_Virtual_Element : in RawDict.Variable_Info_Ref;
         TheOwner            : in Symbol)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   Comp_Unit,
      --#                   Declaration,
      --#                   TheOwner,
      --#                   The_Protected_Type,
      --#                   The_Virtual_Element;
      is
         NewVirtualElement  : Symbol;
      begin
         RawDict.CreateVirtualElement
           (The_Variable      => The_Virtual_Element,
            TheOwner          => TheOwner,
            Comp_Unit         => Comp_Unit,
            Loc               => Declaration.Start_Position,
            TheVirtualElement => NewVirtualElement);
         RawDict.SetNextVirtualElement (NewVirtualElement, RawDict.Get_Type_Virtual_Element_List (Type_Mark => The_Protected_Type));
         RawDict.Set_Type_Virtual_Element_List (Type_Mark => The_Protected_Type,
                                                The_List  => NewVirtualElement);
         RawDict.Set_Variable_Virtual_Element
           (The_Variable    => The_Virtual_Element,
            Virtual_Element => NewVirtualElement);
      end Add_Virtual_Element;

   begin -- AddVirtualElement
      Add_Virtual_Element
        (The_Protected_Type  => RawDict.Get_Type_Info_Ref (ToProtectedType), -- GAA External
         Comp_Unit           => Comp_Unit,
         Declaration         => Declaration,
         The_Virtual_Element => RawDict.Get_Variable_Info_Ref (TheVirtualElement), -- GAA External
         TheOwner            => TheOwner);
   end AddVirtualElement;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   procedure AddPOorSOToSuspendsList
     (TheTaskOrProc : in Symbol;
      Comp_Unit     : in ContextManager.UnitDescriptors;
      Declaration   : in Location;
      ThePOorSO     : in Symbol) is
      The_Suspends_List : Symbol;
      The_Subprogram    : RawDict.Subprogram_Info_Ref;
      The_Task_Type     : RawDict.Type_Info_Ref;
   begin
      RawDict.CreateSuspendsListItem
        (ThePOorSO        => ThePOorSO,
         Comp_Unit        => Comp_Unit,
         Loc              => Declaration.Start_Position,
         SuspendsListItem => The_Suspends_List);
      case RawDict.GetSymbolDiscriminant (TheTaskOrProc) is
         when Subprogram_Symbol =>
            The_Subprogram := RawDict.Get_Subprogram_Info_Ref (Item => TheTaskOrProc); -- GAA External
            RawDict.SetNextSuspendsListItem
              (The_Suspends_List,
               RawDict.Get_Subprogram_Suspends_List (The_Subprogram => The_Subprogram));
            RawDict.Set_Subprogram_Suspends_List (The_Subprogram    => The_Subprogram,
                                                  The_Suspends_List => The_Suspends_List);
         when Type_Symbol =>
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheTaskOrProc)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.AddPOorSOToSuspendsList");
            The_Task_Type := RawDict.Get_Type_Info_Ref (Item => TheTaskOrProc); -- GAA External
            RawDict.SetNextSuspendsListItem
              (The_Suspends_List,
               RawDict.Get_Task_Type_Suspends_List (The_Task_Type => The_Task_Type));
            RawDict.Set_Task_Type_Suspends_List (The_Task_Type     => The_Task_Type,
                                                 The_Suspends_List => The_Suspends_List);
         when others => -- non-exec code
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.AddPOorSOToSuspendsList");
      end case;
   end AddPOorSOToSuspendsList;

   --------------------------------------------------------------------------------
   --  Add_Interrupt_Stream_Mapping
   --------------------------------------------------------------------------------

   procedure AddInterruptStreamMapping
     (Subject             : in Symbol;
      Comp_Unit           : in ContextManager.UnitDescriptors;
      Declaration         : in Location;
      TheInterruptHandler : in LexTokenManager.Lex_String;
      TheInterruptStream  : in LexTokenManager.Lex_String) is

      procedure Add_Interrupt_Stream_Mapping
        (The_Variable          : in RawDict.Variable_Info_Ref;
         Comp_Unit             : in ContextManager.UnitDescriptors;
         Declaration           : in Location;
         The_Interrupt_Handler : in LexTokenManager.Lex_String;
         The_Interrupt_Stream  : in LexTokenManager.Lex_String)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   Comp_Unit,
      --#                   Declaration,
      --#                   The_Interrupt_Handler,
      --#                   The_Interrupt_Stream,
      --#                   The_Variable;
      is
         NewSym           : Symbol;
         The_Own_Variable : RawDict.Own_Variable_Info_Ref;
      begin
         The_Own_Variable := RawDict.Get_Variable_Own_Variable (The_Variable => The_Variable);
         RawDict.CreateInterruptStreamMapping
           (TheHandler         => The_Interrupt_Handler,
            TheInterruptStream => The_Interrupt_Stream,
            Comp_Unit          => Comp_Unit,
            Loc                => Declaration.Start_Position,
            TheMapping         => NewSym);

         RawDict.SetNextInterruptStreamMapping
           (Current => NewSym,
            Next    => RawDict.Get_Own_Variable_Interrupt_Stream_Mappings (The_Own_Variable => The_Own_Variable));

         RawDict.Set_Own_Variable_Interrupt_Stream_Mappings
           (The_Own_Variable              => The_Own_Variable,
            The_Interrupt_Stream_Mappings => NewSym);
      end Add_Interrupt_Stream_Mapping;

   begin -- AddInterruptStreamMapping
      Add_Interrupt_Stream_Mapping
        (The_Variable          => RawDict.Get_Variable_Info_Ref (Subject), -- GAA External
         Comp_Unit             => Comp_Unit,
         Declaration           => Declaration,
         The_Interrupt_Handler => TheInterruptHandler,
         The_Interrupt_Stream  => TheInterruptStream);
   end AddInterruptStreamMapping;

   --------------------------------------------------------------------------------
   --  Copy_Dependency_List
   --------------------------------------------------------------------------------

   procedure CopyDependencyList
     (Abstraction  : in Abstractions;
      TheProcedure : in Symbol;
      FromExport   : in Symbol;
      ToExport     : in Symbol) is

      procedure Copy_Subprogram_Dependency_List
        (The_Subprogram : in RawDict.Subprogram_Info_Ref;
         Abstraction    : in Abstractions;
         FromExport     : in Symbol;
         ToExport       : in Symbol)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   Abstraction,
      --#                   FromExport,
      --#                   The_Subprogram,
      --#                   ToExport;
      is
         The_Dependency : RawDict.Dependency_Info_Ref;
      begin
         if RawDict.GetSymbolDiscriminant (FromExport) = Subprogram_Parameter_Symbol
           and then RawDict.Get_Subprogram_Parameter_Subprogram
           (The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => FromExport)) =
           The_Subprogram then
            The_Dependency :=
              RawDict.Get_Subprogram_Parameter_Dependencies
              (The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => FromExport),
               Abstraction              => Abstraction);
         else
            case RawDict.GetSymbolDiscriminant (FromExport) is
               when Variable_Symbol =>
                  The_Dependency :=
                    RawDict.Get_Global_Variable_Dependencies
                    (The_Global_Variable => Get_Subprogram_Variable_Global_Variable
                       (The_Subprogram => The_Subprogram,
                        Abstraction    => Abstraction,
                        The_Variable   => RawDict.Get_Variable_Info_Ref (Item => FromExport)),
                     Abstraction         => Abstraction);
               when Subprogram_Parameter_Symbol =>
                  The_Dependency :=
                    RawDict.Get_Global_Variable_Dependencies
                    (The_Global_Variable => Get_Subprogram_Parameter_Global_Variable
                       (The_Subprogram           => The_Subprogram,
                        Abstraction              => Abstraction,
                        The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => FromExport)),
                     Abstraction         => Abstraction);
               when others => -- non-exec code
                  The_Dependency := RawDict.Null_Dependency_Info_Ref;
                  SystemErrors.Fatal_Error
                    (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                     Msg     => "in Dictionary.Copy_Subprogram_Dependency_List");
            end case;
         end if;

         if RawDict.GetSymbolDiscriminant (ToExport) = Subprogram_Parameter_Symbol
           and then RawDict.Get_Subprogram_Parameter_Subprogram
           (The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => ToExport)) =
           The_Subprogram then
            RawDict.Set_Subprogram_Parameter_Dependencies
              (The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => ToExport),
               Abstraction              => Abstraction,
               Dependency               => The_Dependency);
         else
            case RawDict.GetSymbolDiscriminant (ToExport) is
               when Variable_Symbol =>
                  RawDict.Set_Global_Variable_Dependencies
                    (The_Global_Variable => Get_Subprogram_Variable_Global_Variable
                       (The_Subprogram => The_Subprogram,
                        Abstraction    => Abstraction,
                        The_Variable   => RawDict.Get_Variable_Info_Ref (Item => ToExport)),
                     Abstraction         => Abstraction,
                     Dependency          => The_Dependency);
               when Subprogram_Parameter_Symbol =>
                  RawDict.Set_Global_Variable_Dependencies
                    (The_Global_Variable => Get_Subprogram_Parameter_Global_Variable
                       (The_Subprogram           => The_Subprogram,
                        Abstraction              => Abstraction,
                        The_Subprogram_Parameter => RawDict.Get_Subprogram_Parameter_Info_Ref (Item => ToExport)),
                     Abstraction         => Abstraction,
                     Dependency          => The_Dependency);
               when others => -- non-exec code
                  SystemErrors.Fatal_Error
                    (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                     Msg     => "in Dictionary.Copy_Subprogram_Dependency_List");
            end case;
         end if;
      end Copy_Subprogram_Dependency_List;

      --------------------------------------------------------------------------------

      procedure Copy_Task_Type_Dependency_List
        (The_Task_Type : in RawDict.Type_Info_Ref;
         Abstraction   : in Abstractions;
         From_Export   : in RawDict.Variable_Info_Ref;
         To_Export     : in RawDict.Variable_Info_Ref)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   Abstraction,
      --#                   From_Export,
      --#                   The_Task_Type,
      --#                   To_Export;
      is
      begin
         RawDict.Set_Global_Variable_Dependencies
           (The_Global_Variable => Get_Task_Type_Variable_Global_Variable
              (The_Task_Type => The_Task_Type,
               Abstraction   => Abstraction,
               The_Variable  => To_Export),
            Abstraction         => Abstraction,
            Dependency          => RawDict.Get_Global_Variable_Dependencies
              (The_Global_Variable => Get_Task_Type_Variable_Global_Variable
                 (The_Task_Type => The_Task_Type,
                  Abstraction   => Abstraction,
                  The_Variable  => From_Export),
               Abstraction         => Abstraction));
      end Copy_Task_Type_Dependency_List;

   begin -- CopyDependencyList
      case RawDict.GetSymbolDiscriminant (TheProcedure) is
         when Subprogram_Symbol =>
            Copy_Subprogram_Dependency_List
              (The_Subprogram => RawDict.Get_Subprogram_Info_Ref (Item => TheProcedure), -- GAA External
               Abstraction    => Abstraction,
               FromExport     => FromExport,
               ToExport       => ToExport);
         when Type_Symbol =>
            SystemErrors.RT_Assert
              (C       => Is_Task_Type (Type_Mark => RawDict.Get_Type_Info_Ref (Item => TheProcedure)), -- GAA External
               Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "in Dictionary.CopyDependencyList");
            Copy_Task_Type_Dependency_List
              (The_Task_Type => RawDict.Get_Type_Info_Ref (Item => TheProcedure), -- GAA External
               Abstraction   => Abstraction,
               From_Export   => RawDict.Get_Variable_Info_Ref (FromExport), -- GAA External
               To_Export     => RawDict.Get_Variable_Info_Ref (ToExport)); -- GAA External
         when others => -- non-exec code
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "in Dictionary.CopyDependencyList");
      end case;
   end CopyDependencyList;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   procedure Add_New_Package
     (Name        : in     LexTokenManager.Lex_String;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      Scope       : in     Scopes;
      The_Parent  : in     RawDict.Package_Info_Ref;
      Child_Sort  : in     PackageSort;
      The_Package :    out RawDict.Package_Info_Ref)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys from *,
   --#                                Child_Sort,
   --#                                Comp_Unit,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Scope,
   --#                                The_Parent &
   --#         The_Package       from Comp_Unit,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Scope,
   --#                                The_Parent;
   --# post The_Package /= RawDict.Null_Package_Info_Ref;
   is
      The_Specification : RawDict.Declaration_Info_Ref;

      procedure Make_Child (The_Parent, The_Package : in RawDict.Package_Info_Ref;
                            Child_Sort              : in PackageSort)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   Child_Sort,
      --#                   The_Package,
      --#                   The_Parent;
      is
      begin
         RawDict.Set_Package_Parent (The_Package => The_Package,
                                     The_Parent  => The_Parent);
         case Child_Sort is
            when Public =>
               if RawDict.Get_Package_First_Public_Child (The_Package => The_Parent) = RawDict.Null_Package_Info_Ref then
                  -- Adding first public child
                  RawDict.Set_Package_First_Public_Child (The_Package => The_Parent,
                                                          The_Child   => The_Package);
                  RawDict.Set_Package_Last_Public_Child (The_Package => The_Parent,
                                                         The_Child   => The_Package);
               else
                  -- Adding a sibling to an existing public child
                  RawDict.Set_Package_Next_Sibling
                    (The_Package => RawDict.Get_Package_Last_Public_Child (The_Package => The_Parent),
                     The_Sibling => The_Package);
                  RawDict.Set_Package_Last_Public_Child (The_Package => The_Parent,
                                                         The_Child   => The_Package);
               end if;
            when PrivateChild =>
               if RawDict.Get_Package_First_Private_Child (The_Package => The_Parent) = RawDict.Null_Package_Info_Ref then
                  -- Adding first private child
                  RawDict.Set_Package_First_Private_Child (The_Package => The_Parent,
                                                           The_Child   => The_Package);
                  RawDict.Set_Package_Last_Private_Child (The_Package => The_Parent,
                                                          The_Child   => The_Package);
               else
                  -- Adding a sibling to an existing private child
                  RawDict.Set_Package_Next_Sibling
                    (The_Package => RawDict.Get_Package_Last_Private_Child (The_Package => The_Parent),
                     The_Sibling => The_Package);
                  RawDict.Set_Package_Last_Private_Child (The_Package => The_Parent,
                                                          The_Child   => The_Package);
               end if;
         end case;
      end Make_Child;

      --------------------------------------------------------------------------------

      procedure Write_Package_Declaration (The_Package : in RawDict.Package_Info_Ref;
                                           Declaration : in Location)
      --# global in     Dict;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Declaration,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                The_Package;
      is
      begin
         if SPARK_IO.Is_Open (Dict.TemporaryFile) then
            Write_String (Dict.TemporaryFile, "package specification of ");
            Write_Name (File => Dict.TemporaryFile,
                        Item => RawDict.Get_Package_Symbol (The_Package));
            Write_String (Dict.TemporaryFile, " is at ");
            Write_Location (File => Dict.TemporaryFile,
                            Loc  => Declaration);
            Write_Line (Dict.TemporaryFile, " ;");
         end if;
      end Write_Package_Declaration;

   begin -- Add_New_Package
      if The_Parent = RawDict.Null_Package_Info_Ref and then Get_Visibility (Scope => Scope) = Local then
         The_Package := RawDict.Get_Package_Info_Ref (LookupImmediateScope (Name, Scope, ProofContext));
      else
         The_Package := RawDict.Null_Package_Info_Ref;
      end if;

      if The_Package = RawDict.Null_Package_Info_Ref then
         if The_Parent = RawDict.Null_Package_Info_Ref then
            Add_Declaration
              (Comp_Unit       => Comp_Unit,
               Loc             => Declaration,
               Scope           => Scope,
               Context         => ProgramContext,
               The_Declaration => The_Specification);
         else
            RawDict.Create_Declaration
              (Context         => ProgramContext,
               Scope           => Scope,
               Comp_Unit       => Comp_Unit,
               Loc             => Declaration.Start_Position,
               The_Declaration => The_Specification);
         end if;
         RawDict.Create_Package
           (Name            => Name,
            The_Declaration => The_Specification,
            Comp_Unit       => Comp_Unit,
            Loc             => Declaration.Start_Position,
            The_Package     => The_Package);
      else
         RawDict.Set_Declaration_Context
           (The_Declaration => RawDict.Get_Package_Specification (The_Package => The_Package),
            Context         => ProgramContext);
         if RawDict.Get_Symbol_Compilation_Unit (Item => RawDict.Get_Package_Symbol (The_Package)) = Comp_Unit then
            --  There was a usage of the package before its actual
            --  declaration -> update the declaration position. The
            --  declaration must be in the same compilation unit as
            --  the usage (in the package specification) to prevent to
            --  get a cross-reference from the package specification
            --  to the package body.
            RawDict.Set_Symbol_Location (Item     => RawDict.Get_Package_Symbol (The_Package),
                                         Location => Declaration.Start_Position);
         end if;
      end if;

      if The_Parent /= RawDict.Null_Package_Info_Ref then
         Make_Child (The_Parent  => The_Parent,
                     The_Package => The_Package,
                     Child_Sort  => Child_Sort);
      end if;

      if Child_Sort = PrivateChild then
         RawDict.Set_Package_Is_Private (The_Package => The_Package);
      end if;
      Write_Package_Declaration (The_Package => The_Package,
                                 Declaration => Declaration);
   end Add_New_Package;

   --------------------------------------------------------------------------------

   procedure Add_Package
     (Name          : in     LexTokenManager.Lex_String;
      Comp_Unit     : in     ContextManager.UnitDescriptors;
      Specification : in     Location;
      Scope         : in     Scopes;
      ThePackage    :    out Symbol) is
      The_Package : RawDict.Package_Info_Ref;
   begin
      Add_New_Package
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Specification,
         Scope       => Scope,
         The_Parent  => RawDict.Null_Package_Info_Ref,
         Child_Sort  => Public,
         The_Package => The_Package);
      ThePackage := RawDict.Get_Package_Symbol (The_Package); -- GAA External
   end Add_Package;

   --------------------------------------------------------------------------------

   procedure AddChildPackage
     (TheParent     : in     Symbol;
      Sort          : in     PackageSort;
      Name          : in     LexTokenManager.Lex_String;
      Comp_Unit     : in     ContextManager.UnitDescriptors;
      Specification : in     Location;
      Scope         : in     Scopes;
      ThePackage    :    out Symbol) is
      The_Package : RawDict.Package_Info_Ref;
   begin
      Add_New_Package
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Specification,
         Scope       => Scope,
         The_Parent  => RawDict.Get_Package_Info_Ref (TheParent), -- GAA External
         Child_Sort  => Sort,
         The_Package => The_Package);
      ThePackage := RawDict.Get_Package_Symbol (The_Package); -- GAA External
   end AddChildPackage;

   --------------------------------------------------------------------------------

   procedure AddPrivatePackage
     (Name          : in     LexTokenManager.Lex_String;
      Comp_Unit     : in     ContextManager.UnitDescriptors;
      Specification : in     Location;
      Scope         : in     Scopes;
      ThePackage    :    out Symbol) is
      The_Package : RawDict.Package_Info_Ref;
   begin
      Add_New_Package
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Specification,
         Scope       => Scope,
         The_Parent  => RawDict.Null_Package_Info_Ref,
         Child_Sort  => PrivateChild,
         The_Package => The_Package);
      ThePackage := RawDict.Get_Package_Symbol (The_Package); -- GAA External
   end AddPrivatePackage;

   --------------------------------------------------------------------------------

   procedure SetPackageElaborateBodyFound (ThePackage : in Symbol) is
   begin
      RawDict.Set_Package_Elaborate_Body_Found (The_Package => RawDict.Get_Package_Info_Ref (ThePackage)); -- GAA External
   end SetPackageElaborateBodyFound;

   --------------------------------------------------------------------------------

   procedure SetPackageAsExtendingAnother (ThePackage          : in Symbol;
                                           ThePackageItExtends : in Symbol) is
   begin
      RawDict.Set_Package_Extends (The_Package => RawDict.Get_Package_Info_Ref (ThePackage), -- GAA External
                                   The_Extends => RawDict.Get_Package_Info_Ref (ThePackageItExtends)); -- GAA External
   end SetPackageAsExtendingAnother;

   --------------------------------------------------------------------------------

   procedure AddPrivatePart (ThePackage  : in Symbol;
                             PrivatePart : in Location;
                             Hidden      : in Boolean) is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         if Hidden then
            Write_String (Dict.TemporaryFile, "hidden ");
         end if;
         Write_String (Dict.TemporaryFile, "private part of ");
         Write_Name (File => Dict.TemporaryFile,
                     Item => ThePackage);
         Write_String (Dict.TemporaryFile, " is at ");
         Write_Location (File => Dict.TemporaryFile,
                         Loc  => PrivatePart);
         Write_Line (Dict.TemporaryFile, " ;");
      end if;
   end AddPrivatePart;

   --------------------------------------------------------------------------------

   procedure AddOwnAnnotation (ThePackage : in Symbol;
                               Annotation : in Location) is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         Write_String (Dict.TemporaryFile, "own annotation of ");
         Write_Name (File => Dict.TemporaryFile,
                     Item => ThePackage);
         Write_String (Dict.TemporaryFile, " is at ");
         Write_Location (File => Dict.TemporaryFile,
                         Loc  => Annotation);
         Write_Line (Dict.TemporaryFile, " ;");
      end if;
   end AddOwnAnnotation;

   --------------------------------------------------------------------------------
   --  Add_Own_Variable
   --------------------------------------------------------------------------------

   procedure Add_Own_Variable_Local
     (Name                : in     LexTokenManager.Lex_String;
      The_Package         : in     RawDict.Package_Info_Ref;
      Mode                : in     Modes;
      Is_Protected        : in     Boolean;
      Is_Interrupt_Stream : in     Boolean;
      Comp_Unit           : in     ContextManager.UnitDescriptors;
      Declaration         : in     Location;
      The_Variable        :    out RawDict.Variable_Info_Ref)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys from *,
   --#                                Comp_Unit,
   --#                                Declaration,
   --#                                Dict,
   --#                                Is_Interrupt_Stream,
   --#                                Is_Protected,
   --#                                LexTokenManager.State,
   --#                                Mode,
   --#                                Name,
   --#                                The_Package &
   --#         The_Variable      from Comp_Unit,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                The_Package;
   --# post The_Variable /= RawDict.Null_Variable_Info_Ref;
   is
      The_Implicit_In_Stream : RawDict.Implicit_In_Stream_Info_Ref;
      The_Own_Variable       : RawDict.Own_Variable_Info_Ref;
   begin
      case Get_Visibility (Scope => Get_Package_Scope (The_Package => The_Package)) is
         when Visible =>
            The_Variable := RawDict.Null_Variable_Info_Ref;
         when Privat | Local  =>
            The_Variable := RawDict.Get_Variable_Info_Ref
              (LookupImmediateScope (Name,
                                     Set_Visibility (The_Visibility => Visible,
                                                     The_Unit       => RawDict.Get_Package_Symbol (The_Package)),
                                     ProofContext));
      end case;
      if The_Variable = RawDict.Null_Variable_Info_Ref then
         RawDict.Create_Variable
           (Name         => Name,
            Comp_Unit    => Comp_Unit,
            Loc          => Declaration.Start_Position,
            The_Variable => The_Variable);
         RawDict.Create_Own_Variable
           (Variable         => The_Variable,
            Owner            => RawDict.Get_Package_Symbol (The_Package),
            Comp_Unit        => Comp_Unit,
            Loc              => Declaration.Start_Position,
            The_Own_Variable => The_Own_Variable);
         RawDict.Set_Variable_Own_Variable (The_Variable => The_Variable,
                                            Own_Variable => The_Own_Variable);
         RawDict.Set_Next_Own_Variable
           (The_Own_Variable => The_Own_Variable,
            Next             => RawDict.Get_Package_Own_Variables (The_Package => The_Package));
         RawDict.Set_Package_Own_Variables (The_Package   => The_Package,
                                            Own_Variables => The_Own_Variable);
      else
         if RawDict.Get_Symbol_Compilation_Unit (Item => RawDict.Get_Variable_Symbol (The_Variable)) = Comp_Unit then
            --  There was a usage of the variable before its actual
            --  declaration -> update the declaration position. The
            --  declaration must be in the same compilation unit as
            --  the usage (in the package specification) to prevent to
            --  get a cross-reference from the package specification
            --  to the package body.
            RawDict.Set_Symbol_Location
              (Item     => RawDict.Get_Variable_Symbol (The_Variable),
               Location => Declaration.Start_Position);
         end if;
         The_Own_Variable := RawDict.Get_Variable_Own_Variable (The_Variable => The_Variable);
      end if;

      RawDict.Set_Own_Variable_Announced (The_Own_Variable => The_Own_Variable);
      RawDict.Set_Own_Variable_Mode (The_Own_Variable => The_Own_Variable,
                                     Mode             => Mode);
      RawDict.Set_Own_Variable_Protected (The_Own_Variable => The_Own_Variable,
                                          Is_Protected     => Is_Protected);
      RawDict.Set_Own_Variable_Is_Interrupt_Stream
        (The_Own_Variable    => The_Own_Variable,
         Is_Interrupt_Stream => Is_Interrupt_Stream);
      AddOtherReference (RawDict.Get_Variable_Symbol (The_Variable), RawDict.Get_Package_Symbol (The_Package), Declaration);

      -- For protected state that has no mode, we need to create an implicit in stream
      -- from which reads of the protected state are deemed to come; this facilitiates
      -- effective flow analysis of shared protected state.
      if Is_Protected and then Mode = DefaultMode then
         RawDict.Create_Implicit_In_Stream
           (The_Own_Variable       => The_Own_Variable,
            Comp_Unit              => Comp_Unit,
            Loc                    => Declaration.Start_Position,
            The_Implicit_In_Stream => The_Implicit_In_Stream);
         RawDict.Set_Own_Variable_Implicit_In_Stream
           (The_Own_Variable       => The_Own_Variable,
            The_Implicit_In_Stream => The_Implicit_In_Stream);
      end if;
   end Add_Own_Variable_Local;

   --------------------------------------------------------------------------------

   procedure Add_Own_Variable
     (Name                : in     LexTokenManager.Lex_String;
      The_Package         : in     Symbol;
      Mode                : in     Modes;
      Is_Protected        : in     Boolean;
      Is_Interrupt_Stream : in     Boolean;
      Comp_Unit           : in     ContextManager.UnitDescriptors;
      Declaration         : in     Location;
      Var_Symbol          :    out Symbol) is
      The_Variable : RawDict.Variable_Info_Ref;
   begin
      Add_Own_Variable_Local
        (Name                => Name,
         The_Package         => RawDict.Get_Package_Info_Ref (The_Package), -- GAA External
         Mode                => Mode,
         Is_Protected        => Is_Protected,
         Is_Interrupt_Stream => Is_Interrupt_Stream,
         Comp_Unit           => Comp_Unit,
         Declaration         => Declaration,
         The_Variable        => The_Variable);
      Var_Symbol := RawDict.Get_Variable_Symbol (The_Variable); -- GAA External
   end Add_Own_Variable;

   --------------------------------------------------------------------------------
   --  Add_Own_Variable_Type
   --------------------------------------------------------------------------------

   procedure Add_Own_Variable_Type
     (Own_Variable   : in RawDict.Variable_Info_Ref;
      Type_Mark      : in RawDict.Type_Info_Ref;
      Type_Reference : in Location)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Own_Variable,
   --#                                Type_Mark &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Own_Variable,
   --#                                Type_Mark,
   --#                                Type_Reference;
   is
   begin
      RawDict.Set_Variable_Type (The_Variable => Own_Variable,
                                 Type_Mark    => Type_Mark);
      -- Remember the abstract type, needed for spec-only view of own variable
      -- type, since the typeinfo gets overwritten during examination of body.
      RawDict.Set_Variable_Abstract_Type (The_Variable       => Own_Variable,
                                          Abstract_Type_Mark => Type_Mark);
      RawDict.Set_Type_Is_Own_Var_Type (Type_Mark => Type_Mark);
      RawDict.Set_Own_Variable_Typed (The_Own_Variable => RawDict.Get_Variable_Own_Variable (The_Variable => Own_Variable));

      if Type_Mark /= Get_Unknown_Type_Mark then
         AddOtherReference (RawDict.Get_Type_Symbol (Type_Mark), Get_Owner (The_Variable => Own_Variable), Type_Reference);
      end if;
   end Add_Own_Variable_Type;

   --------------------------------------------------------------------------------

   procedure AddOwnVariableType (OwnVariable   : in Symbol;
                                 TypeMark      : in Symbol;
                                 TypeReference : in Location) is
   begin
      Add_Own_Variable_Type
        (Own_Variable   => RawDict.Get_Variable_Info_Ref (OwnVariable), -- GAA External
         Type_Mark      => RawDict.Get_Type_Info_Ref (TypeMark), -- GAA External
         Type_Reference => TypeReference);
   end AddOwnVariableType;

   --------------------------------------------------------------------------------
   --  Add_Own_Task
   --------------------------------------------------------------------------------

   procedure AddOwnTask
     (Name        : in     LexTokenManager.Lex_String;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Declaration : in     Location;
      TypeMark    : in     Symbol;
      ThePackage  : in     Symbol;
      TaskSym     :    out Symbol) is
      The_Task : RawDict.Variable_Info_Ref;

      --------------------------------------------------------------------------------

      procedure Add_Own_Task
        (Name        : in     LexTokenManager.Lex_String;
         Comp_Unit   : in     ContextManager.UnitDescriptors;
         Declaration : in     Location;
         Type_Mark   : in     RawDict.Type_Info_Ref;
         The_Package : in     RawDict.Package_Info_Ref;
         The_Task    :    out RawDict.Variable_Info_Ref)
      --# global in out Dict;
      --# derives Dict     from *,
      --#                       Comp_Unit,
      --#                       Declaration,
      --#                       Name,
      --#                       The_Package,
      --#                       Type_Mark &
      --#         The_Task from Comp_Unit,
      --#                       Declaration,
      --#                       Dict,
      --#                       Name;
      --# post The_Task /= RawDict.Null_Variable_Info_Ref;
      is
         OwnTask  : Symbol;
      begin
         RawDict.Create_Variable (Name         => Name,
                                  Comp_Unit    => Comp_Unit,
                                  Loc          => Declaration.Start_Position,
                                  The_Variable => The_Task);
         RawDict.Set_Variable_Type (The_Variable => The_Task,
                                    Type_Mark    => Type_Mark);
         RawDict.CreateOwnTask
           (Variable  => The_Task,
            Owner     => The_Package,
            Comp_Unit => Comp_Unit,
            Loc       => Declaration.Start_Position,
            OwnTask   => OwnTask);
         RawDict.Set_Variable_Own_Task (The_Variable => The_Task,
                                        Own_Task     => OwnTask);
         RawDict.SetNextOwnTask (OwnTask,
                                 RawDict.Get_Package_Task_List (The_Package => The_Package));
         RawDict.Set_Package_Task_List (The_Package => The_Package,
                                        Task_List   => OwnTask);
      end Add_Own_Task;

   begin -- AddOwnTask
      Add_Own_Task
        (Name        => Name,
         Comp_Unit   => Comp_Unit,
         Declaration => Declaration,
         Type_Mark   => RawDict.Get_Type_Info_Ref (TypeMark), -- GAA External
         The_Package => RawDict.Get_Package_Info_Ref (ThePackage), -- GAA External
         The_Task    => The_Task);
      TaskSym := RawDict.Get_Variable_Symbol (The_Task); -- GAA External
   end AddOwnTask;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   procedure AddRefinementDefinition (ThePackage : in Symbol;
                                      Annotation : in Location) is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         Write_String (Dict.TemporaryFile, "refinement definition of ");
         Write_Name (File => Dict.TemporaryFile,
                     Item => ThePackage);
         Write_String (Dict.TemporaryFile, " is at ");
         Write_Location (File => Dict.TemporaryFile,
                         Loc  => Annotation);
         Write_Line (Dict.TemporaryFile, " ;");
      end if;
   end AddRefinementDefinition;

   --------------------------------------------------------------------------------
   --  Add_Constituent_Sym
   --------------------------------------------------------------------------------

   procedure AddConstituentSym
     (ConstituentVariable  : in Symbol;
      Subject              : in Symbol;
      Comp_Unit            : in ContextManager.UnitDescriptors;
      ConstituentReference : in Location) is

      procedure Add_Constituent_Sym
        (Constituent_Variable  : in RawDict.Variable_Info_Ref;
         The_Variable          : in RawDict.Variable_Info_Ref;
         Comp_Unit             : in ContextManager.UnitDescriptors;
         Constituent_Reference : in Location)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Constituent_Reference,
      --#                                Constituent_Variable,
      --#                                The_Variable &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Constituent_Reference,
      --#                                Constituent_Variable,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                The_Variable;
      is
         ThePackage       : Symbol;
         The_Own_Variable : RawDict.Own_Variable_Info_Ref;
         The_Constituent  : RawDict.Constituent_Info_Ref;
      begin
         The_Own_Variable := RawDict.Get_Variable_Own_Variable (The_Variable => The_Variable);
         ThePackage       := RawDict.Get_Own_Variable_Owner (The_Own_Variable => The_Own_Variable);

         RawDict.Create_Constituent
           (The_Own_Variable => The_Own_Variable,
            The_Variable     => Constituent_Variable,
            Mode             => Get_Own_Variable_Mode (The_Variable => Constituent_Variable),
            Comp_Unit        => Comp_Unit,
            Loc              => Constituent_Reference.Start_Position,
            The_Constituent  => The_Constituent);
         RawDict.Set_Variable_Constituent (The_Variable    => Constituent_Variable,
                                           The_Constituent => The_Constituent);

         RawDict.Set_Next_Constituent
           (The_Constituent => The_Constituent,
            Next            => RawDict.Get_Own_Variable_Constituents (The_Own_Variable => The_Own_Variable));
         RawDict.Set_Own_Variable_Constituents (The_Own_Variable => The_Own_Variable,
                                                The_Constituent  => The_Constituent);

         AddOtherReference (RawDict.Get_Variable_Symbol (The_Variable), ThePackage, Null_Location);
         AddOtherReference (RawDict.Get_Variable_Symbol (Constituent_Variable), ThePackage, Constituent_Reference);
      end Add_Constituent_Sym;

   begin -- AddConstituentSym
      Add_Constituent_Sym
        (Constituent_Variable  => RawDict.Get_Variable_Info_Ref (ConstituentVariable), -- GAA External
         The_Variable          => RawDict.Get_Variable_Info_Ref (Subject), -- GAA External
         Comp_Unit             => Comp_Unit,
         Constituent_Reference => ConstituentReference);
   end AddConstituentSym;

   --------------------------------------------------------------------------------
   --  Add_Constituent
   --------------------------------------------------------------------------------

   procedure AddConstituent
     (Name                 : in LexTokenManager.Lex_String;
      Subject              : in Symbol;
      Mode                 : in Modes;
      SubjectReference     : in Location;
      Comp_Unit            : in ContextManager.UnitDescriptors;
      ConstituentReference : in Location) is

      procedure Add_Constituent
        (Name                  : in LexTokenManager.Lex_String;
         The_Subject           : in RawDict.Variable_Info_Ref;
         Mode                  : in Modes;
         Subject_Reference     : in Location;
         Comp_Unit             : in ContextManager.UnitDescriptors;
         Constituent_Reference : in Location)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Constituent_Reference,
      --#                                Mode,
      --#                                Name,
      --#                                The_Subject &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Constituent_Reference,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Name,
      --#                                Subject_Reference,
      --#                                The_Subject;
      is
         ThePackage       : Symbol;
         The_Own_Variable : RawDict.Own_Variable_Info_Ref;
         The_Variable     : RawDict.Variable_Info_Ref;
         The_Constituent  : RawDict.Constituent_Info_Ref;
      begin
         The_Own_Variable := RawDict.Get_Variable_Own_Variable (The_Variable => The_Subject);
         ThePackage       := RawDict.Get_Own_Variable_Owner (The_Own_Variable => The_Own_Variable);

         RawDict.Create_Variable
           (Name         => Name,
            Comp_Unit    => Comp_Unit,
            Loc          => Constituent_Reference.Start_Position,
            The_Variable => The_Variable);
         RawDict.Create_Constituent
           (The_Own_Variable => The_Own_Variable,
            The_Variable     => The_Variable,
            Mode             => Mode,
            Comp_Unit        => Comp_Unit,
            Loc              => Constituent_Reference.Start_Position,
            The_Constituent  => The_Constituent);
         RawDict.Set_Variable_Constituent (The_Variable    => The_Variable,
                                           The_Constituent => The_Constituent);

         RawDict.Set_Next_Constituent
           (The_Constituent => The_Constituent,
            Next            => RawDict.Get_Own_Variable_Constituents (The_Own_Variable => The_Own_Variable));
         RawDict.Set_Own_Variable_Constituents (The_Own_Variable => The_Own_Variable,
                                                The_Constituent  => The_Constituent);

         AddOtherReference (RawDict.Get_Variable_Symbol (The_Subject), ThePackage, Subject_Reference);
         AddOtherReference (RawDict.Get_Variable_Symbol (The_Variable), ThePackage, Constituent_Reference);
         --# accept Flow, 601, SPARK_IO.File_Sys, Mode, "Spurious data coupling via Dictionary";
      end Add_Constituent;

   begin -- AddConstituent
      Add_Constituent
        (Name                  => Name,
         The_Subject           => RawDict.Get_Variable_Info_Ref (Subject), -- GAA External
         Mode                  => Mode,
         Subject_Reference     => SubjectReference,
         Comp_Unit             => Comp_Unit,
         Constituent_Reference => ConstituentReference);
   end AddConstituent;

   --------------------------------------------------------------------------------
   --  Add_Embedded_Constituent
   --------------------------------------------------------------------------------

   procedure AddEmbeddedConstituent
     (PackageName          : in LexTokenManager.Lex_String;
      VariableName         : in LexTokenManager.Lex_String;
      Subject              : in Symbol;
      Mode                 : in Modes;
      SubjectReference     : in Location;
      Comp_Unit            : in ContextManager.UnitDescriptors;
      ConstituentReference : in Location) is

      procedure Add_Embedded_Constituent
        (Package_Name          : in LexTokenManager.Lex_String;
         Variable_Name         : in LexTokenManager.Lex_String;
         The_Subject           : in RawDict.Variable_Info_Ref;
         Mode                  : in Modes;
         Subject_Reference     : in Location;
         Comp_Unit             : in ContextManager.UnitDescriptors;
         Constituent_Reference : in Location)
      --# global in     LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict              from *,
      --#                                Comp_Unit,
      --#                                Constituent_Reference,
      --#                                LexTokenManager.State,
      --#                                Mode,
      --#                                Package_Name,
      --#                                The_Subject,
      --#                                Variable_Name &
      --#         SPARK_IO.File_Sys from *,
      --#                                Comp_Unit,
      --#                                Constituent_Reference,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Package_Name,
      --#                                Subject_Reference,
      --#                                The_Subject,
      --#                                Variable_Name;
      is
         Outer_Package                                  : Symbol;
         Inner_Package                                  : RawDict.Package_Info_Ref;
         The_Specification                              : RawDict.Declaration_Info_Ref;
         The_Own_Variable, The_Constituent_Own_Variable : RawDict.Own_Variable_Info_Ref;
         The_Constituent_Variable                       : RawDict.Variable_Info_Ref;
         The_Constituent                                : RawDict.Constituent_Info_Ref;
      begin
         The_Own_Variable := RawDict.Get_Variable_Own_Variable (The_Variable => The_Subject);
         Outer_Package    := RawDict.Get_Own_Variable_Owner (The_Own_Variable => The_Own_Variable);
         Inner_Package    := RawDict.Get_Package_Info_Ref
           (LookupImmediateScope (Package_Name,
                                  Set_Visibility (The_Visibility => Local,
                                                  The_Unit       => Outer_Package),
                                  ProofContext));
         if Inner_Package = RawDict.Null_Package_Info_Ref then
            Add_Declaration
              (Comp_Unit       => Comp_Unit,
               Loc             => Constituent_Reference,
               Scope           => Set_Visibility (The_Visibility => Local,
                                                  The_Unit       => Outer_Package),
               Context         => ProofContext,
               The_Declaration => The_Specification);
            RawDict.Create_Package
              (Name            => Package_Name,
               The_Declaration => The_Specification,
               Comp_Unit       => Comp_Unit,
               Loc             => Constituent_Reference.Start_Position,
               The_Package     => Inner_Package);
         end if;

         RawDict.Create_Variable
           (Name         => Variable_Name,
            C