/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file implements classes SKGPayeeObject.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgpayeeobject.h"

#include <KLocale>

#include "skgdocumentbank.h"
#include "skgtraces.h"
#include "skgoperationobject.h"

SKGPayeeObject::SKGPayeeObject(SKGDocument* iDocument, int iID): SKGNamedObject(iDocument, "v_payee", iID)
{
}

SKGPayeeObject::~SKGPayeeObject()
{
}

SKGPayeeObject::SKGPayeeObject(const SKGPayeeObject& iObject)
    : SKGNamedObject(iObject)
{
}

SKGPayeeObject::SKGPayeeObject(const SKGObjectBase& iObject)

{
    if (iObject.getRealTable() == "payee") {
        copyFrom(iObject);
    } else {
        *this = SKGNamedObject(iObject.getDocument(), "v_payee", iObject.getID());
    }
}

const SKGPayeeObject& SKGPayeeObject::operator= (const SKGObjectBase& iObject)
{
    copyFrom(iObject);
    return *this;
}

SKGError SKGPayeeObject::createPayee(SKGDocumentBank* iDocument,
                                     const QString& iName,
                                     SKGPayeeObject& oPayee,
                                     bool iSendPopupMessageOnCreation)
{
    SKGError err;
    SKGTRACEINRC(10, "SKGPayeeObject::createPayee", err);

    // Check if refund is already existing
    if (iName.isEmpty()) {
        oPayee = SKGPayeeObject(NULL, 0);
    } else if (iDocument) {
        iDocument->getObject("v_payee", "t_name='" % SKGServices::stringToSqlString(iName) % '\'', oPayee);
        if (oPayee.getID() == 0) {
            // No, we have to create it
            oPayee = SKGPayeeObject(iDocument);
            err = oPayee.setName(iName);
            IFOKDO(err, oPayee.save())

            if (!err && iSendPopupMessageOnCreation && iDocument) {
                err = iDocument->sendMessage(i18nc("Information message", "Payee '%1' has been created" , iName), SKGDocument::Positive);
            }
        }
    }

    return err;
}

SKGError SKGPayeeObject::getOperations(SKGListSKGObjectBase& oOperations) const
{
    SKGError err = getDocument()->getObjects("v_operation",
                   "r_payee_id=" % SKGServices::intToString(getID()) ,
                   oOperations);
    return err;
}

SKGError SKGPayeeObject::setAddress(const QString& iAddress)
{
    return setAttribute("t_address", iAddress);
}

QString SKGPayeeObject::getAddress() const
{
    return getAttribute("t_address");
}

SKGError SKGPayeeObject::bookmark(bool iBookmark)
{
    return setAttribute("t_bookmarked", iBookmark ? "Y" : "N");
}

bool SKGPayeeObject::isBookmarked() const
{
    return (getAttribute("t_bookmarked") == "Y" ? true : false);
}

SKGError SKGPayeeObject::merge(const SKGPayeeObject& iPayee)
{
    SKGError err;

    SKGObjectBase::SKGListSKGObjectBase ops;
    IFOKDO(err, iPayee.getOperations(ops))
    int nb = ops.count();
    for (int i = 0; !err && i < nb; ++i) {
        SKGOperationObject op(ops.at(i));
        err = op.setPayee(*this);
        IFOKDO(err, op.save(true, false))
    }

    IFOKDO(err, iPayee.remove())
    return err;
}

#include "skgpayeeobject.moc"
