# Authors:
#     Endi S. Dewata <edewata@redhat.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#
# Copyright (C) 2015 Red Hat, Inc.
# All rights reserved.
#

from __future__ import absolute_import
from __future__ import print_function
import getopt
import subprocess
import sys
import traceback

import pki.cli
import pki.server.cli.ca
import pki.server.cli.kra
import pki.server.cli.ocsp
import pki.server.cli.tks
import pki.server.cli.tps
import pki.server.cli.http
import pki.server.cli.banner
import pki.server.cli.db
import pki.server.cli.instance
import pki.server.cli.subsystem
import pki.server.cli.migrate
import pki.server.cli.nuxwdog
import pki.server.cli.cert
import pki.server.cli.selftest


class PKIServerCLI(pki.cli.CLI):

    def __init__(self):
        super(PKIServerCLI, self).__init__(
            'pki-server',
            'PKI server command-line interface')

        self.add_module(pki.server.cli.ca.CACLI())
        self.add_module(pki.server.cli.kra.KRACLI())
        self.add_module(pki.server.cli.ocsp.OCSPCLI())
        self.add_module(pki.server.cli.tks.TKSCLI())
        self.add_module(pki.server.cli.tps.TPSCLI())

        self.add_module(pki.server.cli.banner.BannerCLI())
        self.add_module(pki.server.cli.db.DBCLI())
        self.add_module(pki.server.cli.instance.InstanceCLI())
        self.add_module(pki.server.cli.subsystem.SubsystemCLI())
        self.add_module(pki.server.cli.migrate.MigrateCLI())
        self.add_module(pki.server.cli.nuxwdog.NuxwdogCLI())
        self.add_module(pki.server.cli.cert.CertCLI())
        self.add_module(pki.server.cli.selftest.SelfTestCLI())
        self.add_module(pki.server.cli.http.HTTPCLI())

    def get_full_module_name(self, module_name):
        return module_name

    def print_help(self):
        print('Usage: pki-server [OPTIONS]')
        print()
        print('  -v, --verbose                Run in verbose mode.')
        print('      --debug                  Show debug messages.')
        print('      --help                   Show help message.')
        print()

        super(PKIServerCLI, self).print_help()

    def execute(self, argv):
        try:
            opts, args = getopt.getopt(argv[1:], 'v', [
                'verbose', 'debug', 'help'])

        except getopt.GetoptError as e:
            print('ERROR: ' + str(e))
            self.print_help()
            sys.exit(1)

        for o, _ in opts:
            if o in ('-v', '--verbose'):
                self.set_verbose(True)

            elif o == '--debug':
                self.set_verbose(True)
                self.set_debug(True)

            elif o == '--help':
                self.print_help()
                sys.exit()

            else:
                print('ERROR: unknown option ' + o)
                self.print_help()
                sys.exit(1)

        if self.verbose:
            print('Command: %s' % ' '.join(args))

        super(PKIServerCLI, self).execute(args)


if __name__ == '__main__':

    cli = PKIServerCLI()

    try:
        cli.execute(sys.argv)

    except subprocess.CalledProcessError as e:
        if cli.verbose:
            traceback.print_exc()
        print('ERROR: Command: %s' % ' '.join(e.cmd))
        sys.exit(e.returncode)

    except Exception as e:  # pylint: disable=broad-except
        if cli.verbose:
            traceback.print_exc()
        print('ERROR: %s' % e)
        sys.exit(1)
