/* $Id: object.h,v 1.15 2003/06/17 14:45:56 bsenders Exp $ */

#ifndef FFRENZY_OBJECT_H
#define FFRENZY_OBJECT_H

#include "global.h"
#include "model.h"

/** \file
 * Object specific functions and defines.
 */

/* Object settings */
#define OBJ_X_SIZE       15   /**< Default object X size (width). */
#define OBJ_Y_SIZE       15   /**< Default object Y size (length). */

/* Object mechanics settings */
#define OBJ_Y_MAX_SPEED  1000 /**< Maximum Y speed of all objects. */

#define OBJ_Y_GRAV_ACCEL 15   /**< Object gravity constant. */

#define OBJ_X_BOUNCINESS 1    /**< Horizontal bounciness factor of object. */
#define OBJ_Y_BOUNCINESS 2    /**< Vertical bounciness factor of object. */
#define OBJ_Y_BOUNCE_CUTOFF 5 /**< Vertical speed will be set to zero if
				   vertical speed after bounciness is lower
				   than this. */

/** Creates and initializes object with given specs and adds it to the model.
 *  \param m      The model the object needs to be added to.
 *  \param type   Type of the object.
 *  \param owner  Owner (player) of the object (\c NULL if none).
 *  \param x      X coordinate of create position.
 *  \param y      Y coordinate of create position.
 *  \param width  Width of the object.
 *  \param height height of the object. 
 *  \return       Pointer to newly created object.
 */
Model_object *new_object(Model *m, Object_type type, Model_object *owner, 
                         int32_t x, int32_t y, int32_t width, int32_t height);

/** Deletes object which has been mark for deletion and graphics data field
 *  \c gfx_data is cleared.
 *  \param m Model the object needs to be removed from.
 *  \param o Pointer to object.
 */
void del_object(Model *m, Model_object *o);

/** Dumps all known information about object using MDEBUG as one-liner
 *  <type:id size position speed state etc... >.
 *  This function can be used as an Listfunc!
 *  \param object Pointer to object.
 *  \param data   Prefix string (can be NULL for no prefix).
 *  \return TRUE  Keep \a list_foreach or \a list_search going.
 */
int object_dump(void *object, void *data);

/** Marks object as deleted, so that graphics part can remove the object
 *  from the screen and free the graphics data. 
 *  This function can be used as an Listfunc!
 *  \param object    Pointer to object.
 *  \param user_data Dummy arg (use \c NULL).
 *  \return TRUE  Keep \a list_foreach or \a list_search going.
 */
int object_set_deleted(void *object, void *user_data);

/** Check if object is marked for deletion and graphics data
 *  is cleared. If so, delete the object.
 *  \param object Pointer to object.
 *  \param model  Model the object is part of.
 *  \return TRUE  Keep \a list_foreach or \a list_search going.
 */
int object_check_deleted(void *object, void *model);

/* Move object from right passive player/food list to right active
 * player/food list, set active flag en enable gravity.
 * This function assumes object is marked passive!
 */
void object_make_active(Model *m, Model_object *o);

/** Handles one tick for an passive non-scenery object.
 *  If object is able to move down (gravity would move the object),
 *  or desires the move, make the object active.
 *  \param object Pointer to object.
 *  \param model  Model the object is part of.
 *  \return TRUE  Keep \a list_foreach or \a list_search going.
 */
int object_passive_tick(void *object, void *model);

/** Handles one tick for an active (moving) non-scenery object.
 *  If object doesn't want to move anymore, make object passive.
 *  \param object Pointer to object.
 *  \param model  Model the object is part of.
 *  \return TRUE  Keep \a list_foreach or \a list_search going.
 */
int object_active_tick(void *object, void *model);

/** Handles tick for objects (movement and events). 
 *  \li Handle I/O (flags) on players, handle powerups on food.
 *  \li Calculates new position/speed for non-scenery objects.
 *  \param object Pointer to object.
 *  \param model  Model the object is part of.
 *  \return TRUE  Keep \a list_foreach or \a list_search going.
 */
int object_tick(void *object, void *model);

#endif /* FFRENZY_OBJECT_H */
