;;; -*- Mode: Common-Lisp -*-

;;; Copyright (c) 2006, Abhijit 'quasi' Rao.  All rights reserved.
;;; Copyright (c) 2006, Cleartrip Travel Services.
;;; Copyright (c) 2011 Kevin Rosenberg

;;; Redistribution and use in source and binary forms, with or without
;;; modification, are permitted provided that the following conditions
;;; are met:

;;;   * Redistributions of source code must retain the above copyright
;;;     notice, this list of conditions and the following disclaimer.

;;;   * Redistributions in binary form must reproduce the above
;;;     copyright notice, this list of conditions and the following
;;;     disclaimer in the documentation and/or other materials
;;;     provided with the distribution.

;;; THIS SOFTWARE IS PROVIDED BY THE AUTHOR 'AS IS' AND ANY EXPRESSED
;;; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
;;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;;; ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
;;; DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
;;; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
;;; GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
;;; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;; NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;; SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
(in-package #:memcache)

(defvar *memcache* nil
  "Represents a particular Memcached server")

(defvar *use-pool* nil
  "Default value for the USE-POOL keyword parameter in memcached functions")

(defvar *pool-get-trys?* nil
  "If true then it will try to wait and sleep for a while if pool item in unavailable,
if nil then will return immideatly")

(defconstant* +crlf+
  (concatenate 'string
               (string (code-char 13))
               (string (code-char 10))))

(defconstant* +mc-END-ret+
    (concatenate 'string
                 (string "END")
                 (string #\return)))

(defstruct
  (memcache-stats
   (:conc-name mc-stats-)
   (:print-function
    (lambda (struct stream depth)
      (declare (ignore depth))
      (print-unreadable-object (struct stream :type t :identity t)
        (format stream "pid:~A size:~d MB curr:~d total:~D"
                (mc-stats-pid struct)
                (/ (mc-stats-limit-maxbytes struct) 1024 1024)
                (mc-stats-curr-items struct)
                (mc-stats-curr-items-total struct))))))
"The structure which holds the statistics from the memcached server. The fields are :
field-name                 accessor-function                 documentation
----------                 -----------------                 -------------
pid                        mc-stats-pid                      Process id of this server process
uptime                     mc-stats-uptime                   Number of seconds this server has been running
time                       mc-stats-time                     current UNIX time according to the server
version                    mc-stats-version                  Version string of this server
rusage-user                mc-stats-rusage-user              Accumulated user time for this process
rusage-system              mc-stats-rusage-system            Accumulated system time for this process
curr-items                 mc-stats-curr-items               Current number of items stored by the server
curr-items-total           mc-stats-curr-items-total
curr-connections           mc-stats-curr-connections         Number of open connections
total-connections          mc-stats-total-connections        Total number of connections opened since the server started running
connection-structures      mc-stats-connection-structures    Number of connection structures allocated by the server
cmd-get                    mc-stats-cmd-get                  Cumulative number of retrieval requests
cmd-set                    mc-stats-cmd-set                  Cumulative number of storage requests
get-hits                   mc-stats-get-hits                 Number of keys that have been requested and found present
get-misses                 mc-stats-get-misses               Number of items that have been requested and not found
bytes-read                 mc-stats-bytes-read               Total number of bytes read by this server from network
bytes-written              mc-stats-bytes-written            Total number of bytes sent by this server to network
limit-maxbytes             mc-stats-limit-maxbytes           Number of bytes this server is allowed to use for storage.
"
  all-stats
  pid uptime time version rusage-user rusage-system curr-items curr-items-total
  curr-connections total-connections connection-structures cmd-get cmd-set
  get-hits get-misses bytes-read bytes-written limit-maxbytes)

;;;
;;; The main class which represents the memcached server
;;;
(defclass memcache ()
  ((name
    :initarg :name
    :reader name
    :type simple-string
    :documentation "Name of this Memcache instance")
   (host
    :initarg :host
    :initform "127.0.0.1"
    :accessor host
    :type simple-string
    :documentation "The host name of the Memcached server for this instance.")
   (port
    :initarg :port
    :initform 11211
    :accessor port
    :type fixnum
    :documentation "The port on which the Memcached server this instance represents runs")
   (memcached-server-storage-size
    :initform 0
    :reader memcached-server-storage-size
    :type fixnum
    :documentation "Memory allocated to the Memcached Server")
   (pool-size
    :initarg :pool-size
    :initform 2
    :reader pool-size)
   (pool
    :reader pool))
  (:documentation "This class represents an instance of the Memcached server"))


(defconstant* +membase17-stat-names+
  '("accepting_conns" "auth_cmds" "auth_errors" "bucket_active_conns" "bucket_conns"
    "bytes_read" "bytes_written" "cas_badval" "cas_hits" "cas_misses" "cmd_flush"
    "cmd_get" "cmd_set" "conn_yields" "connection_structures" "curr_connections"
    "curr_items" "curr_items_tot" "daemon_connections" "decr_hits" "decr_misses"
    "delete_hits" "delete_misses" "ep_bg_fetched" "ep_commit_num" "ep_commit_time"
    "ep_commit_time_total" "ep_data_age" "ep_data_age_highwat" "ep_db_cleaner_status"
    "ep_db_strategy" "ep_dbinit" "ep_dbname" "ep_dbshards" "ep_diskqueue_drain"
    "ep_diskqueue_fill" "ep_diskqueue_items" "ep_diskqueue_memory"
    "ep_diskqueue_pending" "ep_expired" "ep_flush_all" "ep_flush_duration"
    "ep_flush_duration_highwat" "ep_flush_duration_total" "ep_flush_preempts"
    "ep_flusher_state" "ep_flusher_todo" "ep_io_num_read" "ep_io_num_write"
    "ep_io_read_bytes" "ep_io_write_bytes" "ep_item_begin_failed"
    "ep_item_commit_failed" "ep_item_flush_expired" "ep_item_flush_failed"
    "ep_items_rm_from_checkpoints" "ep_kv_size" "ep_latency_arith_cmd"
    "ep_latency_get_cmd" "ep_latency_store_cmd" "ep_max_data_size" "ep_max_txn_size"
    "ep_mem_high_wat" "ep_mem_low_wat" "ep_min_data_age" "ep_num_active_non_resident"
    "ep_num_checkpoint_remover_runs" "ep_num_eject_failures" "ep_num_eject_replicas"
    "ep_num_expiry_pager_runs" "ep_num_non_resident" "ep_num_not_my_vbuckets"
    "ep_num_pager_runs" "ep_num_value_ejects" "ep_onlineupdate"
    "ep_onlineupdate_revert_add" "ep_onlineupdate_revert_delete"
    "ep_onlineupdate_revert_update" "ep_oom_errors" "ep_overhead" "ep_pending_ops"
    "ep_pending_ops_max" "ep_pending_ops_max_duration" "ep_pending_ops_total"
    "ep_queue_age_cap" "ep_queue_size" "ep_storage_age" "ep_storage_age_highwat"
    "ep_storage_type" "ep_store_max_concurrency" "ep_store_max_readers"
    "ep_store_max_readwrite" "ep_tap_bg_fetch_requeued" "ep_tap_bg_fetched"
    "ep_tap_keepalive" "ep_tmp_oom_errors" "ep_too_old" "ep_too_young"
    "ep_total_cache_size" "ep_total_del_items" "ep_total_enqueued" "ep_total_new_items"
    "ep_total_persisted" "ep_uncommitted_items" "ep_vb_total" "ep_vbucket_del"
    "ep_vbucket_del_fail" "ep_version" "ep_warmed_up" "ep_warmup" "ep_warmup_dups"
    "ep_warmup_oom" "ep_warmup_thread" "ep_warmup_time" "get_hits" "get_misses"
    "incr_hits" "incr_misses" "libevent" "limit_maxbytes" "listen_disabled_num"
    "mem_used" "pid" "pointer_size" "rejected_conns" "rusage_system" "rusage_user"
    "threads" "time" "total_connections" "uptime" "vb_active_curr_items"
    "vb_active_eject" "vb_active_ht_memory" "vb_active_itm_memory" "vb_active_num"
    "vb_active_num_non_resident" "vb_active_ops_create" "vb_active_ops_delete"
    "vb_active_ops_reject" "vb_active_ops_update" "vb_active_perc_mem_resident"
    "vb_active_queue_age" "vb_active_queue_drain" "vb_active_queue_fill"
    "vb_active_queue_memory" "vb_active_queue_pending" "vb_active_queue_size"
    "vb_dead_num" "vb_pending_curr_items" "vb_pending_eject" "vb_pending_ht_memory"
    "vb_pending_itm_memory" "vb_pending_num" "vb_pending_num_non_resident"
    "vb_pending_ops_create" "vb_pending_ops_delete" "vb_pending_ops_reject"
    "vb_pending_ops_update" "vb_pending_perc_mem_resident" "vb_pending_queue_age"
    "vb_pending_queue_drain" "vb_pending_queue_fill" "vb_pending_queue_memory"
    "vb_pending_queue_pending" "vb_pending_queue_size" "vb_replica_curr_items"
    "vb_replica_eject" "vb_replica_ht_memory" "vb_replica_itm_memory" "vb_replica_num"
    "vb_replica_num_non_resident" "vb_replica_ops_create" "vb_replica_ops_delete"
    "vb_replica_ops_reject" "vb_replica_ops_update" "vb_replica_perc_mem_resident"
    "vb_replica_queue_age" "vb_replica_queue_drain" "vb_replica_queue_fill"
    "vb_replica_queue_memory" "vb_replica_queue_pending" "vb_replica_queue_size"
    "version"))
