/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


/**
   \file cdw_checkbox.c

   Implementation of "checkbox" widget.

   A checkbox looks like this:
   [ ]    - unchecked checkbox
   [X]    - checked checkbox.

   A checkbox is placed in given parent window in given place.
   Can be checked on or off. It provides a pointer to which client code
   can assign a callback. The callback will be called when state of the
   checkbox changes.

   A checkbox can be visible (and visitable by use of TAB key) or not.
   Checkbox is visible by default.
*/

#include <stdlib.h>

#include "cdw_checkbox.h"
#include "cdw_debug.h"
#include "cdw_form.h"


/* Definitions made so that symbol of marked checkbox is used
   consistently in this module. */
#define CDW_CHECKBOX_CHECKED   "X"
#define CDW_CHECKBOX_UNCHECKED " "







/**
   \brief Create new check box

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Create new checkbox widget, place it in given position of
   given parent window. Set its initial state.
   Remember to assign value to CDW_CHECKBOX->on_toggle.

   \param parent - parent window, in which the checkbox will be displayed
   \param begin_y - y coordinate of the checkbox
   \param begin_x - x coordinate of the checkbox
   \param checked - initial state of the checkbox

   \return pointer to the widget on success
   \return NULL on failure
*/
CDW_CHECKBOX *cdw_checkbox_new(WINDOW *parent, int begin_y, int begin_x, bool checked)
{
	cdw_assert (parent, "ERROR: parent window can't be NULL\n");

	CDW_CHECKBOX *checkbox = (CDW_CHECKBOX *) malloc(sizeof (CDW_CHECKBOX));
	if (!checkbox) {
		cdw_vdm ("ERROR: failed to malloc checkbox\n");
		return (CDW_CHECKBOX *) NULL;
	}

	cdw_widget_init(&checkbox->widget);
	cdw_widget_add_return_keys(&checkbox->widget, CDW_KEY_ESCAPE, CDW_KEY_TAB, CDW_KEY_BTAB, KEY_UP, KEY_DOWN, 0);
	checkbox->widget.add_return_key = cdw_widget_add_return_key; /* Default function for adding return keys by parent cdw_form. */
	checkbox->widget.type_id = CDW_WIDGET_ID_CHECKBOX;

	checkbox->parent = parent;
	checkbox->begin_y = begin_y;
	checkbox->begin_x = begin_x;
	checkbox->checked = checked;

	checkbox->visible = true;
	checkbox->on_toggle_callback = (cdw_form_widget_function_t) NULL;

	return checkbox;
}





/**
   \brief Change state of the checkbox

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Change state of the checkbox from checked to unchecked,
   or the other way around.
   Return state of a checkbox after the change.

   \param checkbox - checkbox to toggle.

   \return true if checkbox has been toggled on
   \return false if checkbox has been toggled off
*/
bool cdw_checkbox_toggle(CDW_CHECKBOX *checkbox)
{
	if (checkbox->visible) {
		if (checkbox->checked) {
			mvwprintw(checkbox->parent, checkbox->begin_y, checkbox->begin_x, CDW_CHECKBOX_UNCHECKED);
			checkbox->checked = false;
		} else {
			mvwprintw(checkbox->parent, checkbox->begin_y, checkbox->begin_x, CDW_CHECKBOX_CHECKED);
			checkbox->checked = true;
		}
		/* why the cursor moves? why? why? */
		wmove(checkbox->parent, checkbox->begin_y, checkbox->begin_x);
	} else {
		cdw_vdm ("WARNING: called the function for invisible checkbox\n");
	}

	return checkbox->checked;
}





/**
   \brief Get current state of a checkbox

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Get current state of a checkbox widget.
   Return true if checkbox is checked (marked), return false otherwise.

   \param checkbox - checkbox to query

   \return true if checkbox is checked
   \return false if checkbox is not checked
*/
bool cdw_checkbox_get_state(CDW_CHECKBOX *checkbox)
{
	cdw_assert (checkbox, "ERROR: checkbox is NULL\n");

	return checkbox->checked;
}





/**
   \brief Set checkbox to a specified state

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Set given \p checkbox to state \p checked. The \p checked
   argument may be true - the checkbox will become checked,
   or it can be false - the checkbox will become unchecked.
   If state of a checkbox before function call is the same as
   value of \p checked, no change is made to the state of a checkbox.

   Function returns state of a checkbox after the new state
   has been set.

   \param checkbox - checkbox to be altered
   \param checked - intended state of a checkbox, true or false

   \return true if checkbox has been set to "checked"
   \return false if checkbox has been set to "unchecked"
*/
bool cdw_checkbox_set_state(CDW_CHECKBOX *checkbox, bool checked)
{
	cdw_assert (checkbox, "ERROR: checkbox is NULL\n");

	if (checkbox->visible) {
		checkbox->checked = checked;
		if (checkbox->checked) {
			mvwprintw(checkbox->parent, checkbox->begin_y, checkbox->begin_x, CDW_CHECKBOX_CHECKED);
		} else {
			mvwprintw(checkbox->parent, checkbox->begin_y, checkbox->begin_x, CDW_CHECKBOX_UNCHECKED);
		}

		wrefresh(checkbox->parent);
	} else {
		cdw_vdm ("WARNING: called the function for invisible checkbox\n");
	}

	return checkbox->checked;
}





/**
   \brief Set visibility of a checkbox

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Turn checkbox visible or invisible.
   Invisible checkbox is not visible in UI (obvious), and can't be
   visited with key movement (e.g. TAB key).
   Value of \p visible should be true if checkbox should be visible,
   and false otherwise.

   \param checkbox - checkbox to be modified
   \param visible - intended visibility of the checkbox
*/
void cdw_checkbox_set_visibility(CDW_CHECKBOX *checkbox, bool visible)
{
	cdw_assert (checkbox, "ERROR: checkbox is NULL\n");

	checkbox->visible = visible;
	return;
}





/**
   \brief Delete a checkbox

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Deallocate all resources associated with given \p checkbox.
   Set \p *checkbox pointer to NULL. Function accepts NULL argument.

   \param checkbox - pointer to checkbox to be deleted.
*/
void cdw_checkbox_delete(CDW_CHECKBOX **checkbox)
{
	cdw_assert(checkbox, "ERROR: passing to the function a NULL pointer to a widget\n");

	if (!*checkbox) {
		cdw_vdm ("WARNING: passing NULL checkbox to the function\n");
		return;
	}

	free(*checkbox);
	*checkbox = (CDW_CHECKBOX *) NULL;
	return;
}





/**
   \brief Free resources related to checkbox

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Free all memory related to given \p checkbox.
   Function does not set \p checkbox to NULL.

   \param checkbox - checkbox to be freed
*/
void cdw_checkbox_free(CDW_CHECKBOX *checkbox)
{
	if (!checkbox) {
		cdw_vdm ("WARNING: passing NULL checkbox to the function\n");
		return;
	}

	free(checkbox);
	return;
}





/**
   \brief Draw a checkbox - if it is visible

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Draw borders of a checkbox, and draw an 'X' representing its state
   (if the checkbox is checked). Do this only if a checkbox is
   visible. Draw the checkbox in its parent window.

   \param checkbox - checkbox to be drawn
*/
void cdw_checkbox_draw(CDW_CHECKBOX *checkbox)
{
	cdw_assert (checkbox, "ERROR: checkbox is NULL\n");

	if (!checkbox->visible) {
		cdw_vdm ("WARNING: calling the function for invisible checkbox\n");
		return;
	}

	mvwprintw(checkbox->parent,
		  checkbox->begin_y, checkbox->begin_x - 1,
		  "[%s]",
		  checkbox->checked? CDW_CHECKBOX_CHECKED : CDW_CHECKBOX_UNCHECKED);

	redrawwin(checkbox->parent);
	wrefresh(checkbox->parent);

	return;
}





int cdw_checkbox_driver(CDW_CHECKBOX *cb, void *cdw_form)
{
	cdw_assert (cb, "ERROR: NULL pointer to checkbox\n");
	cdw_assert (cb->widget.type_id == CDW_WIDGET_ID_CHECKBOX, "ERROR: this is not a checkbox widget\n");
	cdw_assert (cb->parent, "ERROR: cannot control checkbox with NULL parent\n");

	int key = 'a'; /* Safe initial value. */
	while (!cdw_widget_is_return_key(&cb->widget, key)) {
		key = wgetch(cb->parent);

		if (key == ' ' || key == 'x' || key == 'X') {
			cdw_checkbox_toggle(cb);
			if (cb->on_toggle_callback) {
				cdw_assert (cdw_form, "ERROR: 'cdw_form' argument to callback is NULL\n");
				bool state = cdw_checkbox_get_state(cb);
				cb->on_toggle_callback((cdw_form_t *) cdw_form, &state);
			}
		} else {
			/* Either widget's return key that will be
			   caught by loop condition and handled by
			   widget's parent, or a completely
			   meaningless key that should be ignored. */
			;
		}
	}

	if (key == CDW_KEY_ESCAPE || key == 'q' || key == 'Q') {
		key = CDW_KEY_ESCAPE;
	}

	redrawwin(cb->parent);
	wrefresh(cb->parent);

	return key;
}
