/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

/**
   \file cdw_image_wizard.c

   Wizard for creating ISO9660 or UDF stand-alone image files.

   Most options are available in separate window, opened by clicking
   "Options" window in wizard. The only options available in main
   window of the wizard are image format (ISO9660/UDF), image volume
   string, and path to image file.

   The wizard does not change file extension in image file path if
   user changes image format.
*/

#define _GNU_SOURCE /* asprintf() */
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>

#include "cdw_file_picker.h"
#include "cdw_config_window.h"
#include "gettext.h"
#include "cdw_string.h"
#include "cdw_widgets.h"
#include "cdw_ncurses.h"
#include "cdw_window.h"
#include "cdw_main_window.h"
#include "cdw_debug.h"
#include "cdw_ext_tools.h"
#include "cdw_form.h"
#include "cdw_task.h"
#include "cdw_mkisofs_options.h"
#include "cdw_mkisofsrc.h"
#include "cdw_xorriso_options.h"
#include "cdw_xorrisorc.h"
#include "cdw_mkudffs_options.h"
#include "cdw_tabs_window.h"
#include "cdw_image_wizard.h"
#include "cdw_udf.h"
#include "cdw_iso9660.h"
#include "canonicalize.h"
#include "cdw_fs.h"


enum field_ids {
	f_image_format_l = 0,
	f_image_format_dd,
	f_volume_id_l,
	f_volume_id_i,
	f_image_file_l,
	f_image_file_i,
	f_image_file_b,
	f_options_b,
	f_create_b,
	f_cancel_b,
};


/* Does not include guard. */
#define CDW_IMAGE_WIZARD_N_FIELDS 10

/* ISO9660 + UDF = 2 */
#define CDW_IMAGE_WIZARD_FORMATS_MAX 2



static cdw_id_clabel_t cdw_img_wizard_formats[CDW_IMAGE_WIZARD_FORMATS_MAX] = {
	/* 2TRANS: ISO9660 optical image file format. */
	{ CDW_IMAGE_FORMAT_ISO9660,   gettext_noop("ISO9660") },
	/* 2TRANS: UDF optical image file format. */
	{ CDW_IMAGE_FORMAT_UDF,       gettext_noop("UDF") },
};



static cdw_rv_t cdw_image_wizard_init(cdw_task_t *task);
static cdw_rv_t cdw_image_wizard_build(cdw_task_t *task);
static void     cdw_image_wizard_destroy(void);
static int      cdw_image_wizard_driver(void);
static cdw_rv_t cdw_image_wizard_build_fields(cdw_task_t *task);
static cdw_rv_t cdw_image_wizard_validate_and_save(cdw_task_t *task, int *fi);


static int      cdw_image_wizard_options_window_show(cdw_id_t *pid, int *fi);
static cdw_rv_t cdw_image_wizard_options_window_sub(cdw_task_t *task, cdw_id_t *pid, int *fi);
static int      cdw_image_wizard_options_window_by_callback(cdw_form_t *cdw_form, void *dummy);
static int      cdw_image_wizard_options_window_by_flow(cdw_id_t *pid, int *fi);
static int      cdw_image_wizard_options_window_driver(cdw_task_t *task);
static void     cdw_image_wizard_options_window_destroy(void);


static cdw_rv_t cdw_image_wizard_options_save(cdw_task_t *task);
static cdw_rv_t cdw_image_wizard_options_validate(cdw_task_t *task, cdw_id_t *page_id, int *fi);
static cdw_rv_t cdw_image_wizard_options_validate_closed(cdw_task_t *task);


static int cdw_image_wizard_image_file_selector(void *dummy1, void *dummy2);
static int cdw_image_wizard_on_select_image_format(void *dummy1, void *vid);
static void cdw_image_wizard_on_select_image_format_sub(cdw_form_descr_t *volume_line, const char *label_text);

static void cdw_image_wizard_recreate_volume_id_input(cdw_form_t *cdw_form);

static void cdw_image_wizard_debug_at_exit(cdw_task_t *task);
CDW_DROPDOWN *cdw_img_wizard_make_image_format_dropdown(WINDOW *window, int begin_y, int begin_x, int width);


static struct {
	/* Form used in main window of image wizard. Forms used in
	   "Options" window tabs are elsewhere. */
	cdw_form_t *cdw_form;

	/* UDF/ISO9660. */
	cdw_id_t image_format;

	cdw_task_t *task;

	FIELD *fields[CDW_IMAGE_WIZARD_N_FIELDS + 1];

	/* Text displayed at the bottom of Options window. So it
	   doesn't belong to wizard, rather to Options window. */
	char *window_tip;

} wizard;




static cdw_form_dropdown_maker_t main_dropdown_makers[] = {
	cdw_img_wizard_make_image_format_dropdown
};



/* Some layout constraints. */
static const int first_col =  2;  /* Most of widgets start in leftmost column. */
static const int vid_row   =  4;  /* Volume ID row. */


#define window_n_cols             52
#define window_n_lines            20
#define subwindow_n_cols           (window_n_cols - 2)
#define subwindow_n_lines          (window_n_lines - 4)
#define wide_label_n_cols          (subwindow_n_cols - 2)
#define narrow_label_n_cols        (subwindow_n_cols - 15)

/* The "voli" field will contain "Volume Identifier" for ISO9660 or
   "Volume Identifier" for UDF file system. I would like the field to
   display full string (without any visual truncation) for at least
   the shorter of the two IDs.

   The maximal length for UDF vid is much smaller than for ISO9660, so the
   width (not capacity!) of the field will be equal to maximal length
   of UDF volume identifier.

   I'm choosing UDF VID over UDF LVID, because it's VID that is
   visible in Linux's file managers.
*/
#define voli_field_n_cols          CDW_UDF_VID_LEN_MAX





/**
   \brief Ncurses window showing summary of prepared "create image" task

   A window that shows 'You are creating image' message and some basic
   options. User can press 'More options' button to see and change
   other options. Two other buttons available are 'Create' and
   'Cancel'.

   \param task - variable describing current task

   \return CDW_IMAGE_FORMAT_ISO9660 or CDW_IMAGE_FORMAT_UDF if user selects 'Create' button
   \return CDW_IMAGE_FORMAT_NONE if user presses Escape key or selects 'Cancel' button
   \return CDW_IMAGE_FORMAT_ERROR on errors
*/
int cdw_image_wizard(cdw_task_t *task)
{
	cdw_assert (task->id == CDW_TASK_CREATE_IMAGE_ISO9660
		    || task->id == CDW_TASK_CREATE_IMAGE_UDF,
		    "ERROR: incorrect task id %lld\n", task->id);

	cdw_rv_t crv = cdw_image_wizard_init(task);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to init wizard\n");
		cdw_image_wizard_destroy();
		return CDW_IMAGE_FORMAT_ERROR;
	}
	crv = cdw_image_wizard_build(task);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to build wizard\n");
		cdw_image_wizard_destroy();
		cdw_main_window_wrefresh();
		return CDW_IMAGE_FORMAT_ERROR;
	}

	int rv = cdw_image_wizard_driver();

	cdw_image_wizard_destroy();
	cdw_main_window_wrefresh();

	if (rv == CDW_KEY_ENTER) {
		/* Enter == Enter on Create button == proceed with
		   creating image file. */

#ifndef NDEBUG
		cdw_image_wizard_debug_at_exit(task);
#endif
		return (int) wizard.image_format;
	} else {
		cdw_sdm ("INFO: not attempting creating image, not displaying settings\n");
		return CDW_IMAGE_FORMAT_NONE;
	}
}





/**
   \brief Set initial values of some variables global in the file

   Function sets values of various fields of "wizard" variable.
*/
cdw_rv_t cdw_image_wizard_init(cdw_task_t *task)
{
	wizard.task = task;

	wizard.image_format = CDW_IMAGE_FORMAT_NONE;

	wizard.cdw_form = cdw_form_new(CDW_IMAGE_WIZARD_N_FIELDS);
	if (!wizard.cdw_form) {
		cdw_vdm ("ERROR: failed to create cdw form\n");
		return CDW_ERROR;
	}

	wizard.cdw_form->fields = wizard.fields;
	for (int i = 0; i < CDW_IMAGE_WIZARD_N_FIELDS; i++ ) {
		wizard.cdw_form->fields[i] = (FIELD *) NULL;
	}

	wizard.window_tip = (char *) NULL;

	return CDW_OK;
}





int cdw_image_wizard_driver(void)
{
	int fi = f_cancel_b; /* Initial focus on "Cancel" button. */
	int key = 'a';

	while (key != CDW_KEY_ESCAPE && key != 'q' && key != 'Q') {
		key = cdw_form_driver(wizard.cdw_form, fi);

		/* In top-level driver of wizard we are mainly
		   interested in ENTER being pressed on either
		   "create" or "Cancel" button. If user presses ENTER
		   on "Cancel" button, form driver will return ESCAPE
		   key. */
		if (key == CDW_KEY_ENTER) {
			fi = field_index(current_field(wizard.cdw_form->form));
			if (fi != f_create_b) {
				/* We are not on "Create" button, so do nothing, wait for next key. */
				continue;
			}

			/* Flush. */
			form_driver(wizard.cdw_form->form, REQ_VALIDATION);

			/* Check options in closed "Options" window. */
			cdw_rv_t crv = cdw_image_wizard_options_validate_closed(wizard.task);
			if (crv != CDW_OK) {
				return CDW_KEY_ESCAPE;
			}

			/* Check options in main wizard window. */
			crv = cdw_image_wizard_validate_and_save(wizard.task, &fi);
			if (crv == CDW_NO) {
				/* one of option strings in main
				   wizard window is invalid */

				/* We will move focus to field with
				   incorrect data. Remove focus from
				   current widget. */
				CDW_BUTTON *b = cdw_form_get_button(wizard.cdw_form, f_create_b);
				cdw_button_unfocus(b);

				/* 2TRANS: this is title of dialog window */
				cdw_buttons_dialog(_("Error"),
						   /* 2TRANS: this is message in dialog window */
						   _("One of option fields is incorrect or contains character that is not allowed. Please fix it."),
						   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

				cdw_form_driver_go_to_field(wizard.cdw_form, fi);
				/* Loop. */
			} else {
				/* Fields from Options window and from
				   main page of the wizard are
				   validated and saved into task
				   variable. */
				return CDW_KEY_ENTER;
			}
		} else if (key == CDW_KEY_ESCAPE) {
			/* Either "Cancel" button pressed (which is
			   configured to return Escape), or "Escape"
			   key anywhere else. The result is the
			   same. */
			return CDW_KEY_ESCAPE;
		} else {
			/* Some unspecified key. */
		}
	}

	return CDW_KEY_ESCAPE;
}





/**
   \brief Deallocate all resources allocated by "create image" wizard
*/
void cdw_image_wizard_destroy(void)
{
	/* This order of these four function calls minimizes number of
	   problems reported by valgrind. */
	cdw_form_delete_form_objects(wizard.cdw_form);
	cdw_window_delete(&wizard.cdw_form->subwindow);
	cdw_window_delete(&wizard.cdw_form->window);
	cdw_form_delete(&wizard.cdw_form);

	return;
}





/**
   \brief Print to stderr values of \p task fields set in "create image" wizard

   \param task - task variable, in which some fields were set in "create image" wizard
*/
void cdw_image_wizard_debug_at_exit(cdw_task_t *task)
{
	cdw_task_debug_print_options(task, "at image wizard exit");
	return;
}





/**
   \brief Create all UI elements in wizard window

   \param task - variable describing current task
   \param disc - variable describing disc currently in drive

   \return CDW_OK on success
   \return CDW_ERROR on failure
*/
cdw_rv_t cdw_image_wizard_build(cdw_task_t *task)
{
	int begin_y = ((LINES - window_n_lines) / 2) - 2;
	int begin_x = (COLS - window_n_cols) / 2;
	wizard.cdw_form->window =
		cdw_window_new((WINDOW *) NULL,
			       window_n_lines, window_n_cols,
			       begin_y, begin_x,
			       CDW_COLORS_DIALOG,
			       /* 2TRANS: this is title of wizard
				  window; */
			       _("Image wizard"),
			       /* 2TRANS: this is tip at the bottom
				  of window - user can switch between
				  window elements using tab key */
			       _("Use 'Tab' key to move"));

	if (!wizard.cdw_form->window) {
		cdw_vdm ("ERROR: failed to create window\n");
		return CDW_ERROR;
	}

	wizard.cdw_form->subwindow =
		cdw_window_new(wizard.cdw_form->window,
			       subwindow_n_lines, subwindow_n_cols,
			       3, 1,
			       CDW_COLORS_DIALOG,
			       (char *) NULL, (char *) NULL);

	if (!wizard.cdw_form->subwindow) {
		cdw_vdm ("ERROR: failed to create subwindow\n");
		return CDW_ERROR;
	}

	cdw_rv_t crv = cdw_image_wizard_build_fields(task);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to build fields\n");
		return CDW_ERROR;
	}

	wizard.cdw_form->form = cdw_ncurses_new_form(wizard.cdw_form->window,
						     wizard.cdw_form->subwindow,
						     wizard.cdw_form->fields);
	if (!wizard.cdw_form->form) {
		cdw_vdm ("ERROR: failed to create form\n");
		return CDW_ERROR;
	}


	/* Check current value in "image format" dropdown, and make an
	   input field of appropriate width. */
	cdw_image_wizard_recreate_volume_id_input(wizard.cdw_form);

	/* Make sure that cdw widgets are presented properly. */
	cdw_form_redraw_widgets(wizard.cdw_form);


	wrefresh(wizard.cdw_form->subwindow);
	wrefresh(wizard.cdw_form->window);

	return CDW_OK;
}





cdw_rv_t cdw_image_wizard_build_fields(cdw_task_t *task)
{
	int iso_button_col = first_col + wide_label_n_cols - 5;

	cdw_form_descr_t descr[] = {
		/* type                       begin_y    begin_x              n_cols             n_lines   field enum          data1                             data2 */

		/* 2TRANS: this is label next to input field, where user can select format of image: ISO9660 or UDF. */
		{ CDW_WIDGET_ID_STATIC_LABEL,       1,   first_col,           wide_label_n_cols,       1,  f_image_format_l,   _("Image format:"),                  0 },
		{ CDW_WIDGET_ID_DROPDOWN,           2,   first_col,           narrow_label_n_cols,     1,  f_image_format_dd,  main_dropdown_makers,                0 },

		/* 2TRANS: this is label next to input field, where user can enter a label describing created IMAGE volume */
		{ CDW_WIDGET_ID_DYNAMIC_LABEL,      vid_row,     first_col,   wide_label_n_cols,       1,  f_volume_id_l,      _("Volume ID:"),                     0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,    vid_row + 1, first_col,   voli_field_n_cols,       1,  f_volume_id_i,      task->create_image.iso9660.volume_id,    CDW_ISO9660_VOLI_LEN },

		/* 2TRANS: this is label next to input field, where user can specify path to ISO9660 or UDF file on disc. */
		{ CDW_WIDGET_ID_DYNAMIC_LABEL,      7,   first_col,           wide_label_n_cols - 2,   1,  f_image_file_l,     _("Output image file:"),             0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,    8,   first_col,           wide_label_n_cols - 8,   1,  f_image_file_i,     task->image_file_fullpath,           0 },

		/* 2TRANS: do not translate */
		{ CDW_WIDGET_ID_BUTTON,             8,   iso_button_col,      1,                       1,  f_image_file_b,     _(">"),              CDW_COLORS_DIALOG },

		/* 2TRANS: this is label of a button that opens configuration window; "more options" related to creating IMAGE file */
		{ CDW_WIDGET_ID_BUTTON,            11,   first_col,           1,                       1,  f_options_b,        _("More options"),   CDW_COLORS_DIALOG },

		/* 2TRANS: button label, it refers to creating ISO9660 or UDF image. */
		{ CDW_WIDGET_ID_BUTTON,            14,   3,                   2,                       1,  f_create_b,         _("Create"),         CDW_COLORS_DIALOG },
		/* 2TRANS: button label */
		{ CDW_WIDGET_ID_BUTTON,            14,  15,                   2,                       1,  f_cancel_b,         _("Cancel"),         CDW_COLORS_DIALOG },

		/* guard */
		{ -1,                               0,   0,                   0,                       0,  0,                  (void *) NULL,                       0 }};


	wizard.cdw_form->n_fields = CDW_IMAGE_WIZARD_N_FIELDS;

	cdw_rv_t crv = cdw_form_description_to_fields(descr, wizard.cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_ERROR;
	} else {
		cdw_form_add_return_chars(wizard.cdw_form, CDW_KEY_ENTER, CDW_KEY_ESCAPE, 'q', 'Q', 0);

		cdw_form_set_button_return_key(wizard.cdw_form, f_create_b, CDW_KEY_ENTER);
		cdw_form_set_button_return_key(wizard.cdw_form, f_cancel_b, CDW_KEY_ESCAPE);

		cdw_form_set_widget_callback(wizard.cdw_form, f_image_file_b,    (cdw_form_widget_function_t) cdw_image_wizard_image_file_selector);
		cdw_form_set_widget_callback(wizard.cdw_form, f_options_b,       (cdw_form_widget_function_t) cdw_image_wizard_options_window_by_callback);
		cdw_form_set_widget_callback(wizard.cdw_form, f_image_format_dd, (cdw_form_widget_function_t) cdw_image_wizard_on_select_image_format);

		cdw_form_bind_input_and_label(wizard.cdw_form, f_volume_id_i, f_volume_id_l);
		cdw_form_bind_input_and_label(wizard.cdw_form, f_image_file_i, f_image_file_l);

		return CDW_OK;
	}
}





int cdw_image_wizard_options_window_by_callback(__attribute__((unused)) cdw_form_t *cdw_form, __attribute__((unused)) void *dummy)
{
	return cdw_image_wizard_options_window_show((cdw_id_t *) NULL, (int *) NULL);
}





int cdw_image_wizard_options_window_by_flow(cdw_id_t *pid, int *fi)
{
	return cdw_image_wizard_options_window_show(pid, fi);
}





int cdw_image_wizard_options_window_show(cdw_id_t *pid, int *fi)
{
	CDW_DROPDOWN *dd = cdw_form_get_dropdown(wizard.cdw_form, f_image_format_dd);
	cdw_assert (dd, "ERROR: image format dd is NULL (field id = %d)\n", f_image_format_dd);

	/* FIXME: possible error: where is the code that prevents
	   creating wizard when cdw can't find any tools for creating
	   image of any format? The code should be in cdw_task module -
	   creating new task should fail if external_tools module
	   can't provide any tools. */
	cdw_assert (dd->n_items, "ERROR: wizard created when no tools for stand-alone are available\n");

	wizard.image_format = cdw_dropdown_get_current_item_id(dd);
	cdw_vdm ("INFO: image format = %lld\n", wizard.image_format);


	cdw_rv_t crv = cdw_image_wizard_options_window_sub(wizard.task, pid, fi);

	cdw_main_window_wrefresh();

	redrawwin(wizard.cdw_form->window);
	redrawwin(wizard.cdw_form->subwindow);
	wrefresh(wizard.cdw_form->window);
	wrefresh(wizard.cdw_form->subwindow);

	/* the function needs to return int representing a char */
	if (crv == CDW_OK) {
		return '1';
	} else {
		return '0';
	}
}





int cdw_image_wizard_image_file_selector(__attribute__((unused)) void *dummy1, __attribute__((unused)) void *dummy2)
{
	/* 2TRANS: this is title of dialog window */
	cdw_rv_t crv = cdw_fs_ui_file_picker(_("Path to image file"),
					     /* 2TRANS: this is message in dialog window;
					     below it there is an input field where user can
					     enter path to ISO9660 or UDF image file */
					     _("Please enter FULL path to new image file:"),
					     &(wizard.task->image_file_fullpath),
					     CDW_FS_FILE, R_OK | W_OK, CDW_FS_NEW | CDW_FS_EXISTING);

	cdw_main_window_wrefresh();

	redrawwin(wizard.cdw_form->window);
	wrefresh(wizard.cdw_form->window);
	redrawwin(wizard.cdw_form->subwindow);
	wrefresh(wizard.cdw_form->subwindow);

	char *fp = wizard.task->image_file_fullpath;
	if (crv == CDW_OK) {
		int rv = set_field_buffer(wizard.cdw_form->fields[f_image_file_i], 0, fp);
		if (rv != E_OK) {
			cdw_vdm ("ERROR: failed to set field buffer with string = \"%s\"\n", fp);
		} else {
			form_driver(wizard.cdw_form->form, REQ_END_LINE);
		}
		cdw_vdm ("INFO: file picker returns \"%s\"\n", fp);
		return '1';
	} else if (crv == CDW_CANCEL) {
		cdw_vdm ("INFO: pressed escape in file picker, fullpath is \"%s\"\n", fp);
		return '0';
	} else {
		cdw_vdm ("INFO: file picker returns CDW_ERROR, fullpath is \"%s\"\n", fp);
		return '0';
	}
}





/*
  Change of image format involves changing of "volume ID" input
  field. The ID has different maximum length depending on image
  format. I have to change the maximal capacity of the input field
  accordingly.

  While ncurses allows me to increase the capacity (with
  set_max_field()), it doesn't allow me to decrease it. As a
  workaround I have to delete old ncurses field, and create new
  one. But then I have to inform ncurses form that there has been a
  change in fields table.

  This is why I have free_field() -> unpost_form() ->
  set_form_fields() -> post_form().
*/
int cdw_image_wizard_on_select_image_format(__attribute__((unused)) void *dummy1, void *vid)
{
	/* ID of currently selected list item. */
	cdw_id_t *id = (cdw_id_t *) vid;
	cdw_assert (*id == CDW_IMAGE_FORMAT_ISO9660
		    || *id == CDW_IMAGE_FORMAT_UDF,

		    "ERROR: invalid image format %lld\n", *id);

	wizard.image_format = *id;

	CDW_SAFE_INPUT_LINE *line = cdw_form_get_safe_input_line(wizard.cdw_form, f_volume_id_i);

	/* The "" string is used in form description because upon
	   changing capacity of input field from 32 to 30 chars I may
	   truncate the string. In some situations the truncation
	   won't be obvious to the user, so he will notice the fact
	   only after creating the image (or burning the disc).

	   Completely erasing the ID string is more noticeable and
	   will hopefully prompt user to enter correct ID. */

	/* Initial value of .data2 in volume_line doesn't matter, it
	   will be overwritten inside
	   cdw_image_wizard_on_select_image_format_sub() with proper
	   value before volume_line will be used to create new input
	   line. */
	/*                               type                           begin_y      begin_x    n_cols             n_lines   field enum      data1   data2  */
	cdw_form_descr_t volume_line = { CDW_WIDGET_ID_SAFE_INPUT_LINE, vid_row + 1, first_col, voli_field_n_cols, 1,        f_volume_id_i,  _(""),  CDW_UDF_VID_LEN_MAX };

	if (*id == CDW_IMAGE_FORMAT_ISO9660) {

		if (line->chars_max == CDW_ISO9660_VOLI_LEN) {
			/* The input line already has proper width, so
			   we assume that it is an input field
			   suitable for ISO9660 volume label. Don't
			   replace it with new input field. */
		} else {
			cdw_vdm ("INFO: replacing input field with field for ISO9660\n");

			cdw_assert (CDW_ISO9660_VOLI_LEN == 32, "ERROR: wrong value of volume len\n");
			/* Proper length limit (capacity) of input line. */
			volume_line.data2 = CDW_ISO9660_VOLI_LEN;
			cdw_image_wizard_on_select_image_format_sub(&volume_line, "Volume ID (32 chars max)");
		}

		wizard.task->id = CDW_TASK_CREATE_IMAGE_ISO9660;

		/* As the selected image format changes, so should the
		   tool change as well. */
		/* Do the tool selection even if the 'if ()' above is
		   true - this function may be called in wizard
		   initialization, where tool is not set yet. */
		bool sel = cdw_ext_tools_set_iso9660_sa_tool(&wizard.task->create_image.tool);
		cdw_assert (sel, "ERROR: failed to get tool for sa ISO9660 FS\n");
		cdw_vdm ("INFO: TASK ID = ISO9660, tool = %s\n", wizard.task->create_image.tool.label);

	} else { /* Current selection of "Image format" dropdown is UDF. */

		if (line->chars_max == CDW_UDF_VID_LEN_MAX) {
			/* The input line already has proper width, so
			   we assume that it is an input field
			   suitable for UDF volume label. Don't
			   replace it with new input field. */
		} else {
			cdw_vdm ("INFO: replacing input field with field for UDF\n");

			cdw_assert (CDW_UDF_VID_LEN_MAX == 30, "ERROR: wrong value of volume len\n");
			/* Proper length limit (capacity) of input line. */
			volume_line.data2 = CDW_UDF_VID_LEN_MAX;
			cdw_image_wizard_on_select_image_format_sub(&volume_line, "Volume ID (\"vid\", 30 chars max)");
		}

		wizard.task->id = CDW_TASK_CREATE_IMAGE_UDF;

		/* As the selected image format changes, so should the
		   tool change as well. */
		/* Do the tool selection even if the 'if ()' above is
		   true - this function may be called in wizard
		   initialization, where tool is not set yet. */
		bool sel = cdw_ext_tools_set_udf_sa_tool(&wizard.task->create_image.tool);
		cdw_assert (sel, "ERROR: failed to get tool for stand-alone UDF file system\n");
		cdw_vdm ("INFO: TASK ID = UDF, tool = %s\n", wizard.task->create_image.tool.label);
	}

	return 0;
}





void cdw_image_wizard_on_select_image_format_sub(cdw_form_descr_t *volume_line, const char *label_text)
{
	/* First replace old input field with new one, that has
	   correct capacity. The capacity is already configured by
	   caller as volume_line.data2. */
	cdw_form_replace_field(wizard.cdw_form, volume_line, f_volume_id_i, CDW_WIDGET_ID_SAFE_INPUT_LINE, CDW_WIDGET_ID_SAFE_INPUT_LINE);


	/* Update the text of dynamic label widget. It can be done at
	   any time, the widget should accept a new label at any time
	   of its life, regardless if the widget is bound to any input
	   field or not. */
	CDW_DYNAMIC_LABEL *label = cdw_form_get_dynamic_label(wizard.cdw_form, f_volume_id_l);
	cdw_dynamic_label_set_new(label, label_text);
	cdw_form_bind_input_and_label(wizard.cdw_form, f_volume_id_i, f_volume_id_l);


	/* Make sure that all cdw widgets in the form are presented
	   properly. */
	cdw_form_redraw_widgets(wizard.cdw_form);

	int current_fi = field_index(current_field(wizard.cdw_form->form));
	cdw_form_driver_focus_on(wizard.cdw_form, current_fi);

	return;
}





/* Depending on state of "image format" dropdown, set up proper input
   field and label for "volume id". */
void cdw_image_wizard_recreate_volume_id_input(cdw_form_t *cdw_form)
{
	CDW_DROPDOWN *dd = cdw_form_get_dropdown(cdw_form, f_image_format_dd);
	cdw_id_t id = cdw_dropdown_get_current_item_id(dd);

	cdw_image_wizard_on_select_image_format(NULL, &id);

	return;
}






/* some size and layout constants */
/* width and height of main options window can't be smaller
   than size of minimal supported terminal */

/** \brief Width of options window */
#define OPTIONS_WIDTH 80
/** \brief Height of options window */
#define OPTIONS_HEIGHT 24
/** \brief Width of right-hand area with tabs */
#define TABS_WIDTH 23
/** \brief Width of (most) labels in options window */
#define LABEL_WIDTH 30
/** \brief Width of some fields that should be as wide as possible: input
    fields storing file path, some dropdowns, "other options" fields,
    txt subwins, long labels */
#define WIDE_FIELD_WIDTH (OPTIONS_WIDTH - TABS_WIDTH - 4)
/** \brief Window column of first column of items in options page  */
#define FIRST_COL 1
/** \brief Window column of second column of items in options page  */
#define SECOND_COL (FIRST_COL + LABEL_WIDTH + 2)



/* We may create three different "options" windows:

   - first type for mkisofs (with main options and with meta-data read
     from rc file),

   - second type for xorriso (with main options and with meta-data
     read from rc file);

   - third type for mkudffs (with main options (for mkudffs and its
     helpers) and with udf meta-data);

   Each window type has N pages. We need to distinguish the pages with
   these IDs. */
#define IMAGE_PAGE_ID_MKISOFS_ISO_MAIN    0
#define IMAGE_PAGE_ID_MKISOFS_ISO_META    1
#define IMAGE_PAGE_ID_XORRISO_ISO_MAIN    2
#define IMAGE_PAGE_ID_XORRISO_ISO_META    3
#define IMAGE_PAGE_ID_MKUDFFS_UDF_MAIN    4
#define IMAGE_PAGE_ID_MKUDFFS_UDF_META    5


/* this is first main component of options window: a big table
   aggregating data and functions that prepare the data; the data
   represents widgets displayed in pages of options window */
static cdw_options_page_t c_pages[] = {
	/* 2TRANS: label of a tab in configuration window with options related to ISO9660; probably you don't want to translate this */
	{ false, CDW_MKISOFS_ISO_MAIN_OPTIONS_N_FIELDS,   cdw_mkisofs_options_form,       gettext_noop("ISO9660"),   (cdw_form_t *) NULL },
	/* 2TRANS: label of a tab in configuration window with options related to meta information of ISO9660 file system */
	{ false, CDW_MKISOFS_ISO_META_OPTIONS_N_FIELDS,   cdw_mkisofsrc_options_form,     gettext_noop("Meta data"), (cdw_form_t *) NULL },
	/* 2TRANS: label of a tab in configuration window with options related to ISO9660; probably you don't want to translate this */
	{ false, CDW_XORRISO_ISO_MAIN_OPTIONS_N_FIELDS,   cdw_xorriso_iso_options_form,   gettext_noop("ISO9660"),   (cdw_form_t *) NULL },
	/* 2TRANS: label of a tab in configuration window with options related to meta information of ISO9660 file system */
	{ false, CDW_XORRISO_ISO_META_OPTIONS_N_FIELDS,   cdw_xorrisorc_options_form,     gettext_noop("Meta data"), (cdw_form_t *) NULL },
	/* 2TRANS: label of a tab in configuration window with options related to UDF; probably you don't want to translate this */
	{ false, CDW_MKUDFFS_UDF_MAIN_OPTIONS_N_FIELDS,   cdw_mkudffs_options_form_p1,    gettext_noop("UDF tools"), (cdw_form_t *) NULL },
	/* 2TRANS: label of a tab in configuration window with options related to meta information of UDF file system */
	{ false, CDW_MKUDFFS_UDF_META_OPTIONS_N_FIELDS,   cdw_mkudffs_options_form_p2,    gettext_noop("Meta data"), (cdw_form_t *) NULL }
};


/* this is a pointer to second main component of options window:
   tabbed window, which enables switching between pages */
static cdw_tabs_window_t *c_twindow = (cdw_tabs_window_t *) NULL;




cdw_rv_t cdw_image_wizard_options_window_sub(cdw_task_t *task, cdw_id_t *pid, int *fi)
{
	cdw_assert (wizard.image_format == CDW_IMAGE_FORMAT_ISO9660
		    || wizard.image_format == CDW_IMAGE_FORMAT_UDF,

		    "ERROR: invalid image format %lld\n", wizard.image_format);

	/* 2TRANS: this is message at the bottom of options window;
	   cancel means: quit without saving; '%d' is an integer used to
	   create label of function key, e.g. "F10" */
	int rv = asprintf(&wizard.window_tip,
			  _("Press F%d / F%d key to save changes or ESC to cancel"),
			  CDW_CONFIG_UI_SnC_KEY_B, CDW_CONFIG_UI_SnC_KEY_A);
	if (!wizard.window_tip || rv == -1) {
		cdw_vdm ("ERROR: failed to create window tip\n");
		return CDW_ERROR;
	}

	int n_tabs = 0;
	if (wizard.image_format == CDW_IMAGE_FORMAT_ISO9660) {
		n_tabs = 2; /* mkisofs|xorriso + mkisofsrc|xorrisorc */
		/* 2TRANS: this is title of cdw options main window */
		c_twindow = cdw_tabs_window_init(n_tabs, _("ISO9660 options"), wizard.window_tip);
		cdw_tabs_window_set_geometry(c_twindow, OPTIONS_HEIGHT, OPTIONS_WIDTH, 4, 4, TABS_WIDTH);


		/* prepare tabbed window */
		cdw_id_t id = 0;
		int ind = 0;
		/* tool-specific options: mkisofs or xorriso */
		if (task->create_image.tool.id == CDW_TOOL_MKISOFS) {
			id = IMAGE_PAGE_ID_MKISOFS_ISO_MAIN;
			cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);
			ind++;

			id = IMAGE_PAGE_ID_MKISOFS_ISO_META;
			cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);
			ind++;
		} else {
			id = IMAGE_PAGE_ID_XORRISO_ISO_MAIN;
			cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);
			ind++;

			id = IMAGE_PAGE_ID_XORRISO_ISO_META;
			cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);
			ind++;
		}
	} else { /* wizard.image_format == CDW_IMAGE_FORMAT_UDF */
		n_tabs = 2; /* mkudffs main options + UDF meta data. */
		/* 2TRANS: this is title of cdw options main window */
		c_twindow = cdw_tabs_window_init(n_tabs, _("UDF options"), wizard.window_tip);
		cdw_tabs_window_set_geometry(c_twindow, OPTIONS_HEIGHT, OPTIONS_WIDTH, 4, 4, TABS_WIDTH);

		/* prepare tabbed window */
		int ind = 0;
		cdw_id_t id = IMAGE_PAGE_ID_MKUDFFS_UDF_MAIN;
		cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);
		ind++;

		id = IMAGE_PAGE_ID_MKUDFFS_UDF_META;
		cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);
		ind++;
	}


	cdw_tabs_window_add_return_keys(c_twindow,
					KEY_F(CDW_CONFIG_UI_SnC_KEY_A), KEY_F(CDW_CONFIG_UI_SnC_KEY_B),
					CDW_KEY_ESCAPE, 'q', 'Q', 0);
	c_twindow->driver_reader = cdw_config_ui_tabs_window_form_reader;
	cdw_tabs_window_finalize(c_twindow);

	/* prepare forms in the tabbed window */
	for (int i = 0; i < n_tabs; i++) {
		cdw_id_t id = c_twindow->tabs[i].id;

		c_pages[id].cdw_form = cdw_form_new(c_pages[id].n_fields);
		c_pages[id].cdw_form->form_id = id;
		c_pages[id].cdw_form->window = c_twindow->tabs[i].window;
		c_pages[id].cdw_form->subwindow = c_twindow->tabs[i].subwindow;

		c_pages[id].visible = true;

		cdw_rv_t crv;
		if (id == IMAGE_PAGE_ID_MKUDFFS_UDF_MAIN || id == IMAGE_PAGE_ID_MKUDFFS_UDF_META) {
			crv = c_pages[id].fields_builder(c_pages[id].cdw_form, &(task->create_image.udf), FIRST_COL, SECOND_COL, WIDE_FIELD_WIDTH, LABEL_WIDTH);
		} else {
			crv = c_pages[id].fields_builder(c_pages[id].cdw_form, &(task->create_image.iso9660), FIRST_COL, SECOND_COL, WIDE_FIELD_WIDTH, LABEL_WIDTH);
		}
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to build fields in page #%d\n", i);
			return CDW_ERROR;;
		}

		c_pages[id].cdw_form->form = cdw_ncurses_new_form(c_pages[id].cdw_form->window,
								  c_pages[id].cdw_form->subwindow,
								  c_pages[id].cdw_form->fields);

		for (int j = 0; j < n_tabs; j++) {
			cdw_form_add_return_char(c_pages[id].cdw_form, KEY_F(j + CDW_CONFIG_UI_FX_START_KEY));
		}
		cdw_form_add_return_char(c_pages[id].cdw_form, KEY_F(CDW_CONFIG_UI_SnC_KEY_A));
		cdw_form_add_return_char(c_pages[id].cdw_form, KEY_F(CDW_CONFIG_UI_SnC_KEY_B));
		cdw_form_add_return_char(c_pages[id].cdw_form, 'q'); /* will work only for non-text fields */
	}
	c_twindow->user_data = (void *) c_pages;

	cdw_tabs_window_draw_tabs(c_twindow);
	cdw_config_ui_tabs_window_form_draw(c_twindow);

	if (pid && fi
	    && *pid != -1 && *fi != -1) {

		cdw_tabs_window_show_tab_by_id(c_twindow, *pid);
		cdw_form_driver_go_to_field(c_pages[*pid].cdw_form, *fi);
	}

        int key = cdw_image_wizard_options_window_driver(task);
	cdw_image_wizard_options_window_destroy();
	if (key == CDW_KEY_ENTER) {
		return CDW_OK;
	} else {
		return CDW_CANCEL;
	}
}





/**
   \brief Destructor function for cdw options window
*/
void cdw_image_wizard_options_window_destroy(void)
{
	if (wizard.window_tip) {
		free(wizard.window_tip);
		wizard.window_tip = (char *) NULL;
	}

	for (int i = 0; i < c_twindow->n_tabs; i++) {
		cdw_id_t id = c_twindow->tabs[i].id;
		if (c_pages[id].cdw_form) {
			cdw_form_delete_form_objects(c_pages[id].cdw_form);
		}
		if (c_pages[id].cdw_form) {
			cdw_form_delete(&(c_pages[id].cdw_form));
		}
	}

	cdw_tabs_window_delete(&c_twindow);

	return;
}





/**
   \brief Function reacting to keys pressed when options window is displayed

   Function catches all keys pressed when options window is displayed,
   and reacts to them either by switching pages, or by passing the keys to
   page driver(s).

   Keys configured as hotkeys for "tabbed window" tabs are used to switch
   between pages. F(CDW_CONFIG_SnC_KEY) key is interpreted as "save and close"
   key - values of all option fields are stored in \p config and validated.
   Function then returns CDW_OK if validation is successful, or displays
   incorrect field to user.
   ESCAPE key causes the function to return with CDW_KEY_ESCAPE.

   \param config - variable in which function saves values of all option fields

   \return CDW_KEY_ESCAPE if user pressed ESCAPE key in options window
   \return CDW_KEY_ENTER if user pressed F10 and validation of \p tmp config was successful
*/
int cdw_image_wizard_options_window_driver(cdw_task_t *task)
{
	while (1) {
		int key = cdw_tabs_window_driver(c_twindow);
		if (key == CDW_KEY_ESCAPE || key == 'q' || key == 'Q') {
			break;
		} else if (key == KEY_F(CDW_CONFIG_UI_SnC_KEY_A)
			   || key == KEY_F(CDW_CONFIG_UI_SnC_KEY_B)) { /* SnC = Save and Close */

			/* flush */
			for (int i = 0; i < c_twindow->n_tabs; i++) {
				cdw_id_t id = c_twindow->tabs[i].id;
				form_driver(c_pages[id].cdw_form->form, REQ_VALIDATION);
			}
			cdw_id_t page_id = 0;
			int fi = 0;
			cdw_image_wizard_options_save(task);
			cdw_rv_t valid = cdw_image_wizard_options_validate(task, &page_id, &fi);
			if (valid == CDW_NO) { /* some option field is invalid */

				/* 2TRANS: this is title of dialog window */
				cdw_buttons_dialog(_("Error"),
						   /* 2TRANS: this is message in dialog window */
						   _("One of option fields is incorrect or contains character that is not allowed. Please fix it."),
						   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

				cdw_tabs_window_show_tab_by_id(c_twindow, page_id);
				cdw_form_driver_go_to_field(c_pages[page_id].cdw_form, fi);

				/* loop */
			} else {
				/* options from window with advanced
				   options are validated and saved into
				   task variable */
				return CDW_KEY_ENTER;
			}
		} else {
			/* loop */
		}
	} /* while (1) */

	return CDW_KEY_ESCAPE;
}





cdw_rv_t cdw_image_wizard_options_save(cdw_task_t *task)
{
	cdw_id_t pid;

	if (wizard.image_format == CDW_IMAGE_FORMAT_ISO9660) {
		if (task->create_image.tool.id == CDW_TOOL_MKISOFS) {

			pid = IMAGE_PAGE_ID_MKISOFS_ISO_MAIN;
			cdw_mkisofs_options_save(c_pages[pid].cdw_form, &(task->create_image.iso9660));

			pid = IMAGE_PAGE_ID_MKISOFS_ISO_META;
			cdw_mkisofsrc_options_save(c_pages[pid].cdw_form, &(task->create_image.iso9660));
		} else if (task->create_image.tool.id == CDW_TOOL_XORRISO) {

			pid = IMAGE_PAGE_ID_XORRISO_ISO_MAIN;
			cdw_xorriso_iso_options_save(c_pages[pid].cdw_form, &(task->create_image.iso9660));

			pid = IMAGE_PAGE_ID_XORRISO_ISO_META;
			cdw_xorrisorc_options_save(c_pages[pid].cdw_form, &(task->create_image.iso9660));
		} else {
			cdw_assert (0, "ERROR: invalid tool id for task \"create image\": %s (%lld)\n",
				    cdw_ext_tools_get_tool_name(task->create_image.tool.id),
				    task->create_image.tool.id);
			return CDW_NO;
		}

	} else {
		pid = IMAGE_PAGE_ID_MKUDFFS_UDF_MAIN;
		cdw_mkudffs_options_save_p1(&(task->create_image.udf), c_pages[pid].cdw_form);
		pid = IMAGE_PAGE_ID_MKUDFFS_UDF_META;
		cdw_mkudffs_options_save_p2(&(task->create_image.udf), c_pages[pid].cdw_form);
	}

	return CDW_OK;
}





/**
   \brief Check options for creating image

   Check options used to create ISO9660/UDF image file that *would* be
   displayed in options window. The function should be called when
   "Options" window has not been displayed, i.e. when user pressed
   "Create" button in wizard without visiting "Options" window.

   Some options in such non-visited window may be invalid, this
   function should detect this.

   First it tests contents of task variable, and if there is anything
   wrong with any of the options related to creating image, it creates
   and display options window, with cursor placed in problematic
   field.

   User is given a chance to fix the options in the window. User may
   press Escape key in the window, the function will then return
   CDW_NO. The function also returns CDW_NO if it decides that for
   some reason they weren't fixed by user.

   \param task - task which needs to be validated

   \return CDW_OK if options for creating image are valid
   \return CDW_NO if the options are invalid, or if user has pressed Escape key in options window
*/
cdw_rv_t cdw_image_wizard_options_validate_closed(cdw_task_t *task)
{
	cdw_id_t pid;
	int fi;
	cdw_rv_t valid = cdw_image_wizard_options_validate(task, &pid, &fi);
	if (valid == CDW_NO) { /* some option field is invalid */

		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("One of option fields is incorrect or contains character that is not allowed. Please fix it."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

		int c = cdw_image_wizard_options_window_by_flow(&pid, &fi);
		if (c == '0') {
			return CDW_NO;
		} else {
			/* c == '1', invalid options have been fixed
			   and user pressed "Save and Close"
			   key. Validate again, just to be sure. */
			valid = cdw_image_wizard_options_validate(task, &pid, &fi);
			if (valid == CDW_NO) {
				/* 2TRANS: this is title of dialog window */
				cdw_buttons_dialog(_("Error"),
						   /* 2TRANS: this is message in dialog window */
						   _("The invalid option(s) in options window has not been fixed."),
						   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
				return CDW_NO;
			}
		}
	} else {
		/* Options are valid and user has pressed "save and close" key. */
	}

	return CDW_OK;
}





cdw_rv_t cdw_image_wizard_options_validate(cdw_task_t *task, cdw_id_t *pid, int *fi)
{
	cdw_rv_t crv = CDW_NO;
	if (wizard.image_format == CDW_IMAGE_FORMAT_ISO9660) {
		if (task->create_image.tool.id == CDW_TOOL_MKISOFS) {

			*pid = IMAGE_PAGE_ID_MKISOFS_ISO_MAIN;
			crv = cdw_mkisofs_options_validate(&(task->create_image.iso9660), fi);
			if (crv == CDW_NO) {
				return CDW_NO;
			}

			*pid = IMAGE_PAGE_ID_MKISOFS_ISO_META;
			crv = cdw_mkisofsrc_options_validate(&(task->create_image.iso9660), fi);
			if (crv == CDW_NO) {
				return CDW_NO;
			}


		} else if (task->create_image.tool.id == CDW_TOOL_XORRISO) {

			*pid = IMAGE_PAGE_ID_XORRISO_ISO_MAIN;
			crv = cdw_xorriso_iso_options_validate(&(task->create_image.iso9660), fi);
			if (crv == CDW_NO) {
				return CDW_NO;
			}

			*pid = IMAGE_PAGE_ID_XORRISO_ISO_META;
			crv = cdw_xorrisorc_options_validate(&(task->create_image.iso9660), fi);
			if (crv == CDW_NO) {
				return CDW_NO;
			}
		} else {
			cdw_assert (0, "ERROR: invalid tool id for task \"create image\": %lld\n",
				    task->create_image.tool.id);
			return CDW_NO;
		}

	} else {
		*pid = IMAGE_PAGE_ID_MKUDFFS_UDF_MAIN;
		crv = cdw_mkudffs_options_validate_p1(&(task->create_image.udf), fi);
		if (crv == CDW_NO) {
			return CDW_NO;
		}

		*pid = IMAGE_PAGE_ID_MKUDFFS_UDF_META;
		crv = cdw_mkudffs_options_validate_p2(&(task->create_image.udf), fi);
		if (crv == CDW_NO) {
			return CDW_NO;
		}
	}

	return CDW_OK;
}





/**
   \brief Validate and save values of options
   available in main page of the wizard

   Values of options will be saved in \p task.
   If any value is invalid, index of corresponding field
   will be saved in \fi, and function will return CDW_NO.

   \param task - variable describing current task
   \param fi - field index

   \return CDW_OK on success
   \return CDW_NO if any value in main wizard page is invalid
*/
cdw_rv_t cdw_image_wizard_validate_and_save(cdw_task_t *task, int *fi)
{
	/* First validate. */
	/* FIXME: review whole code base and replace field_buffer()
	   with cdw_ncurses_get_field_string() where possible, so that
	   we get a nicely trimmed string. */
	const char *s = field_buffer(wizard.cdw_form->fields[f_volume_id_i], 0);
	cdw_rv_t crv = cdw_string_security_parser(s, (char *) NULL);
	if (crv == CDW_NO) {
		*fi = f_volume_id_i;
		return CDW_NO;
	}

	s = cdw_ncurses_get_field_string(wizard.cdw_form->fields[f_image_file_i]);
	crv = cdw_string_security_parser(s, (char *) NULL);
	if (crv == CDW_NO) {
		*fi = f_image_file_i;
		return CDW_NO;
	}

	char *canonical_fullpath = canonicalize_filename_mode(s, CAN_MISSING);
	if (!canonical_fullpath) {
		/* canonicalization of path failed */
		*fi = f_image_file_i;
		return CDW_NO;
	} else {
		cdw_vdm ("INFO: canonicalized fullpath is \"%s\"\n", canonical_fullpath);

		int fs_check = cdw_fs_check_fullpath(canonical_fullpath, CDW_FS_FILE, W_OK | R_OK, CDW_FS_EXISTING);

		free(canonical_fullpath);
		canonical_fullpath = (char *) NULL;
		if (fs_check & CDW_FS_CHECK_EXISTS) {
			cdw_rv_t eh = cdw_fs_errno_handler(EEXIST);
			if (eh == CDW_OK) {
				; /* Pass to next check. */
			} else {
				/* Don't overwrite file, allow user to modify file fullpath. */
				*fi = f_image_file_i;
				return CDW_NO;
			}
		}
	}



	/* Save only when valid. */
	s = cdw_form_get_string(wizard.cdw_form, f_volume_id_i);
	if (wizard.image_format == CDW_IMAGE_FORMAT_ISO9660) {
		strncpy(task->create_image.iso9660.volume_id, s, CDW_ISO9660_VOLI_LEN);
		task->create_image.iso9660.volume_id[CDW_ISO9660_VOLI_LEN] = '\0';

	} else { /* CDW_IMAGE_FORMAT_UDF */
		strncpy(task->create_image.udf.mkudffs_vid, s, CDW_UDF_VID_LEN_MAX);
		task->create_image.udf.mkudffs_vid[CDW_UDF_VID_LEN_MAX] = '\0';
	}

	s = cdw_form_get_string(wizard.cdw_form, f_image_file_i);
	cdw_string_set(&(task->image_file_fullpath), s);

	return CDW_OK;
}





/**
   Dropdown list to select image file format: ISO9660 or UDF.
*/
CDW_DROPDOWN *cdw_img_wizard_make_image_format_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	/* Decide on default item in dropdown, taking into
	   consideration an ID of the task and availability of
	   tools. */
	cdw_id_t dd_id = CDW_IMAGE_FORMAT_ISO9660;     /* Initial value. */
	if (wizard.task->id == CDW_TASK_CREATE_IMAGE_ISO9660) {
		if (cdw_ext_tools_get_n_iso9660_handlers() > 0) {
			dd_id = CDW_IMAGE_FORMAT_ISO9660;
		} else {
			if (cdw_ext_tools_udf_sa_tools_tools_available()) {
				dd_id = CDW_IMAGE_FORMAT_UDF;
			} else {
				cdw_assert (0, "ERROR: no tools available for creating sa image, the function shouldn't have been called at all\n");
				return (CDW_DROPDOWN *) NULL;
			}
		}
	} else if (wizard.task->id == CDW_TASK_CREATE_IMAGE_UDF) {
		if (cdw_ext_tools_udf_sa_tools_tools_available()) {
			dd_id = CDW_IMAGE_FORMAT_UDF;
		} else {
			if (cdw_ext_tools_get_n_iso9660_handlers() > 0) {
				dd_id = CDW_IMAGE_FORMAT_ISO9660;
			} else {
				cdw_assert (0, "ERROR: no tools available for creating sa image, the function shouldn't have been called at all\n");
				return (CDW_DROPDOWN *) NULL;
			}
		}
	} else {
		cdw_assert (0, "ERROR: invalid task id %lld\n", wizard.task->id);
		return (CDW_DROPDOWN *) NULL;
	}


	cdw_id_clabel_t items[CDW_IMAGE_WIZARD_FORMATS_MAX];
	int n_items = 0;


	/* Put available options into dropdown. */

	if (cdw_ext_tools_get_n_iso9660_handlers() > 0) {

		/* There is some tool needed to create ISO9660 image -
		   either mkisofs (or its replacement genisoimage), or
		   xorriso. */

		items[n_items].id = cdw_img_wizard_formats[CDW_IMAGE_FORMAT_ISO9660].id;
		items[n_items].label = cdw_img_wizard_formats[CDW_IMAGE_FORMAT_ISO9660].label;
		n_items++;
	}

	if (cdw_ext_tools_udf_sa_tools_tools_available()) {
		/* Full set of tools needed to create UDF file is present.
		   FIXME: the "1" above is only temporary - the list will be longer. */

		items[n_items].id = cdw_img_wizard_formats[CDW_IMAGE_FORMAT_UDF].id;
		items[n_items].label = cdw_img_wizard_formats[CDW_IMAGE_FORMAT_UDF].label;
		n_items++;
	}

	CDW_DROPDOWN *dd = cdw_dropdown_maker_wrapper(window, begin_y, begin_x, width, n_items, items);
	cdw_dropdown_set_current_item_by_id(dd, dd_id);

	return dd;
}
