/*
** (c) 1996-2000 The Regents of the University of California (through
** E.O. Lawrence Berkeley National Laboratory), subject to approval by
** the U.S. Department of Energy.  Your use of this software is under
** license -- the license agreement is attached and included in the
** directory as license.txt or you may contact Berkeley Lab's Technology
** Transfer Department at TTD@lbl.gov.  NOTICE OF U.S. GOVERNMENT RIGHTS.
** The Software was developed under funding from the U.S. Government
** which consequently retains certain rights as follows: the
** U.S. Government has been granted for itself and others acting on its
** behalf a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, and perform publicly
** and display publicly.  Beginning five (5) years after the date
** permission to assert copyright is obtained from the U.S. Department of
** Energy, and subject to any subsequent five (5) year renewals, the
** U.S. Government is granted for itself and others acting on its behalf
** a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, distribute copies to
** the public, perform publicly and display publicly, and to permit
** others to do so.
*/


//
// $Id: AmrDeriveIntegrate.cpp,v 1.17 2002/03/28 01:34:18 vince Exp $
//
// This is a version of AmrDerive.cpp that calculates integrals of
// quantities and writes out scalars instead of plotfiles.
//

#include <new>
#include <iostream>
#include <cstdlib>
#include <cstring>

#include <unistd.h>

#include "REAL.H"
#include "Box.H"
#include "FArrayBox.H"
#include "ParmParse.H"
#include "ParallelDescriptor.H"
#include "DataServices.H"
#include "Utility.H"
#include "VisMF.H"
#include "Derived.H"

static
void
PrintUsage (char* progName)
{
    std::cout << "\nUsage:\n"
         << progName
         << "\n\tinfile = inputFileName"
         << "\n\t[-help]"
         << "\n\n";
    exit(1);
}

static
Real
SumThisComp(AmrData &amrData, const int ncomp)
{
    Real sum(0.0);
    int finest_level(amrData.FinestLevel());
    FArrayBox fab;

    for(int iLevel(0); iLevel <= finest_level; ++iLevel) {
        const Real *dx = amrData.DxLevel()[iLevel].dataPtr();

	MultiFab &adGrids = amrData.GetGrids(iLevel, ncomp);
        MFIter ntmfi(adGrids);

        for( ; ntmfi.isValid(); ++ntmfi) {
            //
            // Make copy of FAB so that we can change it.
            //
            fab.resize(ntmfi.fabbox(), 1);

            fab.copy(adGrids[ntmfi], 0, 0, 1);  // src has only one component

            if(iLevel < finest_level) {
              const BoxArray& f_box = amrData.boxArray(iLevel+1);

              for(int j(0); j < f_box.size(); ++j) {
                Box c_box = BoxLib::coarsen(f_box[j],amrData.RefRatio()[iLevel]);
                c_box &= ntmfi.validbox();
                if(c_box.ok()) {
                  fab.setVal(0.0, c_box, 0);
		}
              }
            }
            Real s;
            const Real *dat = fab.dataPtr();
            const int *dlo  = fab.loVect();
            const int *dhi  = fab.hiVect();
            const int *lo   = ntmfi.validbox().loVect();
            const int *hi   = ntmfi.validbox().hiVect();
            int nz          = hi[1]-lo[1]+1;
            Array<Real> tmp(nz);
            FORT_SUMMASS(dat, ARLIM(dlo), ARLIM(dhi), ARLIM(lo), ARLIM(hi),
                         dx, &s, tmp.dataPtr());
            sum += s;
        }
    }

    ParallelDescriptor::ReduceRealSum(sum);

    amrData.FlushGrids(ncomp);

    return sum;
}

static
Real
SumConc (Array<MultiFab*>& mfout,
         AmrData&          amrData)
{
    Real sum = 0;
    //
    // The component number of concentration in the MultiFabs.
    //
    const int ncomp = 0;

    int finest_level = mfout.size() - 1;

    FArrayBox fab;

    for(int iLevel = 0; iLevel <= finest_level; ++iLevel)
    {
        const Real* dx = amrData.DxLevel()[iLevel].dataPtr();

        //MultiFabIterator ntmfi(*mfout[iLevel]);
        MFIter ntmfi(*mfout[iLevel]);

        for( ; ntmfi.isValid(); ++ntmfi)
        {
            //
            // Make copy of FAB so that we can change it.
            //
            fab.resize(ntmfi.fabbox(), 1);

            //fab.copy(ntmfi(), ncomp, 0, 1);
            fab.copy((*mfout[iLevel])[ntmfi], ncomp, 0, 1);

            if(iLevel < finest_level)
            {
                const BoxArray& f_box = mfout[iLevel+1]->boxArray();

                for(int j = 0; j < f_box.size(); j++)
                {
                    Box c_box = BoxLib::coarsen(f_box[j],amrData.RefRatio()[iLevel]);
                    c_box &= ntmfi.validbox();
                    if(c_box.ok())
                        fab.setVal(0.0, c_box, 0);
                }
            }
            Real s;
            const Real* dat = fab.dataPtr();
            const int* dlo  = fab.loVect();
            const int* dhi  = fab.hiVect();
            const int* lo   = ntmfi.validbox().loVect();
            const int* hi   = ntmfi.validbox().hiVect();
            int nz          = hi[1]-lo[1]+1;
            Array<Real> tmp(nz);
            FORT_SUMCONC(dat,ARLIM(dlo),ARLIM(dhi),ARLIM(lo),ARLIM(hi),
                         dx,&s,tmp.dataPtr());
            sum += s;
        }
    }

    ParallelDescriptor::ReduceRealSum(sum);

    return sum;
}

static
Real
SumVort (Array<MultiFab*>& mfout,
         AmrData&          amrData)
{
    Real sum = 0;
    //
    // The component number of vorticity in the MultiFabs.
    //
    const int ncomp = 1;

    int finest_level = mfout.size() - 1;

    FArrayBox fab;

    for(int iLevel = 0; iLevel <= finest_level; ++iLevel)
    {
        const Real* dx = amrData.DxLevel()[iLevel].dataPtr();

        MFIter ntmfi(*mfout[iLevel]);

        for( ; ntmfi.isValid(); ++ntmfi)
        {
            //
            // Make copy of FAB so that we can change it.
            //
            fab.resize(ntmfi.fabbox(), 1);

            //fab.copy(ntmfi(), ncomp, 0, 1);
            fab.copy((*mfout[iLevel])[ntmfi], ncomp, 0, 1);

            if(iLevel < finest_level)
            {
                const BoxArray& f_box = mfout[iLevel+1]->boxArray();

                for(int j = 0; j < f_box.size(); j++)
                {
                    Box c_box = BoxLib::coarsen(f_box[j],amrData.RefRatio()[iLevel]);
                    c_box &= ntmfi.validbox();
                    if(c_box.ok())
                        fab.setVal(0.0, c_box, 0);
                }
            }
            Real s;
            const Real* dat = fab.dataPtr();
            const int* dlo  = fab.loVect();
            const int* dhi  = fab.hiVect();
            const int* lo   = ntmfi.validbox().loVect();
            const int* hi   = ntmfi.validbox().hiVect();
            int nz          = hi[1]-lo[1]+1;
            Array<Real> tmp(nz);
            FORT_SUMVORT(dat,ARLIM(dlo),ARLIM(dhi),ARLIM(lo),ARLIM(hi),
                         dx,&s,tmp.dataPtr());
            sum += s;
        }
    }

    ParallelDescriptor::ReduceRealSum(sum);

    return sum;
}

int
main (int   argc,
      char* argv[])
{
    BoxLib::Initialize(argc,argv);

    if(argc == 1)
        PrintUsage(argv[0]);

    int iLevel;
    ParallelDescriptor::StartParallel(&argc, &argv);

    ParmParse pp;

    if(pp.contains("help"))
        PrintUsage(argv[0]);

    FArrayBox::setFormat(FABio::FAB_IEEE_32);
    //
    // Set default info.
    //
    Real gam = 1.667;
    FORT_SETGAMMA(&gam);
    //
    // Scan the arguments.
    //
    std::string iFile;
    pp.query("infile", iFile);
    if(iFile.empty()) {
      BoxLib::Abort("You must specify `infile'");
    }

    const int ntype = 0;
    //Array<std::string> derives(ntype);
    Array<std::string> derives(1);
    derives[0] = "logden";
    //derives[1] = "vorticity";

    DataServices::SetBatchMode();
    FileType fileType(NEWPLT);
    
    DataServices dataServices(iFile, fileType);

    if( ! dataServices.AmrDataOk()) {
        // This calls ParallelDescriptor::EndParallel() and exit()
        DataServices::Dispatch(DataServices::ExitRequest, NULL);
    }
    AmrData &amrData = dataServices.AmrDataRef();

    BL_ASSERT(amrData.NComp() == NSTATECOMPS);

    int finestLevel = amrData.FinestLevel();

    Array<MultiFab *> mfout(finestLevel + 1);

    FArrayBox srcFab;

    if(ntype > 0) {
      for(iLevel = 0; iLevel <= finestLevel; ++iLevel) {
        const BoxArray& ba = amrData.boxArray(iLevel);

        mfout[iLevel] = new MultiFab(ba, ntype, 0);

        MultiFab& mfo = *mfout[iLevel];

        for(int nt = 0; nt < ntype; ++nt) {
            int whichDerived  = -1;

            for(int nder = 0; nder < NumDeriveFunc; ++nder) {
                if(derives[nt] == DerivedQuants[nder].name) {
                    whichDerived = nder;
                    break;
                }
            }
            BL_ASSERT(whichDerived >= 0);

            Array<MultiFab*> mfin(amrData.NComp());
            int validComp  = -1;
            for(int iComp = 0; iComp < NSTATECOMPS; ++iComp) {
                if(DerivedQuants[whichDerived].derMap[iComp] == 1) {
                    mfin[iComp] = &amrData.GetGrids(iLevel, iComp);
                    validComp = iComp;
                }
            }
            BL_ASSERT(validComp >= 0);

            MFIter ntmfi(*mfin[validComp]);

            for( ; ntmfi.isValid(); ++ntmfi) {
                int ng = ntmfi.index();

                srcFab.resize(ntmfi.validbox(), NSTATECOMPS);

                for(int iComp = 0; iComp < NSTATECOMPS; ++iComp) {
                    if(DerivedQuants[whichDerived].derMap[iComp] == 1) {
                        srcFab.copy((*mfin[iComp])[ng], 0, iComp, 1);
                    }
                }

                const Real* srcptr = srcFab.dataPtr();
                const int*  losrc  = srcFab.loVect();
                const int*  hisrc  = srcFab.hiVect();
                FArrayBox& destFab = mfo[ng];
                Real* destptr      = destFab.dataPtr(nt);
                const int* lodest  = destFab.loVect();
                const int* hidest  = destFab.hiVect();

                DerivedQuants[whichDerived].func(srcptr,
                                        losrc[0],losrc[1],losrc[2],
                                        hisrc[0],hisrc[1],hisrc[2],
                                        amrData.NComp(),
                                        amrData.GridLocLo()[iLevel][ng].dataPtr(),
                                        amrData.GridLocHi()[iLevel][ng].dataPtr(),
                                        destptr,
                                        lodest[0],lodest[1],lodest[2],
                                        hidest[0],hidest[1],hidest[2],
                                        lodest,hidest);
            }
        }
      }
    }  // end if(ntype > 0)

    for(int iYSum(6); iYSum <= 11; ++iYSum) {  // just do rho.Y(iYSum)
      const std::string sCompName = amrData.PlotVarNames()[iYSum];
      Real sumComp = SumThisComp(amrData, iYSum);
      //Real sum_conc = SumConc(mfout,amrData);
      //Real sum_vort = SumVort(mfout,amrData);

      if(ParallelDescriptor::IOProcessor()) {
        const Array<Real>& size = amrData.ProbSize();

        Real volume = D_TERM(size[0],*size[1],*size[2]);

        BL_ASSERT(volume > 0);

        std::cout << "TIME = " << amrData.Time()
             << "  " << sCompName << " = "
             << sumComp
             //<< ','
             //<< sum_conc
             //<< ','
             //<< sum_vort/volume
             << '\n';
      }
    }

    //DataServices::Dispatch(DataServices::ExitRequest, NULL);
    BoxLib::Finalize();
    return 0;

}
