#ifndef __AUTHNLIB_OPENSSLUTIL_H__
#define __AUTHNLIB_OPENSSLUTIL_H__

#include <pthread.h>
#include <list>
#include <map>
#include <string>
#include <openssl/asn1.h>
#include <openssl/pem.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>
#include <openssl/pkcs12.h>
#include <openssl/err.h>
#include <openssl/ocsp.h>

#include "cache.h"
#include "datetime.h"
#include "canlxx.h"
#include "opensslextutil.h"

namespace AuthN {
namespace OpenSSL {

  std::string convert_rfc2253_to_ossdn(const std::string& str);

  std::string convert_ossdn_to_rfc2253(const std::string& str);

  class OpenSSLUtil {
    public:
      enum Credformat {CRED_PEM, CRED_DER, CRED_PKCS12, CRED_UNKNOWN};
      //Parse the string for certificate and get the format of it
      static Credformat getFormat_str(const std::string& source);


  };

  class OSSLItem {
    public:
      enum Type { TypeCert, TypeReq, TypeCRL, TypeKey };
      OSSLItem();
      OSSLItem(const OSSLItem& other);
      ~OSSLItem();

      OSSLItem& operator=(const OSSLItem& other);

      void reset();

      operator bool(void) const {
        return (cert != NULL || req != NULL || crl != NULL || pkey != NULL || pkcs12 != NULL);
      }

      bool operator!(void) const {
        return (!cert && !req && !crl && !pkey && !pkcs12);
      }

      bool invalid() const;

      Status tostrDER(std::string& out) const;

      Status tostrPEM(std::string& out, bool enc = false, const std::string& passphrase = "") const;

      Status tostrPKCS12(std::string& out) const;

      Status tofileDER(const char* filename);

      Status tofilePEM(const char* filename);

      Status tofilePKCS12(const char* filename);

      Status fromstrDER(const std::string& in, Type t);

      Status fromstrPEM(const std::string& in, Type t);

      Status fromstrPKCS12(const std::string& in);
  
      Status fromstr(const std::string& in, Type t);

      Status fromfile(const char* filename, Type t);

    //private:
      X509* cert;
      X509_REQ* req;
      X509_CRL* crl;
      EVP_PKEY* pkey;
      STACK_OF(X509)* cert_chain;
      PKCS12* pkcs12;
      Context* logctx;
  };

  /// Container for the option information for signing a certificate
  class CertificateOptions {
    public:
      CertificateOptions() : isCA_(false), pathLimit_(-1), serial_(-1), start_(-1), end_(-1) {};

      /// Copy constructor
      CertificateOptions(const CertificateOptions &from);
      ~CertificateOptions() { };

      /// Assignment operator
      CertificateOptions& operator=(const CertificateOptions &from);

      /// Check if the object is valid
      bool isValid() const;

      operator bool(void) const {
        return true; 
      }

      bool operator!(void) const {
        return false;
      }

      /// Get the subject of the certificate
      std::string subject() const { return subject_; };

      /// List the constraints on this certificate
      Constraints constraints() const { return certConstraints_; };

      /// Get the list of the policies on this certificate
      std::vector<std::string> policies() const { return certPolicies_; };

      /// Get the list of URI locations for CRL files
      std::vector<std::string> crlLocations() const { return crlLocations_; };

      /// Get the list of URI locations for issuer certificate files
      std::vector<std::string> issuerLocations() const { return issuerLocations_; };

      /// Get the list of URI locations for OCSP services
      std::vector<std::string> ocspLocations() const { return ocspLocations_; };

      /// Get the SPKAC challenge
      std::string challenge() const { return challenge_; };

      /// Check if the certificate is a CA cert
      bool isCA() const { return isCA_; };

      /// Get the path limit on this certificate
      int pathLimit() const { return pathLimit_; };

      /// Get the serial number for the certificate
      long serialNumber() const { return serial_; };

      /// Get the beginning time the certificate is valid
      AuthN::Utils::Time notValidBefore() const { return start_; };
      
      /// Get the end time the certificate is valid
      AuthN::Utils::Time notValidAfter() const { return end_; };

      /// Get the proxy policy for the proxy certificate
      std::string proxyPolicy() const{ return proxyPolicy_; };

      /// Set the subject of the certificate
      void setSubject(const std::string& subject) { subject_ = subject; };

      /// Set the constraints on the certificate
      void setConstraints(const Constraints& constraints) { certConstraints_ = constraints; };

      /// Set the policies on the certificate
      void setPolicies(const std::vector<std::string>& policies) { certPolicies_ = policies; };

      /// Set the CRL locations of the certificate
      /// each location/URI refers to the same CRL.
      void setCRLLocations(const std::vector<std::string>& locations) { crlLocations_ = locations; };

      /// Set the issuer certificate locations of the certificate
      /// each location/URI refers to the same issuer file.
      void setIssuerLocations(const std::vector<std::string>& locations) { issuerLocations_ = locations; };

      /// Set the OCSP service locations of the certificate
      void setOCSPLocations(const std::vector<std::string>& locations) { ocspLocations_ = locations; };

      /// Set SPKAC challenge
      void setChallenge(const std::string& challenge) { challenge_ = challenge; };

      /// Set the certificate to be a CA cert
      void setAsCA() { isCA_ = true; }; 

      /// Set the certificate to be a user cert (default)
      void setAsUser() { isCA_ = false; };

      /// Set the path len limit
      void setPathLimit(int pathLimit) { pathLimit_ = pathLimit; };

      /// Set the serial number property on this certificate
      void setSerialNumber(const long serial) { serial_ = serial; };
 
      /// Set the validity period for the certificate
      void setValidityPeriod(const AuthN::Utils::Time &start, const AuthN::Utils::Time &end) { start_ = start; end_ = end; };

      /// Set the validity period for the certificate
      void setValidityPeriod(const AuthN::Utils::Time &start, const int hours) { start_ = start; end_ = start_ + hours * 3600; };

      /// Set the proxy policy for the proxy certificate
      void setProxyPolicy(const std::string& proxypolicy) { proxyPolicy_ = proxypolicy; };

    private:
      std::string subject_;
      Constraints certConstraints_;
      std::vector<std::string> certPolicies_;
      std::vector<std::string> crlLocations_;
      std::vector<std::string> issuerLocations_;
      std::vector<std::string> ocspLocations_;
      bool isCA_;
      int pathLimit_;
      long serial_;
      AuthN::Utils::Time start_;
      AuthN::Utils::Time end_;
      std::string proxyPolicy_;  
      std::string challenge_;
  };


  /// Containter for the information of certificate or CSR 
  /// If an item is not secified for CSR also, then it is for certificate only 
  class CertContextProps {
    public:
      /// Standard Constructor
      CertContextProps() : serial(-1), isCA(false), isProxy(false), is_selfsigned(false), pathLimit(-1) {};

      CertContextProps& operator=(const CertContextProps& arg) {
        version = arg.version;
        start 	= arg.start;
        end 	= arg.end;
        subject = arg.subject;
        issuer 	= arg.issuer;
        identity= arg.identity;
        hash 	= arg.hash;
        constraints = arg.constraints;
        policies	= arg.policies;
        crl_locations	= arg.crl_locations;
        issuer_locations= arg.issuer_locations;
        ocsp_locations	= arg.ocsp_locations;
        serial	= arg.serial;
        isCA	= arg.isCA;
        isProxy	= arg.isProxy;
        is_selfsigned	= arg.is_selfsigned;
        pathLimit	= arg.pathLimit;
        proxyPolicy	= arg.proxyPolicy;
        sigData	= arg.sigData;
        sigAlg	= arg.sigAlg;
        subjectId	= arg.subjectId;
        issuerId	= arg.issuerId;
      };

      /// The X.509 certificate version
      int version;

      /// The start time
      AuthN::Utils::Time start;

      /// The expiration time
      AuthN::Utils::Time end;

      /// The subject information, for CSR also
      std::string subject;

      /// The issuer information
      std::string issuer;

      /// The identity information, in the case of proxy certificte, identity could be different with subject.
      std::string identity;

      /// The hash value of the subject
      std::string hash;

      /// The constraints, for CSR also
      Constraints constraints;

      /// The policies, for CSR also
      std::vector<std::string> policies;

      /// The list of URIs for CRLs
      std::vector<std::string> crl_locations;

      /// The list of URIs for issuer certificates
      std::vector<std::string> issuer_locations;

      /// The list of URIs for OCSP services (OCSP responder)
      std::vector<std::string> ocsp_locations;

      /// The certificate serial number
      long serial;

      /// The certificate is a CA or not, or the CSR is a request to be a CA or not, for CSR also
      bool isCA;

      /// The certificate is a proxy or not
      bool isProxy;

      /// The certificate is self-signed or not
      bool is_selfsigned;

      /// The path limit, for CSR also
      int pathLimit;

      /// The proxy policy, for CSR also
      AuthN::Credentials::Extension proxyPolicy;

      /// The signature data, for CSR also
      std::string sigData;

      /// The signature algorithm used to create the signature, for CSR also
      std::string sigAlg;

      std::string subjectId;

      IssuerId issuerId;
  };

  class KeyContext;
  class CSRContext;
  class CRLContext;
  class OCSPContext;

  class CertContext {
    private:
      OSSLItem cert_;
      OSSLItem key_;
      CertContextProps props_;
      std::string conf_file;
      Context* context;
      Status last_error_;

      AuthN::Utils::Cache* ocsp_cache_;
      std::string ocsp_cache_file;
      AuthN::Utils::Cache* create_cache();
      void remove_cache();

      CertContext():context(NULL), ocsp_cache_(NULL) {};

    public:
      /// Standard constructor
      CertContext(Context& ctx);
 
      /// Constructor 
      CertContext(Context& ctx, const char* cert_filename);

      /// Constructor
      CertContext(Context& ctx, const char* cert_filename, const char* key_filename);

      /// Constructor
      CertContext(Context& ctx, const std::string& certstr);

      /// Constructor
      CertContext(Context& ctx, const std::string& certstr, const std::string& keystr);

      /// Deconstructor
      ~CertContext();

      /// Set path to openssl configuration file
      void setOSSLConf(const std::string conf_f) { conf_file = conf_f; };
 
      /// Create a self-signed certificate based on the given options
      Status createSelfSigned(const CertificateOptions& opts, KeyContext& key);

      /// Sign a certificate based on the given options and certificate request
      CertContext* signRequest(const CertificateOptions& opts, CSRContext& csr);

      /// Get the properties about the certificate
      const CertContextProps* getProps() const;

      /// Set the properties about the certificate according to the attached certificate object
      void setProps();

      void certToDER(std::string& out) const;

      void certToPEM(std::string& out) const;

      bool certFromDER(const std::string& in);

      bool certFromPEM(const std::string& in);

      bool certFromStr(const std::string& in);

      void certFromX509(X509* x509);

      void certFromX509(X509* x509, STACK_OF(X509)* chain);

      void keyToDER(std::string& out) const;

      void keyToPEM(std::string& out, bool enc = false, const std::string& passphrase = "") const;

      bool keyFromDER(const std::string& in);

      bool keyFromPEM(const std::string& in);

      bool keyFromStr(const std::string& in);

      CertContext* getChainCert(int pos);

      bool getCertExtension(int pos, AuthN::Credentials::Extension& ext) const;

      bool getCertExtension(const std::string& name, AuthN::Credentials::Extension& ext) const;

      bool getCertAttributes(const std::string& name, std::list<std::string>& attrs) const;

      /// Get X509 object
      X509* getCert();

      /// Get EVP_PKEY object
      EVP_PKEY* getKey();

      /// Check if this certificate is equal to another certificate,
      bool operator==(const CertContext& other);

      /// Check if this certificate is an issuer of another
      bool isIssuerOf(const CertContext *other) const;

      /// Check the validity of this certificate, the CA and CRL will 
      /// be fetched from files that configured in Context
      Status validate(const AuthN::Validator::ValidationMode& mode, AuthN::Context* ctx = NULL);

      /// Check the validity of this certificate
      Status validate(const AuthN::Validator::ValidationMode& mode, const std::vector<CertContext*>& trusted, const std::vector<CertContext*>& untrusted, const std::vector<CRLContext*>&crls, AuthN::Context* ctx = NULL);

      /// Check the validity of a certificate chain
      /// this method does not make use of the certificate in this object
      Status validate_chain(const std::vector<CertContext*>& chain, const std::vector<CertContext*>& trusted, const std::vector<CRLContext*>&crls, AuthN::Context* ctx= NULL);

      operator bool(void) const {
        return (cert_ && ((cert_.cert) != NULL));
      }

      bool operator!(void) const {
        return (!cert_ || !(cert_.cert));
      }

      void setOCSPCache(const std::string& cache_file);

      friend class AuthN::IONetwork;
      friend class AuthN::IOSocket;
  };

  class KeyContext {
    private:
      OSSLItem key_;
      std::string conf_file;
      Context* context;
      Status last_error_;
 
      KeyContext() : context(NULL) { };

    public:
    
      KeyContext(Context& ctx) : context(&ctx) { key_.logctx = context; };

      /// Set path to openssl configuration file
      void setOSSLConf(const std::string conf_f) { conf_file = conf_f; };

      /// Create key 
      Status createKey(int bits = 1024);
 
      /// Convert a public key to DER format
      void publicToDER(std::string& out) const;

      /// Convert a public key to PEM format
      void publicToPEM(std::string& out) const;

      /// Convert a private key to DER format
      void privateToDER(std::string& out, const std::string& passphrase = "") const;

      /// Convert a private key to PEM format
      void privateToPEM(std::string& out, bool enc = false, const std::string& passphrase = "") const;

      /// Read input from string and convert it into a private key
      bool privateFromStr(const std::string& in, const std::string& passphrase = ""); 

      /// Get EVP_PKEY object
      EVP_PKEY* getKey();

      /// Show if the EVP_PKEY object contains private key information
      bool isPrivate();

      operator bool(void) const {
        return (key_ && ((key_.pkey) != NULL));
      }

      bool operator!(void) const {
        return (!key_ || !(key_.pkey));
      }

  };

  class CSRContext {
    private:
      OSSLItem csr_;
      CertContextProps props_;
      std::string conf_file;
      Context* context;
      std::vector<AuthN::Credentials::Extension> exts_;
      std::map<std::string, std::string> attrs_;
      std::string default_md;
      Status last_error_;

      /// Standard constructor
      CSRContext() : context(NULL) { };

    public:
      /// Constructor
      CSRContext(Context& ctx) : context(&ctx) { csr_.logctx = context; };

      /// Constructor
      CSRContext(Context& ctx, const char* csr_filename);

      /// Constructor
      CSRContext(Context& ctx, const std::string& csrstr);

      /// Set path to openssl configuration file
      void setOSSLConf(const std::string conf_f) { conf_file = conf_f; };

      /// Create a certificate based on the given options and certificate request
      bool createRequest(const CertificateOptions& opts, KeyContext& key);

      /// Get the properties about the certificate
      const CertContextProps* getProps() const;

      /// Set the properties about the certificate according to the attached certificate object
      void setProps();

      void csrToDER(std::string& out) const;

      void csrToPEM(std::string& out) const;

      bool csrFromDER(const std::string& in);

      bool csrFromPEM(const std::string& in);

      Status getCSRExtension(int pos, AuthN::Credentials::Extension& ext) const;

      Status getCSRExtension(const std::string& name, AuthN::Credentials::Extension& ext) const;

      Status setCSRExtension(AuthN::Credentials::Extension& ext);

      Status setCSRExtension(AuthN::Credentials::Extension& ext, KeyContext& key);

      Status getCSRAttribute(const std::string& name, std::list<std::string>& attr) const;

      Status setCSRAttribute(const std::string& name, const std::string& attrs);

      Status setCSRAttribute(const std::string& name,
    	const std::string& attrs, KeyContext& key);

      /// Get X509_REQ object
      X509_REQ* getReq();

      /// Show if the X509_REQ structure including signature
      bool isSigned();

      /// Check if this csr is equal to another csr
      bool operator==(const CSRContext& other);

      operator bool(void) const {
        return (csr_ && ((csr_.req) != NULL));
      }

      bool operator!(void) const {
        return (!csr_ || !(csr_.req));
      }

      friend class AuthN::ProxyCredentialsRequest;
      friend class AuthN::Credentials;
  };


  /// Container for the information of CRL
  class CRLContextProps {
    public:
      enum Reason
      {
        Unspecified,        // reason is unknown
        KeyCompromise,      // private key has been compromised
        CACompromise,       // certificate authority has been compromised
        AffiliationChanged,
        Superseded,         // certificate has been superseded
        CessationOfOperation,
        CertificateHold,    // certificate is on hold
        RemoveFromCRL,      // certificate was previously in a CRL, but is now valid
        PrivilegeWithdrawn,
        AACompromise        // attribute authority has been compromised
      };

      struct CRLEntry {
        long serial;
        AuthN::Utils::Time rev_time;
        Reason reason;
        std::string cert_issuer;
      };

      /// Standard Constructor
      CRLContextProps() : number(-1) {};

      /// The issuer of the CRL
      std::string issuer;

      /// The CRL number
      int number;

      /// The time this CRL was created
      AuthN::Utils::Time this_update;

      /// The time this CRL expires, and also the time next CRL is fetched
      AuthN::Utils::Time next_update;

      /// The revoked certificate
      std::map<long, CRLEntry> revoked;

      /// The signature data, for CSR also
      std::string sigData;

      /// The signature algorithm used to create the signature, for CSR also
      std::string sigAlg;

      /// X509v3 Issuing Distrubution Point
      std::string issuing_dp;

      /// X509v3 Delta CRL Indicator
      std::string delta_indicator;

      /// X509v3 Freshest CRL, a.k.a. Delta CRL Distribution Point
      std::string delta_dp;

      IssuerId issuerId;
  };

  class CRLContext {
    private:
      OSSLItem crl_;
      CRLContextProps props_;
      std::string conf_file;
      Context* context;

    public:
      /// Standard constructor
      CRLContext() : context(NULL) { };

      /// Constructor
      CRLContext(Context* ctx) : context(ctx) { crl_.logctx = context; };

      /// Constructor
      CRLContext(Context* ctx, const char* crl_filename);

      /// Constructor
      CRLContext(Context* ctx, const std::string& crlstr);

      /// Set path to openssl configuration file
      void setOSSLConf(const std::string conf_f) { conf_file = conf_f; };

      /// Get the properties about the certificate
      const CRLContextProps* getProps() const;

      /// Set the properties about the certificate according to the attached crl object
      void setProps();

      void crlToDER(std::string& out) const;

      void crlToPEM(std::string& out) const;

      bool crlFromDER(const std::string& in);

      bool crlFromPEM(const std::string& in);

      void crlFromCRL(X509_CRL* crl);

      /// Get X509_CRL object
      X509_CRL* getCRL();

      /// Check if this certificate is equal to another certificate,
      bool operator==(const CRLContext& other);

  };

  //query the OCSP responder to get the certificate status, according to rfc2560
  //this is as the alternative of crl
  class OCSPContext {
    public:
      enum CertStatus { CertStatusGood = 0, CertStatusRevoked = 1, CertStatusUnknown =2, CertStatusNotRequired =3 };

    private:
      std::string ocsp_responder_url_;
      Context* context_;
      X509* cert_;
      OCSP_CERTID* cert_id_; 
      std::string cached_id_;
      X509* issuer_;
      CertContext* request_signer_;
      SSL* ssl_;
      AuthN::Utils::Cache* cache_; 
      OCSP_REQUEST* req_;
      OCSP_RESPONSE* resp_;
      AuthN::Utils::Time valid_till_;
      CertStatus cert_status_;
      Status last_error_;

    public:
      /// Standard constructor
      //OCSPContext() : context_(NULL), cache_(NULL), cert_(NULL), ssl_(NULL) { };

      /// Constructor
      //OCSPContext(Context* ctx) : context_(ctx), cache_(NULL), cert_(NULL), ssl_(NULL) { };

      /// Constructor
      OCSPContext(Context* ctx, X509* cert, AuthN::Utils::Cache* cache, const std::string& resp_url= "");

      /// Constructor
      OCSPContext(Context* ctx, SSL* ssl, AuthN::Utils::Cache* cache, const std::string& resp_url= "");

      /// Destructor
      ~OCSPContext();

      /// Make OCSP request
      Status MakeOCSPRequest();

      /// Get OCSP request, with string format
      Status GetOCSPRequest(std::string& reqstr);
   
      /// Retrieve the OCSP response from cache
      Status RetrieveFromCache();

      /// Retrieve the OCSP response from cache, with string format
      Status RetrieveFromCache(std::string& respstr);

      /// Get the OCSP response from this object, with string format
      Status GetOCSPResponse(std::string& respstr);

      /// Store the OCSP response to cache
      Status StoreToCache();

      /// Store the OCSP response to cache, with string format
      Status StoreToCache(const std::string& respstr);

      /// Check the validity of the OCSP response
      Status CheckOCSPResponse();

      /// Check the validity of the OCSP response with string format
      Status CheckOCSPResponse(const std::string& reqstr, const std::string& respstr);

      /// Check the validity of the OCSP response with string format
      Status CheckOCSPResponse(const std::string& respstr);

      /// Query the OCSP responder to get the certificate status, according to rfc2560
      Status QueryOCSPResponder(const std::string& responder_url = "");

      /// Query the OCSP responder to get the certificate status, according to rfc2560
      // Status QueryOCSPResponder(const std::string& reqstr, std::string& respstr, const std::string& responder_url = "");

      /// Set the certificate and key, which will be used for 
      /// signing the OCSP request
      void SetRequestSigner(const std::string& certstr, const std::string& keystr);

      void SetRequestSigner(const char* certfile, const char* keyfile);

      CertStatus GetCertStatus(void) { return cert_status_; };

      AuthN::Utils::Time GetValidTill(void) { return valid_till_; };
  };

  void ssl_stapling_init(void);

  Status ssl_stapling_init_CertStatusContext(Context* logctx, SSL_CTX* sslctx, X509* cert,
      const std::string& cache_file, long cache_interval, const std::string& resp_url);


  // callback for OCSP stapling server side, it should be used by
  // SSL_CTX_set_tlsext_status_cb to setup the callback.
  // SSL_CTX_set_tlsext_status_arg could be used to 
  // setup the argument for this callback.
  // But an alternative approach by inserting the X509 extension
  // is used, because by using SSL_CTX_set_tlsext_status_arg,
  // it seems we have no way to delete the object that is created
  // for "arg".
  int ocsp_stapling_server_callback(SSL *s, void *arg);

  // callback for OCSP stapling client side
  int ocsp_stapling_client_callback(SSL *s, void *arg);

  class CredentialError : public Status {
    public:
      // Constructor
      /** This is the constructor of the CredentialError class.
       * @param what  An explanation of the error.
       */
      CredentialError(const std::string& what="");
  };

  typedef enum {CRED_PEM, CRED_DER, CRED_PKCS, CRED_UNKNOWN} Credformat;

  Credformat getFormat(Context& context, BIO * in);

  class AutoBIO {
   private:
    BIO* bio_;
   public:
    AutoBIO(BIO* bio):bio_(bio) { };
    ~AutoBIO(void) { if(bio_) { (BIO_set_close(bio_,BIO_CLOSE) == 1); BIO_free_all(bio_); } };
    operator bool(void) { return (bio_ != NULL); };
    operator BIO*(void) { return bio_; };
    BIO& operator*(void) const { return *bio_; };
    BIO* operator->(void) const { return bio_; };
  };

  class AutoX509 {
   private:
    X509* x509_;
   public:
    AutoX509(X509* x509):x509_(x509) { };
    ~AutoX509(void) { if(x509_) { X509_free(x509_); } };
    operator bool(void) { return (x509_ != NULL); };
    operator X509*(void) { return x509_; };
    X509& operator*(void) const { return *x509_; };
    X509* operator->(void) const { return x509_; };
  };

  class AutoX509NAME {
   private:
    X509_NAME* x509name_;
   public:
    AutoX509NAME(X509_NAME* x509name):x509name_(x509name) { };
    ~AutoX509NAME(void) { if(x509name_) { X509_NAME_free(x509name_); } };
    operator bool(void) { return (x509name_ != NULL); };
    operator X509_NAME*(void) { return x509name_; };
    X509_NAME& operator*(void) const { return *x509name_; };
    X509_NAME* operator->(void) const { return x509name_; };
  };

  class AutoX509REQ {
   private:
    X509_REQ* x509req_;
   public:
    AutoX509REQ(X509_REQ* x509req):x509req_(x509req) { };
    ~AutoX509REQ(void) { if(x509req_) { X509_REQ_free(x509req_); } };
    operator bool(void) { return (x509req_ != NULL); };
    operator X509_REQ*(void) { return x509req_; };
    X509_REQ& operator*(void) const { return *x509req_; };
    X509_REQ* operator->(void) const { return x509req_; };
  };

  class AutoEVPPKEY {
   private:
    EVP_PKEY* key_;
   public:
    AutoEVPPKEY(EVP_PKEY* key):key_(key) { };
    ~AutoEVPPKEY(void) { if(key_) { EVP_PKEY_free(key_); } };
    operator bool(void) { return (key_ != NULL); };
    operator EVP_PKEY*(void) { return key_; };
    EVP_PKEY& operator*(void) const { return *key_; };
    EVP_PKEY* operator->(void) const { return key_; };
  };
 
  AuthN::Utils::Time asn1_to_utctime(const ASN1_UTCTIME *s);

#define PASS_MIN_LENGTH 4
#define PASS_MAX_LENGTH 20
//  typedef struct pw_cb_data {
//    const void *password;
//    const char *prompt_info;
//  } PW_CB_DATA;
//  int passwordcb(char *buf, int bufsiz, int verify, void *cb_tmp);

  typedef struct { std::string subject_name; std::string policy; } ProxyPolicy;
  typedef std::list<ProxyPolicy> ProxyPolicyList;
  typedef struct {
    int inhibit_anypolicy;  //This is for the checking of "X509v3 Inhibit Any Policy"
    int inhibit_policymapping; //This if for the checking of "Inhibit Policy Mapping"
                               // part of "X509v3 Policy Constraints"
  } PolicyContext; //The context for handling certificate policy

  typedef struct { 
    std::string ca_path;
    ProxyPolicyList ppl;
    int validation_mode;
    PolicyContext policy_context;
    int path_len;  //Handle the pathLenConstraint
    Context* context; // or use copy?
    OCSPContext::CertStatus cert_status;
  } ValidationContext;

  int get_proxy_auth_ex_data_idx(void);
  int verify_callback(int ok, X509_STORE_CTX *ctx);


  int check_issued(X509_STORE_CTX* ctx, X509* x, X509* issuer);

  X509_STORE* setup_verify(Context& context, const std::string& CAfile, const std::string& CApath);

  std::string get_cert_hash(X509* cert);

  std::string get_crl_hash(X509_CRL* cert);


  class OpenSSLInitializer {
   private:
    static pthread_mutex_t * ssl_locks;
    static int ssl_locks_num;
    static unsigned long ssl_id_cb(void);
    static void ssl_locking_cb(int mode, int n, const char * s_, int n_);
   public:
    /// Creating instance of this class will initialize OpenSSL.
    OpenSSLInitializer(void);
    /// It is advisable to call this method upon exiting thread
    /// to avoid memory leaks.
    static void ThreadCleanup(void);
  };

  // This class can be used to perform password request through UI_* interface
  class ContextOpenSSLUI: public Context {
   public:
    ContextOpenSSLUI(void);
    virtual Context& Copy(void) const;
    virtual std::string Password(const std::string& message);
    virtual void Message(const std::string& message);
    void SetPassword(const std::string& pass);
    static int passwordcb(char *buf, int bufsize, int verify, void *cb_tmp);
   private:
    std::string password;
  };
}
}

#endif // __AUTHNLIB_OPENSSLUTIL_H__
