/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2016 Univ. Grenoble Alpes, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include "VtkMeshComponent.h"

// -- CamiTK stuff
#include <Application.h>
#include <MeshComponent.h>
#include <Geometry.h>
#include <InteractiveViewer.h>
#include <Property.h>

// -- vtk stuff
#include <vtkPolyData.h>
#include <vtkCleanPolyData.h>
#include <vtkPointData.h>
#include <vtkCell.h>
#include <vtkTriangleFilter.h>
#include <vtkGeometryFilter.h>
#include <vtkPolyDataNormals.h>
#include <vtkDoubleArray.h>
#include <vtkProperty.h>

using namespace camitk;


// -------------------- default constructor  --------------------
VtkMeshComponent::VtkMeshComponent (const QString& fileName)  throw(AbortException) : MeshComponent ( fileName ) {

    whatIsIt = VtkMeshUtil::typeOf(myFileName.toStdString());

    if (whatIsIt == VtkMeshUtil::UNKNOWN) {
        throw AbortException("Unable to recognize this vtk file\n(it is neither an unstructured grid nor a poly data).");
    }

    setName(VtkMeshUtil::getVtkPointSetHeaderString(myFileName.toStdString()).c_str());

    myProperties = NULL;
    initialPointData = NULL;
    demoPointData = NULL;

    // build the vtk data structure from the file
    vtkSmartPointer<vtkPointSet> data = VtkMeshUtil::buildVtkPointSet(myFileName.toStdString().c_str(), whatIsIt);

    // instanciate the Geometry
    initRepresentation(data);

    // initialize point data value (for demo or display)
    initPointData();
}

// -------------------- destructor --------------------
VtkMeshComponent::~VtkMeshComponent() {
    setPointData ( NULL );
    InteractiveViewer::get3DViewer()->setColorScale ( false );
}

// -------------------- initPointData --------------------
void VtkMeshComponent::initPointData() {
    // create a new vtkDataArray to store the demo value
    demoPointData = vtkSmartPointer<vtkDoubleArray>::New();

    int numberOfPoints = getPointSet()->GetNumberOfPoints();
    demoPointData->SetNumberOfValues ( numberOfPoints );
    demoPointData->SetName ( "the point data" );
    for ( vtkIdType i = 0; i < numberOfPoints; ++i ) {
        demoPointData->SetValue ( i, 0.0 );
    }
    // backup the read point data
    initialPointData = vtkSmartPointer<vtkPointData>::New();
    initialPointData->ShallowCopy(myGeometry->getPointSet()->GetPointData());

    // remove the point data
    setPointData(NULL);
}

// -------------------- pointPicked --------------------
void VtkMeshComponent::pointPicked ( vtkIdType pointId, bool sel/* picking does not change the selection state, do not bother with the 2nd parameter*/ ) {
    MeshComponent::pointPicked(pointId, sel);
    // picking show demo point data
    demoPointData->SetValue ( pointId, 1.0 - double ( pointId ) / double ( getPointSet()->GetNumberOfPoints() ) );
    demoPointData->Modified(); // force an update on the vtk pipeline
    refresh(); // refresh display
}

// -------------------- cellPicked --------------------
void VtkMeshComponent::cellPicked ( vtkIdType cellId, bool  sel /* picking does not change the selection state, do not bother with the 2nd parameter*/ ) {
    MeshComponent::cellPicked(cellId, sel);
    // change data for all cell points
    vtkSmartPointer<vtkCell> c = getPointSet()->GetCell ( cellId );
    for ( vtkIdType i = 0; i < c->GetNumberOfPoints(); i++ ) {
        vtkIdType pointId = c->GetPointId ( i );
        demoPointData->SetValue ( pointId, 1.0 - double ( pointId ) / double ( getPointSet()->GetNumberOfPoints() ) );
    }
    demoPointData->Modified(); // force an update on the vtk pipeline
    refresh(); // refresh display
}

// -------------------- showPointData --------------------
void VtkMeshComponent::showPointData(VTK_COMPONENT_POINT_DATA_TYPE type) {
    switch (type) {
    case NONE:
        setPointData ( NULL );
        InteractiveViewer::get3DViewer()->setColorScale ( false );
        break;
    case DEMO:
        setPointData ( demoPointData );
        InteractiveViewer::get3DViewer()->setColorScaleMinMax ( 0.0, getPointSet()->GetNumberOfPoints() );
        InteractiveViewer::get3DViewer()->setColorScaleTitle("Demo Point Data");
        InteractiveViewer::get3DViewer()->setColorScale ( true );
        break;
    case INITIAL:
        myGeometry->getPointSet()->GetPointData()->ShallowCopy(initialPointData);
        if (initialPointData->GetNumberOfArrays() > 0) {
            double range[2];
            myGeometry->getPointSet()->GetPointData()->GetScalars()->GetRange ( range );
            InteractiveViewer::get3DViewer()->setColorScaleMinMax ( range[0], range[1] );
            InteractiveViewer::get3DViewer()->setColorScaleTitle("VTK Point Data");
            InteractiveViewer::get3DViewer()->setColorScale ( true );
        } else {
            InteractiveViewer::get3DViewer()->setColorScale ( false );
        }
        break;
    }
    refresh();
}

// -------------------- exportMDL --------------------
bool VtkMeshComponent::exportMDL ( std::string filename ) {
    if ( myGeometry ) {
        int i;
        // MDL format supports only triangles. Hence, this dataset triangularized before
        // being exported (this operation is then NOT reversible).
        // if the dataset is a volumetric mesh (e.g. hexahedrons), only the external surface is exported.

        //extract external surface
        vtkSmartPointer<vtkGeometryFilter> geomF = vtkSmartPointer<vtkGeometryFilter>::New();
        geomF->SetInputData(getPointSet());


        // triangles
        vtkSmartPointer<vtkTriangleFilter> triangleF = vtkSmartPointer<vtkTriangleFilter>::New();
        triangleF->SetInputData(geomF->GetOutput());

        // clean unused
        vtkSmartPointer<vtkCleanPolyData> cleaner = vtkSmartPointer<vtkCleanPolyData>::New();
        cleaner->SetInputData(triangleF->GetOutput());
        cleaner->Update();

        vtkSmartPointer<vtkPolyData> ds = cleaner->GetOutput();



        //--- write as .mdl file (Registration format)
        std::ofstream o ( filename.c_str() );

        //--- name
        o << "[Name, STRING]" << std::endl;
        o << getName().toStdString() << std::endl;
        o << std::endl;


        //--- vertices
        o << "[Vertices, ARRAY1<POINT3D>]" << std::endl;
        o << ds->GetPoints()->GetNumberOfPoints() << std::endl;

        double pt[3];

        for ( i = 0; i < ds->GetPoints()->GetNumberOfPoints(); i++ ) {
            ds->GetPoints()->GetPoint ( i, pt );
            o << pt[0] << " " << pt[1] << " " << pt[2] << std::endl;
        }

        o << std::endl;


        //--- normals
        o << "[Normals, ARRAY1<VECTOR3D>]" << std::endl;

        if ( ds->GetPointData() && ds->GetPointData()->GetNormals() ) {
            // save existing normals
            o << ds->GetPointData()->GetNormals()->GetNumberOfTuples() << std::endl;

            for ( i = 0; i < ds->GetPointData()->GetNormals()->GetNumberOfTuples(); i++ ) {
                ds->GetPointData()->GetNormals()->GetTuple ( i, pt );
                o << pt[0] << " " << pt[1] << " " << pt[2] << std::endl;
            }
        } else {
            // compute the normals
            vtkSmartPointer<vtkPolyDataNormals> pNormals = vtkSmartPointer<vtkPolyDataNormals>::New();
            pNormals->SetInputData( ds );
            pNormals->Update();

            o << pNormals->GetOutput()->GetPointData()->GetNormals()->GetNumberOfTuples() << std::endl;

            for ( i = 0; i < pNormals->GetOutput()->GetPointData()->GetNormals()->GetNumberOfTuples(); i++ ) {
                pNormals->GetOutput()->GetPointData()->GetNormals()->GetTuple ( i, pt );
                o << pt[0] << " " << pt[1] << " " << pt[2] << std::endl;
            }

        }

        o << std::endl;


        //--- triangles
        int j;
        o << "[Triangles, ARRAY1<STRING>]" << std::endl;
        o << ds->GetNumberOfCells() << std::endl;

        for ( i = 0; i < ds->GetNumberOfCells(); i++ ) {
            // write it twice
            for ( j = 0; j < ds->GetCell ( i )->GetNumberOfPoints(); j++ )
                o << ds->GetCell ( i )->GetPointId ( j ) << " ";

            for ( j = 0; j < ds->GetCell ( i )->GetNumberOfPoints(); j++ )
                o << ds->GetCell ( i )->GetPointId ( j ) << " ";

            o << std::endl;
        }
        return true;

    } else
        return false;
}

//------------------------ getPixmap ---------------------
#include "vtklogo_20x20.xpm"
QPixmap * VtkMeshComponent::myPixmap = NULL;
QPixmap VtkMeshComponent::getIcon() {
    if ( !myPixmap ) {
        myPixmap = new QPixmap ( vtklogo_20x20 );
    }
    return ( *myPixmap );
}

// -------------------- initDynamicProperties --------------------
void VtkMeshComponent::initDynamicProperties() {

    MeshComponent::initDynamicProperties();

    // add a dynamic property to manage the surface color
    double initialValue = 0.0;
    if( getActor( InterfaceGeometry::Surface ) )
        initialValue = getActor( InterfaceGeometry::Surface )->GetProperty()->GetAmbient();
    Property *ambiant = new Property("Ambient", initialValue, tr("Ambient Lightning Coefficient. Warning: this cannot be saved in the VTK file."), "%");
    ambiant->setAttribute("minimum",0.0);
    ambiant->setAttribute("maximum",1.0);
    ambiant->setAttribute("decimals",2);
    ambiant->setAttribute("singleStep",0.1);
    addProperty(ambiant);

    if( getActor( InterfaceGeometry::Surface ) )
        initialValue = getActor( InterfaceGeometry::Surface )->GetProperty()->GetDiffuse();
    Property *diffuse = new Property("Diffuse", initialValue, tr("Diffuse Lightning Coefficient. Warning: this cannot be saved in the VTK file."), "%");
    diffuse->setAttribute("minimum",0.0);
    diffuse->setAttribute("maximum",1.0);
    diffuse->setAttribute("decimals",2);
    diffuse->setAttribute("singleStep",0.1);
    addProperty(diffuse);

    if( getActor( InterfaceGeometry::Surface ) )
        initialValue = getActor( InterfaceGeometry::Surface )->GetProperty()->GetSpecular();
    Property *specular = new Property("Specular", initialValue, tr("Specular Lightning Coefficient. Warning: this cannot be saved in the VTK file."), "%");
    specular->setAttribute("minimum",0.0);
    specular->setAttribute("maximum",1.0);
    specular->setAttribute("decimals",2);
    specular->setAttribute("singleStep",0.1);
    addProperty(specular);

    if( getActor( InterfaceGeometry::Surface ) )
        initialValue = getActor( InterfaceGeometry::Surface )->GetProperty()->GetSpecularPower();
    Property *specularPower = new Property("Specular Power", initialValue, tr("Specular Power. Warning: this cannot be saved in the VTK file."), "%");
    specularPower->setAttribute("minimum",0.0);
    specularPower->setAttribute("maximum",128.0);
    addProperty(specularPower);

    if( getActor( InterfaceGeometry::Surface ) )
        initialValue = getActor( InterfaceGeometry::Surface )->GetProperty()->GetOpacity();
    Property *opacity = new Property("Opacity", initialValue, tr("The object's opacity. 1.0 is totally opaque and 0.0 is completely transparent. Warning: this cannot be saved in the VTK file."), "%");
    opacity->setAttribute("minimum",0.0);
    opacity->setAttribute("maximum",1.0);
    opacity->setAttribute("decimals",2);
    opacity->setAttribute("singleStep",0.1);
    addProperty(opacity);

    double initialColor[3] = {0.8,0.8,0.8};
    if( getActor( InterfaceGeometry::Surface ) )
        getActor( InterfaceGeometry::Surface )->GetProperty()->GetColor(initialColor);
    QColor initialQColor;
    initialQColor.setRgbF(initialColor[0],initialColor[1],initialColor[2]);
    addProperty(new Property("Specular Color", initialQColor, tr("Specular Surface Color"), "RGB"));

}

// -------------------- updateProperty  --------------------
void VtkMeshComponent::updateProperty(QString name, QVariant value) {
    if( name == "Ambient" ) {
        if( getActor( InterfaceGeometry::Surface ) )
            getActor( InterfaceGeometry::Surface )->GetProperty()->SetAmbient( value.toDouble());
    } else if( name == "Diffuse" ) {
        if( getActor( InterfaceGeometry::Surface ) )
            getActor( InterfaceGeometry::Surface )->GetProperty()->SetDiffuse( value.toDouble() );
    } else if( name == "Specular" ) {
        if( getActor( InterfaceGeometry::Surface ) )
            getActor( InterfaceGeometry::Surface )->GetProperty()->SetSpecular( value.toDouble() );
    } else if( name == "Specular Power" ) {
        if( getActor( InterfaceGeometry::Surface ) )
            getActor( InterfaceGeometry::Surface )->GetProperty()->SetSpecularPower( value.toDouble() );
    } else if( name == "Opacity" ) {
        if( getActor( InterfaceGeometry::Surface ) )
            getActor( InterfaceGeometry::Surface )->GetProperty()->SetOpacity( value.toDouble() );
    } else if( name == "Specular Color" ) {
        if( getActor( InterfaceGeometry::Surface ) ) {
            QColor newColor = value.value<QColor>();
            getActor( InterfaceGeometry::Surface )->GetProperty()->SetSpecularColor( newColor.redF(), newColor.greenF(), newColor.blueF() );
        }
    } else {
        // just in case there is an inherited property to manage
        MeshComponent::updateProperty(name, value);
    }

    // refresh the viewers
    refresh();
}


