/*  BEGIN software license
 *
 *  msXpertSuite - mass spectrometry software suite
 *  -----------------------------------------------
 *  Copyright(C) 2009, 2017 Filippo Rusconi
 *
 *  http://www.msxpertsuite.org
 *
 *  This file is part of the msXpertSuite project.
 *
 *  The msXpertSuite project is the successor of the massXpert project. This
 *  project now includes various independent modules:
 *  
 *  - massXpert, model polymer chemistries and simulate mass spectrometric data;
 *  - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * END software license
 */



#pragma once


/////////////////////// Qt includes
#include <QObject>
#include <QList>


/////////////////////// Local includes
#include <libmass/Trace.hpp>
#include <libmass/DataPoint.hpp>


namespace msXpSlibmass
{


	//! The MassSpectrum class provides a mass spectrum.
	/*!

		Since a mass spectrum is a collection of DataPoint instances acquired at a
		specific point in time, it needs to be characterized by a retention time
		member datum (\c m_rt). In experiments of ion mobility mass spectrometry,
		another value is needed to fully characterize the mass spectrum, the drift
		time value (\c m_dt).

		The MassSpectrum class derives from the Trace class. As such it inherits a
		set of methods required to combine traces into a single trace. Other
		specialized functions are specific of MassSpectrum.

		The key value of the DataPoint instances of the MassSpectrum should be
		construed as m/z values, and the value of the DataPoint instances should be
		construed as the intensity. Finally, in a MassSpectrum, DataPoints instances
		may be construed as mass peaks.

*/
	class MassSpectrum 
		: 
			public Trace
	{
		friend class MassSpectrumJs;

		private:

		//! Index of the bin that was last modified.
		/*!

			This value has the same function as \c Trace::m_lastModifIdx but in cases
			where binning is required.

*/
		int m_lastBinIdx = 0;

		//! Retention time at which this mass spectrum was acquired.
		double m_rt = 0.0;

		//! Drift time at which this mass spectrum was acquired.
		double m_dt = 0.0;

		//! Greatest difference between two consecutive m/z values in the spectrum
		/*!

			This value is useful when performing alignment of the spectra of an
			acquisition.

*/
		double m_greatestStep = qSNaN();

		//! Number of bins in the spectrum.
		/*! 

			The number of bins is useful for spectral alignment.

*/
		int m_binCount = 0;

		//! Tells if the spectrum can undergo a binning process
		bool m_isBinnable = false;	

		//! The smallest m/z value contained in the spectrum list being processed
		/*!

			This value is useful when setting-up the binning framework in \c this
			MassSpectrum.  It stores the smallest m/z value of a list of mass spectra.

*/
		double m_binMinMz = 0;

		//! The greatest m/z value contained in the spectrum list being processed
		/*!

			This value is useful when setting-up the binning framework in \c this
			MassSpectrum.  It stores the greatest m/z value of a list of mass spectra.

*/
		double m_binMaxMz = 0;

		//! Value by which the mass spectra need to be shifted when aligned.
		/*!

			This value holds the m/z delta value by which this mass spectrum needs
			to be shifted in order to align it properly.

*/
		double m_mzShift = 0;

		public:

		//! Maximum acceptable value of the bin size average.
		static double m_maxAvgMzBinSize;

		//! Maximum acceptable value of the bin size standard deviation.
		static double m_maxBinSizeStdDev;

		MassSpectrum();
		MassSpectrum(const MassSpectrum &other);
		MassSpectrum(const QString &title);
		virtual ~MassSpectrum();

		MassSpectrum &operator=(const MassSpectrum &other);

		void reset() override;

		double rt() const;
		double &rrt();

		double dt() const;
		double &rdt();

		// The overriding functions below cannot shadow the base class ones.
		using Trace::initialize;

		int initializeRt(const QList<double> &mzList, const QList<double> &iList,
				double rt);

		int initializeDt(const QList<double> &mzList, const QList<double> &iList,
				double dt);

		int initializeRt(const QList<double> &mzList, const QList<double> &iList,
				double rt, double mzStart, double mzEnd);

		int initializeDt(const QList<double> &mzList, const QList<double> &iList,
				double dt, double mzStart, double mzEnd);

		int initialize(const QByteArray *keyByteArray,
				const QByteArray *valByteArray, int compressionType, 
				double keyStart = qSNaN(), double keyEnd = qSNaN()) override;

		int initialize(const Trace &trace);

		bool setupBinnability(const QList<MassSpectrum *> &massSpectra);
		double fillInBins(QList<double> *binList);
		double binSizeAvg(const QList<double> &binList);
		double binSizeStdDev(const QList<double> &binList);
		bool isBinnable(const QList<double> &binList);
		void setupBins(const QList<double> &binList);
		int binIndex(double mz);

		double determineMzShift(const MassSpectrum &massSpectrum);
		double determineMzShift(double otherMzValue);

		// The overriding functions below cannot shadow the base class ones.
		using Trace::combine;
		using Trace::subtract;

		int combineBinned(const DataPoint &dataPoint);
		int subtractBinned(const DataPoint &dataPoint);

		int combine(const MassSpectrum &other);
		int subtract(const MassSpectrum &other);
		int combine(const MassSpectrum &other, double mzStart, double mzEnd);
		int subtract(const MassSpectrum &other, double mzStart, double mzEnd);

		int combine(const QList<MassSpectrum *> &massSpectra);
		int subtract(const QList<MassSpectrum *> &massSpectra);
		int combine(const QList<MassSpectrum *> &massSpectra,
				double mzStart, double mzEnd);
		int subtract(const QList<MassSpectrum *> &massSpectra,
				double mzStart, double mzEnd);

		int combine(const QList<double> &mzList, const QList<double> &iList);
		int subtract(const QList<double> &mzList, const QList<double> &iList);
		int combine(const QList<double> &mzList, const QList<double> &iList,
				double mzStart, double mzEnd);
		int subtract(const QList<double> &mzList, const QList<double> &iList,
				double mzStart, double mzEnd);

		double tic();
		double tic(double mzStart, double mzEnd);
		static double tic(const QList<MassSpectrum *> &massSpectra);

		static void mzMinMax(const QList<MassSpectrum *> &massSpectra,
				double *minMz, double *maxMz);

		static bool areSpectraSameLength(const QList<MassSpectrum *> &massSpectra);
	};

} // namespace msXpSlibmass


Q_DECLARE_METATYPE(msXpSlibmass::MassSpectrum);
Q_DECLARE_METATYPE(msXpSlibmass::MassSpectrum*);

extern int massSpectrumMetaTypeId ;
