#!/usr/bin/env python
#-*- coding: utf-8 -*-
# memaker.py - A gtk based avatar creation program.
# Copyright 2008 The MeMaker Project
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. See the file LICENSE.
# If not, see <http://www.gnu.org/licenses/>.
import sys, os, fileinput, rsvg
import motor
import gtk
import cairo
import pynotify
import webbrowser
import shutil
import ConfigParser
from xdg import BaseDirectory as xdg

import locale
import gettext

pynotify.init("MeMaker")

try:
    import Image
except ImportError:
    print ('Couldn\'t find PIL,  install \'python-imaging\' for autogenerated thumbnails.')
    print ('Continuing without the feature...')
    notifier = pynotify.Notification(("Recommended Module Not Found"), 
	                                 ("Install 'python-imaging' for better thumbnails."),
	                                 "dialog-info")
    notifier.show()
    from image_loader_dummy import *
else:
    from image_loader import *
os.chdir(os.sys.path[0])

AUTHORSFILE = '/usr/share/doc/memaker/AUTHORS'
DATADIR = '/usr/share/memaker/data'
THEMESDIR = '/usr/share/memaker/themes'
APP_NAME = "memaker"

class MeMakerGui:
    
    class featuresObject():
        pass
        
    class runningHeads():
        pass
        
    def __init__(self):
        if "--debug" in sys.argv:
            print "Debug-Mode ON"
            self.debug = True
        else:
            self.debug = False
        #self.debug = True
        
        # Translations
        locale_dir = "/usr/share/locale"
        for module in (gettext, locale):
            module.bindtextdomain(APP_NAME, locale_dir)
            module.textdomain(APP_NAME)
        if hasattr(module, 'bind_textdomain_codeset'):
            module.bind_textdomain_codeset(APP_NAME,'UTF-8')
        gettext.install(APP_NAME, unicode = True)
        self._ = _
        
        self.featureList = ["Head", "Hair", "Eye", "Mouth", "Ear", "Glasses", "Nose", "Eyebrow", "Hat", "Accessory", "Beard"]
        gtkbuilderfile = DATADIR + "/memaker.ui"
        self.windowname = "meMaker"
        self.currentFace = "Unset"
        self.meMakerWin = gtk.Builder()
        self.meMakerWin.add_from_file(gtkbuilderfile)
        self.meMakerWin.set_translation_domain(APP_NAME)
        #Create the default directory structure if not there now....
        self.dataHome = xdg.xdg_data_home+"/memaker"
        self.confHome = xdg.xdg_config_home+"/memaker"
        self.cacheHome = xdg.xdg_cache_home+"/memaker"
        self.mkdirCheck(self.dataHome+"/themes")
        self.mkdirCheck(self.confHome)
        self.mkdirCheck(self.cacheHome)
        #Remove any old configuration, it is very buggy
        if os.path.isdir(os.path.expanduser('~/.MeMaker')):
            shutil.rmtree(os.path.expanduser('~/.MeMaker'))
        #Create the window system for memaker
        windowMain = self.meMakerWin.get_object("meMaker")
        windowMain.set_app_paintable(True)
        #Getting and connecting the widgets
        self.meMakerWin.get_object("aboutMenuItem").connect('activate', self.loadAbout)
        self.meMakerWin.get_object("menuItemPreferences").connect('activate', self.loadPreferences)
        self.meMakerWin.get_object("menuItemReset").connect('activate', self.avatarReset)
        self.meMakerWin.get_object("linkbuttonDownload").connect('clicked', self.linkClicked)
        self.meMakerWin.get_object("menuItemQuit").connect('activate', self.saveAndQuit)
        self.meMakerWin.get_object("meMaker").connect('destroy', self.saveAndQuit)
        for x in ['svg', 'png', 'bmp', 'gnome', 'launchpad']:
            self.meMakerWin.get_object("menuItemSaveAs"+x+"").connect('activate', self.saveAs, x)
		
        self.featuresMade = {}
        for feat in self.featureList:
            self.createUpDownClear(feat)
            self.connectIconviews(feat)
        if self.debug: print self._("Loading Up Down and Clear Buttons...")

        #Connect to the combobox and add all the themes to the combobox
        comboboxThemePicker = self.meMakerWin.get_object("comboboxThemes")
        listStore = gtk.ListStore(gtk.gdk.Pixbuf, str)
        comboboxThemePicker.set_model(listStore)
        px = gtk.CellRendererPixbuf()
        text = gtk.CellRendererText()
        comboboxThemePicker.pack_start(px, False)
        comboboxThemePicker.pack_start(text, False)
        comboboxThemePicker.add_attribute(px, "pixbuf", 0)
        comboboxThemePicker.add_attribute(text, "text", 1)
        locations = [ THEMESDIR + "/", self.dataHome+"/themes/"] #The areas that will load themes
        themesList = []
        themesListB = []
        for location in locations:
            for items in os.listdir(location):
                if os.path.isdir(location+items):
                    themesList.append(items)
                    themesListB.append(location + items + "/")
        #TODO, clean up this very dirty hack.  My gosh!
        #Creating the list of themes with their picture in nice combo box
        count = 0
        for items in themesList:
            themeIcon = themesListB[count] + items + ".png"
            if os.path.isfile(themeIcon):
                image = gtk.gdk.pixbuf_new_from_file(themeIcon)
            else:
                print self._("The theme '%s' is missing a theme icon. Please report this to the theme's creator") % items.capitalize()
            count += 1
            listStore.append((image, items.capitalize()))
        #For each theme start up it's object stack and then assign it to the the dictionary for reference
        #The reference is the actually the location of the themes.
        #Would this take up to much memory?
        self.loadedHeads =  {}
        #for items in themesListB:
            #self.loadedHeads[items] = motor.ObjectStack(self.debug)
        #Try to load the old ObjectStakcs and then see if it works.  If it doesn't load ignore and move on
        try:
            self.loadPickledHead()
        except:
            if self.debug:print self._("Error found in the pickledHeads file ignoring and moving on...")
        #Now lets check the loaded heads... do we have the same list as the themesList?
        #print themesListB
        #print self.loadedHeads.keys()
        for items in themesListB:
            if items not in self.loadedHeads.keys():
                print self._("Looks like a new theme was added.  We will need to reset everything to adjust.")
                for items in themesListB:
                    self.loadedHeads[items] = motor.ObjectStack(self.debug)
                
        linkbuttonDownload = self.meMakerWin.get_object("linkbuttonDownload")
        linkbuttonDownload.hide()
        self.avatarPicture = self.meMakerWin.get_object("imageAvatar")
        self.clearIt = rsvg.Handle( DATADIR + "/clearIt.svg")
        self.avatarPicture.set_size_request(400, 400)
        self.avatarPicture.connect("expose-event", self.draw_scene)
        #Show the window now so all major changes to the gui are applied first.
        windowMain.show()
        #Now to load the last theme used
        self.memakerConf = ConfigParser.SafeConfigParser()
        try:
            self.memakerConf.read(self.confHome+"/conf.conf")
            themeToLoad = self.memakerConf.get('main', 'lastTheme')
            themeToLoad = self.memakerConf.get('main', 'launchpadAuth')
        except:
            if self.debug: print self._("No conf.conf file found. Creating a new one with the default theme.")
            self.memakerConf = ConfigParser.SafeConfigParser()
            self.memakerConf.add_section('main')
            themeToLoad = THEMESDIR + "/cocoHead/"
            self.memakerConf.set('main', 'lastTheme', themeToLoad)
            self.memakerConf.set('main', 'launchpadAuth', "False")
            self.memakerConfData = open(self.confHome+"/conf.conf", "w")
            self.memakerConf.write(self.memakerConfData)
            self.memakerConfData.close()
        if os.path.isdir(themeToLoad):
            self.loadFeatures(themeToLoad)
        else:
            if self.debug: print self._("Previously used theme was not found.  Was it deleted/moved?")
            if self.debug: print self._("Falling back to default theme...")
            themeToLoad = THEMESDIR + "/cocoHead/"
            self.loadFeatures(themeToLoad)
        #Even more dirty code, just plain yucky, I am ashamed of myself!
        ###############################################################
        count = 0
        if len(themesList) <= 1:
            comboboxThemePicker.hide()
            linkbuttonDownload.show()
        else:
            linkbuttonDownload.hide()
            for items in themesListB:
                if themeToLoad == items:
                    comboboxThemePicker.set_active(count)
                count += 1
            comboboxThemePicker.show()
        ###############################################################
        comboboxThemePicker.connect('changed', self.themeChanged, themesListB)
        
        # Disable launchpad stuff if not authorized yet
        self.memakerConf.read(self.confHome+"/conf.conf")
        if self.memakerConf.get('main', 'launchpadAuth') == "False":
            self.meMakerWin.get_object("menuItemSaveAslaunchpad").set_sensitive(False)
            self.meMakerWin.get_object("menuItemSaveAslaunchpad").set_tooltip_text(("You have not yet authorized MeMaker for use with your Launchpad account. Please do so through Edit > Preferences."))
        # Disable restore stuff if we have no backups
        self.meMakerWin.get_object('menuItemRestore').connect('activate', self.checkBackups)
        self.meMakerWin.get_object('menuitemRestoregnome').connect('activate', self.restoreGnome)
        self.meMakerWin.get_object('menuitemRestorelaunchpad').connect('activate', self.restoreLaunchpad)
        self.meMakerWin.get_object("dialogLaunchpadAuth").connect('delete-event', lambda x, y=None : self.meMakerWin.get_object("dialogLaunchpadAuth").hide())
        self.meMakerWin.get_object("buttonAuthCancel").connect("clicked", lambda x : self.meMakerWin.get_object("dialogLaunchpadAuth").hide())
        self.meMakerWin.get_object("buttonApplyAuth").connect("clicked", lambda x : self.meMakerWin.get_object("dialogLaunchpadAuth").hide())
        self.meMakerWin.get_object("buttonApplyAuth").set_sensitive(False)
        self.meMakerWin.get_object("buttonAuthForward1").connect("clicked", self.getAuthPre)
        self.meMakerWin.get_object("buttonAuthForward2").connect("clicked", self.getAuthPost)
        self.meMakerWin.get_object("buttonStartAssistant").connect('clicked', self.getAuthMethod)
        
    def mkdirCheck(self, newdir):
        """
        Check if directory exist and create it if it doesn't
        """
        if os.path.isdir(newdir):
            pass
        elif os.path.isfile(newdir):
            raise OSError("a file with the same name as the desired " \
                          "dir, '%s', already exists." % newdir)
        else:
            head, tail = os.path.split(newdir)
            if head and not os.path.isdir(head):
                print ("A required Directory in your Home Directory was not found .... making:"), head
                os.makedirs(head)
            if tail:
                print ("A required Directory in your Home Directory was not found .... making:"), newdir
                os.makedirs(newdir)
    
    def connectIconviews(self, feat):
        """
        Connect the feature thumbnails to their corresponding functions
        """
        tempvar = self.featuresObject()
        tempvar.model = gtk.ListStore(str, gtk.gdk.Pixbuf)
        tempvar.iconview = self.meMakerWin.get_object('iconview'+feat)
        tempvar.iconview.set_model(tempvar.model)
        tempvar.iconview.set_pixbuf_column(1)
        tempvar.iconview.connect('selection-changed', self.featureClicked, tempvar.model)
        self.featuresMade[feat] = tempvar
        
    def createUpDownClear(self, feat):
        """
        Add the buttons for removing, raising and lowering features
        """
        buttons = self.meMakerWin.get_object("buttonRemove"+feat)
        buttons.connect('clicked', self.removeFeature)
        buttons.set_tooltip_text(self._("Remove the %s from your avatar.") % feat.lower())
        buttons = self.meMakerWin.get_object("buttonUp"+feat)
        buttons.connect('clicked', self.arrowUpClicked)
        buttons.set_tooltip_text(self._("Bring the %s closer.") % feat.lower())
        buttons = self.meMakerWin.get_object("buttonDown"+feat)
        buttons.connect('clicked', self.arrowDownClicked)
        buttons.set_tooltip_text(self._("Push the %s away.") % feat.lower())

    def linkClicked(self, linkButton, location = "http://memaker.org"):
        """
        Open the memaker website in the user's default browser
        """
        webbrowser.open_new(location)

    def saveAs(self, widget, data):
        """
        The actions that need to be taken when saving
        """
        status = self.meMakerWin.get_object("statusbar")
        if data == "svg":
            locationSave = self.saveLoadFeatures(gtk.FILE_CHOOSER_ACTION_SAVE, "svg", FILE_EXT = {self._("Scalable Vector Graphic | SVG"):"svg"})
            status.push(1, self._("Saving..."))
            try:
                open(locationSave, "w").write(self.currentFace.printMe())
                status.push(1, self._("Saved."))
            except Exception, e:
                self.genericDialog((str(e)), self._("An error occured whilst saving the file."), gtk.BUTTONS_OK, gtk.MESSAGE_ERROR)
                status.push(1, self._("An error occured whilst saving."))
        svgHandle = rsvg.Handle(data = self.currentFace.printMe())
        cr = self.avatarPicture.window.cairo_create()
        self.clearIt.render_cairo(cr)
        pixbuf = svgHandle.get_pixbuf()
        if data == "png":
            locationSave = self.saveLoadFeatures(gtk.FILE_CHOOSER_ACTION_SAVE, "png", FILE_EXT = {self._("Portable Network Graphic | PNG"):"png"})
            status.push(1, self._("Saving..."))
            try:
                pixbuf.save(locationSave, 'png')
                status.push(1, self._("Saved."))
            except Exception, e:
                self.genericDialog((str(e)), self._("An error occured whilst saving the file."), gtk.BUTTONS_OK, gtk.MESSAGE_ERROR)
                status.push(1, self._("An error occured whilst saving."))
        elif data == "bmp":
            locationSave = self.saveLoadFeatures(gtk.FILE_CHOOSER_ACTION_SAVE, "bmp", FILE_EXT = {self._("Bitmap Image | BMP"):"bmp"})
            status.push(1, self._("Saving..."))
            try:
                pixbuf.save(locationSave, 'bmp')
                status.push(1, self._("Saved."))
            except Exception, e:
                self.genericDialog((str(e)), self._("An error occured whilst saving the file."), gtk.BUTTONS_OK, gtk.MESSAGE_ERROR)
                status.push(1, self._("An error occured whilst saving."))
        elif data == "launchpad":
            status.push(1, self._("Uploading new mugshot..."))
            import launchpad_memaker
            launchpad_memaker.load_auth(self.confHome+"/launchpad_auth")
            old_mugshot = open(self.cacheHome+'/launchpad_mugshot_old.png', 'w')
            tmpMugshot = self.cacheHome+'/tmp.launchpad_mugshot.png'
            try:
                old_mugshot.write(launchpad_memaker.get_old_mugshot(self.cacheHome))
                old_mugshot.close()
                pixbuf = pixbuf.scale_simple(192,192,gtk.gdk.INTERP_BILINEAR)
                pixbuf.save(tmpMugshot, "png")
                launchpad_memaker.set_new_mugshot(open(tmpMugshot).read(), self.cacheHome)
                status.push(1, self._("Done."))
                notifier = pynotify.Notification(self._("Mugshot Updated"), self._("Your Launchpad mugshot has been updated."))
                notifier.set_icon_from_pixbuf(gtk.gdk.pixbuf_new_from_file(tmpMugshot))
                notifier.show()
                os.remove(tmpMugshot)
            except Exception, e:
                self.genericDialog((str(e)), self._("Please check your connection to the internet or try again in a few minutes"), gtk.BUTTONS_OK, gtk.MESSAGE_ERROR)
                status.push(1, self._("An error occured."))
        elif data == "gnome":
            backupFile = self.cacheHome+"/.face"
            faceLocation = os.path.expanduser("~/.face")                     
            try:
                shutil.move(faceLocation, backupFile)
            except IOError, inst:
                if self.debug : print "No Backup"
            status.push(1, self._("Saving..."))
            pixbuf = svgHandle.get_pixbuf()
            pixbuf = pixbuf.scale_simple(96, 96, gtk.gdk.INTERP_BILINEAR)
            try:
                pixbuf.save(faceLocation, 'png')
                notifier = pynotify.Notification(self._("Profile Updated"), self._("Your GNOME avatar has been updated."))
                notifier.set_icon_from_pixbuf(pixbuf.scale_simple(50, 50, gtk.gdk.INTERP_BILINEAR))
                notifier.show()
                status.push(1, self._("Saved."))
            except IOError, e:
                self.genericDialog((str(e)), self._("An error occured whilst saving the file."), gtk.BUTTONS_OK, gtk.MESSAGE_ERROR)
                status.push(1, self._("An error occured whilst saving."))
        
    def themeChanged(self, comboboxThemePicker, themesListB):
        """
        Restore the proper theme for extraction
        """
        self.loadFeatures(themesListB[comboboxThemePicker.get_active()])

    def loadFeatures(self, themeLocation):
        """
        Load the features from themeLocation and place it in the iconview as thumbnails
        """
        self.lastThemeUsed = themeLocation
        self.currentFace = self.loadedHeads[themeLocation]
        self.applyChanges()
        notebookFeatures = self.meMakerWin.get_object("notebookFeatures")
        notebookFeatures.set_current_page(0)
        progressbarLoading = self.meMakerWin.get_object("progressbarLoading")
        progressbarLoading.show()
        comboboxThemePicker = self.meMakerWin.get_object("comboboxThemes")
        comboboxThemePicker.hide()
        featListLoading = []
        for feat in self.featureList:
            self.featuresMade[feat].model.clear()
            featListLoading.append({'text':self._('Loading %ss ...')% feat, 'path':themeLocation + feat+'/', 'featureType':feat})
        #This is going to now show the progress window...
        count = 0.0
        loaderLength = len(featListLoading) + 1.0
        image_loader = ImageLoader(50, 50, '#000000')
        for feature in featListLoading:
            count += 1
            progressbarLoading.set_text(feature['text'])
            progressbarLoading.set_fraction(count/loaderLength)
            while gtk.events_pending():
                gtk.main_iteration()
            locationsToLoad = [os.path.expanduser(feature['path'])]
            for locationTheme in locationsToLoad:
                for items in (os.listdir(locationTheme)):
                    if os.path.isfile(locationTheme+items) == True:
                        pixbuf = image_loader.load_cached_image(locationTheme+items)
                        scaled_buf = pixbuf.scale_simple(50, 50, gtk.gdk.INTERP_BILINEAR)
                        pictureLocation = (locationTheme+items)
                        for feat in self.featureList:
                            if feature['featureType'] == feat:
                                self.featuresMade[feat].model.append([pictureLocation , scaled_buf])
                    else:
                        if self.debug:print(self._("Found a folder, but I'm not loading what's inside."))
        #Hide out the loader
        image_loader.close()
        progressbarLoading.hide()
        comboboxThemePicker.show()

    def genericDialog(self, string1, string2, dialogbuttons, dialogtype):
        """
        Constructor that returns a generic dialog that can be used for errors
        """
        dialogError = gtk.MessageDialog(parent=None, flags=0, type=dialogtype, \
        buttons=dialogbuttons, message_format=string1)
        dialogError.format_secondary_text(string2)
        dialogError.run()
        dialogError.hide()
        return dialogError

    def removeFeature(self, widget):
        """
        Remove a feature from the self.currentFace objectstack
        """
        self.currentFace.deleteFeature(self.getSelectedPage())
        self.applyChanges()

    def arrowUpClicked(self, widget):
        """
        Apply changes after the up-arrow has been clicked
        """
        #Take a snapshot before
        svgHandle = rsvg.Handle(data = self.currentFace.printMe())
        image1 = svgHandle.get_pixbuf()
        if self.currentFace.raiseFeature(self.getSelectedPage()):
            self.applyChanges()
            return
        svgHandle = rsvg.Handle(data = self.currentFace.printMe())
        image2 = svgHandle.get_pixbuf()
        while self.currentFace.hasFeature(self.getSelectedPage()) and self.notChanged(image1, image2):
            if self.currentFace.raiseFeature(self.getSelectedPage()):
                break
            svgHandle = rsvg.Handle(data = self.currentFace.printMe())
            image2 = svgHandle.get_pixbuf()
        self.applyChanges()

    def arrowDownClicked(self, widget):
        """
        Apply changes after the down-arrow has been clicked
        """
        #Take a snapshot before
        svgHandle = rsvg.Handle(data = self.currentFace.printMe())
        image1 = svgHandle.get_pixbuf()
        if self.currentFace.lowerFeature(self.getSelectedPage()):
            self.applyChanges()
            return
        svgHandle = rsvg.Handle(data = self.currentFace.printMe())
        image2 = svgHandle.get_pixbuf()
        while self.currentFace.hasFeature(self.getSelectedPage()) and self.notChanged(image1, image2):
            if self.currentFace.lowerFeature(self.getSelectedPage()):
                break
            svgHandle = rsvg.Handle(data = self.currentFace.printMe())
            image2 = svgHandle.get_pixbuf()
        self.applyChanges()

    def getSelectedPage(self):
        """
        Take the current notbook page and return it back to the caller with the name of the notebook
        """
        notebookFeatures = self.meMakerWin.get_object("notebookFeatures")
        notebookIndex = notebookFeatures.get_current_page()
        return self.featureList[notebookIndex].lower()

    def avatarReset(self, widget = None):
        """
        Reset the avatar image to nothing
        """
        notebookFeatures = self.meMakerWin.get_object("notebookFeatures")
        notebookFeatures.set_current_page(0)
        self.currentFace.clearMe()
        self.applyChanges()

    def loadAbout(self, widget):
        """
        Open the about dialog
        """
		# Parse our AUTHORS file
        f = open(AUTHORSFILE, 'r')
        auth = f.read().split('--')
        f.close()
        frmAbout = self.meMakerWin.get_object("aboutdialogMeMaker")
        frmAbout.set_authors(auth[2].split("\n")[1:])
        frmAbout.set_artists(auth[4].split("\n")[1:])
        result = frmAbout.run()
        frmAbout.hide()
        return result
    
    def loadPreferences(self, widget):
        """
        Open the preferences dialog
        """
        preferencesWindow = self.meMakerWin.get_object("preferencesDialog")
        try: 
            import launchpadlib
            self.meMakerWin.get_object("labelLaunchpadInstalled").hide()
        except ImportError:
            self.meMakerWin.get_object("labelLaunchpadInstalled").show()
            self.meMakerWin.get_object("vboxPrefLaunchpad").set_sensitive(False)
        self.memakerConf.read(self.confHome+"/conf.conf")
        #if self.memakerConf.get('main', 'launchpadAuth') == "True":
            #self.meMakerWin.get_object("authorizeToggle").set_tooltip_text(self._("MeMaker is currently authorized for use with your Launchpad account."))
        result = preferencesWindow.run()
        preferencesWindow.hide()
        return result

    def featureClicked(self, iconviewList, modelTheme):
        """
        Figure out what was clicked and then apply that change to 
        the current face object stack and redraw the drawing area.
        """
        try:
            selectedItems = iconviewList.get_selected_items()[0][0]
            locationFeature = modelTheme[selectedItems][0]
        except IndexError:
            return
        self.currentFace.addFeature(locationFeature, self.getSelectedPage())
        iconviewList.unselect_all()
        self.applyChanges()

    def applyChanges(self):
        """
        Redraw the image to make sure it is all shown on the screen.
        """
        self.avatarPicture.queue_draw()

    def draw_scene(self, a, v):
        """
        Draw the current face Object stack with cairo
        """
        svgHandle = rsvg.Handle(data = self.currentFace.printMe())
        cr = self.avatarPicture.window.cairo_create()
        self.clearIt.render_cairo(cr)
        svgHandle.render_cairo(cr)

    def saveLoadFeatures(self, dialog_action, file_type, FILE_EXT):
        """
        dialog_action - The open or save mode for the dialog either
        gtk.FILE_CHOOSER_ACTION_OPEN, gtk.FILE_CHOOSER_ACTION_SAVE
            file_name - Default name when doing a save
        """
        file_name = self._("MyAvatar")        
        if (dialog_action==gtk.FILE_CHOOSER_ACTION_OPEN):
            dialog_buttons = (gtk.STOCK_CANCEL
                                , gtk.RESPONSE_CANCEL
                                , gtk.STOCK_OPEN
                                , gtk.RESPONSE_OK)
        else:
            dialog_buttons = (gtk.STOCK_CANCEL
                                , gtk.RESPONSE_CANCEL
                                , gtk.STOCK_SAVE
                                , gtk.RESPONSE_OK)

        file_dialog = gtk.FileChooserDialog(title=self._("MeMaker - Save Avatar")
                    , action=dialog_action
                    , buttons=dialog_buttons)
        if (dialog_action==gtk.FILE_CHOOSER_ACTION_SAVE):
            file_dialog.set_current_name(file_name+"."+file_type)
            for extension in FILE_EXT.keys():
                filters = gtk.FileFilter()
                filters.set_name(extension)
                filters.add_pattern("*." + FILE_EXT[extension])
                file_dialog.add_filter(filters)
        filters = gtk.FileFilter()
        filters.set_name(self._("All files"))
        filters.add_pattern("*")
        file_dialog.add_filter(filters)
        #Path to save the avatar
        file_dialog.set_current_folder(os.path.expanduser("~/"))
        #Init the return value
        result = ""
        if file_dialog.run() == gtk.RESPONSE_OK:
            result = file_dialog.get_filename()
        #Update preferred save location
        self.prefSaveLocation = file_dialog.get_current_folder()
        file_dialog.destroy()
        #Lets see if the user put an extension on or not.
        #TODO This feels dirty, could someone clean it up?
        for items in FILE_EXT:
            if result.endswith("."+FILE_EXT[items]):
                if self.debug: print self._("saving as"), result
                return result
            else:
                result = result+"."+FILE_EXT[items]
                if self.debug: print self._("saving as..."), result
                return result

    def loadPickledHead(self):
        """
        Restore the pickled file pickledHeads to restore all of the heads or use.
        """
        picklePlace = self.cacheHome+"/pickledHeads"
        if os.path.isfile(picklePlace):
            fileToPickle = open(picklePlace, "r")
            self.loadedHeads = pickle.load(fileToPickle)
        else:
            if self.debug: print self._("Previously session not found. Continuing without loading the last session.")
            
    def saveAndQuit(self, window):
        """
        Pickle all the face objects to a file called pickledHeads and quit memaker.
        """
        window.hide()
        while gtk.events_pending():
            gtk.main_iteration()
        if self.debug: print self._("Saving your session and quiting...")
        picklePlace = self.cacheHome+"/pickledHeads"
        fileToUnpickle = open(picklePlace, "w")
        for obst in self.loadedHeads:
            self.loadedHeads[obst].debug = False                    # if you used --debug only once, make sure it is not applied on next startup
        pickle.dump(self.loadedHeads, fileToUnpickle)
        if self.debug: print self._("Saving the last theme..."), self.lastThemeUsed
        self.memakerConf.read(self.confHome+"/conf.conf")
        self.memakerConf.set('main', 'lastTheme', self.lastThemeUsed)
        self.memakerConfData = open(self.confHome+"/conf.conf", 'w')
        self.memakerConf.write(self.memakerConfData)
        self.memakerConfData.close()
        gtk.main_quit()

    def notChanged(self, originalPixbuf, newPixbuf):
        """
        Takes two pixbufs and returns if they are the exact same or different. True/False
        """
        return originalPixbuf.get_pixels() == newPixbuf.get_pixels()
        
    def getAuthPost(self, widget):
        """
        Get authorization from Launchpad
        """
        import launchpad_memaker
        launchpad_memaker.get_auth2(self.confHome+"/launchpad_auth")
        self.memakerConf.read(self.confHome+"/conf.conf")
        self.memakerConf.set('main', 'launchpadAuth', "True")
        self.memakerConfData = open(self.confHome+"/conf.conf", 'w')
        self.memakerConf.write(self.memakerConfData)
        self.memakerConfData.close()
        #self.meMakerWin.get_object("authorizeToggle").set_tooltip_text(self._("MeMaker <b>is</b> currently\n authorized for use with\n your Launchpad account."))
        self.meMakerWin.get_object("buttonApplyAuth").set_sensitive(True)
        self.meMakerWin.get_object("buttonAuthForward2").set_sensitive(False)
        self.meMakerWin.get_object("menuItemSaveAslaunchpad").set_sensitive(True)
        self.meMakerWin.get_object("menuItemSaveAslaunchpad").set_tooltip_text('')
        self.meMakerWin.get_object('notebookAuth').next_page()
        
    def getAuthPre(self, widget):
        """
        Get authorization from the user
        """
        import launchpad_memaker
        try:
            launchpad_memaker.get_auth1()
            self.meMakerWin.get_object("buttonAuthForward1").hide()
            self.meMakerWin.get_object("buttonAuthForward2").show()
            self.meMakerWin.get_object('notebookAuth').next_page()
        except Exception, e:
                self.genericDialog((str(e)), self._("Please check your connection to the internet or try again in a few minutes"), gtk.BUTTONS_OK, gtk.MESSAGE_ERROR)
        
    def getAuthMethod(self, widget):
        """
        The routine of getting authentication from Launchpad
        """
        import launchpad_memaker
        #Reset everything
        self.meMakerWin.get_object("preferencesDialog").hide()
        self.meMakerWin.get_object("buttonAuthForward1").show()
        self.meMakerWin.get_object("buttonAuthForward2").hide()
        self.meMakerWin.get_object("buttonAuthForward2").set_sensitive(True)
        self.meMakerWin.get_object("buttonApplyAuth").set_sensitive(False)
        self.meMakerWin.get_object('notebookAuth').set_current_page(0)
        dialogLaunchpadAuth = self.meMakerWin.get_object("dialogLaunchpadAuth")
        dialogLaunchpadAuth.run()
        
    def hideAuthDialog(self, widget, event=None):
        self.meMakerWin.get_object("dialogLaunchpadAuth").hide()
        
    def checkBackups(self, widget):
        """
        Check if we have backups, if we do offer to restore from them
        """
        if not os.path.isfile(self.cacheHome+"/.face"):
            self.meMakerWin.get_object('menuitemRestoregnome').hide()
        if os.path.isfile(self.cacheHome+"/.face"):
            self.meMakerWin.get_object('menuitemRestoregnome').show()
        if os.path.isfile(self.cacheHome+'/launchpad_mugshot_old.png'):
            self.meMakerWin.get_object('menuitemRestorelaunchpad').show()
        if not os.path.isfile(self.cacheHome+'/launchpad_mugshot_old.png'):
            self.meMakerWin.get_object('menuitemRestorelaunchpad').hide()
        if os.path.isfile(self.cacheHome+'/launchpad_mugshot_old.png') or os.path.isfile(self.cacheHome+"/.face"):
            self.meMakerWin.get_object('menuitemNoRestore').hide()
        if not os.path.isfile(self.cacheHome+'/launchpad_mugshot_old.png') and not os.path.isfile(self.cacheHome+"/.face"):
            self.meMakerWin.get_object('menuitemNoRestore').show()
        self.meMakerWin.get_object('menuitemNoRestore').set_sensitive(False)
        
    def restoreGnome(self, widget):
        backupFile = self.cacheHome+"/.face"
        faceLocation = os.path.expanduser("~/.face")      
        shutil.move(backupFile, faceLocation)
        image = gtk.gdk.pixbuf_new_from_file(faceLocation)
        notifier = pynotify.Notification(self._("GNOME Avatar Set Back"), self._("The change you made to your GNOME Avatar was undone."))
        notifier.set_icon_from_pixbuf(image.scale_simple(50, 50, gtk.gdk.INTERP_BILINEAR))
        notifier.show()
        
    def restoreLaunchpad(self, widget):
        import launchpad_memaker
        launchpad_memaker.load_auth(self.confHome+"/launchpad_auth")
        old_mugshot = open(self.cacheHome+'/launchpad_mugshot_old.png', 'r')
        try:
            launchpad_memaker.set_new_mugshot(old_mugshot.read(), self.cacheHome)
            notifier = pynotify.Notification(self._("Mugshot Restored"), self._("Your Launchpad mugshot has been restored."))
            notifier.set_icon_from_pixbuf(gtk.gdk.pixbuf_new_from_file(self.cacheHome+'/launchpad_mugshot_old.png'))
            notifier.show()
            os.remove(self.cacheHome+'/launchpad_mugshot_old.png')
        except Exception, e:
            self.genericDialog((str(e)), self._("Please check your connection to the internet or try again in a few minutes"), gtk.BUTTONS_OK, gtk.MESSAGE_ERROR)

if __name__ == "__main__":
    app = MeMakerGui()
    gtk.main()
