// ---------------------------------------------------------------------------
// - Importer.hpp                                                            -
// - afnix:sps module - sps objects importer class definition                -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2012 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_IMPORTER_HPP
#define  AFNIX_IMPORTER_HPP

#ifndef  AFNIX_SHEET_HPP
#include "Sheet.hpp"
#endif

namespace afnix {

  /// The Recording class is a base class that permits to import data
  /// into a record. The base method operates with an input stream and
  /// attempts to read a literal. When read, the literal is added to the
  /// record until the stream is closed or the end-of-stream is reached.
  /// @author amaury darsch

  class Recording : public Object {
  protected:
    /// the importation record
    Record* p_ircd;

  public:
    /// create an empty importer
    Recording (void);

    /// create a importer with a record
    /// @param rcd the record to import into
    Recording (Record* rcd);

    /// destroy this importer
    ~Recording (void);

    /// @return the object name
    String repr (void) const;

    /// @return the importation record
    virtual Record* getrcd (void) const;

    /// set the importation record
    /// @param rcd the importation record to set
    virtual void setrcd (Record* rcd);

    /// import cons data into the record
    /// @param cons the cons cell to use
    virtual void import (Cons* cons);

    /// import stream data into the record
    /// @param is the input stream to use
    virtual void import (InputStream* is);

  private:
    // make the copy constructor private
    Recording (const Recording&);
    // make the assignment operator private
    Recording& operator = (const Recording&);

  public:
    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
                   Vector* argv);
  };

  /// The Sheeting class is a base class that permits to import data
  /// into a sheet. The base method operates with an input stream and
  /// attempts to build a cons cell that will be used as a sheet record.
  /// The importation operates as long as the stream is valid.
  /// @author amaury darsch

  class Sheeting : public Object {
  protected:
    /// the importation sheet
    Sheet* p_isht;

  public:
    /// create an empty importer
    Sheeting (void);

    /// create a importer with a record
    /// @param sht the sheet to import into
    Sheeting (Sheet* sht);

    /// destroy this importer
    ~Sheeting (void);

    /// @return the object name
    String repr (void) const;

    /// @return the importation sheet
    virtual Sheet* getsht (void) const;

    /// set the importation sheet
    /// @param sht the sheet to set
    virtual void setsht (Sheet* sht);

    /// import cons data into the sheet
    /// @param cons the cons cell to use
    virtual void import (Cons* cons);

    /// import stream data into the sheet
    /// @param is the input stream to use
    virtual void import (InputStream* is);

  private:
    // make the copy constructor private
    Sheeting (const Sheeting&);
    // make the assignment operator private
    Sheeting& operator = (const Sheeting&);

  public:
    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);
    
    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
                   Vector* argv);
  };
}

#endif
