!{\src2tex{textfont=tt}}
!!****f* ABINIT/linopt
!! NAME
!! linopt
!!
!! FUNCTION
!! This routine compute optical frequency dependent dielectric function
!! for semiconductors
!!
!! COPYRIGHT
!! Copyright (C) 2002-2007 ABINIT group (SSharma)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!  nspin=number of spins(integer)
!!  omega=crystal volume in au (real)
!!  nkpt=total number of kpoints (integer)
!!  wkpt(nkpt)=weights of kpoints (real)
!!  nsymcrys=number of crystal symmetry operations(integer)
!!  symcrys(3,3,nsymcrys)=symmetry operations in cartisian coordinates(real)
!!  nstval=total number of valence states(integer)
!!  occv(nstval,nspin,nkpt)=occupation number for each band(real)
!!  evalv(nstval,nspin,nkpt)=eigen value for each band in Ha(real)
!!  efermi=Fermi energy in Ha(real)
!!  pmat(nstval,nstval,nkpt,3,nspin)=momentum matrix elements in cartesian coordinates(complex)
!!  v1,v2=desired component of the dielectric function(integer) 1=x,2=y,3=z
!!  nmesh=desired number of energy mesh points(integer)
!!  de=desired step in energy(real); nmesh*de=maximum energy
!!  sc=scissors shift in Ha(real)
!!  brod=broadening in Ha(real)
!!  fnam=root for filename that will contain the output filename will be trim(fnam)//'-linopt.out'
!!
!! OUTPUT
!!  Dielectric function for semiconductors, on a desired energy mesh and for a desired
!!  direction of polarisation. The output is in a file named trim(fnam)//'-linopt.out' and contains
!!  Im(\epsilon_{v1v2}(\omega), Re(\epsilon_{v1v2}(\omega) and abs(\epsilon_{v1v2}(\omega).
!!  Comment:
!!  Right now the routine sums over the kpoints. In future linear tetrahedron method should be
!!  useful.
!!
!! PARENTS
!!      optic
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine linopt(nspin,omega,nkpt,wkpt,nsymcrys,symcrys,nstval,occv,evalv,efermi,pmat, &
  v1,v2,nmesh,de,sc,brod,fnam)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!no_abirules
integer, intent(in) :: nspin
real(dp), intent(in) :: omega
integer, intent(in) :: nkpt
real(dp), intent(in) :: wkpt(nkpt)
integer, intent(in) :: nsymcrys
real(dp), intent(in) :: symcrys(3,3,nsymcrys)
integer, intent(in) :: nstval
real(dp), intent(in) :: occv(nstval,nspin,nkpt)
real(dp), intent(in) :: evalv(nstval,nspin,nkpt)
real(dp), intent(in) :: efermi
complex(dp), intent(in) :: pmat(nstval,nstval,nkpt,3,nspin)
integer, intent(in) :: v1
integer, intent(in) :: v2
integer, intent(in) :: nmesh
real(dp), intent(in) :: de
real(dp), intent(in) :: sc
real(dp), intent(in) :: brod
character(256), intent(in) :: fnam

!Local variables -------------------------
!no_abirules
integer :: isp
integer :: i,j,isym,lx,ly,ik
integer :: ist1,ist2,iw
real(dp) :: e1,e2,e12
real(dp) :: ha2ev
real(dp) :: const,emin,emax
real(dp) :: corec,t1,t2,ene
complex(dp) :: b11,b12
complex(dp) :: ieta,w
character(256) :: fnam1
! local allocatable arrays
real(dp), allocatable :: s(:,:)
real(dp), allocatable :: sym(:,:)
complex(dp), allocatable :: chi(:)
complex(dp), allocatable :: eps(:)

! *********************************************************************

! fool proof:
! check polarisation
if (v1.le.0.or.v2.le.0.or.v1.gt.3.or.v2.gt.3) then
  print *,'---------------------------------------------'
  print *,'    Error in linopt:                         '
  print *,'    the polarisation directions incorrect    '
  print *,'    1=x and 2=y and 3=z                      '
  print *,'---------------------------------------------'
  stop
end if
! number of energy mesh points
if (nmesh.le.0) then
  print *,'---------------------------------------------'
  print *,'    Error in linopt:                         '
  print *,'    number of energy mesh points incorrect   '
  print *,'    number has to integer greater than 0     '
  print *,'    nmesh*de = max energy for calculation    '
  print *,'---------------------------------------------'
  stop
end if
! step in energy
if (de.le.0._dp) then
  print *,'---------------------------------------------'
  print *,'    Error in linopt:                         '
  print *,'    energy step is incorrect                 '
  print *,'    number has to real greater than 0.0      '
  print *,'    nmesh*de = max energy for calculation    '
  print *,'---------------------------------------------'
  stop
end if
! scissors operator
if (sc.lt.0._dp) then
  print *,'---------------------------------------------'
  print *,'    Error in linopt:                         '
  print *,'    scissors shift is incorrect              '
  print *,'    number has to real greater than 0.0      '
  print *,'---------------------------------------------'
  stop
end if
! broadening
if (brod.gt.0.009) then
  print *,'---------------------------------------------'
  print *,'    ATTENTION: broadening is quite high      '
  print *,'    ideally should be less than 0.005        '
  print *,'---------------------------------------------'
else if (brod.gt.0.015) then
  print *,'----------------------------------------'
  print *,'    ATTENTION: broadening is too high   '
  print *,'    ideally should be less than 0.005   '
  print *,'----------------------------------------'
end if
! fool proof end
!
! allocate local arrays
allocate(chi(nmesh),eps(nmesh))
allocate(s(3,3),sym(3,3))
ieta=(0._dp,1._dp)*brod
const=1._dp/(omega*dble(nsymcrys))
ha2ev=13.60569172*2._dp
! output file names
fnam1=trim(fnam)//'-linopt.out'
! construct symmetrisation tensor
sym(:,:)=0._dp
do isym=1,nsymcrys
  s(:,:)=symcrys(:,:,isym)
  do i=1,3
    do j=1,3
      sym(i,j)=sym(i,j)+s(i,v1)*s(j,v2)
    end do
  end do
end do
! calculate the energy window
emin=0._dp
emax=0._dp
do ik=1,nkpt
  do isp=1,nspin
    do ist1=1,nstval
      emin=min(emin,evalv(ist1,isp,ik))
      emax=max(emax,evalv(ist1,isp,ik))
    end do
  end do
end do
! start calculating linear optical response
chi(:)=0._dp
do ik=1,nkpt
  print *,ik,'of',nkpt
  do isp=1,nspin
    do ist1=1,nstval
      e1=evalv(ist1,isp,ik)
      if (e1.lt.efermi) then
        do ist2=ist1,nstval
          e2=evalv(ist2,isp,ik)
          if (e2.gt.efermi) then
! scissors correction of mommentum matrix
            e12=e1-e2-sc
            corec=e12/(e12+sc)
            b11=0._dp
! symmetrization of mommentum matrix
            do lx=1,3
              do ly=1,3
                b11=b11+(sym(lx,ly)*pmat(ist1,ist2,ik,lx,isp)* &
                 conjg(pmat(ist1,ist2,ik,ly,isp)))
              end do
            end do
            b12=b11*const*corec*corec*(1._dp/(e12**2))
! calculate on the desired energy grid
            do iw=2,nmesh
              w=(iw-1)*de+ieta
              chi(iw)=chi(iw)+(wkpt(ik)*(occv(ist1,isp,ik)-occv(ist2,isp,ik))* &
               (b12/(-e12-w)))
            end do
! end loops over states
          end if
        end do
      end if
    end do
! end loop over spins
  end do
! end loop over k-points
end do

! open the output files
open(92,file=fnam1,action='WRITE',form='FORMATTED')
! write the output
write(92, '(a)' ) ' # Energy(eV)         Im(chi1(w))'
write(92, '(a,2i3,a)' )' #calculated the component:',v1,v2,'of linear susceptibility'
print *, 'calculated the component:',v1,v2,'of linear susceptibility'
write(92, '(a,2es16.6)' ) ' #broadening:', real(ieta),aimag(ieta)
print *, ' with broadening:',ieta
write(92, '(a,es16.6)' ) ' #scissors shift:',sc
print *,'and scissors shift:',sc
write(92, '(a,es16.6,a,es16.6,a)' ) ' #energy window:',(emax-emin)*ha2ev,'eV',(emax-emin),'Ha'
print *,'energy window:',(emax-emin)*ha2ev,'eV',(emax-emin),'Ha'
eps(:)=0._dp
do iw=2,nmesh
  ene=(iw-1)*de
  ene=ene*ha2ev
  eps(iw)=1._dp+4._dp*pi*chi(iw)
  write(92, '(2es16.6)' ) ene,aimag(eps(iw))
end do
write(92,*)
write(92,*)
write(92, '(a)' ) ' # Energy(eV)         Re(chi1(w))'
do iw=2,nmesh
  ene=(iw-1)*de
  ene=ene*ha2ev
  write(92, '(2es16.6)' ) ene,dble(eps(iw))
end do
write(92,*)
write(92,*)
write(92, '(a)' )' # Energy(eV)         abs(chi1(w))'
do iw=2,nmesh
  ene=(iw-1)*de
  ene=ene*ha2ev
  write(92, '(2es16.6)' ) ene,abs(eps(iw))
end do

!  close output file
close(92)
! deallocate local arrays
deallocate(s,sym,chi,eps)

return

end subroutine linopt
!!***
