/**
 * @brief Low level backlight driver for Intel GMA950 graphics card.
 *
 * This module contains the low level driver to control a LCD backlight
 * connected to the Intel GMA950 graphics card. This card is used in the
 * new MacBooks from Apple.
 *
 * This program was written after Julian Blache reverse engineered
 * the AppleIntelIntegratedFramebuffer.kext kernel extension in
 * Mac OS X and played with the register at the memory location
 * found therein. The following description is based on his experiences:
 *
 * The register appears to have two halves.
 *                 yyyyyyyyyyyyyyy0xxxxxxxxxxxxxxx0
 *
 * The top (y) bits appear to be the maximum brightness level and the
 * bottom (x) bits are the current brightness level. 0s are always 0.
 * The brightness level is, therefore, x/y.
 *
 * At a freshly booted Macbook, y is set to 0x94 and x is set to 0x1f.
 * Going below 0x1f produces odd results.  For example, if you come from
 * above, the backlight will completely turn off at 0x12 (18).  Coming
 * from below, however, you need to get to 0x15 (21) before the backlight
 * comes back on.
 *
 * Since there is no clear threshold, it looks like that this value
 * specifies a raw voltage. It seems that the bootup value of 0x1f
 * corresponds to the lowest level that Mac OS X will set the backlight.
 *
 * Because of this a value of 0x1f is taken as minimum brightness. The
 * maximum value is defined by the upper 15 bits.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation
 * (http://www.gnu.org/licenses/gpl.html)
 *
 * @file    src/driver_backlight_gma950.c
 * @author  Matthias Grimm <matthias.grimm@users.sourceforge.net>
 * @author  Stefan Bruda <bruda@cs.ubishops.ca> 
 * @author  Nicolas Boichat <nicolas@boichat.ch>
 * @author  Julien Blache <jb@jblache.org>
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdlib.h>
#include <sys/mman.h>

#include <pci/pci.h>

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>

#include "gettext_macros.h"
#include "class_backlight.h"
#include "driver_backlight_gma950.h"

#define REGISTER_OFFSET 0x00061254

/**
 * @brief Address in memory of the Intel GMA950 graphics card
 *
 */
static long address;

/**
 * @brief Length of memory segment the Intel GMA950 graphics card uses
 */
static long length;

/**
 * @brief Maximum brightbess level of backlight
 *
 * This value may vary between different systems so it will queried
 * from the graphics card at init time.
 */
static int max_brightness;

/**
 * @brief Exit function of the driver - cleans up all ressources
 *
 * This function will be called when the work is done and the driver
 * should be unloaded. It frees all allocated ressources.
 */
void
driver_backlight_gma950_exit ()
{
}

/**
 * @brief Get the current brightness level from the device
 *
 * <b>This function is part of the public interface of the driver.</b>
 *
 * It returns the current brightness level of the LCD backlight.
 *
 * Because the brightness levels beween 0 and 0x1f aren't very usefull,
 * they will be masked out here. That means the read out brightness
 * values are transformed as follows:
 * @li  0x00...0x1e -> 0
 * @li  0x1f - 0x1e -> 1
 * @li  0x20 - 0x1e -> 2
 * @li  :
 * 
 * @return  transformed brightness level or -1 on error
 */
int
gma950bl_get_brightness ()
{
	int fd, rc = -1;
	char *memory;

	if ((fd = open("/dev/mem", O_RDWR)) >= 0) {
		memory = mmap(NULL, length, PROT_READ|PROT_WRITE, MAP_SHARED, fd, address);
		if (memory != MAP_FAILED) {
			rc = (INREG(REGISTER_OFFSET) >> 1) & 0x7fff;  /* backlight value */
        	munmap(memory, length);
		}
		close(fd);
		rc = rc < GMA950_BRIGHTNESS_MIN ? 0 : rc - (GMA950_BRIGHTNESS_MIN - 1);
	}
	return rc;
}

/**
 * @brief Get the maximum brightness level the device supports
 *
 * <b>This function is part of the public interface of the driver.</b>
 *
 * It returns the maximum brightness level the LCD backlight
 * device supports. The number of steps is hardware dependent and
 * must be read from the graphics card. The level was probed in init
 * phase.
 *
 * Bacause the brightness levels beween 0 and 0x1f aren't very usefull,
 * they will be masked out here. That means the maximum brightness
 * value returned by this function will be the hardware reported maximum
 * brightness value minus 0x1f.
 *
 * @return  maximum supported brightness level of the device.
 */
int
gma950bl_get_brightness_max ()
{
	return max_brightness - GMA950_BRIGHTNESS_MIN;
}

/**
 * @brief Change the current brightness level.
 *
 * <b>This function is part of the public interface of the driver.</b>
 *
 * This function sets a new brightness level. The given level must be
 * valid for this device, that means it must be lower than the level
 * gma950_get_brightness_max() returns. No further check is done on that.
 *
 * Because the brightness levels beween 0 and 0x1f aren't very usefull,
 * they will be masked out here. That means the given brightness value
 * must be in range 0 .. gma950bl_get_brightness_max() and does not
 * contain any gap. The value will then be transformed as follows.
 * @li  0 -> 0
 * @li  1 -> 0x1f = 1 + 0x1e
 * @li  2 -> 0x20 = 2 + 0x1e
 * @li  :
 *
 * @param  val   new brightness level
 */
void
gma950bl_set_brightness (int val)
{
	int fd;
	char *memory;

	if (val >= 0) {
		if ((fd = open("/dev/mem", O_RDWR)) >= 0) {
			memory = mmap(NULL, length, PROT_READ|PROT_WRITE, MAP_SHARED, fd, address);
			if (memory != MAP_FAILED) {
				val = val ? val + (GMA950_BRIGHTNESS_MIN - 1) : 0;
				OUTREG(REGISTER_OFFSET, (max_brightness << 17) | (val << 1));
        		munmap(memory, length);
			}
			close (fd);
		}
	}
}

static struct driver_backlight driver_backlight_gma950 = {
	.name               = N_("Intel GMA950 Backlight Driver"),
	.get_brightness     = gma950bl_get_brightness,
	.get_brightness_max = gma950bl_get_brightness_max,
	.set_brightness     = gma950bl_set_brightness,
	.driver_exit        = driver_backlight_gma950_exit,
};

/**
 * @brief Constructor of a GMA950 backlight driver object
 *
 * This function probes for a Intel GMA950 graphics card on the
 * PCI bus and if one is found, saves the memory address and
 * the seqment length. The other functions of this driver rely
 * on this data to change the LCD brightness.
 *
 * @return  Initializes backlight driver object or NULL
 */
struct driver_backlight *
driver_backlight_gma950_init ()
{
	struct pci_access *pacc;
	struct pci_dev *dev;
	char *memory;
	int fd;

	/* initialize the globals */
	address        = 0;
	length         = 0;
	max_brightness = 0;

	pacc = pci_alloc();
	pci_init(pacc);
	pci_scan_bus(pacc);

	for(dev=pacc->devices; dev; dev=dev->next) {  /* Iterate over all devices */
		pci_fill_info(dev, PCI_FILL_IDENT | PCI_FILL_BASES);
		if ((dev->vendor_id == PCI_ID_VENDOR_INTEL) &&
			(dev->device_id == PCI_ID_PRODUCT_GMA950)) {
			address = dev->base_addr[0];
			length = dev->size[0];
		}
	}
	pci_cleanup(pacc);

	if (!address)
		return NULL;

	if ((fd = open("/dev/mem", O_RDWR)) >= 0) {
		memory = mmap(NULL, length, PROT_READ|PROT_WRITE, MAP_SHARED, fd, address);
		if (memory != MAP_FAILED) {
			max_brightness = INREG(REGISTER_OFFSET) >> 17;  /* max backlight value */
        	munmap(memory, length);
		}
		close(fd);
	}
	
	if (!max_brightness)
		return NULL;

	return &driver_backlight_gma950;
}

