// This file is part of the AspectC++ compiler 'ac++'.
// Copyright (C) 1999-2003  The 'ac++' developers (see aspectc.org)
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef ACMODEL

#include <sstream>
using std::stringstream;
using std::ostringstream;

#include "JoinPointLoc.h"
#include "AdviceInfo.h"
#include "PointCutEvaluator.h"

#include "Puma/CTypeInfo.h"
#include "Puma/CTree.h"

JoinPointLoc::JoinPointLoc (RepoXMLNode jn, RepoXMLNode::iter &curr) :
  _parent (0), _plan (0), _trans (0) {
  map (_map);
  id (jn.get_int_prop ("id"));
  if (jn.has_prop ("sig"))
    _sig = jn.get_str_prop ("sig");
  // TODO: temporary hack:
  if (jn.has_prop ("plan")) {
    _plan = (JoinPointPlan*)1;
  }
  if (jn.has_prop ("tunits")) {
    _tunits.from_string (jn.get_str_prop ("tunits"));
  }
  while (curr != jn.end_child () && (*curr).has_name ("src")) {
    RepoXMLNode fn = *curr;
    _source_locs.insert (SourceLoc (fn));
    ++curr;
  }
}

RepoXMLNode JoinPointLoc::make_xml (RepoXMLNode parent) const {
  RepoXMLNode jpn = parent.make_child (type_str ());
  jpn.set_int_prop ("id", id ());
  // TODO: temporary hack:
  if (plan ()) {
    jpn.set_int_prop ("plan", 1);
  }
  if (_tunits.size () > 0)
    jpn.set_str_prop ("tunits", _tunits.to_string ().c_str ());
  for (SSet::const_iterator iter = source_locs ().begin ();
    iter != source_locs ().end (); ++iter)
    (*iter).make_xml (jpn);
  return jpn;
}

IdElementMap *JoinPointLoc::_map = 0;

JoinPointLoc *JoinPointLoc::create (RepoXMLNode node) {
  RepoXMLNode::iter curr_child = node.first_child ();
  if (node.has_name ("function"))
    return new JPL_Function (node, curr_child);
  if (node.has_name ("class"))
    return new JPL_Class (node, curr_child);
  if (node.has_name ("namespace"))
    return new JPL_Namespace (node, curr_child);
  if (node.has_name ("aspect"))
    return new JPL_Aspect (node, curr_child);
  if (node.has_name ("exec"))
    return new JPL_Method (node, curr_child);
  if (node.has_name ("call"))
    return new JPL_MethodCall (node, curr_child);
  if (node.has_name ("construction"))
    return new JPL_Construction (node, curr_child);
  if (node.has_name ("destruction"))
    return new JPL_Destruction (node, curr_child);
  if (node.has_name ("advice-before") ||
      node.has_name ("advice-after") ||
      node.has_name ("advice-around"))
    return new JPL_AdviceCode (node, curr_child);
  if (node.has_name ("intro"))
    return new JPL_Introduction (node, curr_child);
  if (node.has_name ("slice-dep-base") ||
      node.has_name ("slice-dep-member") ||
      node.has_name ("slice-class"))
    return new JPL_ClassSlice (node, curr_child);
  return 0;
}

void JoinPointLoc::parent (JPL_Name *p) {
  _parent = p;
  p->add_child (this);
}

void JoinPointLoc::dump (int indent) const {
  for (int i = 0; i < indent; i++) cout << "  ";
  cout << type_str () << " " << id () << " " << signature ();
  if (_tunits.size () > 0)
    cout << " tunits=" << _tunits.to_string () << endl;
  cout << endl;
  for (SSet::const_iterator iter = source_locs ().begin ();
    iter != source_locs ().end (); ++iter)
    (*iter).print (indent + 1);
}

/// check whether this join point is seen by a specific tranlation unit
bool JoinPointLoc::is_known_in_tunit (File *tunit) {
  int tunit_id = tunit->id ();
  if (_tunits.size () > 0)
    return _tunits.find (tunit_id) != _tunits.end ();
  for (SSet::const_iterator iter = source_locs ().begin ();
    iter != source_locs ().end (); ++iter) {
    int src_file_id = iter->file_id ();
    if (((File*)map (src_file_id))->is_known_in_tunit (tunit))
      return true;
  }
  return false;
}

string JoinPointLoc::filename () const {
	if (source_locs ().size () == 0)
		return "<unknown>";
		 
	SSet::const_iterator last; 
  for (SSet::const_iterator iter = source_locs ().begin ();
    iter != source_locs ().end (); ++iter) {
  	last = iter;
  	if (iter->kind () == SLK_DEF)
  		break;
  }
  return ((File*)map (last->file_id ()))->name ();
}

int JoinPointLoc::line () const {
	if (source_locs ().size () == 0)
		return -1;
		 
	SSet::const_iterator last; 
  for (SSet::const_iterator iter = source_locs ().begin ();
    iter != source_locs ().end (); ++iter) {
  	last = iter;
  	if (iter->kind () == SLK_DEF)
  		break;
  }
  return last->line ();
}

void JPL_Name::dump (int indent) const {
  JoinPointLoc::dump (indent);
  if (type () == Aspect || type () == Class) {
    JPL_Class *cls = (JPL_Class*)this;

    const set<int> &bases = cls->base_class_ids ();
    for (set<int>::const_iterator i = bases.begin (); i != bases.end (); ++i) {
      JPL_Class *base = (JPL_Class*)map (*i);
      for (int i = 0; i < (indent + 1); i++) cout << "  ";
      cout << "  base class " << base->signature () << endl;
    }
  }
  for (CList::const_iterator iter = children ().begin ();
    iter != children ().end (); ++iter)
    (*iter)->dump (indent + 1);
}

static void set_str (RepoXMLNode node, const char *a, const char *v) {
  node.set_str_prop (a, v);
}

RepoXMLNode JPL_Name::make_xml (RepoXMLNode parent) const {
  RepoXMLNode jpn = JoinPointLoc::make_xml (parent);
  set_str (jpn, "sig", signature ());
  if (_is_built_in)
    jpn.set_int_prop("builtin", 1);
  for (CList::const_iterator iter = children ().begin ();
    iter != children ().end (); ++iter)
    (*iter)->make_xml (jpn);
  return jpn;
}

JPL_Name::JPL_Name (RepoXMLNode node, RepoXMLNode::iter &curr) :
  JoinPointLoc (node, curr) {
  _is_built_in = node.has_prop ("builtin");
  while (curr != node.end_child ()) {
    JoinPointLoc *jpl = create (*curr);
    if (!jpl)
      break;
    jpl->parent (this);
    ++curr;
  }
}

JPL_Namespace::JPL_Namespace (RepoXMLNode node, RepoXMLNode::iter &curr) :
  JPL_Name (node, curr){
}

JPL_Class::JPL_Class (RepoXMLNode node, RepoXMLNode::iter &curr) :
  JPL_Name (node, curr) {
  while (curr != node.end_child () && (*curr).has_name ("base")) {
    int base_id = (*curr).get_int_prop ("id");
    _base_ids.insert (base_id);
    ++curr;
  }
}

RepoXMLNode JPL_Class::make_xml (RepoXMLNode parent) const {
  RepoXMLNode jpn = JPL_Name::make_xml (parent);
  for (set<int>::const_iterator iter = _base_ids.begin ();
    iter != _base_ids.end (); ++iter) {
    RepoXMLNode base_node = jpn.make_child ("base");
    base_node.set_int_prop ("id", *iter);
  }
  return jpn;
}

JPL_Function::JPL_Function (RepoXMLNode node, RepoXMLNode::iter &curr) :
  JPL_Name (node, curr) {
}

JPL_Function::~JPL_Function () {
  for (vector<JPL_Type*>::iterator iter = _arg_types.begin ();
       iter != _arg_types.end (); ++iter)
    delete *iter;
}


JPL_MethodCall::~JPL_MethodCall () {
  for (vector<JPL_Type*>::iterator iter = _arg_types.begin ();
       iter != _arg_types.end (); ++iter)
    delete *iter;
}

// save as XML
RepoXMLNode JPL_MethodCall::make_xml (RepoXMLNode parent) const {
  RepoXMLNode jpn = JPL_Code::make_xml (parent);
  jpn.set_int_prop ("target", _target_id);
  return jpn;
}

bool JPL_Function::has_same_name_and_args (const JPL_Function &func) const {
  if (!(arg_count () == func.arg_count () && name () == func.name ()))
    return false;
  for (int a = 0; a < arg_count (); a++)
    if (strcmp (arg_type (a).signature (), func.arg_type (a).signature ()) != 0)
      return false;
  return true;
}

JPL_AdviceCode::JPL_AdviceCode (RepoXMLNode node, RepoXMLNode::iter &curr) :
  JPL_Advice (0), JPL_Function (node, curr) {
  if (node.has_name ("advice-before"))
    _type = ADVICE_BEFORE;
  else if (node.has_name ("advice-after"))
    _type = ADVICE_AFTER;
  else 
    _type = ADVICE_AROUND;
}

RepoXMLNode JPL_AdviceCode::make_xml (RepoXMLNode parent) const {
  RepoXMLNode jpn = JPL_Name::make_xml (parent);
  return jpn;
}

JPL_Introduction::JPL_Introduction (RepoXMLNode node, RepoXMLNode::iter &curr) :
  JPL_Advice (0), JoinPointLoc (node, curr) {
  _slice_id = node.get_int_prop ("slice");
}

RepoXMLNode JPL_Introduction::make_xml (RepoXMLNode parent) const {
  RepoXMLNode jpn = JoinPointLoc::make_xml (parent);
  jpn.set_int_prop ("slice", _slice_id);
  return jpn;
}

JPL_ClassSlice::JPL_ClassSlice (RepoXMLNode node, RepoXMLNode::iter &curr) :
  JPL_Name (node, curr) {
  if (node.has_name ("slice-dep-base"))
    _type = CS_OLD_BASE;
  else if (node.has_name ("slice-dep-member"))
    _type = CS_OLD_OTHER;
  else 
    _type = CS_NORMAL;
}

RepoXMLNode JPL_ClassSlice::make_xml (RepoXMLNode parent) const {
  RepoXMLNode jpn = JPL_Name::make_xml (parent);
  return jpn;
}

#endif // !ACMODEL
